/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures a hyperparameter tuning job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTuningJobConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTuningJobConfig.Builder, HyperParameterTuningJobConfig> {
    private static final SdkField<String> STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HyperParameterTuningJobConfig::strategyAsString)).setter(setter(Builder::strategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Strategy").build()).build();

    private static final SdkField<HyperParameterTuningJobObjective> HYPER_PARAMETER_TUNING_JOB_OBJECTIVE_FIELD = SdkField
            .<HyperParameterTuningJobObjective> builder(MarshallingType.SDK_POJO)
            .getter(getter(HyperParameterTuningJobConfig::hyperParameterTuningJobObjective))
            .setter(setter(Builder::hyperParameterTuningJobObjective))
            .constructor(HyperParameterTuningJobObjective::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HyperParameterTuningJobObjective")
                    .build()).build();

    private static final SdkField<ResourceLimits> RESOURCE_LIMITS_FIELD = SdkField
            .<ResourceLimits> builder(MarshallingType.SDK_POJO).getter(getter(HyperParameterTuningJobConfig::resourceLimits))
            .setter(setter(Builder::resourceLimits)).constructor(ResourceLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceLimits").build()).build();

    private static final SdkField<ParameterRanges> PARAMETER_RANGES_FIELD = SdkField
            .<ParameterRanges> builder(MarshallingType.SDK_POJO).getter(getter(HyperParameterTuningJobConfig::parameterRanges))
            .setter(setter(Builder::parameterRanges)).constructor(ParameterRanges::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterRanges").build()).build();

    private static final SdkField<String> TRAINING_JOB_EARLY_STOPPING_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(HyperParameterTuningJobConfig::trainingJobEarlyStoppingTypeAsString))
            .setter(setter(Builder::trainingJobEarlyStoppingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingJobEarlyStoppingType")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STRATEGY_FIELD,
            HYPER_PARAMETER_TUNING_JOB_OBJECTIVE_FIELD, RESOURCE_LIMITS_FIELD, PARAMETER_RANGES_FIELD,
            TRAINING_JOB_EARLY_STOPPING_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String strategy;

    private final HyperParameterTuningJobObjective hyperParameterTuningJobObjective;

    private final ResourceLimits resourceLimits;

    private final ParameterRanges parameterRanges;

    private final String trainingJobEarlyStoppingType;

    private HyperParameterTuningJobConfig(BuilderImpl builder) {
        this.strategy = builder.strategy;
        this.hyperParameterTuningJobObjective = builder.hyperParameterTuningJobObjective;
        this.resourceLimits = builder.resourceLimits;
        this.parameterRanges = builder.parameterRanges;
        this.trainingJobEarlyStoppingType = builder.trainingJobEarlyStoppingType;
    }

    /**
     * <p>
     * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training job
     * it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To randomly search, set it to
     * <code>Random</code>. For information about search strategies, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     * Hyperparameter Tuning Works</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link HyperParameterTuningJobStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #strategyAsString}.
     * </p>
     * 
     * @return Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
     *         training job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To
     *         randomly search, set it to <code>Random</code>. For information about search strategies, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     *         Hyperparameter Tuning Works</a>.
     * @see HyperParameterTuningJobStrategyType
     */
    public HyperParameterTuningJobStrategyType strategy() {
        return HyperParameterTuningJobStrategyType.fromValue(strategy);
    }

    /**
     * <p>
     * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training job
     * it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To randomly search, set it to
     * <code>Random</code>. For information about search strategies, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     * Hyperparameter Tuning Works</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link HyperParameterTuningJobStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #strategyAsString}.
     * </p>
     * 
     * @return Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
     *         training job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To
     *         randomly search, set it to <code>Random</code>. For information about search strategies, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
     *         Hyperparameter Tuning Works</a>.
     * @see HyperParameterTuningJobStrategyType
     */
    public String strategyAsString() {
        return strategy;
    }

    /**
     * <p>
     * The <a>HyperParameterTuningJobObjective</a> object that specifies the objective metric for this tuning job.
     * </p>
     * 
     * @return The <a>HyperParameterTuningJobObjective</a> object that specifies the objective metric for this tuning
     *         job.
     */
    public HyperParameterTuningJobObjective hyperParameterTuningJobObjective() {
        return hyperParameterTuningJobObjective;
    }

    /**
     * <p>
     * The <a>ResourceLimits</a> object that specifies the maximum number of training jobs and parallel training jobs
     * for this tuning job.
     * </p>
     * 
     * @return The <a>ResourceLimits</a> object that specifies the maximum number of training jobs and parallel training
     *         jobs for this tuning job.
     */
    public ResourceLimits resourceLimits() {
        return resourceLimits;
    }

    /**
     * <p>
     * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches.
     * </p>
     * 
     * @return The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job
     *         searches.
     */
    public ParameterRanges parameterRanges() {
        return parameterRanges;
    }

    /**
     * <p>
     * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This can be
     * one of the following values (the default value is <code>OFF</code>):
     * </p>
     * <dl>
     * <dt>OFF</dt>
     * <dd>
     * <p>
     * Training jobs launched by the hyperparameter tuning job do not use early stopping.
     * </p>
     * </dd>
     * <dt>AUTO</dt>
     * <dd>
     * <p>
     * Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
     * better than previously completed training jobs. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop Training
     * Jobs Early</a>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #trainingJobEarlyStoppingType} will return {@link TrainingJobEarlyStoppingType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #trainingJobEarlyStoppingTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This
     *         can be one of the following values (the default value is <code>OFF</code>):</p>
     *         <dl>
     *         <dt>OFF</dt>
     *         <dd>
     *         <p>
     *         Training jobs launched by the hyperparameter tuning job do not use early stopping.
     *         </p>
     *         </dd>
     *         <dt>AUTO</dt>
     *         <dd>
     *         <p>
     *         Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
     *         perform better than previously completed training jobs. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
     *         Training Jobs Early</a>.
     *         </p>
     *         </dd>
     * @see TrainingJobEarlyStoppingType
     */
    public TrainingJobEarlyStoppingType trainingJobEarlyStoppingType() {
        return TrainingJobEarlyStoppingType.fromValue(trainingJobEarlyStoppingType);
    }

    /**
     * <p>
     * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This can be
     * one of the following values (the default value is <code>OFF</code>):
     * </p>
     * <dl>
     * <dt>OFF</dt>
     * <dd>
     * <p>
     * Training jobs launched by the hyperparameter tuning job do not use early stopping.
     * </p>
     * </dd>
     * <dt>AUTO</dt>
     * <dd>
     * <p>
     * Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to perform
     * better than previously completed training jobs. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop Training
     * Jobs Early</a>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #trainingJobEarlyStoppingType} will return {@link TrainingJobEarlyStoppingType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #trainingJobEarlyStoppingTypeAsString}.
     * </p>
     * 
     * @return Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This
     *         can be one of the following values (the default value is <code>OFF</code>):</p>
     *         <dl>
     *         <dt>OFF</dt>
     *         <dd>
     *         <p>
     *         Training jobs launched by the hyperparameter tuning job do not use early stopping.
     *         </p>
     *         </dd>
     *         <dt>AUTO</dt>
     *         <dd>
     *         <p>
     *         Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
     *         perform better than previously completed training jobs. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
     *         Training Jobs Early</a>.
     *         </p>
     *         </dd>
     * @see TrainingJobEarlyStoppingType
     */
    public String trainingJobEarlyStoppingTypeAsString() {
        return trainingJobEarlyStoppingType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(strategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hyperParameterTuningJobObjective());
        hashCode = 31 * hashCode + Objects.hashCode(resourceLimits());
        hashCode = 31 * hashCode + Objects.hashCode(parameterRanges());
        hashCode = 31 * hashCode + Objects.hashCode(trainingJobEarlyStoppingTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTuningJobConfig)) {
            return false;
        }
        HyperParameterTuningJobConfig other = (HyperParameterTuningJobConfig) obj;
        return Objects.equals(strategyAsString(), other.strategyAsString())
                && Objects.equals(hyperParameterTuningJobObjective(), other.hyperParameterTuningJobObjective())
                && Objects.equals(resourceLimits(), other.resourceLimits())
                && Objects.equals(parameterRanges(), other.parameterRanges())
                && Objects.equals(trainingJobEarlyStoppingTypeAsString(), other.trainingJobEarlyStoppingTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("HyperParameterTuningJobConfig").add("Strategy", strategyAsString())
                .add("HyperParameterTuningJobObjective", hyperParameterTuningJobObjective())
                .add("ResourceLimits", resourceLimits()).add("ParameterRanges", parameterRanges())
                .add("TrainingJobEarlyStoppingType", trainingJobEarlyStoppingTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Strategy":
            return Optional.ofNullable(clazz.cast(strategyAsString()));
        case "HyperParameterTuningJobObjective":
            return Optional.ofNullable(clazz.cast(hyperParameterTuningJobObjective()));
        case "ResourceLimits":
            return Optional.ofNullable(clazz.cast(resourceLimits()));
        case "ParameterRanges":
            return Optional.ofNullable(clazz.cast(parameterRanges()));
        case "TrainingJobEarlyStoppingType":
            return Optional.ofNullable(clazz.cast(trainingJobEarlyStoppingTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTuningJobConfig, T> g) {
        return obj -> g.apply((HyperParameterTuningJobConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTuningJobConfig> {
        /**
         * <p>
         * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training
         * job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To randomly search,
         * set it to <code>Random</code>. For information about search strategies, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         * Hyperparameter Tuning Works</a>.
         * </p>
         * 
         * @param strategy
         *        Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
         *        training job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To
         *        randomly search, set it to <code>Random</code>. For information about search strategies, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         *        Hyperparameter Tuning Works</a>.
         * @see HyperParameterTuningJobStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobStrategyType
         */
        Builder strategy(String strategy);

        /**
         * <p>
         * Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the training
         * job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To randomly search,
         * set it to <code>Random</code>. For information about search strategies, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         * Hyperparameter Tuning Works</a>.
         * </p>
         * 
         * @param strategy
         *        Specifies how hyperparameter tuning chooses the combinations of hyperparameter values to use for the
         *        training job it launches. To use the Bayesian search stategy, set this to <code>Bayesian</code>. To
         *        randomly search, set it to <code>Random</code>. For information about search strategies, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html">How
         *        Hyperparameter Tuning Works</a>.
         * @see HyperParameterTuningJobStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HyperParameterTuningJobStrategyType
         */
        Builder strategy(HyperParameterTuningJobStrategyType strategy);

        /**
         * <p>
         * The <a>HyperParameterTuningJobObjective</a> object that specifies the objective metric for this tuning job.
         * </p>
         * 
         * @param hyperParameterTuningJobObjective
         *        The <a>HyperParameterTuningJobObjective</a> object that specifies the objective metric for this tuning
         *        job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hyperParameterTuningJobObjective(HyperParameterTuningJobObjective hyperParameterTuningJobObjective);

        /**
         * <p>
         * The <a>HyperParameterTuningJobObjective</a> object that specifies the objective metric for this tuning job.
         * </p>
         * This is a convenience that creates an instance of the {@link HyperParameterTuningJobObjective.Builder}
         * avoiding the need to create one manually via {@link HyperParameterTuningJobObjective#builder()}.
         *
         * When the {@link Consumer} completes, {@link HyperParameterTuningJobObjective.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #hyperParameterTuningJobObjective(HyperParameterTuningJobObjective)}.
         * 
         * @param hyperParameterTuningJobObjective
         *        a consumer that will call methods on {@link HyperParameterTuningJobObjective.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hyperParameterTuningJobObjective(HyperParameterTuningJobObjective)
         */
        default Builder hyperParameterTuningJobObjective(
                Consumer<HyperParameterTuningJobObjective.Builder> hyperParameterTuningJobObjective) {
            return hyperParameterTuningJobObjective(HyperParameterTuningJobObjective.builder()
                    .applyMutation(hyperParameterTuningJobObjective).build());
        }

        /**
         * <p>
         * The <a>ResourceLimits</a> object that specifies the maximum number of training jobs and parallel training
         * jobs for this tuning job.
         * </p>
         * 
         * @param resourceLimits
         *        The <a>ResourceLimits</a> object that specifies the maximum number of training jobs and parallel
         *        training jobs for this tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLimits(ResourceLimits resourceLimits);

        /**
         * <p>
         * The <a>ResourceLimits</a> object that specifies the maximum number of training jobs and parallel training
         * jobs for this tuning job.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLimits.Builder} avoiding the need to
         * create one manually via {@link ResourceLimits#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLimits.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceLimits(ResourceLimits)}.
         * 
         * @param resourceLimits
         *        a consumer that will call methods on {@link ResourceLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceLimits(ResourceLimits)
         */
        default Builder resourceLimits(Consumer<ResourceLimits.Builder> resourceLimits) {
            return resourceLimits(ResourceLimits.builder().applyMutation(resourceLimits).build());
        }

        /**
         * <p>
         * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches.
         * </p>
         * 
         * @param parameterRanges
         *        The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job
         *        searches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterRanges(ParameterRanges parameterRanges);

        /**
         * <p>
         * The <a>ParameterRanges</a> object that specifies the ranges of hyperparameters that this tuning job searches.
         * </p>
         * This is a convenience that creates an instance of the {@link ParameterRanges.Builder} avoiding the need to
         * create one manually via {@link ParameterRanges#builder()}.
         *
         * When the {@link Consumer} completes, {@link ParameterRanges.Builder#build()} is called immediately and its
         * result is passed to {@link #parameterRanges(ParameterRanges)}.
         * 
         * @param parameterRanges
         *        a consumer that will call methods on {@link ParameterRanges.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterRanges(ParameterRanges)
         */
        default Builder parameterRanges(Consumer<ParameterRanges.Builder> parameterRanges) {
            return parameterRanges(ParameterRanges.builder().applyMutation(parameterRanges).build());
        }

        /**
         * <p>
         * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This can
         * be one of the following values (the default value is <code>OFF</code>):
         * </p>
         * <dl>
         * <dt>OFF</dt>
         * <dd>
         * <p>
         * Training jobs launched by the hyperparameter tuning job do not use early stopping.
         * </p>
         * </dd>
         * <dt>AUTO</dt>
         * <dd>
         * <p>
         * Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
         * perform better than previously completed training jobs. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         * Training Jobs Early</a>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param trainingJobEarlyStoppingType
         *        Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
         *        This can be one of the following values (the default value is <code>OFF</code>):</p>
         *        <dl>
         *        <dt>OFF</dt>
         *        <dd>
         *        <p>
         *        Training jobs launched by the hyperparameter tuning job do not use early stopping.
         *        </p>
         *        </dd>
         *        <dt>AUTO</dt>
         *        <dd>
         *        <p>
         *        Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely
         *        to perform better than previously completed training jobs. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         *        Training Jobs Early</a>.
         *        </p>
         *        </dd>
         * @see TrainingJobEarlyStoppingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingJobEarlyStoppingType
         */
        Builder trainingJobEarlyStoppingType(String trainingJobEarlyStoppingType);

        /**
         * <p>
         * Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job. This can
         * be one of the following values (the default value is <code>OFF</code>):
         * </p>
         * <dl>
         * <dt>OFF</dt>
         * <dd>
         * <p>
         * Training jobs launched by the hyperparameter tuning job do not use early stopping.
         * </p>
         * </dd>
         * <dt>AUTO</dt>
         * <dd>
         * <p>
         * Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely to
         * perform better than previously completed training jobs. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         * Training Jobs Early</a>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param trainingJobEarlyStoppingType
         *        Specifies whether to use early stopping for training jobs launched by the hyperparameter tuning job.
         *        This can be one of the following values (the default value is <code>OFF</code>):</p>
         *        <dl>
         *        <dt>OFF</dt>
         *        <dd>
         *        <p>
         *        Training jobs launched by the hyperparameter tuning job do not use early stopping.
         *        </p>
         *        </dd>
         *        <dt>AUTO</dt>
         *        <dd>
         *        <p>
         *        Amazon SageMaker stops training jobs launched by the hyperparameter tuning job when they are unlikely
         *        to perform better than previously completed training jobs. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-early-stopping.html">Stop
         *        Training Jobs Early</a>.
         *        </p>
         *        </dd>
         * @see TrainingJobEarlyStoppingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingJobEarlyStoppingType
         */
        Builder trainingJobEarlyStoppingType(TrainingJobEarlyStoppingType trainingJobEarlyStoppingType);
    }

    static final class BuilderImpl implements Builder {
        private String strategy;

        private HyperParameterTuningJobObjective hyperParameterTuningJobObjective;

        private ResourceLimits resourceLimits;

        private ParameterRanges parameterRanges;

        private String trainingJobEarlyStoppingType;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTuningJobConfig model) {
            strategy(model.strategy);
            hyperParameterTuningJobObjective(model.hyperParameterTuningJobObjective);
            resourceLimits(model.resourceLimits);
            parameterRanges(model.parameterRanges);
            trainingJobEarlyStoppingType(model.trainingJobEarlyStoppingType);
        }

        public final String getStrategyAsString() {
            return strategy;
        }

        @Override
        public final Builder strategy(String strategy) {
            this.strategy = strategy;
            return this;
        }

        @Override
        public final Builder strategy(HyperParameterTuningJobStrategyType strategy) {
            this.strategy(strategy == null ? null : strategy.toString());
            return this;
        }

        public final void setStrategy(String strategy) {
            this.strategy = strategy;
        }

        public final HyperParameterTuningJobObjective.Builder getHyperParameterTuningJobObjective() {
            return hyperParameterTuningJobObjective != null ? hyperParameterTuningJobObjective.toBuilder() : null;
        }

        @Override
        public final Builder hyperParameterTuningJobObjective(HyperParameterTuningJobObjective hyperParameterTuningJobObjective) {
            this.hyperParameterTuningJobObjective = hyperParameterTuningJobObjective;
            return this;
        }

        public final void setHyperParameterTuningJobObjective(
                HyperParameterTuningJobObjective.BuilderImpl hyperParameterTuningJobObjective) {
            this.hyperParameterTuningJobObjective = hyperParameterTuningJobObjective != null ? hyperParameterTuningJobObjective
                    .build() : null;
        }

        public final ResourceLimits.Builder getResourceLimits() {
            return resourceLimits != null ? resourceLimits.toBuilder() : null;
        }

        @Override
        public final Builder resourceLimits(ResourceLimits resourceLimits) {
            this.resourceLimits = resourceLimits;
            return this;
        }

        public final void setResourceLimits(ResourceLimits.BuilderImpl resourceLimits) {
            this.resourceLimits = resourceLimits != null ? resourceLimits.build() : null;
        }

        public final ParameterRanges.Builder getParameterRanges() {
            return parameterRanges != null ? parameterRanges.toBuilder() : null;
        }

        @Override
        public final Builder parameterRanges(ParameterRanges parameterRanges) {
            this.parameterRanges = parameterRanges;
            return this;
        }

        public final void setParameterRanges(ParameterRanges.BuilderImpl parameterRanges) {
            this.parameterRanges = parameterRanges != null ? parameterRanges.build() : null;
        }

        public final String getTrainingJobEarlyStoppingTypeAsString() {
            return trainingJobEarlyStoppingType;
        }

        @Override
        public final Builder trainingJobEarlyStoppingType(String trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType = trainingJobEarlyStoppingType;
            return this;
        }

        @Override
        public final Builder trainingJobEarlyStoppingType(TrainingJobEarlyStoppingType trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType(trainingJobEarlyStoppingType == null ? null : trainingJobEarlyStoppingType
                    .toString());
            return this;
        }

        public final void setTrainingJobEarlyStoppingType(String trainingJobEarlyStoppingType) {
            this.trainingJobEarlyStoppingType = trainingJobEarlyStoppingType;
        }

        @Override
        public HyperParameterTuningJobConfig build() {
            return new HyperParameterTuningJobConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
