/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakera2iruntime;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ConflictException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DeleteHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DeleteHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DescribeHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DescribeHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.InternalServerException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.SageMakerA2IRuntimeException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.SageMakerA2IRuntimeRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StartHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StartHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StopHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StopHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ThrottlingException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ValidationException;
import software.amazon.awssdk.services.sagemakera2iruntime.paginators.ListHumanLoopsIterable;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.DeleteHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.DescribeHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.ListHumanLoopsRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.StartHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.StopHumanLoopRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SageMakerA2IRuntimeClient}.
 *
 * @see SageMakerA2IRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerA2IRuntimeClient implements SageMakerA2IRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultSageMakerA2IRuntimeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSageMakerA2IRuntimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified human loop for a flow definition.
     * </p>
     *
     * @param deleteHumanLoopRequest
     * @return Result of the DeleteHumanLoop operation returned by the service.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.DeleteHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/DeleteHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHumanLoopResponse deleteHumanLoop(DeleteHumanLoopRequest deleteHumanLoopRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            SageMakerA2IRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHumanLoopResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHumanLoop");

            return clientHandler.execute(new ClientExecutionParams<DeleteHumanLoopRequest, DeleteHumanLoopResponse>()
                    .withOperationName("DeleteHumanLoop").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteHumanLoopRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteHumanLoopRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified human loop.
     * </p>
     *
     * @param describeHumanLoopRequest
     * @return Result of the DescribeHumanLoop operation returned by the service.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.DescribeHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/DescribeHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeHumanLoopResponse describeHumanLoop(DescribeHumanLoopRequest describeHumanLoopRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, SageMakerA2IRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeHumanLoopResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHumanLoop");

            return clientHandler.execute(new ClientExecutionParams<DescribeHumanLoopRequest, DescribeHumanLoopResponse>()
                    .withOperationName("DescribeHumanLoop").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeHumanLoopRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeHumanLoopRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about human loops, given the specified parameters. If a human loop was deleted, it will not
     * be included.
     * </p>
     *
     * @param listHumanLoopsRequest
     * @return Result of the ListHumanLoops operation returned by the service.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.ListHumanLoops
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/ListHumanLoops"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHumanLoopsResponse listHumanLoops(ListHumanLoopsRequest listHumanLoopsRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            SageMakerA2IRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHumanLoopsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHumanLoopsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHumanLoopsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHumanLoops");

            return clientHandler.execute(new ClientExecutionParams<ListHumanLoopsRequest, ListHumanLoopsResponse>()
                    .withOperationName("ListHumanLoops").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHumanLoopsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHumanLoopsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about human loops, given the specified parameters. If a human loop was deleted, it will not
     * be included.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHumanLoops(software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakera2iruntime.paginators.ListHumanLoopsIterable responses = client.listHumanLoopsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.sagemakera2iruntime.paginators.ListHumanLoopsIterable responses = client
     *             .listHumanLoopsPaginator(request);
     *     for (software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.sagemakera2iruntime.paginators.ListHumanLoopsIterable responses = client.listHumanLoopsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHumanLoops(software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHumanLoopsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.ListHumanLoops
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/ListHumanLoops"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHumanLoopsIterable listHumanLoopsPaginator(ListHumanLoopsRequest listHumanLoopsRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, InternalServerException,
            AwsServiceException, SdkClientException, SageMakerA2IRuntimeException {
        return new ListHumanLoopsIterable(this, applyPaginatorUserAgent(listHumanLoopsRequest));
    }

    /**
     * <p>
     * Starts a human loop, provided that at least one activation condition is met.
     * </p>
     *
     * @param startHumanLoopRequest
     * @return Result of the StartHumanLoop operation returned by the service.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws ServiceQuotaExceededException
     *         You exceeded your service quota. Delete some resources or request an increase in your service quota.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws ConflictException
     *         Your request has the same name as another active human loop but has different input data. You cannot
     *         start two human loops with the same name and different input data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.StartHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/StartHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartHumanLoopResponse startHumanLoop(StartHumanLoopRequest startHumanLoopRequest) throws ValidationException,
            ThrottlingException, ServiceQuotaExceededException, InternalServerException, ConflictException, AwsServiceException,
            SdkClientException, SageMakerA2IRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartHumanLoopResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartHumanLoop");

            return clientHandler.execute(new ClientExecutionParams<StartHumanLoopRequest, StartHumanLoopResponse>()
                    .withOperationName("StartHumanLoop").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startHumanLoopRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartHumanLoopRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the specified human loop.
     * </p>
     *
     * @param stopHumanLoopRequest
     * @return Result of the StopHumanLoop operation returned by the service.
     * @throws ValidationException
     *         The request isn't valid. Check the syntax and try again.
     * @throws ResourceNotFoundException
     *         We couldn't find the requested resource.
     * @throws ThrottlingException
     *         You exceeded the maximum number of requests.
     * @throws InternalServerException
     *         We couldn't process your request because of an issue with the server. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SageMakerA2IRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SageMakerA2IRuntimeClient.StopHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/StopHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopHumanLoopResponse stopHumanLoop(StopHumanLoopRequest stopHumanLoopRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, InternalServerException, AwsServiceException, SdkClientException,
            SageMakerA2IRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopHumanLoopResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopHumanLoop");

            return clientHandler.execute(new ClientExecutionParams<StopHumanLoopRequest, StopHumanLoopResponse>()
                    .withOperationName("StopHumanLoop").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopHumanLoopRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopHumanLoopRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerA2IRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SageMakerA2IRuntimeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
