/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakera2iruntime;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakera2iruntime.internal.SageMakerA2IRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ConflictException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DeleteHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DeleteHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DescribeHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.DescribeHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.InternalServerException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ListHumanLoopsResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.SageMakerA2IRuntimeException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StartHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StartHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StopHumanLoopRequest;
import software.amazon.awssdk.services.sagemakera2iruntime.model.StopHumanLoopResponse;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ThrottlingException;
import software.amazon.awssdk.services.sagemakera2iruntime.model.ValidationException;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.DeleteHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.DescribeHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.ListHumanLoopsRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.StartHumanLoopRequestMarshaller;
import software.amazon.awssdk.services.sagemakera2iruntime.transform.StopHumanLoopRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SageMakerA2IRuntimeAsyncClient}.
 *
 * @see SageMakerA2IRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerA2IRuntimeAsyncClient implements SageMakerA2IRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSageMakerA2IRuntimeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SageMakerA2IRuntimeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSageMakerA2IRuntimeAsyncClient(SageMakerA2IRuntimeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified human loop for a flow definition.
     * </p>
     * <p>
     * If the human loop was deleted, this operation will return a <code>ResourceNotFoundException</code>.
     * </p>
     *
     * @param deleteHumanLoopRequest
     * @return A Java Future containing the result of the DeleteHumanLoop operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same AWS Region as your request, and try your request again.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerA2IRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerA2IRuntimeAsyncClient.DeleteHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/DeleteHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHumanLoopResponse> deleteHumanLoop(DeleteHumanLoopRequest deleteHumanLoopRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHumanLoopRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHumanLoop");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteHumanLoopResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHumanLoopResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHumanLoopRequest, DeleteHumanLoopResponse>()
                            .withOperationName("DeleteHumanLoop")
                            .withMarshaller(new DeleteHumanLoopRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHumanLoopRequest));
            CompletableFuture<DeleteHumanLoopResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified human loop. If the human loop was deleted, this operation will return a
     * <code>ResourceNotFoundException</code> error.
     * </p>
     *
     * @param describeHumanLoopRequest
     * @return A Java Future containing the result of the DescribeHumanLoop operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same AWS Region as your request, and try your request again.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerA2IRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerA2IRuntimeAsyncClient.DescribeHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/DescribeHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHumanLoopResponse> describeHumanLoop(DescribeHumanLoopRequest describeHumanLoopRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeHumanLoopRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHumanLoop");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeHumanLoopResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHumanLoopResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHumanLoopRequest, DescribeHumanLoopResponse>()
                            .withOperationName("DescribeHumanLoop")
                            .withMarshaller(new DescribeHumanLoopRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeHumanLoopRequest));
            CompletableFuture<DescribeHumanLoopResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about human loops, given the specified parameters. If a human loop was deleted, it will not
     * be included.
     * </p>
     *
     * @param listHumanLoopsRequest
     * @return A Java Future containing the result of the ListHumanLoops operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same AWS Region as your request, and try your request again.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerA2IRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerA2IRuntimeAsyncClient.ListHumanLoops
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/ListHumanLoops"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHumanLoopsResponse> listHumanLoops(ListHumanLoopsRequest listHumanLoopsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHumanLoopsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHumanLoopsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHumanLoops");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHumanLoopsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHumanLoopsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHumanLoopsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHumanLoopsRequest, ListHumanLoopsResponse>()
                            .withOperationName("ListHumanLoops")
                            .withMarshaller(new ListHumanLoopsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listHumanLoopsRequest));
            CompletableFuture<ListHumanLoopsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a human loop, provided that at least one activation condition is met.
     * </p>
     *
     * @param startHumanLoopRequest
     * @return A Java Future containing the result of the StartHumanLoop operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your AWS account. For a list of
     *         Amazon A2I service quotes, see <a href="https://docs.aws.amazon.com/general/latest/gr/a2i.html">Amazon
     *         Augmented AI Service Quotes</a>. Delete some resources or request an increase in your service quota. You
     *         can request a quota increase using Service Quotas or the AWS Support Center. To request an increase, see
     *         <a href="https://docs.aws.amazon.com/general/latest/gr/aws_service_limits.html">AWS Service Quotas</a> in
     *         the <i>AWS General Reference</i>.</li>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>ConflictException Your request has the same name as another active human loop but has different input
     *         data. You cannot start two human loops with the same name and different input data.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerA2IRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerA2IRuntimeAsyncClient.StartHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/StartHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartHumanLoopResponse> startHumanLoop(StartHumanLoopRequest startHumanLoopRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startHumanLoopRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartHumanLoop");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartHumanLoopResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartHumanLoopResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartHumanLoopRequest, StartHumanLoopResponse>()
                            .withOperationName("StartHumanLoop")
                            .withMarshaller(new StartHumanLoopRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startHumanLoopRequest));
            CompletableFuture<StartHumanLoopResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the specified human loop.
     * </p>
     *
     * @param stopHumanLoopRequest
     * @return A Java Future containing the result of the StopHumanLoop operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. Check the syntax and try again.</li>
     *         <li>ResourceNotFoundException We couldn't find the requested resource. Check that your resources exists
     *         and were created in the same AWS Region as your request, and try your request again.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException We couldn't process your request because of an issue with the server. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerA2IRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerA2IRuntimeAsyncClient.StopHumanLoop
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-a2i-runtime-2019-11-07/StopHumanLoop"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopHumanLoopResponse> stopHumanLoop(StopHumanLoopRequest stopHumanLoopRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopHumanLoopRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopHumanLoopRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker A2I Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopHumanLoop");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopHumanLoopResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopHumanLoopResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopHumanLoopResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopHumanLoopRequest, StopHumanLoopResponse>()
                            .withOperationName("StopHumanLoop")
                            .withMarshaller(new StopHumanLoopRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopHumanLoopRequest));
            CompletableFuture<StopHumanLoopResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SageMakerA2IRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerA2IRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SageMakerA2IRuntimeServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = SageMakerA2IRuntimeServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
