/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemakerfeaturestoreruntime;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.AccessForbiddenException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.BatchGetRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.BatchGetRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.DeleteRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.DeleteRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.GetRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.GetRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.InternalFailureException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.PutRecordRequest;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.PutRecordResponse;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.SageMakerFeatureStoreRuntimeException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.model.ValidationErrorException;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.BatchGetRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.DeleteRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.GetRecordRequestMarshaller;
import software.amazon.awssdk.services.sagemakerfeaturestoreruntime.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SageMakerFeatureStoreRuntimeAsyncClient}.
 *
 * @see SageMakerFeatureStoreRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSageMakerFeatureStoreRuntimeAsyncClient implements SageMakerFeatureStoreRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSageMakerFeatureStoreRuntimeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SageMakerFeatureStoreRuntimeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSageMakerFeatureStoreRuntimeAsyncClient(
            SageMakerFeatureStoreRuntimeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves a batch of <code>Records</code> from a <code>FeatureGroup</code>.
     * </p>
     *
     * @param batchGetRecordRequest
     * @return A Java Future containing the result of the BatchGetRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.BatchGetRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/BatchGetRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetRecordResponse> batchGetRecord(BatchGetRecordRequest batchGetRecordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetRecordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetRecordRequest, BatchGetRecordResponse>()
                            .withOperationName("BatchGetRecord")
                            .withMarshaller(new BatchGetRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetRecordRequest));
            CompletableFuture<BatchGetRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Record</code> from a <code>FeatureGroup</code>. When the <code>DeleteRecord</code> API is called
     * a new record will be added to the <code>OfflineStore</code> and the <code>Record</code> will be removed from the
     * <code>OnlineStore</code>. This record will have a value of <code>True</code> in the <code>is_deleted</code>
     * column.
     * </p>
     *
     * @param deleteRecordRequest
     * @return A Java Future containing the result of the DeleteRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.DeleteRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/DeleteRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRecordResponse> deleteRecord(DeleteRecordRequest deleteRecordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRecordRequest, DeleteRecordResponse>()
                            .withOperationName("DeleteRecord").withMarshaller(new DeleteRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRecordRequest));
            CompletableFuture<DeleteRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use for <code>OnlineStore</code> serving from a <code>FeatureStore</code>. Only the latest records stored in the
     * <code>OnlineStore</code> can be retrieved. If no Record with <code>RecordIdentifierValue</code> is found, then an
     * empty result is returned.
     * </p>
     *
     * @param getRecordRequest
     * @return A Java Future containing the result of the GetRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>ResourceNotFoundException A resource that is required to perform an action was not found.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.GetRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/GetRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecordResponse> getRecord(GetRecordRequest getRecordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecordRequest, GetRecordResponse>().withOperationName("GetRecord")
                            .withMarshaller(new GetRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecordRequest));
            CompletableFuture<GetRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used for data ingestion into the <code>FeatureStore</code>. The <code>PutRecord</code> API writes to both the
     * <code>OnlineStore</code> and <code>OfflineStore</code>. If the record is the latest record for the
     * <code>recordIdentifier</code>, the record is written to both the <code>OnlineStore</code> and
     * <code>OfflineStore</code>. If the record is a historic record, it is written only to the
     * <code>OfflineStore</code>.
     * </p>
     *
     * @param putRecordRequest
     * @return A Java Future containing the result of the PutRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationErrorException There was an error validating your request.</li>
     *         <li>InternalFailureException An internal failure occurred. Try your request again. If the problem
     *         persists, contact Amazon Web Services customer support.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>AccessForbiddenException You do not have permission to perform an action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SageMakerFeatureStoreRuntimeException Base class for all service exceptions. Unknown exceptions will
     *         be thrown as an instance of this type.</li>
     *         </ul>
     * @sample SageMakerFeatureStoreRuntimeAsyncClient.PutRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/sagemaker-featurestore-runtime-2020-07-01/PutRecord"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecordResponse> putRecord(PutRecordRequest putRecordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SageMaker FeatureStore Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecord");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>().withOperationName("PutRecord")
                            .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRecordRequest));
            CompletableFuture<PutRecordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SageMakerFeatureStoreRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SageMakerFeatureStoreRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationError")
                                .exceptionBuilderSupplier(ValidationErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailable")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessForbidden")
                                .exceptionBuilderSupplier(AccessForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailure")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
