/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.scheduler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The templated target type for the Amazon ECS <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html"> <code>RunTask</code> </a> API
 * operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EcsParameters implements SdkPojo, Serializable, ToCopyableBuilder<EcsParameters.Builder, EcsParameters> {
    private static final SdkField<List<CapacityProviderStrategyItem>> CAPACITY_PROVIDER_STRATEGY_FIELD = SdkField
            .<List<CapacityProviderStrategyItem>> builder(MarshallingType.LIST)
            .memberName("CapacityProviderStrategy")
            .getter(getter(EcsParameters::capacityProviderStrategy))
            .setter(setter(Builder::capacityProviderStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityProviderStrategy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CapacityProviderStrategyItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(CapacityProviderStrategyItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_ECS_MANAGED_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableECSManagedTags").getter(getter(EcsParameters::enableECSManagedTags))
            .setter(setter(Builder::enableECSManagedTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableECSManagedTags").build())
            .build();

    private static final SdkField<Boolean> ENABLE_EXECUTE_COMMAND_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableExecuteCommand").getter(getter(EcsParameters::enableExecuteCommand))
            .setter(setter(Builder::enableExecuteCommand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableExecuteCommand").build())
            .build();

    private static final SdkField<String> GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Group")
            .getter(getter(EcsParameters::group)).setter(setter(Builder::group))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Group").build()).build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LaunchType").getter(getter(EcsParameters::launchTypeAsString)).setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchType").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("NetworkConfiguration")
            .getter(getter(EcsParameters::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkConfiguration").build())
            .build();

    private static final SdkField<List<PlacementConstraint>> PLACEMENT_CONSTRAINTS_FIELD = SdkField
            .<List<PlacementConstraint>> builder(MarshallingType.LIST)
            .memberName("PlacementConstraints")
            .getter(getter(EcsParameters::placementConstraints))
            .setter(setter(Builder::placementConstraints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementConstraints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlacementConstraint> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlacementConstraint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PlacementStrategy>> PLACEMENT_STRATEGY_FIELD = SdkField
            .<List<PlacementStrategy>> builder(MarshallingType.LIST)
            .memberName("PlacementStrategy")
            .getter(getter(EcsParameters::placementStrategy))
            .setter(setter(Builder::placementStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementStrategy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlacementStrategy> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlacementStrategy::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlatformVersion").getter(getter(EcsParameters::platformVersion))
            .setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformVersion").build()).build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PropagateTags").getter(getter(EcsParameters::propagateTagsAsString))
            .setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PropagateTags").build()).build();

    private static final SdkField<String> REFERENCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReferenceId").getter(getter(EcsParameters::referenceId)).setter(setter(Builder::referenceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceId").build()).build();

    private static final SdkField<List<Map<String, String>>> TAGS_FIELD = SdkField
            .<List<Map<String, String>>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(EcsParameters::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, String>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Integer> TASK_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TaskCount").getter(getter(EcsParameters::taskCount)).setter(setter(Builder::taskCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskCount").build()).build();

    private static final SdkField<String> TASK_DEFINITION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskDefinitionArn").getter(getter(EcsParameters::taskDefinitionArn))
            .setter(setter(Builder::taskDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskDefinitionArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CAPACITY_PROVIDER_STRATEGY_FIELD, ENABLE_ECS_MANAGED_TAGS_FIELD, ENABLE_EXECUTE_COMMAND_FIELD, GROUP_FIELD,
            LAUNCH_TYPE_FIELD, NETWORK_CONFIGURATION_FIELD, PLACEMENT_CONSTRAINTS_FIELD, PLACEMENT_STRATEGY_FIELD,
            PLATFORM_VERSION_FIELD, PROPAGATE_TAGS_FIELD, REFERENCE_ID_FIELD, TAGS_FIELD, TASK_COUNT_FIELD,
            TASK_DEFINITION_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<CapacityProviderStrategyItem> capacityProviderStrategy;

    private final Boolean enableECSManagedTags;

    private final Boolean enableExecuteCommand;

    private final String group;

    private final String launchType;

    private final NetworkConfiguration networkConfiguration;

    private final List<PlacementConstraint> placementConstraints;

    private final List<PlacementStrategy> placementStrategy;

    private final String platformVersion;

    private final String propagateTags;

    private final String referenceId;

    private final List<Map<String, String>> tags;

    private final Integer taskCount;

    private final String taskDefinitionArn;

    private EcsParameters(BuilderImpl builder) {
        this.capacityProviderStrategy = builder.capacityProviderStrategy;
        this.enableECSManagedTags = builder.enableECSManagedTags;
        this.enableExecuteCommand = builder.enableExecuteCommand;
        this.group = builder.group;
        this.launchType = builder.launchType;
        this.networkConfiguration = builder.networkConfiguration;
        this.placementConstraints = builder.placementConstraints;
        this.placementStrategy = builder.placementStrategy;
        this.platformVersion = builder.platformVersion;
        this.propagateTags = builder.propagateTags;
        this.referenceId = builder.referenceId;
        this.tags = builder.tags;
        this.taskCount = builder.taskCount;
        this.taskDefinitionArn = builder.taskDefinitionArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the CapacityProviderStrategy property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCapacityProviderStrategy() {
        return capacityProviderStrategy != null && !(capacityProviderStrategy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The capacity provider strategy to use for the task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapacityProviderStrategy} method.
     * </p>
     * 
     * @return The capacity provider strategy to use for the task.
     */
    public final List<CapacityProviderStrategyItem> capacityProviderStrategy() {
        return capacityProviderStrategy;
    }

    /**
     * <p>
     * Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon ECS
     * Resources</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * 
     * @return Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your
     *         Amazon ECS Resources</a> in the <i>Amazon ECS Developer Guide</i>.
     */
    public final Boolean enableECSManagedTags() {
        return enableECSManagedTags;
    }

    /**
     * <p>
     * Whether or not to enable the execute command functionality for the containers in this task. If true, this enables
     * execute command functionality on all containers in the task.
     * </p>
     * 
     * @return Whether or not to enable the execute command functionality for the containers in this task. If true, this
     *         enables execute command functionality on all containers in the task.
     */
    public final Boolean enableExecuteCommand() {
        return enableExecuteCommand;
    }

    /**
     * <p>
     * Specifies an ECS task group for the task. The maximum length is 255 characters.
     * </p>
     * 
     * @return Specifies an ECS task group for the task. The maximum length is 255 characters.
     */
    public final String group() {
        return group;
    }

    /**
     * <p>
     * Specifies the launch type on which your task is running. The launch type that you specify here must match one of
     * the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in the
     * Regions where Fargate with Amazon ECS is supported. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon ECS</a>
     * in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return Specifies the launch type on which your task is running. The launch type that you specify here must match
     *         one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported
     *         only in the Regions where Fargate with Amazon ECS is supported. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon
     *         ECS</a> in the <i>Amazon ECS Developer Guide</i>.
     * @see LaunchType
     */
    public final LaunchType launchType() {
        return LaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * Specifies the launch type on which your task is running. The launch type that you specify here must match one of
     * the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in the
     * Regions where Fargate with Amazon ECS is supported. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon ECS</a>
     * in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return Specifies the launch type on which your task is running. The launch type that you specify here must match
     *         one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported
     *         only in the Regions where Fargate with Amazon ECS is supported. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon
     *         ECS</a> in the <i>Amazon ECS Developer Guide</i>.
     * @see LaunchType
     */
    public final String launchTypeAsString() {
        return launchType;
    }

    /**
     * <p>
     * This structure specifies the network configuration for an ECS task.
     * </p>
     * 
     * @return This structure specifies the network configuration for an ECS task.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlacementConstraints property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlacementConstraints() {
        return placementConstraints != null && !(placementConstraints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of placement constraint objects to use for the task. You can specify up to 10 constraints per task
     * (including constraints in the task definition and those specified at runtime).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlacementConstraints} method.
     * </p>
     * 
     * @return An array of placement constraint objects to use for the task. You can specify up to 10 constraints per
     *         task (including constraints in the task definition and those specified at runtime).
     */
    public final List<PlacementConstraint> placementConstraints() {
        return placementConstraints;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlacementStrategy property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlacementStrategy() {
        return placementStrategy != null && !(placementStrategy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The task placement strategy for a task or service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlacementStrategy} method.
     * </p>
     * 
     * @return The task placement strategy for a task or service.
     */
    public final List<PlacementStrategy> placementStrategy() {
        return placementStrategy;
    }

    /**
     * <p>
     * Specifies the platform version for the task. Specify only the numeric portion of the platform version, such as
     * <code>1.1.0</code>.
     * </p>
     * 
     * @return Specifies the platform version for the task. Specify only the numeric portion of the platform version,
     *         such as <code>1.1.0</code>.
     */
    public final String platformVersion() {
        return platformVersion;
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the tags
     * are not propagated. Tags can only be propagated to the task during task creation. To add tags to a task after
     * task creation, use Amazon ECS's <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html"> <code>TagResource</code>
     * </a> API action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition to the task. If no value is specified,
     *         the tags are not propagated. Tags can only be propagated to the task during task creation. To add tags to
     *         a task after task creation, use Amazon ECS's <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
     *         <code>TagResource</code> </a> API action.
     * @see PropagateTags
     */
    public final PropagateTags propagateTags() {
        return PropagateTags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the tags
     * are not propagated. Tags can only be propagated to the task during task creation. To add tags to a task after
     * task creation, use Amazon ECS's <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html"> <code>TagResource</code>
     * </a> API action.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition to the task. If no value is specified,
     *         the tags are not propagated. Tags can only be propagated to the task during task creation. To add tags to
     *         a task after task creation, use Amazon ECS's <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
     *         <code>TagResource</code> </a> API action.
     * @see PropagateTags
     */
    public final String propagateTagsAsString() {
        return propagateTags;
    }

    /**
     * <p>
     * The reference ID to use for the task.
     * </p>
     * 
     * @return The reference ID to use for the task.
     */
    public final String referenceId() {
        return referenceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key and
     * an optional value, both of which you define. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html"> <code>RunTask</code> </a> in
     * the <i>Amazon ECS API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a
     *         key and an optional value, both of which you define. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html"> <code>RunTask</code>
     *         </a> in the <i>Amazon ECS API Reference</i>.
     */
    public final List<Map<String, String>> tags() {
        return tags;
    }

    /**
     * <p>
     * The number of tasks to create based on <code>TaskDefinition</code>. The default is <code>1</code>.
     * </p>
     * 
     * @return The number of tasks to create based on <code>TaskDefinition</code>. The default is <code>1</code>.
     */
    public final Integer taskCount() {
        return taskCount;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task definition to use if the event target is an Amazon ECS task.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task definition to use if the event target is an Amazon ECS task.
     */
    public final String taskDefinitionArn() {
        return taskDefinitionArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityProviderStrategy() ? capacityProviderStrategy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(enableECSManagedTags());
        hashCode = 31 * hashCode + Objects.hashCode(enableExecuteCommand());
        hashCode = 31 * hashCode + Objects.hashCode(group());
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlacementConstraints() ? placementConstraints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPlacementStrategy() ? placementStrategy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(referenceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(taskCount());
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinitionArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EcsParameters)) {
            return false;
        }
        EcsParameters other = (EcsParameters) obj;
        return hasCapacityProviderStrategy() == other.hasCapacityProviderStrategy()
                && Objects.equals(capacityProviderStrategy(), other.capacityProviderStrategy())
                && Objects.equals(enableECSManagedTags(), other.enableECSManagedTags())
                && Objects.equals(enableExecuteCommand(), other.enableExecuteCommand()) && Objects.equals(group(), other.group())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && hasPlacementConstraints() == other.hasPlacementConstraints()
                && Objects.equals(placementConstraints(), other.placementConstraints())
                && hasPlacementStrategy() == other.hasPlacementStrategy()
                && Objects.equals(placementStrategy(), other.placementStrategy())
                && Objects.equals(platformVersion(), other.platformVersion())
                && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString())
                && Objects.equals(referenceId(), other.referenceId()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(taskCount(), other.taskCount())
                && Objects.equals(taskDefinitionArn(), other.taskDefinitionArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EcsParameters")
                .add("CapacityProviderStrategy", hasCapacityProviderStrategy() ? capacityProviderStrategy() : null)
                .add("EnableECSManagedTags", enableECSManagedTags()).add("EnableExecuteCommand", enableExecuteCommand())
                .add("Group", group()).add("LaunchType", launchTypeAsString())
                .add("NetworkConfiguration", networkConfiguration())
                .add("PlacementConstraints", hasPlacementConstraints() ? placementConstraints() : null)
                .add("PlacementStrategy", hasPlacementStrategy() ? placementStrategy() : null)
                .add("PlatformVersion", platformVersion()).add("PropagateTags", propagateTagsAsString())
                .add("ReferenceId", referenceId()).add("Tags", hasTags() ? tags() : null).add("TaskCount", taskCount())
                .add("TaskDefinitionArn", taskDefinitionArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CapacityProviderStrategy":
            return Optional.ofNullable(clazz.cast(capacityProviderStrategy()));
        case "EnableECSManagedTags":
            return Optional.ofNullable(clazz.cast(enableECSManagedTags()));
        case "EnableExecuteCommand":
            return Optional.ofNullable(clazz.cast(enableExecuteCommand()));
        case "Group":
            return Optional.ofNullable(clazz.cast(group()));
        case "LaunchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "NetworkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "PlacementConstraints":
            return Optional.ofNullable(clazz.cast(placementConstraints()));
        case "PlacementStrategy":
            return Optional.ofNullable(clazz.cast(placementStrategy()));
        case "PlatformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "PropagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        case "ReferenceId":
            return Optional.ofNullable(clazz.cast(referenceId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "TaskCount":
            return Optional.ofNullable(clazz.cast(taskCount()));
        case "TaskDefinitionArn":
            return Optional.ofNullable(clazz.cast(taskDefinitionArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EcsParameters, T> g) {
        return obj -> g.apply((EcsParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EcsParameters> {
        /**
         * <p>
         * The capacity provider strategy to use for the task.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy to use for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy to use for the task.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy to use for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy to use for the task.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.scheduler.model.CapacityProviderStrategyItem.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.scheduler.model.CapacityProviderStrategyItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.scheduler.model.CapacityProviderStrategyItem.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #capacityProviderStrategy(List<CapacityProviderStrategyItem>)}.
         * 
         * @param capacityProviderStrategy
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.scheduler.model.CapacityProviderStrategyItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityProviderStrategy(java.util.Collection<CapacityProviderStrategyItem>)
         */
        Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy);

        /**
         * <p>
         * Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon
         * ECS Resources</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param enableECSManagedTags
         *        Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your
         *        Amazon ECS Resources</a> in the <i>Amazon ECS Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableECSManagedTags(Boolean enableECSManagedTags);

        /**
         * <p>
         * Whether or not to enable the execute command functionality for the containers in this task. If true, this
         * enables execute command functionality on all containers in the task.
         * </p>
         * 
         * @param enableExecuteCommand
         *        Whether or not to enable the execute command functionality for the containers in this task. If true,
         *        this enables execute command functionality on all containers in the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableExecuteCommand(Boolean enableExecuteCommand);

        /**
         * <p>
         * Specifies an ECS task group for the task. The maximum length is 255 characters.
         * </p>
         * 
         * @param group
         *        Specifies an ECS task group for the task. The maximum length is 255 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder group(String group);

        /**
         * <p>
         * Specifies the launch type on which your task is running. The launch type that you specify here must match one
         * of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in
         * the Regions where Fargate with Amazon ECS is supported. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon
         * ECS</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        Specifies the launch type on which your task is running. The launch type that you specify here must
         *        match one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is
         *        supported only in the Regions where Fargate with Amazon ECS is supported. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on
         *        Amazon ECS</a> in the <i>Amazon ECS Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * Specifies the launch type on which your task is running. The launch type that you specify here must match one
         * of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in
         * the Regions where Fargate with Amazon ECS is supported. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on Amazon
         * ECS</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        Specifies the launch type on which your task is running. The launch type that you specify here must
         *        match one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is
         *        supported only in the Regions where Fargate with Amazon ECS is supported. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html">AWS Fargate on
         *        Amazon ECS</a> in the <i>Amazon ECS Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(LaunchType launchType);

        /**
         * <p>
         * This structure specifies the network configuration for an ECS task.
         * </p>
         * 
         * @param networkConfiguration
         *        This structure specifies the network configuration for an ECS task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * This structure specifies the network configuration for an ECS task.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * An array of placement constraint objects to use for the task. You can specify up to 10 constraints per task
         * (including constraints in the task definition and those specified at runtime).
         * </p>
         * 
         * @param placementConstraints
         *        An array of placement constraint objects to use for the task. You can specify up to 10 constraints per
         *        task (including constraints in the task definition and those specified at runtime).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementConstraints(Collection<PlacementConstraint> placementConstraints);

        /**
         * <p>
         * An array of placement constraint objects to use for the task. You can specify up to 10 constraints per task
         * (including constraints in the task definition and those specified at runtime).
         * </p>
         * 
         * @param placementConstraints
         *        An array of placement constraint objects to use for the task. You can specify up to 10 constraints per
         *        task (including constraints in the task definition and those specified at runtime).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementConstraints(PlacementConstraint... placementConstraints);

        /**
         * <p>
         * An array of placement constraint objects to use for the task. You can specify up to 10 constraints per task
         * (including constraints in the task definition and those specified at runtime).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.scheduler.model.PlacementConstraint.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.scheduler.model.PlacementConstraint#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.scheduler.model.PlacementConstraint.Builder#build()} is called
         * immediately and its result is passed to {@link #placementConstraints(List<PlacementConstraint>)}.
         * 
         * @param placementConstraints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.scheduler.model.PlacementConstraint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placementConstraints(java.util.Collection<PlacementConstraint>)
         */
        Builder placementConstraints(Consumer<PlacementConstraint.Builder>... placementConstraints);

        /**
         * <p>
         * The task placement strategy for a task or service.
         * </p>
         * 
         * @param placementStrategy
         *        The task placement strategy for a task or service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementStrategy(Collection<PlacementStrategy> placementStrategy);

        /**
         * <p>
         * The task placement strategy for a task or service.
         * </p>
         * 
         * @param placementStrategy
         *        The task placement strategy for a task or service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementStrategy(PlacementStrategy... placementStrategy);

        /**
         * <p>
         * The task placement strategy for a task or service.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.scheduler.model.PlacementStrategy.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.scheduler.model.PlacementStrategy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.scheduler.model.PlacementStrategy.Builder#build()} is called
         * immediately and its result is passed to {@link #placementStrategy(List<PlacementStrategy>)}.
         * 
         * @param placementStrategy
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.scheduler.model.PlacementStrategy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placementStrategy(java.util.Collection<PlacementStrategy>)
         */
        Builder placementStrategy(Consumer<PlacementStrategy.Builder>... placementStrategy);

        /**
         * <p>
         * Specifies the platform version for the task. Specify only the numeric portion of the platform version, such
         * as <code>1.1.0</code>.
         * </p>
         * 
         * @param platformVersion
         *        Specifies the platform version for the task. Specify only the numeric portion of the platform version,
         *        such as <code>1.1.0</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the
         * tags are not propagated. Tags can only be propagated to the task during task creation. To add tags to a task
         * after task creation, use Amazon ECS's <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
         * <code>TagResource</code> </a> API action.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition to the task. If no value is
         *        specified, the tags are not propagated. Tags can only be propagated to the task during task creation.
         *        To add tags to a task after task creation, use Amazon ECS's <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
         *        <code>TagResource</code> </a> API action.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition to the task. If no value is specified, the
         * tags are not propagated. Tags can only be propagated to the task during task creation. To add tags to a task
         * after task creation, use Amazon ECS's <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
         * <code>TagResource</code> </a> API action.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition to the task. If no value is
         *        specified, the tags are not propagated. Tags can only be propagated to the task during task creation.
         *        To add tags to a task after task creation, use Amazon ECS's <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html">
         *        <code>TagResource</code> </a> API action.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(PropagateTags propagateTags);

        /**
         * <p>
         * The reference ID to use for the task.
         * </p>
         * 
         * @param referenceId
         *        The reference ID to use for the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceId(String referenceId);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html"> <code>RunTask</code> </a>
         * in the <i>Amazon ECS API Reference</i>.
         * </p>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html">
         *        <code>RunTask</code> </a> in the <i>Amazon ECS API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<? extends Map<String, String>> tags);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html"> <code>RunTask</code> </a>
         * in the <i>Amazon ECS API Reference</i>.
         * </p>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html">
         *        <code>RunTask</code> </a> in the <i>Amazon ECS API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String>... tags);

        /**
         * <p>
         * The number of tasks to create based on <code>TaskDefinition</code>. The default is <code>1</code>.
         * </p>
         * 
         * @param taskCount
         *        The number of tasks to create based on <code>TaskDefinition</code>. The default is <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskCount(Integer taskCount);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task definition to use if the event target is an Amazon ECS task.
         * </p>
         * 
         * @param taskDefinitionArn
         *        The Amazon Resource Name (ARN) of the task definition to use if the event target is an Amazon ECS
         *        task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinitionArn(String taskDefinitionArn);
    }

    static final class BuilderImpl implements Builder {
        private List<CapacityProviderStrategyItem> capacityProviderStrategy = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableECSManagedTags;

        private Boolean enableExecuteCommand;

        private String group;

        private String launchType;

        private NetworkConfiguration networkConfiguration;

        private List<PlacementConstraint> placementConstraints = DefaultSdkAutoConstructList.getInstance();

        private List<PlacementStrategy> placementStrategy = DefaultSdkAutoConstructList.getInstance();

        private String platformVersion;

        private String propagateTags;

        private String referenceId;

        private List<Map<String, String>> tags = DefaultSdkAutoConstructList.getInstance();

        private Integer taskCount;

        private String taskDefinitionArn;

        private BuilderImpl() {
        }

        private BuilderImpl(EcsParameters model) {
            capacityProviderStrategy(model.capacityProviderStrategy);
            enableECSManagedTags(model.enableECSManagedTags);
            enableExecuteCommand(model.enableExecuteCommand);
            group(model.group);
            launchType(model.launchType);
            networkConfiguration(model.networkConfiguration);
            placementConstraints(model.placementConstraints);
            placementStrategy(model.placementStrategy);
            platformVersion(model.platformVersion);
            propagateTags(model.propagateTags);
            referenceId(model.referenceId);
            tags(model.tags);
            taskCount(model.taskCount);
            taskDefinitionArn(model.taskDefinitionArn);
        }

        public final List<CapacityProviderStrategyItem.Builder> getCapacityProviderStrategy() {
            List<CapacityProviderStrategyItem.Builder> result = CapacityProviderStrategyCopier
                    .copyToBuilder(this.capacityProviderStrategy);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCapacityProviderStrategy(
                Collection<CapacityProviderStrategyItem.BuilderImpl> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copyFromBuilder(capacityProviderStrategy);
        }

        @Override
        public final Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copy(capacityProviderStrategy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy) {
            capacityProviderStrategy(Arrays.asList(capacityProviderStrategy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy) {
            capacityProviderStrategy(Stream.of(capacityProviderStrategy)
                    .map(c -> CapacityProviderStrategyItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEnableECSManagedTags() {
            return enableECSManagedTags;
        }

        public final void setEnableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
        }

        @Override
        public final Builder enableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
            return this;
        }

        public final Boolean getEnableExecuteCommand() {
            return enableExecuteCommand;
        }

        public final void setEnableExecuteCommand(Boolean enableExecuteCommand) {
            this.enableExecuteCommand = enableExecuteCommand;
        }

        @Override
        public final Builder enableExecuteCommand(Boolean enableExecuteCommand) {
            this.enableExecuteCommand = enableExecuteCommand;
            return this;
        }

        public final String getGroup() {
            return group;
        }

        public final void setGroup(String group) {
            this.group = group;
        }

        @Override
        public final Builder group(String group) {
            this.group = group;
            return this;
        }

        public final String getLaunchType() {
            return launchType;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(LaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final List<PlacementConstraint.Builder> getPlacementConstraints() {
            List<PlacementConstraint.Builder> result = PlacementConstraintsCopier.copyToBuilder(this.placementConstraints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlacementConstraints(Collection<PlacementConstraint.BuilderImpl> placementConstraints) {
            this.placementConstraints = PlacementConstraintsCopier.copyFromBuilder(placementConstraints);
        }

        @Override
        public final Builder placementConstraints(Collection<PlacementConstraint> placementConstraints) {
            this.placementConstraints = PlacementConstraintsCopier.copy(placementConstraints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementConstraints(PlacementConstraint... placementConstraints) {
            placementConstraints(Arrays.asList(placementConstraints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementConstraints(Consumer<PlacementConstraint.Builder>... placementConstraints) {
            placementConstraints(Stream.of(placementConstraints).map(c -> PlacementConstraint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<PlacementStrategy.Builder> getPlacementStrategy() {
            List<PlacementStrategy.Builder> result = PlacementStrategiesCopier.copyToBuilder(this.placementStrategy);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlacementStrategy(Collection<PlacementStrategy.BuilderImpl> placementStrategy) {
            this.placementStrategy = PlacementStrategiesCopier.copyFromBuilder(placementStrategy);
        }

        @Override
        public final Builder placementStrategy(Collection<PlacementStrategy> placementStrategy) {
            this.placementStrategy = PlacementStrategiesCopier.copy(placementStrategy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementStrategy(PlacementStrategy... placementStrategy) {
            placementStrategy(Arrays.asList(placementStrategy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementStrategy(Consumer<PlacementStrategy.Builder>... placementStrategy) {
            placementStrategy(Stream.of(placementStrategy).map(c -> PlacementStrategy.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final String getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(PropagateTags propagateTags) {
            this.propagateTags(propagateTags == null ? null : propagateTags.toString());
            return this;
        }

        public final String getReferenceId() {
            return referenceId;
        }

        public final void setReferenceId(String referenceId) {
            this.referenceId = referenceId;
        }

        @Override
        public final Builder referenceId(String referenceId) {
            this.referenceId = referenceId;
            return this;
        }

        public final Collection<? extends Map<String, String>> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags;
        }

        public final void setTags(Collection<? extends Map<String, String>> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Collection<? extends Map<String, String>> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Map<String, String>... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final Integer getTaskCount() {
            return taskCount;
        }

        public final void setTaskCount(Integer taskCount) {
            this.taskCount = taskCount;
        }

        @Override
        public final Builder taskCount(Integer taskCount) {
            this.taskCount = taskCount;
            return this;
        }

        public final String getTaskDefinitionArn() {
            return taskDefinitionArn;
        }

        public final void setTaskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
        }

        @Override
        public final Builder taskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
            return this;
        }

        @Override
        public EcsParameters build() {
            return new EcsParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
