/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateSecretRequest extends SecretsManagerRequest implements
        ToCopyableBuilder<CreateSecretRequest.Builder, CreateSecretRequest> {
    private final String name;

    private final String clientRequestToken;

    private final String description;

    private final String kmsKeyId;

    private final ByteBuffer secretBinary;

    private final String secretString;

    private final List<Tag> tags;

    private CreateSecretRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.clientRequestToken = builder.clientRequestToken;
        this.description = builder.description;
        this.kmsKeyId = builder.kmsKeyId;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Specifies the friendly name of the new secret.
     * </p>
     * 
     * @return Specifies the friendly name of the new secret.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * (Optional) If you include <code>SecretString</code> or <code>SecretBinary</code>, then an initial version is
     * created as part of the secret, and this parameter specifies a unique identifier for the new version.
     * </p>
     * <note>
     * <p>
     * If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter empty. The
     * CLI or SDK generates a random UUID for you and includes as the value for this parameter in the request. If you
     * don't use the SDK and instead generate a raw HTTP request to the Secrets Manager service endpoint, then you must
     * generate a <code>ClientRequestToken</code> yourself for the new version and include that value in the request.
     * </p>
     * </note>
     * <p>
     * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
     * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
     * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of your
     * versions within the specified secret.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
     * version of the secret is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> and
     * <code>SecretBinary</code> values are the same as those in the request, then the request is ignored (the operation
     * is idempotent).
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> and
     * <code>SecretBinary</code> values are different from those in the request then the request fails because you
     * cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This value becomes the <code>SecretVersionId</code> of the new version.
     * </p>
     * 
     * @return (Optional) If you include <code>SecretString</code> or <code>SecretBinary</code>, then an initial version
     *         is created as part of the secret, and this parameter specifies a unique identifier for the new version.
     *         </p> <note>
     *         <p>
     *         If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter
     *         empty. The CLI or SDK generates a random UUID for you and includes as the value for this parameter in the
     *         request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager service
     *         endpoint, then you must generate a <code>ClientRequestToken</code> yourself for the new version and
     *         include that value in the request.
     *         </p>
     *         </note>
     *         <p>
     *         This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation
     *         of duplicate versions if there are failures and retries during a rotation. We recommend that you generate
     *         a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure
     *         uniqueness of your versions within the specified secret.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a
     *         new version of the secret is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> and
     *         <code>SecretBinary</code> values are the same as those in the request, then the request is ignored (the
     *         operation is idempotent).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> and
     *         <code>SecretBinary</code> values are different from those in the request then the request fails because
     *         you cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This value becomes the <code>SecretVersionId</code> of the new version.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * (Optional) Specifies a user-provided description of the secret.
     * </p>
     * 
     * @return (Optional) Specifies a user-provided description of the secret.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * (Optional) Specifies the ARN or alias of the AWS KMS customer master key (CMK) to be used to encrypt the
     * <code>SecretString</code> or <code>SecretBinary</code> values in the versions stored in this secret.
     * </p>
     * <p>
     * If you don't specify this value, then Secrets Manager defaults to using the AWS account's default CMK (the one
     * named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't yet exist, then Secrets Manager
     * creates it for you automatically the first time it needs to encrypt a version's <code>SecretString</code> or
     * <code>SecretBinary</code> fields.
     * </p>
     * <important>
     * <p>
     * You can use the account's default CMK to encrypt and decrypt only if you call this operation using credentials
     * from the same account that owns the secret. If the secret is in a different account, then you must create a
     * custom CMK and specify the ARN in this field.
     * </p>
     * </important>
     * 
     * @return (Optional) Specifies the ARN or alias of the AWS KMS customer master key (CMK) to be used to encrypt the
     *         <code>SecretString</code> or <code>SecretBinary</code> values in the versions stored in this secret.</p>
     *         <p>
     *         If you don't specify this value, then Secrets Manager defaults to using the AWS account's default CMK
     *         (the one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't yet exist, then
     *         Secrets Manager creates it for you automatically the first time it needs to encrypt a version's
     *         <code>SecretString</code> or <code>SecretBinary</code> fields.
     *         </p>
     *         <important>
     *         <p>
     *         You can use the account's default CMK to encrypt and decrypt only if you call this operation using
     *         credentials from the same account that owns the secret. If the secret is in a different account, then you
     *         must create a custom CMK and specify the ARN in this field.
     *         </p>
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use this
     * parameter in the command-line tools, we recommend that you store your binary data in a file and then use the
     * appropriate technique for your tool to pass the contents of the file as a parameter.
     * </p>
     * <p>
     * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot both
     * be empty.
     * </p>
     * <p>
     * This parameter is not available using the Secrets Manager console. It can be accessed only by using the AWS CLI
     * or one of the AWS SDKs.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To
     *         use this parameter in the command-line tools, we recommend that you store your binary data in a file and
     *         then use the appropriate technique for your tool to pass the contents of the file as a parameter.</p>
     *         <p>
     *         Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
     *         cannot both be empty.
     *         </p>
     *         <p>
     *         This parameter is not available using the Secrets Manager console. It can be accessed only by using the
     *         AWS CLI or one of the AWS SDKs.
     */
    public ByteBuffer secretBinary() {
        return secretBinary == null ? null : secretBinary.asReadOnlyBuffer();
    }

    /**
     * <p>
     * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
     * </p>
     * <p>
     * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot both
     * be empty.
     * </p>
     * <p>
     * If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected secret text
     * in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a JSON
     * structure of key/value pairs that the Lambda rotation function knows how to parse.
     * </p>
     * <p>
     * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value pairs.
     * For information on how to format a JSON parameter for the various command line tool environments, see <a
     * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON for
     * Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     * </p>
     * <p>
     * <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
     * </p>
     * <p>
     * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes to
     * avoid confusion with the double quotes required in the JSON text.
     * </p>
     * 
     * @return (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.</p>
     *         <p>
     *         Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
     *         cannot both be empty.
     *         </p>
     *         <p>
     *         If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected
     *         secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
     *         information as a JSON structure of key/value pairs that the Lambda rotation function knows how to parse.
     *         </p>
     *         <p>
     *         For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
     *         pairs. For information on how to format a JSON parameter for the various command line tool environments,
     *         see <a
     *         href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
     *         JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     *         </p>
     *         <p>
     *         <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
     *         </p>
     *         <p>
     *         If your command-line tool or SDK requires quotation marks around the parameter, you should use single
     *         quotes to avoid confusion with the double quotes required in the JSON text.
     */
    public String secretString() {
        return secretString;
    }

    /**
     * <p>
     * (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key" and "Value"
     * pair of strings. This operation only appends tags to the existing list of tags. To remove tags, you must use
     * <a>UntagResource</a>.
     * </p>
     * <important>
     * <ul>
     * <li>
     * <p>
     * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with key
     * "abc".
     * </p>
     * </li>
     * <li>
     * <p>
     * If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy, then adding or
     * removing a tag can change permissions. If the successful completion of this operation would result in you losing
     * your permissions for this secret, then this operation is blocked and returns an <code>Access Denied</code> error.
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * This parameter requires a JSON text string argument. For information on how to format a JSON parameter for the
     * various command line tool environments, see <a
     * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON for
     * Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     * </p>
     * <p>
     * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
     * </p>
     * <p>
     * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes to
     * avoid confusion with the double quotes required in the JSON text.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per secret—50
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length—127 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length—255 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS use. You can't
     * edit or delete tag names or values with this prefix. Tags with this prefix do not count against your tags per
     * secret limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema will be used across multiple services and resources, remember that other services might
     * have restrictions on allowed characters. Generally allowed characters are: letters, spaces, and numbers
     * representable in UTF-8, plus the following special characters: + - = . _ : / @.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key" and
     *         "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove tags,
     *         you must use <a>UntagResource</a>.</p> <important>
     *         <ul>
     *         <li>
     *         <p>
     *         Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
     *         with key "abc".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy, then
     *         adding or removing a tag can change permissions. If the successful completion of this operation would
     *         result in you losing your permissions for this secret, then this operation is blocked and returns an
     *         <code>Access Denied</code> error.
     *         </p>
     *         </li>
     *         </ul>
     *         </important>
     *         <p>
     *         This parameter requires a JSON text string argument. For information on how to format a JSON parameter
     *         for the various command line tool environments, see <a
     *         href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
     *         JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     *         </p>
     *         <p>
     *         <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
     *         </p>
     *         <p>
     *         If your command-line tool or SDK requires quotation marks around the parameter, you should use single
     *         quotes to avoid confusion with the double quotes required in the JSON text.
     *         </p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per secret—50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length—127 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length—255 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS use.
     *         You can't edit or delete tag names or values with this prefix. Tags with this prefix do not count against
     *         your tags per secret limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema will be used across multiple services and resources, remember that other services
     *         might have restrictions on allowed characters. Generally allowed characters are: letters, spaces, and
     *         numbers representable in UTF-8, plus the following special characters: + - = . _ : / @.
     *         </p>
     *         </li>
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSecretRequest)) {
            return false;
        }
        CreateSecretRequest other = (CreateSecretRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(description(), other.description()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(secretBinary(), other.secretBinary()) && Objects.equals(secretString(), other.secretString())
                && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateSecretRequest").add("Name", name()).add("ClientRequestToken", clientRequestToken())
                .add("Description", description()).add("KmsKeyId", kmsKeyId()).add("SecretBinary", secretBinary())
                .add("SecretString", secretString()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "ClientRequestToken":
            return Optional.of(clazz.cast(clientRequestToken()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "KmsKeyId":
            return Optional.of(clazz.cast(kmsKeyId()));
        case "SecretBinary":
            return Optional.of(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.of(clazz.cast(secretString()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SecretsManagerRequest.Builder, CopyableBuilder<Builder, CreateSecretRequest> {
        /**
         * <p>
         * Specifies the friendly name of the new secret.
         * </p>
         * 
         * @param name
         *        Specifies the friendly name of the new secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * (Optional) If you include <code>SecretString</code> or <code>SecretBinary</code>, then an initial version is
         * created as part of the secret, and this parameter specifies a unique identifier for the new version.
         * </p>
         * <note>
         * <p>
         * If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter empty.
         * The CLI or SDK generates a random UUID for you and includes as the value for this parameter in the request.
         * If you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager service endpoint,
         * then you must generate a <code>ClientRequestToken</code> yourself for the new version and include that value
         * in the request.
         * </p>
         * </note>
         * <p>
         * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
         * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
         * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of
         * your versions within the specified secret.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
         * version of the secret is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> and
         * <code>SecretBinary</code> values are the same as those in the request, then the request is ignored (the
         * operation is idempotent).
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> and
         * <code>SecretBinary</code> values are different from those in the request then the request fails because you
         * cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This value becomes the <code>SecretVersionId</code> of the new version.
         * </p>
         * 
         * @param clientRequestToken
         *        (Optional) If you include <code>SecretString</code> or <code>SecretBinary</code>, then an initial
         *        version is created as part of the secret, and this parameter specifies a unique identifier for the new
         *        version. </p> <note>
         *        <p>
         *        If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter
         *        empty. The CLI or SDK generates a random UUID for you and includes as the value for this parameter in
         *        the request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager
         *        service endpoint, then you must generate a <code>ClientRequestToken</code> yourself for the new
         *        version and include that value in the request.
         *        </p>
         *        </note>
         *        <p>
         *        This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental
         *        creation of duplicate versions if there are failures and retries during a rotation. We recommend that
         *        you generate a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value
         *        to ensure uniqueness of your versions within the specified secret.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret
         *        then a new version of the secret is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> and
         *        <code>SecretBinary</code> values are the same as those in the request, then the request is ignored
         *        (the operation is idempotent).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> and
         *        <code>SecretBinary</code> values are different from those in the request then the request fails
         *        because you cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new
         *        version.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This value becomes the <code>SecretVersionId</code> of the new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * (Optional) Specifies a user-provided description of the secret.
         * </p>
         * 
         * @param description
         *        (Optional) Specifies a user-provided description of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * (Optional) Specifies the ARN or alias of the AWS KMS customer master key (CMK) to be used to encrypt the
         * <code>SecretString</code> or <code>SecretBinary</code> values in the versions stored in this secret.
         * </p>
         * <p>
         * If you don't specify this value, then Secrets Manager defaults to using the AWS account's default CMK (the
         * one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't yet exist, then Secrets
         * Manager creates it for you automatically the first time it needs to encrypt a version's
         * <code>SecretString</code> or <code>SecretBinary</code> fields.
         * </p>
         * <important>
         * <p>
         * You can use the account's default CMK to encrypt and decrypt only if you call this operation using
         * credentials from the same account that owns the secret. If the secret is in a different account, then you
         * must create a custom CMK and specify the ARN in this field.
         * </p>
         * </important>
         * 
         * @param kmsKeyId
         *        (Optional) Specifies the ARN or alias of the AWS KMS customer master key (CMK) to be used to encrypt
         *        the <code>SecretString</code> or <code>SecretBinary</code> values in the versions stored in this
         *        secret.</p>
         *        <p>
         *        If you don't specify this value, then Secrets Manager defaults to using the AWS account's default CMK
         *        (the one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't yet exist, then
         *        Secrets Manager creates it for you automatically the first time it needs to encrypt a version's
         *        <code>SecretString</code> or <code>SecretBinary</code> fields.
         *        </p>
         *        <important>
         *        <p>
         *        You can use the account's default CMK to encrypt and decrypt only if you call this operation using
         *        credentials from the same account that owns the secret. If the secret is in a different account, then
         *        you must create a custom CMK and specify the ARN in this field.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use
         * this parameter in the command-line tools, we recommend that you store your binary data in a file and then use
         * the appropriate technique for your tool to pass the contents of the file as a parameter.
         * </p>
         * <p>
         * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot
         * both be empty.
         * </p>
         * <p>
         * This parameter is not available using the Secrets Manager console. It can be accessed only by using the AWS
         * CLI or one of the AWS SDKs.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param secretBinary
         *        (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret.
         *        To use this parameter in the command-line tools, we recommend that you store your binary data in a
         *        file and then use the appropriate technique for your tool to pass the contents of the file as a
         *        parameter.</p>
         *        <p>
         *        Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
         *        cannot both be empty.
         *        </p>
         *        <p>
         *        This parameter is not available using the Secrets Manager console. It can be accessed only by using
         *        the AWS CLI or one of the AWS SDKs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(ByteBuffer secretBinary);

        /**
         * <p>
         * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
         * </p>
         * <p>
         * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot
         * both be empty.
         * </p>
         * <p>
         * If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected secret
         * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
         * JSON structure of key/value pairs that the Lambda rotation function knows how to parse.
         * </p>
         * <p>
         * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
         * pairs. For information on how to format a JSON parameter for the various command line tool environments, see
         * <a href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         * JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         * </p>
         * <p>
         * <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
         * </p>
         * <p>
         * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes
         * to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * 
         * @param secretString
         *        (Optional) Specifies text data that you want to encrypt and store in this new version of the
         *        secret.</p>
         *        <p>
         *        Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
         *        cannot both be empty.
         *        </p>
         *        <p>
         *        If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected
         *        secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
         *        information as a JSON structure of key/value pairs that the Lambda rotation function knows how to
         *        parse.
         *        </p>
         *        <p>
         *        For storing multiple values, we recommend that you use a JSON text string argument and specify
         *        key/value pairs. For information on how to format a JSON parameter for the various command line tool
         *        environments, see <a
         *        href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         *        JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         *        </p>
         *        <p>
         *        <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
         *        </p>
         *        <p>
         *        If your command-line tool or SDK requires quotation marks around the parameter, you should use single
         *        quotes to avoid confusion with the double quotes required in the JSON text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key" and
         * "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove tags, you
         * must use <a>UntagResource</a>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * </li>
         * <li>
         * <p>
         * If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy, then
         * adding or removing a tag can change permissions. If the successful completion of this operation would result
         * in you losing your permissions for this secret, then this operation is blocked and returns an
         * <code>Access Denied</code> error.
         * </p>
         * </li>
         * </ul>
         * </important>
         * <p>
         * This parameter requires a JSON text string argument. For information on how to format a JSON parameter for
         * the various command line tool environments, see <a
         * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes
         * to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret—50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length—127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length—255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS use. You
         * can't edit or delete tag names or values with this prefix. Tags with this prefix do not count against your
         * tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema will be used across multiple services and resources, remember that other services
         * might have restrictions on allowed characters. Generally allowed characters are: letters, spaces, and numbers
         * representable in UTF-8, plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key"
         *        and "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove
         *        tags, you must use <a>UntagResource</a>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
         *        with key "abc".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy,
         *        then adding or removing a tag can change permissions. If the successful completion of this operation
         *        would result in you losing your permissions for this secret, then this operation is blocked and
         *        returns an <code>Access Denied</code> error.
         *        </p>
         *        </li>
         *        </ul>
         *        </important>
         *        <p>
         *        This parameter requires a JSON text string argument. For information on how to format a JSON parameter
         *        for the various command line tool environments, see <a
         *        href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         *        JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         *        </p>
         *        <p>
         *        <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         *        </p>
         *        <p>
         *        If your command-line tool or SDK requires quotation marks around the parameter, you should use single
         *        quotes to avoid confusion with the double quotes required in the JSON text.
         *        </p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per secret—50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length—127 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length—255 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS
         *        use. You can't edit or delete tag names or values with this prefix. Tags with this prefix do not count
         *        against your tags per secret limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema will be used across multiple services and resources, remember that other
         *        services might have restrictions on allowed characters. Generally allowed characters are: letters,
         *        spaces, and numbers representable in UTF-8, plus the following special characters: + - = . _ : / @.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key" and
         * "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove tags, you
         * must use <a>UntagResource</a>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * </li>
         * <li>
         * <p>
         * If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy, then
         * adding or removing a tag can change permissions. If the successful completion of this operation would result
         * in you losing your permissions for this secret, then this operation is blocked and returns an
         * <code>Access Denied</code> error.
         * </p>
         * </li>
         * </ul>
         * </important>
         * <p>
         * This parameter requires a JSON text string argument. For information on how to format a JSON parameter for
         * the various command line tool environments, see <a
         * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes
         * to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret—50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length—127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length—255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS use. You
         * can't edit or delete tag names or values with this prefix. Tags with this prefix do not count against your
         * tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema will be used across multiple services and resources, remember that other services
         * might have restrictions on allowed characters. Generally allowed characters are: letters, spaces, and numbers
         * representable in UTF-8, plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key"
         *        and "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove
         *        tags, you must use <a>UntagResource</a>.</p> <important>
         *        <ul>
         *        <li>
         *        <p>
         *        Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
         *        with key "abc".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy,
         *        then adding or removing a tag can change permissions. If the successful completion of this operation
         *        would result in you losing your permissions for this secret, then this operation is blocked and
         *        returns an <code>Access Denied</code> error.
         *        </p>
         *        </li>
         *        </ul>
         *        </important>
         *        <p>
         *        This parameter requires a JSON text string argument. For information on how to format a JSON parameter
         *        for the various command line tool environments, see <a
         *        href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         *        JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         *        </p>
         *        <p>
         *        <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         *        </p>
         *        <p>
         *        If your command-line tool or SDK requires quotation marks around the parameter, you should use single
         *        quotes to avoid confusion with the double quotes required in the JSON text.
         *        </p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per secret—50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length—127 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length—255 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS
         *        use. You can't edit or delete tag names or values with this prefix. Tags with this prefix do not count
         *        against your tags per secret limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema will be used across multiple services and resources, remember that other
         *        services might have restrictions on allowed characters. Generally allowed characters are: letters,
         *        spaces, and numbers representable in UTF-8, plus the following special characters: + - = . _ : / @.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * (Optional) Specifies a list of user-defined tags that are attached to the secret. Each tag is a "Key" and
         * "Value" pair of strings. This operation only appends tags to the existing list of tags. To remove tags, you
         * must use <a>UntagResource</a>.
         * </p>
         * <important>
         * <ul>
         * <li>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * </li>
         * <li>
         * <p>
         * If you check tags in IAM policy <code>Condition</code> elements as part of your security strategy, then
         * adding or removing a tag can change permissions. If the successful completion of this operation would result
         * in you losing your permissions for this secret, then this operation is blocked and returns an
         * <code>Access Denied</code> error.
         * </p>
         * </li>
         * </ul>
         * </important>
         * <p>
         * This parameter requires a JSON text string argument. For information on how to format a JSON parameter for
         * the various command line tool environments, see <a
         * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes
         * to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret—50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length—127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length—255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because it is reserved for AWS use. You
         * can't edit or delete tag names or values with this prefix. Tags with this prefix do not count against your
         * tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema will be used across multiple services and resources, remember that other services
         * might have restrictions on allowed characters. Generally allowed characters are: letters, spaces, and numbers
         * representable in UTF-8, plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SecretsManagerRequest.BuilderImpl implements Builder {
        private String name;

        private String clientRequestToken;

        private String description;

        private String kmsKeyId;

        private ByteBuffer secretBinary;

        private String secretString;

        private List<Tag> tags;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSecretRequest model) {
            super(model);
            name(model.name);
            clientRequestToken(model.clientRequestToken);
            description(model.description);
            kmsKeyId(model.kmsKeyId);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary;
        }

        @Override
        public final Builder secretBinary(ByteBuffer secretBinary) {
            this.secretBinary = StandardMemberCopier.copy(secretBinary);
            return this;
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            this.secretBinary = StandardMemberCopier.copy(secretBinary);
        }

        public final String getSecretString() {
            return secretString;
        }

        @Override
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListTypeCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListTypeCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public CreateSecretRequest build() {
            return new CreateSecretRequest(this);
        }
    }
}
