/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.secretsmanager.transform.SecretVersionsListEntryMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains information about one version of a secret.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SecretVersionsListEntry implements StructuredPojo,
        ToCopyableBuilder<SecretVersionsListEntry.Builder, SecretVersionsListEntry> {
    private final String versionId;

    private final List<String> versionStages;

    private final Instant lastAccessedDate;

    private final Instant createdDate;

    private SecretVersionsListEntry(BuilderImpl builder) {
        this.versionId = builder.versionId;
        this.versionStages = builder.versionStages;
        this.lastAccessedDate = builder.lastAccessedDate;
        this.createdDate = builder.createdDate;
    }

    /**
     * <p>
     * The unique version identifier of this version of the secret.
     * </p>
     * 
     * @return The unique version identifier of this version of the secret.
     */
    public String versionId() {
        return versionId;
    }

    /**
     * <p>
     * An array of staging labels that are currently associated with this version of the secret.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of staging labels that are currently associated with this version of the secret.
     */
    public List<String> versionStages() {
        return versionStages;
    }

    /**
     * <p>
     * The date that this version of the secret was last accessed. Note that the resolution of this field is at the date
     * level and does not include the time.
     * </p>
     * 
     * @return The date that this version of the secret was last accessed. Note that the resolution of this field is at
     *         the date level and does not include the time.
     */
    public Instant lastAccessedDate() {
        return lastAccessedDate;
    }

    /**
     * <p>
     * The date and time this version of the secret was created.
     * </p>
     * 
     * @return The date and time this version of the secret was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(versionStages());
        hashCode = 31 * hashCode + Objects.hashCode(lastAccessedDate());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecretVersionsListEntry)) {
            return false;
        }
        SecretVersionsListEntry other = (SecretVersionsListEntry) obj;
        return Objects.equals(versionId(), other.versionId()) && Objects.equals(versionStages(), other.versionStages())
                && Objects.equals(lastAccessedDate(), other.lastAccessedDate())
                && Objects.equals(createdDate(), other.createdDate());
    }

    @Override
    public String toString() {
        return ToString.builder("SecretVersionsListEntry").add("VersionId", versionId()).add("VersionStages", versionStages())
                .add("LastAccessedDate", lastAccessedDate()).add("CreatedDate", createdDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VersionId":
            return Optional.of(clazz.cast(versionId()));
        case "VersionStages":
            return Optional.of(clazz.cast(versionStages()));
        case "LastAccessedDate":
            return Optional.of(clazz.cast(lastAccessedDate()));
        case "CreatedDate":
            return Optional.of(clazz.cast(createdDate()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SecretVersionsListEntryMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, SecretVersionsListEntry> {
        /**
         * <p>
         * The unique version identifier of this version of the secret.
         * </p>
         * 
         * @param versionId
         *        The unique version identifier of this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * An array of staging labels that are currently associated with this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        An array of staging labels that are currently associated with this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * An array of staging labels that are currently associated with this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        An array of staging labels that are currently associated with this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        /**
         * <p>
         * The date that this version of the secret was last accessed. Note that the resolution of this field is at the
         * date level and does not include the time.
         * </p>
         * 
         * @param lastAccessedDate
         *        The date that this version of the secret was last accessed. Note that the resolution of this field is
         *        at the date level and does not include the time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAccessedDate(Instant lastAccessedDate);

        /**
         * <p>
         * The date and time this version of the secret was created.
         * </p>
         * 
         * @param createdDate
         *        The date and time this version of the secret was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);
    }

    static final class BuilderImpl implements Builder {
        private String versionId;

        private List<String> versionStages;

        private Instant lastAccessedDate;

        private Instant createdDate;

        private BuilderImpl() {
        }

        private BuilderImpl(SecretVersionsListEntry model) {
            versionId(model.versionId);
            versionStages(model.versionStages);
            lastAccessedDate(model.lastAccessedDate);
            createdDate(model.createdDate);
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        public final Collection<String> getVersionStages() {
            return versionStages;
        }

        @Override
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        public final Instant getLastAccessedDate() {
            return lastAccessedDate;
        }

        @Override
        public final Builder lastAccessedDate(Instant lastAccessedDate) {
            this.lastAccessedDate = lastAccessedDate;
            return this;
        }

        public final void setLastAccessedDate(Instant lastAccessedDate) {
            this.lastAccessedDate = lastAccessedDate;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public SecretVersionsListEntry build() {
            return new SecretVersionsListEntry(this);
        }
    }
}
