/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.core.runtime.StandardMemberCopier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class UpdateSecretRequest extends SecretsManagerRequest implements
        ToCopyableBuilder<UpdateSecretRequest.Builder, UpdateSecretRequest> {
    private final String secretId;

    private final String clientRequestToken;

    private final String description;

    private final String kmsKeyId;

    private final ByteBuffer secretBinary;

    private final String secretString;

    private UpdateSecretRequest(BuilderImpl builder) {
        super(builder);
        this.secretId = builder.secretId;
        this.clientRequestToken = builder.clientRequestToken;
        this.description = builder.description;
        this.kmsKeyId = builder.kmsKeyId;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
    }

    /**
     * <p>
     * Specifies the secret that you want to update or to which you want to add a new version. You can specify either
     * the Amazon Resource Name (ARN) or the friendly name of the secret.
     * </p>
     * 
     * @return Specifies the secret that you want to update or to which you want to add a new version. You can specify
     *         either the Amazon Resource Name (ARN) or the friendly name of the secret.
     */
    public String secretId() {
        return secretId;
    }

    /**
     * <p>
     * (Optional) If you want to add a new version to the secret, this parameter specifies a unique identifier for the
     * new version that helps ensure idempotency.
     * </p>
     * <p>
     * If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter empty. The
     * CLI or SDK generates a random UUID for you and includes that in the request. If you don't use the SDK and instead
     * generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
     * <code>ClientRequestToken</code> yourself for new versions and include that value in the request.
     * </p>
     * <p>
     * You typically only need to interact with this value if you implement your own retry logic and want to ensure that
     * a given secret is not created twice. We recommend that you generate a <a
     * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness within
     * the specified secret.
     * </p>
     * <p>
     * Secrets Manager uses this value to prevent the accidental creation of duplicate versions if there are failures
     * and retries during the Lambda rotation function's processing.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
     * version of the secret is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> and
     * <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the operation
     * is idempotent).
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> and
     * <code>SecretBinary</code> values are different from the request then an error occurs because you cannot modify an
     * existing secret value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This value becomes the <code>SecretVersionId</code> of the new version.
     * </p>
     * 
     * @return (Optional) If you want to add a new version to the secret, this parameter specifies a unique identifier
     *         for the new version that helps ensure idempotency. </p>
     *         <p>
     *         If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter
     *         empty. The CLI or SDK generates a random UUID for you and includes that in the request. If you don't use
     *         the SDK and instead generate a raw HTTP request to the Secrets Manager service endpoint, then you must
     *         generate a <code>ClientRequestToken</code> yourself for new versions and include that value in the
     *         request.
     *         </p>
     *         <p>
     *         You typically only need to interact with this value if you implement your own retry logic and want to
     *         ensure that a given secret is not created twice. We recommend that you generate a <a
     *         href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness
     *         within the specified secret.
     *         </p>
     *         <p>
     *         Secrets Manager uses this value to prevent the accidental creation of duplicate versions if there are
     *         failures and retries during the Lambda rotation function's processing.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a
     *         new version of the secret is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> and
     *         <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
     *         operation is idempotent).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> and
     *         <code>SecretBinary</code> values are different from the request then an error occurs because you cannot
     *         modify an existing secret value.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This value becomes the <code>SecretVersionId</code> of the new version.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * (Optional) Specifies a user-provided description of the secret.
     * </p>
     * 
     * @return (Optional) Specifies a user-provided description of the secret.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * (Optional) Specifies the ARN or alias of the KMS customer master key (CMK) to be used to encrypt the protected
     * text in the versions of this secret.
     * </p>
     * <p>
     * If you don't specify this value, then Secrets Manager defaults to using the default CMK in the account (the one
     * named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't exist, then Secrets Manager creates
     * it for you automatically the first time it needs to encrypt a version's <code>Plaintext</code> or
     * <code>PlaintextString</code> fields.
     * </p>
     * <important>
     * <p>
     * You can only use the account's default CMK to encrypt and decrypt if you call this operation using credentials
     * from the same account that owns the secret. If the secret is in a different account, then you must create a
     * custom CMK and provide the ARN in this field.
     * </p>
     * </important>
     * 
     * @return (Optional) Specifies the ARN or alias of the KMS customer master key (CMK) to be used to encrypt the
     *         protected text in the versions of this secret.</p>
     *         <p>
     *         If you don't specify this value, then Secrets Manager defaults to using the default CMK in the account
     *         (the one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't exist, then Secrets
     *         Manager creates it for you automatically the first time it needs to encrypt a version's
     *         <code>Plaintext</code> or <code>PlaintextString</code> fields.
     *         </p>
     *         <important>
     *         <p>
     *         You can only use the account's default CMK to encrypt and decrypt if you call this operation using
     *         credentials from the same account that owns the secret. If the secret is in a different account, then you
     *         must create a custom CMK and provide the ARN in this field.
     *         </p>
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use this
     * parameter in the command-line tools, we recommend that you store your binary data in a file and then use the
     * appropriate technique for your tool to pass the contents of the file as a parameter. Either
     * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
     * empty.
     * </p>
     * <p>
     * This parameter is not accessible using the Secrets Manager console.
     * </p>
     * <p>
     * This method will return a new read-only {@code ByteBuffer} each time it is invoked.
     * </p>
     * 
     * @return (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To
     *         use this parameter in the command-line tools, we recommend that you store your binary data in a file and
     *         then use the appropriate technique for your tool to pass the contents of the file as a parameter. Either
     *         <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both
     *         be empty.</p>
     *         <p>
     *         This parameter is not accessible using the Secrets Manager console.
     */
    public ByteBuffer secretBinary() {
        return secretBinary == null ? null : secretBinary.asReadOnlyBuffer();
    }

    /**
     * <p>
     * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret. Either
     * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
     * empty.
     * </p>
     * <p>
     * If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected secret
     * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
     * JSON structure of key/value pairs that the default Lambda rotation function knows how to parse.
     * </p>
     * <p>
     * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value pairs.
     * For information on how to format a JSON parameter for the various command line tool environments, see <a
     * href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON for
     * Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     * </p>
     * <p>
     * <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
     * </p>
     * <p>
     * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes to
     * avoid confusion with the double quotes required in the JSON text.
     * </p>
     * 
     * @return (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
     *         Either <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They
     *         cannot both be empty.</p>
     *         <p>
     *         If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected
     *         secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
     *         information as a JSON structure of key/value pairs that the default Lambda rotation function knows how to
     *         parse.
     *         </p>
     *         <p>
     *         For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
     *         pairs. For information on how to format a JSON parameter for the various command line tool environments,
     *         see <a
     *         href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
     *         JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
     *         </p>
     *         <p>
     *         <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
     *         </p>
     *         <p>
     *         If your command-line tool or SDK requires quotation marks around the parameter, you should use single
     *         quotes to avoid confusion with the double quotes required in the JSON text.
     */
    public String secretString() {
        return secretString;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(secretId());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateSecretRequest)) {
            return false;
        }
        UpdateSecretRequest other = (UpdateSecretRequest) obj;
        return Objects.equals(secretId(), other.secretId()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(description(), other.description()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(secretBinary(), other.secretBinary()) && Objects.equals(secretString(), other.secretString());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateSecretRequest").add("SecretId", secretId())
                .add("ClientRequestToken", clientRequestToken()).add("Description", description()).add("KmsKeyId", kmsKeyId())
                .add("SecretBinary", secretBinary()).add("SecretString", secretString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SecretId":
            return Optional.of(clazz.cast(secretId()));
        case "ClientRequestToken":
            return Optional.of(clazz.cast(clientRequestToken()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "KmsKeyId":
            return Optional.of(clazz.cast(kmsKeyId()));
        case "SecretBinary":
            return Optional.of(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.of(clazz.cast(secretString()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SecretsManagerRequest.Builder, CopyableBuilder<Builder, UpdateSecretRequest> {
        /**
         * <p>
         * Specifies the secret that you want to update or to which you want to add a new version. You can specify
         * either the Amazon Resource Name (ARN) or the friendly name of the secret.
         * </p>
         * 
         * @param secretId
         *        Specifies the secret that you want to update or to which you want to add a new version. You can
         *        specify either the Amazon Resource Name (ARN) or the friendly name of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretId(String secretId);

        /**
         * <p>
         * (Optional) If you want to add a new version to the secret, this parameter specifies a unique identifier for
         * the new version that helps ensure idempotency.
         * </p>
         * <p>
         * If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter empty.
         * The CLI or SDK generates a random UUID for you and includes that in the request. If you don't use the SDK and
         * instead generate a raw HTTP request to the Secrets Manager service endpoint, then you must generate a
         * <code>ClientRequestToken</code> yourself for new versions and include that value in the request.
         * </p>
         * <p>
         * You typically only need to interact with this value if you implement your own retry logic and want to ensure
         * that a given secret is not created twice. We recommend that you generate a <a
         * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness
         * within the specified secret.
         * </p>
         * <p>
         * Secrets Manager uses this value to prevent the accidental creation of duplicate versions if there are
         * failures and retries during the Lambda rotation function's processing.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
         * version of the secret is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> and
         * <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
         * operation is idempotent).
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> and
         * <code>SecretBinary</code> values are different from the request then an error occurs because you cannot
         * modify an existing secret value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This value becomes the <code>SecretVersionId</code> of the new version.
         * </p>
         * 
         * @param clientRequestToken
         *        (Optional) If you want to add a new version to the secret, this parameter specifies a unique
         *        identifier for the new version that helps ensure idempotency. </p>
         *        <p>
         *        If you use the AWS CLI or one of the AWS SDK to call this operation, then you can leave this parameter
         *        empty. The CLI or SDK generates a random UUID for you and includes that in the request. If you don't
         *        use the SDK and instead generate a raw HTTP request to the Secrets Manager service endpoint, then you
         *        must generate a <code>ClientRequestToken</code> yourself for new versions and include that value in
         *        the request.
         *        </p>
         *        <p>
         *        You typically only need to interact with this value if you implement your own retry logic and want to
         *        ensure that a given secret is not created twice. We recommend that you generate a <a
         *        href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure
         *        uniqueness within the specified secret.
         *        </p>
         *        <p>
         *        Secrets Manager uses this value to prevent the accidental creation of duplicate versions if there are
         *        failures and retries during the Lambda rotation function's processing.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret
         *        then a new version of the secret is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> and
         *        <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
         *        operation is idempotent).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> and
         *        <code>SecretBinary</code> values are different from the request then an error occurs because you
         *        cannot modify an existing secret value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This value becomes the <code>SecretVersionId</code> of the new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * (Optional) Specifies a user-provided description of the secret.
         * </p>
         * 
         * @param description
         *        (Optional) Specifies a user-provided description of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * (Optional) Specifies the ARN or alias of the KMS customer master key (CMK) to be used to encrypt the
         * protected text in the versions of this secret.
         * </p>
         * <p>
         * If you don't specify this value, then Secrets Manager defaults to using the default CMK in the account (the
         * one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't exist, then Secrets Manager
         * creates it for you automatically the first time it needs to encrypt a version's <code>Plaintext</code> or
         * <code>PlaintextString</code> fields.
         * </p>
         * <important>
         * <p>
         * You can only use the account's default CMK to encrypt and decrypt if you call this operation using
         * credentials from the same account that owns the secret. If the secret is in a different account, then you
         * must create a custom CMK and provide the ARN in this field.
         * </p>
         * </important>
         * 
         * @param kmsKeyId
         *        (Optional) Specifies the ARN or alias of the KMS customer master key (CMK) to be used to encrypt the
         *        protected text in the versions of this secret.</p>
         *        <p>
         *        If you don't specify this value, then Secrets Manager defaults to using the default CMK in the account
         *        (the one named <code>aws/secretsmanager</code>). If a KMS CMK with that name doesn't exist, then
         *        Secrets Manager creates it for you automatically the first time it needs to encrypt a version's
         *        <code>Plaintext</code> or <code>PlaintextString</code> fields.
         *        </p>
         *        <important>
         *        <p>
         *        You can only use the account's default CMK to encrypt and decrypt if you call this operation using
         *        credentials from the same account that owns the secret. If the secret is in a different account, then
         *        you must create a custom CMK and provide the ARN in this field.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use
         * this parameter in the command-line tools, we recommend that you store your binary data in a file and then use
         * the appropriate technique for your tool to pass the contents of the file as a parameter. Either
         * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
         * empty.
         * </p>
         * <p>
         * This parameter is not accessible using the Secrets Manager console.
         * </p>
         * <p>
         * To preserve immutability, the remaining bytes in the provided buffer will be copied into a new buffer when
         * set.
         * </p>
         *
         * @param secretBinary
         *        (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret.
         *        To use this parameter in the command-line tools, we recommend that you store your binary data in a
         *        file and then use the appropriate technique for your tool to pass the contents of the file as a
         *        parameter. Either <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not
         *        both. They cannot both be empty.</p>
         *        <p>
         *        This parameter is not accessible using the Secrets Manager console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(ByteBuffer secretBinary);

        /**
         * <p>
         * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret. Either
         * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
         * empty.
         * </p>
         * <p>
         * If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected secret
         * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
         * JSON structure of key/value pairs that the default Lambda rotation function knows how to parse.
         * </p>
         * <p>
         * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
         * pairs. For information on how to format a JSON parameter for the various command line tool environments, see
         * <a href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         * JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         * </p>
         * <p>
         * <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
         * </p>
         * <p>
         * If your command-line tool or SDK requires quotation marks around the parameter, you should use single quotes
         * to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * 
         * @param secretString
         *        (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
         *        Either <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They
         *        cannot both be empty.</p>
         *        <p>
         *        If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected
         *        secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
         *        information as a JSON structure of key/value pairs that the default Lambda rotation function knows how
         *        to parse.
         *        </p>
         *        <p>
         *        For storing multiple values, we recommend that you use a JSON text string argument and specify
         *        key/value pairs. For information on how to format a JSON parameter for the various command line tool
         *        environments, see <a
         *        href="http://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
         *        JSON for Parameters</a> in the <i>AWS CLI User Guide</i>. For example:
         *        </p>
         *        <p>
         *        <code>[{"username":"bob"},{"password":"abc123xyz456"}]</code>
         *        </p>
         *        <p>
         *        If your command-line tool or SDK requires quotation marks around the parameter, you should use single
         *        quotes to avoid confusion with the double quotes required in the JSON text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SecretsManagerRequest.BuilderImpl implements Builder {
        private String secretId;

        private String clientRequestToken;

        private String description;

        private String kmsKeyId;

        private ByteBuffer secretBinary;

        private String secretString;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateSecretRequest model) {
            super(model);
            secretId(model.secretId);
            clientRequestToken(model.clientRequestToken);
            description(model.description);
            kmsKeyId(model.kmsKeyId);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
        }

        public final String getSecretId() {
            return secretId;
        }

        @Override
        public final Builder secretId(String secretId) {
            this.secretId = secretId;
            return this;
        }

        public final void setSecretId(String secretId) {
            this.secretId = secretId;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary;
        }

        @Override
        public final Builder secretBinary(ByteBuffer secretBinary) {
            this.secretBinary = StandardMemberCopier.copy(secretBinary);
            return this;
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            this.secretBinary = StandardMemberCopier.copy(secretBinary);
        }

        public final String getSecretString() {
            return secretString;
        }

        @Override
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public UpdateSecretRequest build() {
            return new UpdateSecretRequest(this);
        }
    }
}
