/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetSecretValueResponse extends SecretsManagerResponse implements
        ToCopyableBuilder<GetSecretValueResponse.Builder, GetSecretValueResponse> {
    private final String arn;

    private final String name;

    private final String versionId;

    private final SdkBytes secretBinary;

    private final String secretString;

    private final List<String> versionStages;

    private final Instant createdDate;

    private GetSecretValueResponse(BuilderImpl builder) {
        super(builder);
        this.arn = builder.arn;
        this.name = builder.name;
        this.versionId = builder.versionId;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.versionStages = builder.versionStages;
        this.createdDate = builder.createdDate;
    }

    /**
     * <p>
     * The ARN of the secret.
     * </p>
     * 
     * @return The ARN of the secret.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The friendly name of the secret.
     * </p>
     * 
     * @return The friendly name of the secret.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier of this version of the secret.
     * </p>
     * 
     * @return The unique identifier of this version of the secret.
     */
    public String versionId() {
        return versionId;
    }

    /**
     * <p>
     * The decrypted part of the protected secret information that was originally provided as binary data in the form of
     * a byte array. The response parameter represents the binary data as a <a
     * href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
     * </p>
     * <p>
     * This parameter is not used if the secret is created by the Secrets Manager console.
     * </p>
     * <p>
     * If you store custom information in this field of the secret, then you must code your Lambda rotation function to
     * parse and interpret whatever you store in the <code>SecretString</code> or <code>SecretBinary</code> fields.
     * </p>
     * 
     * @return The decrypted part of the protected secret information that was originally provided as binary data in the
     *         form of a byte array. The response parameter represents the binary data as a <a
     *         href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.</p>
     *         <p>
     *         This parameter is not used if the secret is created by the Secrets Manager console.
     *         </p>
     *         <p>
     *         If you store custom information in this field of the secret, then you must code your Lambda rotation
     *         function to parse and interpret whatever you store in the <code>SecretString</code> or
     *         <code>SecretBinary</code> fields.
     */
    public SdkBytes secretBinary() {
        return secretBinary;
    }

    /**
     * <p>
     * The decrypted part of the protected secret information that was originally provided as a string.
     * </p>
     * <p>
     * If you create this secret by using the Secrets Manager console then only the <code>SecretString</code> parameter
     * contains data. Secrets Manager stores the information as a JSON structure of key/value pairs that the Lambda
     * rotation function knows how to parse.
     * </p>
     * <p>
     * If you store custom information in the secret by using the <a>CreateSecret</a>, <a>UpdateSecret</a>, or
     * <a>PutSecretValue</a> API operations instead of the Secrets Manager console, or by using the <b>Other secret
     * type</b> in the console, then you must code your Lambda rotation function to parse and interpret those values.
     * </p>
     * 
     * @return The decrypted part of the protected secret information that was originally provided as a string.</p>
     *         <p>
     *         If you create this secret by using the Secrets Manager console then only the <code>SecretString</code>
     *         parameter contains data. Secrets Manager stores the information as a JSON structure of key/value pairs
     *         that the Lambda rotation function knows how to parse.
     *         </p>
     *         <p>
     *         If you store custom information in the secret by using the <a>CreateSecret</a>, <a>UpdateSecret</a>, or
     *         <a>PutSecretValue</a> API operations instead of the Secrets Manager console, or by using the <b>Other
     *         secret type</b> in the console, then you must code your Lambda rotation function to parse and interpret
     *         those values.
     */
    public String secretString() {
        return secretString;
    }

    /**
     * <p>
     * A list of all of the staging labels currently attached to this version of the secret.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of all of the staging labels currently attached to this version of the secret.
     */
    public List<String> versionStages() {
        return versionStages;
    }

    /**
     * <p>
     * The date and time that this version of the secret was created.
     * </p>
     * 
     * @return The date and time that this version of the secret was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(versionStages());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetSecretValueResponse)) {
            return false;
        }
        GetSecretValueResponse other = (GetSecretValueResponse) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(versionId(), other.versionId()) && Objects.equals(secretBinary(), other.secretBinary())
                && Objects.equals(secretString(), other.secretString()) && Objects.equals(versionStages(), other.versionStages())
                && Objects.equals(createdDate(), other.createdDate());
    }

    @Override
    public String toString() {
        return ToString.builder("GetSecretValueResponse").add("ARN", arn()).add("Name", name()).add("VersionId", versionId())
                .add("SecretBinary", secretBinary()).add("SecretString", secretString()).add("VersionStages", versionStages())
                .add("CreatedDate", createdDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "SecretBinary":
            return Optional.ofNullable(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.ofNullable(clazz.cast(secretString()));
        case "VersionStages":
            return Optional.ofNullable(clazz.cast(versionStages()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SecretsManagerResponse.Builder, CopyableBuilder<Builder, GetSecretValueResponse> {
        /**
         * <p>
         * The ARN of the secret.
         * </p>
         * 
         * @param arn
         *        The ARN of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The friendly name of the secret.
         * </p>
         * 
         * @param name
         *        The friendly name of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier of this version of the secret.
         * </p>
         * 
         * @param versionId
         *        The unique identifier of this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * The decrypted part of the protected secret information that was originally provided as binary data in the
         * form of a byte array. The response parameter represents the binary data as a <a
         * href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.
         * </p>
         * <p>
         * This parameter is not used if the secret is created by the Secrets Manager console.
         * </p>
         * <p>
         * If you store custom information in this field of the secret, then you must code your Lambda rotation function
         * to parse and interpret whatever you store in the <code>SecretString</code> or <code>SecretBinary</code>
         * fields.
         * </p>
         * 
         * @param secretBinary
         *        The decrypted part of the protected secret information that was originally provided as binary data in
         *        the form of a byte array. The response parameter represents the binary data as a <a
         *        href="https://tools.ietf.org/html/rfc4648#section-4">base64-encoded</a> string.</p>
         *        <p>
         *        This parameter is not used if the secret is created by the Secrets Manager console.
         *        </p>
         *        <p>
         *        If you store custom information in this field of the secret, then you must code your Lambda rotation
         *        function to parse and interpret whatever you store in the <code>SecretString</code> or
         *        <code>SecretBinary</code> fields.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(SdkBytes secretBinary);

        /**
         * <p>
         * The decrypted part of the protected secret information that was originally provided as a string.
         * </p>
         * <p>
         * If you create this secret by using the Secrets Manager console then only the <code>SecretString</code>
         * parameter contains data. Secrets Manager stores the information as a JSON structure of key/value pairs that
         * the Lambda rotation function knows how to parse.
         * </p>
         * <p>
         * If you store custom information in the secret by using the <a>CreateSecret</a>, <a>UpdateSecret</a>, or
         * <a>PutSecretValue</a> API operations instead of the Secrets Manager console, or by using the <b>Other secret
         * type</b> in the console, then you must code your Lambda rotation function to parse and interpret those
         * values.
         * </p>
         * 
         * @param secretString
         *        The decrypted part of the protected secret information that was originally provided as a string.</p>
         *        <p>
         *        If you create this secret by using the Secrets Manager console then only the <code>SecretString</code>
         *        parameter contains data. Secrets Manager stores the information as a JSON structure of key/value pairs
         *        that the Lambda rotation function knows how to parse.
         *        </p>
         *        <p>
         *        If you store custom information in the secret by using the <a>CreateSecret</a>, <a>UpdateSecret</a>,
         *        or <a>PutSecretValue</a> API operations instead of the Secrets Manager console, or by using the
         *        <b>Other secret type</b> in the console, then you must code your Lambda rotation function to parse and
         *        interpret those values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * A list of all of the staging labels currently attached to this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        A list of all of the staging labels currently attached to this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * A list of all of the staging labels currently attached to this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        A list of all of the staging labels currently attached to this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        /**
         * <p>
         * The date and time that this version of the secret was created.
         * </p>
         * 
         * @param createdDate
         *        The date and time that this version of the secret was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);
    }

    static final class BuilderImpl extends SecretsManagerResponse.BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String versionId;

        private SdkBytes secretBinary;

        private String secretString;

        private List<String> versionStages = DefaultSdkAutoConstructList.getInstance();

        private Instant createdDate;

        private BuilderImpl() {
        }

        private BuilderImpl(GetSecretValueResponse model) {
            super(model);
            arn(model.arn);
            name(model.name);
            versionId(model.versionId);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            versionStages(model.versionStages);
            createdDate(model.createdDate);
        }

        public final String getARN() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setARN(String arn) {
            this.arn = arn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary == null ? null : secretBinary.asByteBuffer();
        }

        @Override
        public final Builder secretBinary(SdkBytes secretBinary) {
            this.secretBinary = StandardMemberCopier.copy(secretBinary);
            return this;
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            secretBinary(secretBinary == null ? null : SdkBytes.fromByteBuffer(secretBinary));
        }

        public final String getSecretString() {
            return secretString;
        }

        @Override
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        public final Collection<String> getVersionStages() {
            return versionStages;
        }

        @Override
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public GetSecretValueResponse build() {
            return new GetSecretValueResponse(this);
        }
    }
}
