/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.beans.Transient;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSecretRequest extends SecretsManagerRequest implements
        ToCopyableBuilder<CreateSecretRequest.Builder, CreateSecretRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateSecretRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(CreateSecretRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateSecretRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CreateSecretRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<SdkBytes> SECRET_BINARY_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("SecretBinary").getter(getter(CreateSecretRequest::secretBinary)).setter(setter(Builder::secretBinary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretBinary").build()).build();

    private static final SdkField<String> SECRET_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretString").getter(getter(CreateSecretRequest::secretString)).setter(setter(Builder::secretString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretString").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateSecretRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ReplicaRegionType>> ADD_REPLICA_REGIONS_FIELD = SdkField
            .<List<ReplicaRegionType>> builder(MarshallingType.LIST)
            .memberName("AddReplicaRegions")
            .getter(getter(CreateSecretRequest::addReplicaRegions))
            .setter(setter(Builder::addReplicaRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddReplicaRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReplicaRegionType> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReplicaRegionType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> FORCE_OVERWRITE_REPLICA_SECRET_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ForceOverwriteReplicaSecret")
            .getter(getter(CreateSecretRequest::forceOverwriteReplicaSecret))
            .setter(setter(Builder::forceOverwriteReplicaSecret))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForceOverwriteReplicaSecret")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, DESCRIPTION_FIELD, KMS_KEY_ID_FIELD, SECRET_BINARY_FIELD, SECRET_STRING_FIELD,
            TAGS_FIELD, ADD_REPLICA_REGIONS_FIELD, FORCE_OVERWRITE_REPLICA_SECRET_FIELD));

    private final String name;

    private final String clientRequestToken;

    private final String description;

    private final String kmsKeyId;

    private final SdkBytes secretBinary;

    private final String secretString;

    private final List<Tag> tags;

    private final List<ReplicaRegionType> addReplicaRegions;

    private final Boolean forceOverwriteReplicaSecret;

    private CreateSecretRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.clientRequestToken = builder.clientRequestToken;
        this.description = builder.description;
        this.kmsKeyId = builder.kmsKeyId;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.tags = builder.tags;
        this.addReplicaRegions = builder.addReplicaRegions;
        this.forceOverwriteReplicaSecret = builder.forceOverwriteReplicaSecret;
    }

    /**
     * <p>
     * The name of the new secret.
     * </p>
     * <p>
     * The secret name can contain ASCII letters, numbers, and the following characters: /_+=.@-
     * </p>
     * <p>
     * Do not end your secret name with a hyphen followed by six characters. If you do so, you risk confusion and
     * unexpected results when searching for a secret by partial ARN. Secrets Manager automatically adds a hyphen and
     * six random characters after the secret name at the end of the ARN.
     * </p>
     * 
     * @return The name of the new secret.</p>
     *         <p>
     *         The secret name can contain ASCII letters, numbers, and the following characters: /_+=.@-
     *         </p>
     *         <p>
     *         Do not end your secret name with a hyphen followed by six characters. If you do so, you risk confusion
     *         and unexpected results when searching for a secret by partial ARN. Secrets Manager automatically adds a
     *         hyphen and six random characters after the secret name at the end of the ARN.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * If you include <code>SecretString</code> or <code>SecretBinary</code>, then Secrets Manager creates an initial
     * version for the secret, and this parameter specifies the unique identifier for the new version.
     * </p>
     * <note>
     * <p>
     * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation, then you
     * can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it as the value for
     * this parameter in the request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets
     * Manager service endpoint, then you must generate a <code>ClientRequestToken</code> yourself for the new version
     * and include the value in the request.
     * </p>
     * </note>
     * <p>
     * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
     * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
     * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of your
     * versions within the specified secret.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
     * version of the secret is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and the version <code>SecretString</code> and
     * <code>SecretBinary</code> values are the same as those in the request, then the request is ignored.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> and
     * <code>SecretBinary</code> values are different from those in the request, then the request fails because you
     * cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This value becomes the <code>VersionId</code> of the new version.
     * </p>
     * 
     * @return If you include <code>SecretString</code> or <code>SecretBinary</code>, then Secrets Manager creates an
     *         initial version for the secret, and this parameter specifies the unique identifier for the new version.
     *         </p> <note>
     *         <p>
     *         If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation,
     *         then you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it
     *         as the value for this parameter in the request. If you don't use the SDK and instead generate a raw HTTP
     *         request to the Secrets Manager service endpoint, then you must generate a <code>ClientRequestToken</code>
     *         yourself for the new version and include the value in the request.
     *         </p>
     *         </note>
     *         <p>
     *         This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation
     *         of duplicate versions if there are failures and retries during a rotation. We recommend that you generate
     *         a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure
     *         uniqueness of your versions within the specified secret.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a
     *         new version of the secret is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and the version <code>SecretString</code> and
     *         <code>SecretBinary</code> values are the same as those in the request, then the request is ignored.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> and
     *         <code>SecretBinary</code> values are different from those in the request, then the request fails because
     *         you cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This value becomes the <code>VersionId</code> of the new version.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The description of the secret.
     * </p>
     * 
     * @return The description of the secret.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN, key ID, or alias of the KMS key that Secrets Manager uses to encrypt the secret value in the secret.
     * </p>
     * <p>
     * To use a KMS key in a different account, use the key ARN or the alias ARN.
     * </p>
     * <p>
     * If you don't specify this value, then Secrets Manager uses the key <code>aws/secretsmanager</code>. If that key
     * doesn't yet exist, then Secrets Manager creates it for you automatically the first time it encrypts the secret
     * value.
     * </p>
     * <p>
     * If the secret is in a different Amazon Web Services account from the credentials calling the API, then you can't
     * use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a customer managed KMS
     * key.
     * </p>
     * 
     * @return The ARN, key ID, or alias of the KMS key that Secrets Manager uses to encrypt the secret value in the
     *         secret.</p>
     *         <p>
     *         To use a KMS key in a different account, use the key ARN or the alias ARN.
     *         </p>
     *         <p>
     *         If you don't specify this value, then Secrets Manager uses the key <code>aws/secretsmanager</code>. If
     *         that key doesn't yet exist, then Secrets Manager creates it for you automatically the first time it
     *         encrypts the secret value.
     *         </p>
     *         <p>
     *         If the secret is in a different Amazon Web Services account from the credentials calling the API, then
     *         you can't use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a
     *         customer managed KMS key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The binary data to encrypt and store in the new version of the secret. We recommend that you store your binary
     * data in a file and then pass the contents of the file as a parameter.
     * </p>
     * <p>
     * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
     * </p>
     * <p>
     * This parameter is not available in the Secrets Manager console.
     * </p>
     * 
     * @return The binary data to encrypt and store in the new version of the secret. We recommend that you store your
     *         binary data in a file and then pass the contents of the file as a parameter.</p>
     *         <p>
     *         Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
     *         </p>
     *         <p>
     *         This parameter is not available in the Secrets Manager console.
     */
    public final SdkBytes secretBinary() {
        return secretBinary;
    }

    /**
     * <p>
     * The text data to encrypt and store in this new version of the secret. We recommend you use a JSON structure of
     * key/value pairs for your secret value.
     * </p>
     * <p>
     * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
     * </p>
     * <p>
     * If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected secret text
     * in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a JSON
     * structure of key/value pairs that a Lambda rotation function can parse.
     * </p>
     * 
     * @return The text data to encrypt and store in this new version of the secret. We recommend you use a JSON
     *         structure of key/value pairs for your secret value.</p>
     *         <p>
     *         Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
     *         </p>
     *         <p>
     *         If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected
     *         secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
     *         information as a JSON structure of key/value pairs that a Lambda rotation function can parse.
     */
    public final String secretString() {
        return secretString;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text string, for
     * example:
     * </p>
     * <p>
     * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
     * </p>
     * <p>
     * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with key
     * "abc".
     * </p>
     * <p>
     * If you check tags in permissions policies as part of your security strategy, then adding or removing a tag can
     * change permissions. If the completion of this operation would result in you losing your permissions for this
     * secret, then Secrets Manager blocks the operation and returns an <code>Access Denied</code> error. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
     * >Control access to secrets using tags</a> and <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
     * >Limit access to identities with tags that match secrets' tags</a>.
     * </p>
     * <p>
     * For information about how to format a JSON parameter for the various command line tool environments, see <a
     * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON for
     * Parameters</a>. If your command-line tool or SDK requires quotation marks around the parameter, you should use
     * single quotes to avoid confusion with the double quotes required in the JSON text.
     * </p>
     * <p>
     * The following restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per secret: 50
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length: 127 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length: 255 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves it for
     * Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with this prefix do
     * not count against your tags per secret limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you use your tagging schema across multiple services and resources, other services might have restrictions on
     * allowed characters. Generally allowed characters: letters, spaces, and numbers representable in UTF-8, plus the
     * following special characters: + - = . _ : / @.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text
     *         string, for example:</p>
     *         <p>
     *         <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
     *         </p>
     *         <p>
     *         Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
     *         with key "abc".
     *         </p>
     *         <p>
     *         If you check tags in permissions policies as part of your security strategy, then adding or removing a
     *         tag can change permissions. If the completion of this operation would result in you losing your
     *         permissions for this secret, then Secrets Manager blocks the operation and returns an
     *         <code>Access Denied</code> error. For more information, see <a href=
     *         "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
     *         >Control access to secrets using tags</a> and <a href=
     *         "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
     *         >Limit access to identities with tags that match secrets' tags</a>.
     *         </p>
     *         <p>
     *         For information about how to format a JSON parameter for the various command line tool environments, see
     *         <a
     *         href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using
     *         JSON for Parameters</a>. If your command-line tool or SDK requires quotation marks around the parameter,
     *         you should use single quotes to avoid confusion with the double quotes required in the JSON text.
     *         </p>
     *         <p>
     *         The following restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per secret: 50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length: 127 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length: 255 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves
     *         it for Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with
     *         this prefix do not count against your tags per secret limit.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you use your tagging schema across multiple services and resources, other services might have
     *         restrictions on allowed characters. Generally allowed characters: letters, spaces, and numbers
     *         representable in UTF-8, plus the following special characters: + - = . _ : / @.
     *         </p>
     *         </li>
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the AddReplicaRegions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAddReplicaRegions() {
        return addReplicaRegions != null && !(addReplicaRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Regions and KMS keys to replicate secrets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAddReplicaRegions} method.
     * </p>
     * 
     * @return A list of Regions and KMS keys to replicate secrets.
     */
    public final List<ReplicaRegionType> addReplicaRegions() {
        return addReplicaRegions;
    }

    /**
     * <p>
     * Specifies whether to overwrite a secret with the same name in the destination Region.
     * </p>
     * 
     * @return Specifies whether to overwrite a secret with the same name in the destination Region.
     */
    public final Boolean forceOverwriteReplicaSecret() {
        return forceOverwriteReplicaSecret;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAddReplicaRegions() ? addReplicaRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(forceOverwriteReplicaSecret());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSecretRequest)) {
            return false;
        }
        CreateSecretRequest other = (CreateSecretRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(description(), other.description()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(secretBinary(), other.secretBinary()) && Objects.equals(secretString(), other.secretString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && hasAddReplicaRegions() == other.hasAddReplicaRegions()
                && Objects.equals(addReplicaRegions(), other.addReplicaRegions())
                && Objects.equals(forceOverwriteReplicaSecret(), other.forceOverwriteReplicaSecret());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateSecretRequest").add("Name", name()).add("ClientRequestToken", clientRequestToken())
                .add("Description", description()).add("KmsKeyId", kmsKeyId())
                .add("SecretBinary", secretBinary() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretString", secretString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Tags", hasTags() ? tags() : null)
                .add("AddReplicaRegions", hasAddReplicaRegions() ? addReplicaRegions() : null)
                .add("ForceOverwriteReplicaSecret", forceOverwriteReplicaSecret()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "SecretBinary":
            return Optional.ofNullable(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.ofNullable(clazz.cast(secretString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "AddReplicaRegions":
            return Optional.ofNullable(clazz.cast(addReplicaRegions()));
        case "ForceOverwriteReplicaSecret":
            return Optional.ofNullable(clazz.cast(forceOverwriteReplicaSecret()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateSecretRequest, T> g) {
        return obj -> g.apply((CreateSecretRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SecretsManagerRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateSecretRequest> {
        /**
         * <p>
         * The name of the new secret.
         * </p>
         * <p>
         * The secret name can contain ASCII letters, numbers, and the following characters: /_+=.@-
         * </p>
         * <p>
         * Do not end your secret name with a hyphen followed by six characters. If you do so, you risk confusion and
         * unexpected results when searching for a secret by partial ARN. Secrets Manager automatically adds a hyphen
         * and six random characters after the secret name at the end of the ARN.
         * </p>
         * 
         * @param name
         *        The name of the new secret.</p>
         *        <p>
         *        The secret name can contain ASCII letters, numbers, and the following characters: /_+=.@-
         *        </p>
         *        <p>
         *        Do not end your secret name with a hyphen followed by six characters. If you do so, you risk confusion
         *        and unexpected results when searching for a secret by partial ARN. Secrets Manager automatically adds
         *        a hyphen and six random characters after the secret name at the end of the ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If you include <code>SecretString</code> or <code>SecretBinary</code>, then Secrets Manager creates an
         * initial version for the secret, and this parameter specifies the unique identifier for the new version.
         * </p>
         * <note>
         * <p>
         * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation, then
         * you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes it as the
         * value for this parameter in the request. If you don't use the SDK and instead generate a raw HTTP request to
         * the Secrets Manager service endpoint, then you must generate a <code>ClientRequestToken</code> yourself for
         * the new version and include the value in the request.
         * </p>
         * </note>
         * <p>
         * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
         * duplicate versions if there are failures and retries during a rotation. We recommend that you generate a <a
         * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness of
         * your versions within the specified secret.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
         * version of the secret is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and the version <code>SecretString</code> and
         * <code>SecretBinary</code> values are the same as those in the request, then the request is ignored.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> and
         * <code>SecretBinary</code> values are different from those in the request, then the request fails because you
         * cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new version.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This value becomes the <code>VersionId</code> of the new version.
         * </p>
         * 
         * @param clientRequestToken
         *        If you include <code>SecretString</code> or <code>SecretBinary</code>, then Secrets Manager creates an
         *        initial version for the secret, and this parameter specifies the unique identifier for the new
         *        version. </p> <note>
         *        <p>
         *        If you use the Amazon Web Services CLI or one of the Amazon Web Services SDKs to call this operation,
         *        then you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes
         *        it as the value for this parameter in the request. If you don't use the SDK and instead generate a raw
         *        HTTP request to the Secrets Manager service endpoint, then you must generate a
         *        <code>ClientRequestToken</code> yourself for the new version and include the value in the request.
         *        </p>
         *        </note>
         *        <p>
         *        This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental
         *        creation of duplicate versions if there are failures and retries during a rotation. We recommend that
         *        you generate a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value
         *        to ensure uniqueness of your versions within the specified secret.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret
         *        then a new version of the secret is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and the version <code>SecretString</code> and
         *        <code>SecretBinary</code> values are the same as those in the request, then the request is ignored.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> and
         *        <code>SecretBinary</code> values are different from those in the request, then the request fails
         *        because you cannot modify an existing version. Instead, use <a>PutSecretValue</a> to create a new
         *        version.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This value becomes the <code>VersionId</code> of the new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The description of the secret.
         * </p>
         * 
         * @param description
         *        The description of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN, key ID, or alias of the KMS key that Secrets Manager uses to encrypt the secret value in the secret.
         * </p>
         * <p>
         * To use a KMS key in a different account, use the key ARN or the alias ARN.
         * </p>
         * <p>
         * If you don't specify this value, then Secrets Manager uses the key <code>aws/secretsmanager</code>. If that
         * key doesn't yet exist, then Secrets Manager creates it for you automatically the first time it encrypts the
         * secret value.
         * </p>
         * <p>
         * If the secret is in a different Amazon Web Services account from the credentials calling the API, then you
         * can't use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a customer
         * managed KMS key.
         * </p>
         * 
         * @param kmsKeyId
         *        The ARN, key ID, or alias of the KMS key that Secrets Manager uses to encrypt the secret value in the
         *        secret.</p>
         *        <p>
         *        To use a KMS key in a different account, use the key ARN or the alias ARN.
         *        </p>
         *        <p>
         *        If you don't specify this value, then Secrets Manager uses the key <code>aws/secretsmanager</code>. If
         *        that key doesn't yet exist, then Secrets Manager creates it for you automatically the first time it
         *        encrypts the secret value.
         *        </p>
         *        <p>
         *        If the secret is in a different Amazon Web Services account from the credentials calling the API, then
         *        you can't use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a
         *        customer managed KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The binary data to encrypt and store in the new version of the secret. We recommend that you store your
         * binary data in a file and then pass the contents of the file as a parameter.
         * </p>
         * <p>
         * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
         * </p>
         * <p>
         * This parameter is not available in the Secrets Manager console.
         * </p>
         * 
         * @param secretBinary
         *        The binary data to encrypt and store in the new version of the secret. We recommend that you store
         *        your binary data in a file and then pass the contents of the file as a parameter.</p>
         *        <p>
         *        Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
         *        </p>
         *        <p>
         *        This parameter is not available in the Secrets Manager console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(SdkBytes secretBinary);

        /**
         * <p>
         * The text data to encrypt and store in this new version of the secret. We recommend you use a JSON structure
         * of key/value pairs for your secret value.
         * </p>
         * <p>
         * Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
         * </p>
         * <p>
         * If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected secret
         * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
         * JSON structure of key/value pairs that a Lambda rotation function can parse.
         * </p>
         * 
         * @param secretString
         *        The text data to encrypt and store in this new version of the secret. We recommend you use a JSON
         *        structure of key/value pairs for your secret value.</p>
         *        <p>
         *        Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both.
         *        </p>
         *        <p>
         *        If you create a secret by using the Secrets Manager console then Secrets Manager puts the protected
         *        secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
         *        information as a JSON structure of key/value pairs that a Lambda rotation function can parse.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text string,
         * for example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * <p>
         * If you check tags in permissions policies as part of your security strategy, then adding or removing a tag
         * can change permissions. If the completion of this operation would result in you losing your permissions for
         * this secret, then Secrets Manager blocks the operation and returns an <code>Access Denied</code> error. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
         * >Control access to secrets using tags</a> and <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
         * >Limit access to identities with tags that match secrets' tags</a>.
         * </p>
         * <p>
         * For information about how to format a JSON parameter for the various command line tool environments, see <a
         * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a>. If your command-line tool or SDK requires quotation marks around the parameter, you
         * should use single quotes to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret: 50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length: 127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length: 255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves it
         * for Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with this
         * prefix do not count against your tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you use your tagging schema across multiple services and resources, other services might have restrictions
         * on allowed characters. Generally allowed characters: letters, spaces, and numbers representable in UTF-8,
         * plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text
         *        string, for example:</p>
         *        <p>
         *        <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         *        </p>
         *        <p>
         *        Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
         *        with key "abc".
         *        </p>
         *        <p>
         *        If you check tags in permissions policies as part of your security strategy, then adding or removing a
         *        tag can change permissions. If the completion of this operation would result in you losing your
         *        permissions for this secret, then Secrets Manager blocks the operation and returns an
         *        <code>Access Denied</code> error. For more information, see <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
         *        >Control access to secrets using tags</a> and <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
         *        >Limit access to identities with tags that match secrets' tags</a>.
         *        </p>
         *        <p>
         *        For information about how to format a JSON parameter for the various command line tool environments,
         *        see <a
         *        href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json"
         *        >Using JSON for Parameters</a>. If your command-line tool or SDK requires quotation marks around the
         *        parameter, you should use single quotes to avoid confusion with the double quotes required in the JSON
         *        text.
         *        </p>
         *        <p>
         *        The following restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per secret: 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length: 127 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length: 255 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services
         *        reserves it for Amazon Web Services use. You can't edit or delete tag names or values with this
         *        prefix. Tags with this prefix do not count against your tags per secret limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you use your tagging schema across multiple services and resources, other services might have
         *        restrictions on allowed characters. Generally allowed characters: letters, spaces, and numbers
         *        representable in UTF-8, plus the following special characters: + - = . _ : / @.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text string,
         * for example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * <p>
         * If you check tags in permissions policies as part of your security strategy, then adding or removing a tag
         * can change permissions. If the completion of this operation would result in you losing your permissions for
         * this secret, then Secrets Manager blocks the operation and returns an <code>Access Denied</code> error. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
         * >Control access to secrets using tags</a> and <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
         * >Limit access to identities with tags that match secrets' tags</a>.
         * </p>
         * <p>
         * For information about how to format a JSON parameter for the various command line tool environments, see <a
         * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a>. If your command-line tool or SDK requires quotation marks around the parameter, you
         * should use single quotes to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret: 50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length: 127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length: 255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves it
         * for Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with this
         * prefix do not count against your tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you use your tagging schema across multiple services and resources, other services might have restrictions
         * on allowed characters. Generally allowed characters: letters, spaces, and numbers representable in UTF-8,
         * plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text
         *        string, for example:</p>
         *        <p>
         *        <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         *        </p>
         *        <p>
         *        Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one
         *        with key "abc".
         *        </p>
         *        <p>
         *        If you check tags in permissions policies as part of your security strategy, then adding or removing a
         *        tag can change permissions. If the completion of this operation would result in you losing your
         *        permissions for this secret, then Secrets Manager blocks the operation and returns an
         *        <code>Access Denied</code> error. For more information, see <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
         *        >Control access to secrets using tags</a> and <a href=
         *        "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
         *        >Limit access to identities with tags that match secrets' tags</a>.
         *        </p>
         *        <p>
         *        For information about how to format a JSON parameter for the various command line tool environments,
         *        see <a
         *        href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json"
         *        >Using JSON for Parameters</a>. If your command-line tool or SDK requires quotation marks around the
         *        parameter, you should use single quotes to avoid confusion with the double quotes required in the JSON
         *        text.
         *        </p>
         *        <p>
         *        The following restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per secret: 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length: 127 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length: 255 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services
         *        reserves it for Amazon Web Services use. You can't edit or delete tag names or values with this
         *        prefix. Tags with this prefix do not count against your tags per secret limit.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you use your tagging schema across multiple services and resources, other services might have
         *        restrictions on allowed characters. Generally allowed characters: letters, spaces, and numbers
         *        representable in UTF-8, plus the following special characters: + - = . _ : / @.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags to attach to the secret. Each tag is a key and value pair of strings in a JSON text string,
         * for example:
         * </p>
         * <p>
         * <code>[{"Key":"CostCenter","Value":"12345"},{"Key":"environment","Value":"production"}]</code>
         * </p>
         * <p>
         * Secrets Manager tag key names are case sensitive. A tag with the key "ABC" is a different tag from one with
         * key "abc".
         * </p>
         * <p>
         * If you check tags in permissions policies as part of your security strategy, then adding or removing a tag
         * can change permissions. If the completion of this operation would result in you losing your permissions for
         * this secret, then Secrets Manager blocks the operation and returns an <code>Access Denied</code> error. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#tag-secrets-abac"
         * >Control access to secrets using tags</a> and <a href=
         * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_examples.html#auth-and-access_tags2"
         * >Limit access to identities with tags that match secrets' tags</a>.
         * </p>
         * <p>
         * For information about how to format a JSON parameter for the various command line tool environments, see <a
         * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-using-param.html#cli-using-param-json">Using JSON
         * for Parameters</a>. If your command-line tool or SDK requires quotation marks around the parameter, you
         * should use single quotes to avoid confusion with the double quotes required in the JSON text.
         * </p>
         * <p>
         * The following restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per secret: 50
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length: 127 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length: 255 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves it
         * for Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with this
         * prefix do not count against your tags per secret limit.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you use your tagging schema across multiple services and resources, other services might have restrictions
         * on allowed characters. Generally allowed characters: letters, spaces, and numbers representable in UTF-8,
         * plus the following special characters: + - = . _ : / @.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A list of Regions and KMS keys to replicate secrets.
         * </p>
         * 
         * @param addReplicaRegions
         *        A list of Regions and KMS keys to replicate secrets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addReplicaRegions(Collection<ReplicaRegionType> addReplicaRegions);

        /**
         * <p>
         * A list of Regions and KMS keys to replicate secrets.
         * </p>
         * 
         * @param addReplicaRegions
         *        A list of Regions and KMS keys to replicate secrets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addReplicaRegions(ReplicaRegionType... addReplicaRegions);

        /**
         * <p>
         * A list of Regions and KMS keys to replicate secrets.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ReplicaRegionType>.Builder} avoiding
         * the need to create one manually via {@link List<ReplicaRegionType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ReplicaRegionType>.Builder#build()} is called immediately
         * and its result is passed to {@link #addReplicaRegions(List<ReplicaRegionType>)}.
         * 
         * @param addReplicaRegions
         *        a consumer that will call methods on {@link List<ReplicaRegionType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #addReplicaRegions(List<ReplicaRegionType>)
         */
        Builder addReplicaRegions(Consumer<ReplicaRegionType.Builder>... addReplicaRegions);

        /**
         * <p>
         * Specifies whether to overwrite a secret with the same name in the destination Region.
         * </p>
         * 
         * @param forceOverwriteReplicaSecret
         *        Specifies whether to overwrite a secret with the same name in the destination Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceOverwriteReplicaSecret(Boolean forceOverwriteReplicaSecret);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SecretsManagerRequest.BuilderImpl implements Builder {
        private String name;

        private String clientRequestToken;

        private String description;

        private String kmsKeyId;

        private SdkBytes secretBinary;

        private String secretString;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<ReplicaRegionType> addReplicaRegions = DefaultSdkAutoConstructList.getInstance();

        private Boolean forceOverwriteReplicaSecret;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSecretRequest model) {
            super(model);
            name(model.name);
            clientRequestToken(model.clientRequestToken);
            description(model.description);
            kmsKeyId(model.kmsKeyId);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            tags(model.tags);
            addReplicaRegions(model.addReplicaRegions);
            forceOverwriteReplicaSecret(model.forceOverwriteReplicaSecret);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        @Transient
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        @Transient
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary == null ? null : secretBinary.asByteBuffer();
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            secretBinary(secretBinary == null ? null : SdkBytes.fromByteBuffer(secretBinary));
        }

        @Override
        @Transient
        public final Builder secretBinary(SdkBytes secretBinary) {
            this.secretBinary = secretBinary;
            return this;
        }

        public final String getSecretString() {
            return secretString;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        @Override
        @Transient
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListTypeCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListTypeCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListTypeCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<ReplicaRegionType.Builder> getAddReplicaRegions() {
            List<ReplicaRegionType.Builder> result = AddReplicaRegionListTypeCopier.copyToBuilder(this.addReplicaRegions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAddReplicaRegions(Collection<ReplicaRegionType.BuilderImpl> addReplicaRegions) {
            this.addReplicaRegions = AddReplicaRegionListTypeCopier.copyFromBuilder(addReplicaRegions);
        }

        @Override
        @Transient
        public final Builder addReplicaRegions(Collection<ReplicaRegionType> addReplicaRegions) {
            this.addReplicaRegions = AddReplicaRegionListTypeCopier.copy(addReplicaRegions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder addReplicaRegions(ReplicaRegionType... addReplicaRegions) {
            addReplicaRegions(Arrays.asList(addReplicaRegions));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder addReplicaRegions(Consumer<ReplicaRegionType.Builder>... addReplicaRegions) {
            addReplicaRegions(Stream.of(addReplicaRegions).map(c -> ReplicaRegionType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getForceOverwriteReplicaSecret() {
            return forceOverwriteReplicaSecret;
        }

        public final void setForceOverwriteReplicaSecret(Boolean forceOverwriteReplicaSecret) {
            this.forceOverwriteReplicaSecret = forceOverwriteReplicaSecret;
        }

        @Override
        @Transient
        public final Builder forceOverwriteReplicaSecret(Boolean forceOverwriteReplicaSecret) {
            this.forceOverwriteReplicaSecret = forceOverwriteReplicaSecret;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSecretRequest build() {
            return new CreateSecretRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
