/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.beans.Transient;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutSecretValueRequest extends SecretsManagerRequest implements
        ToCopyableBuilder<PutSecretValueRequest.Builder, PutSecretValueRequest> {
    private static final SdkField<String> SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretId").getter(getter(PutSecretValueRequest::secretId)).setter(setter(Builder::secretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretId").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(PutSecretValueRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<SdkBytes> SECRET_BINARY_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("SecretBinary").getter(getter(PutSecretValueRequest::secretBinary)).setter(setter(Builder::secretBinary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretBinary").build()).build();

    private static final SdkField<String> SECRET_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretString").getter(getter(PutSecretValueRequest::secretString)).setter(setter(Builder::secretString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretString").build()).build();

    private static final SdkField<List<String>> VERSION_STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VersionStages")
            .getter(getter(PutSecretValueRequest::versionStages))
            .setter(setter(Builder::versionStages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SECRET_ID_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, SECRET_BINARY_FIELD, SECRET_STRING_FIELD, VERSION_STAGES_FIELD));

    private final String secretId;

    private final String clientRequestToken;

    private final SdkBytes secretBinary;

    private final String secretString;

    private final List<String> versionStages;

    private PutSecretValueRequest(BuilderImpl builder) {
        super(builder);
        this.secretId = builder.secretId;
        this.clientRequestToken = builder.clientRequestToken;
        this.secretBinary = builder.secretBinary;
        this.secretString = builder.secretString;
        this.versionStages = builder.versionStages;
    }

    /**
     * <p>
     * Specifies the secret to which you want to add a new version. You can specify either the Amazon Resource Name
     * (ARN) or the friendly name of the secret. The secret must already exist.
     * </p>
     * <p>
     * For an ARN, we recommend that you specify a complete ARN rather than a partial ARN.
     * </p>
     * 
     * @return Specifies the secret to which you want to add a new version. You can specify either the Amazon Resource
     *         Name (ARN) or the friendly name of the secret. The secret must already exist.</p>
     *         <p>
     *         For an ARN, we recommend that you specify a complete ARN rather than a partial ARN.
     */
    public final String secretId() {
        return secretId;
    }

    /**
     * <p>
     * (Optional) Specifies a unique identifier for the new version of the secret.
     * </p>
     * <note>
     * <p>
     * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDK to call this operation, then you can
     * leave this parameter empty. The CLI or SDK generates a random UUID for you and includes that in the request. If
     * you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager service endpoint, then you
     * must generate a <code>ClientRequestToken</code> yourself for new versions and include that value in the request.
     * </p>
     * </note>
     * <p>
     * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
     * duplicate versions if there are failures and retries during the Lambda rotation function's processing. We
     * recommend that you generate a <a href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a>
     * value to ensure uniqueness within the specified secret.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
     * version of the secret is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and that version's <code>SecretString</code> or
     * <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the operation
     * is idempotent).
     * </p>
     * </li>
     * <li>
     * <p>
     * If a version with this value already exists and the version of the <code>SecretString</code> and
     * <code>SecretBinary</code> values are different from those in the request then the request fails because you
     * cannot modify an existing secret version. You can only create new versions to store new secret values.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This value becomes the <code>VersionId</code> of the new version.
     * </p>
     * 
     * @return (Optional) Specifies a unique identifier for the new version of the secret. </p> <note>
     *         <p>
     *         If you use the Amazon Web Services CLI or one of the Amazon Web Services SDK to call this operation, then
     *         you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes that in
     *         the request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager
     *         service endpoint, then you must generate a <code>ClientRequestToken</code> yourself for new versions and
     *         include that value in the request.
     *         </p>
     *         </note>
     *         <p>
     *         This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation
     *         of duplicate versions if there are failures and retries during the Lambda rotation function's processing.
     *         We recommend that you generate a <a
     *         href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness
     *         within the specified secret.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a
     *         new version of the secret is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and that version's <code>SecretString</code> or
     *         <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
     *         operation is idempotent).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a version with this value already exists and the version of the <code>SecretString</code> and
     *         <code>SecretBinary</code> values are different from those in the request then the request fails because
     *         you cannot modify an existing secret version. You can only create new versions to store new secret
     *         values.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This value becomes the <code>VersionId</code> of the new version.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use this
     * parameter in the command-line tools, we recommend that you store your binary data in a file and then use the
     * appropriate technique for your tool to pass the contents of the file as a parameter. Either
     * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
     * empty.
     * </p>
     * <p>
     * This parameter is not accessible if the secret using the Secrets Manager console.
     * </p>
     * <p/>
     * 
     * @return (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To
     *         use this parameter in the command-line tools, we recommend that you store your binary data in a file and
     *         then use the appropriate technique for your tool to pass the contents of the file as a parameter. Either
     *         <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both
     *         be empty.
     *         </p>
     *         <p>
     *         This parameter is not accessible if the secret using the Secrets Manager console.
     *         </p>
     */
    public final SdkBytes secretBinary() {
        return secretBinary;
    }

    /**
     * <p>
     * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret. Either
     * <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot both be
     * empty.
     * </p>
     * <p>
     * If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected secret
     * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
     * JSON structure of key/value pairs that the default Lambda rotation function knows how to parse.
     * </p>
     * <p>
     * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value pairs.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters.html">Specifying parameter values for
     * the Amazon Web Services CLI</a> in the Amazon Web Services CLI User Guide.
     * </p>
     * 
     * @return (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
     *         Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
     *         cannot both be empty.</p>
     *         <p>
     *         If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected
     *         secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
     *         information as a JSON structure of key/value pairs that the default Lambda rotation function knows how to
     *         parse.
     *         </p>
     *         <p>
     *         For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
     *         pairs. For more information, see <a
     *         href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters.html">Specifying parameter
     *         values for the Amazon Web Services CLI</a> in the Amazon Web Services CLI User Guide.
     */
    public final String secretString() {
        return secretString;
    }

    /**
     * For responses, this returns true if the service returned a value for the VersionStages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVersionStages() {
        return versionStages != null && !(versionStages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Optional) Specifies a list of staging labels that are attached to this version of the secret. These staging
     * labels are used to track the versions through the rotation process by the Lambda rotation function.
     * </p>
     * <p>
     * A staging label must be unique to a single version of the secret. If you specify a staging label that's already
     * associated with a different version of the same secret then that staging label is automatically removed from the
     * other version and attached to this version.
     * </p>
     * <p>
     * If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves the staging
     * label <code>AWSCURRENT</code> to this new version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVersionStages} method.
     * </p>
     * 
     * @return (Optional) Specifies a list of staging labels that are attached to this version of the secret. These
     *         staging labels are used to track the versions through the rotation process by the Lambda rotation
     *         function.</p>
     *         <p>
     *         A staging label must be unique to a single version of the secret. If you specify a staging label that's
     *         already associated with a different version of the same secret then that staging label is automatically
     *         removed from the other version and attached to this version.
     *         </p>
     *         <p>
     *         If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves the
     *         staging label <code>AWSCURRENT</code> to this new version.
     */
    public final List<String> versionStages() {
        return versionStages;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(secretId());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(secretBinary());
        hashCode = 31 * hashCode + Objects.hashCode(secretString());
        hashCode = 31 * hashCode + Objects.hashCode(hasVersionStages() ? versionStages() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutSecretValueRequest)) {
            return false;
        }
        PutSecretValueRequest other = (PutSecretValueRequest) obj;
        return Objects.equals(secretId(), other.secretId()) && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(secretBinary(), other.secretBinary()) && Objects.equals(secretString(), other.secretString())
                && hasVersionStages() == other.hasVersionStages() && Objects.equals(versionStages(), other.versionStages());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutSecretValueRequest").add("SecretId", secretId())
                .add("ClientRequestToken", clientRequestToken())
                .add("SecretBinary", secretBinary() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretString", secretString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("VersionStages", hasVersionStages() ? versionStages() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SecretId":
            return Optional.ofNullable(clazz.cast(secretId()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "SecretBinary":
            return Optional.ofNullable(clazz.cast(secretBinary()));
        case "SecretString":
            return Optional.ofNullable(clazz.cast(secretString()));
        case "VersionStages":
            return Optional.ofNullable(clazz.cast(versionStages()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutSecretValueRequest, T> g) {
        return obj -> g.apply((PutSecretValueRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SecretsManagerRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutSecretValueRequest> {
        /**
         * <p>
         * Specifies the secret to which you want to add a new version. You can specify either the Amazon Resource Name
         * (ARN) or the friendly name of the secret. The secret must already exist.
         * </p>
         * <p>
         * For an ARN, we recommend that you specify a complete ARN rather than a partial ARN.
         * </p>
         * 
         * @param secretId
         *        Specifies the secret to which you want to add a new version. You can specify either the Amazon
         *        Resource Name (ARN) or the friendly name of the secret. The secret must already exist.</p>
         *        <p>
         *        For an ARN, we recommend that you specify a complete ARN rather than a partial ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretId(String secretId);

        /**
         * <p>
         * (Optional) Specifies a unique identifier for the new version of the secret.
         * </p>
         * <note>
         * <p>
         * If you use the Amazon Web Services CLI or one of the Amazon Web Services SDK to call this operation, then you
         * can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes that in the
         * request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets Manager service
         * endpoint, then you must generate a <code>ClientRequestToken</code> yourself for new versions and include that
         * value in the request.
         * </p>
         * </note>
         * <p>
         * This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental creation of
         * duplicate versions if there are failures and retries during the Lambda rotation function's processing. We
         * recommend that you generate a <a
         * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure uniqueness
         * within the specified secret.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret then a new
         * version of the secret is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and that version's <code>SecretString</code> or
         * <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
         * operation is idempotent).
         * </p>
         * </li>
         * <li>
         * <p>
         * If a version with this value already exists and the version of the <code>SecretString</code> and
         * <code>SecretBinary</code> values are different from those in the request then the request fails because you
         * cannot modify an existing secret version. You can only create new versions to store new secret values.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This value becomes the <code>VersionId</code> of the new version.
         * </p>
         * 
         * @param clientRequestToken
         *        (Optional) Specifies a unique identifier for the new version of the secret. </p> <note>
         *        <p>
         *        If you use the Amazon Web Services CLI or one of the Amazon Web Services SDK to call this operation,
         *        then you can leave this parameter empty. The CLI or SDK generates a random UUID for you and includes
         *        that in the request. If you don't use the SDK and instead generate a raw HTTP request to the Secrets
         *        Manager service endpoint, then you must generate a <code>ClientRequestToken</code> yourself for new
         *        versions and include that value in the request.
         *        </p>
         *        </note>
         *        <p>
         *        This value helps ensure idempotency. Secrets Manager uses this value to prevent the accidental
         *        creation of duplicate versions if there are failures and retries during the Lambda rotation function's
         *        processing. We recommend that you generate a <a
         *        href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID-type</a> value to ensure
         *        uniqueness within the specified secret.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the <code>ClientRequestToken</code> value isn't already associated with a version of the secret
         *        then a new version of the secret is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and that version's <code>SecretString</code> or
         *        <code>SecretBinary</code> values are the same as those in the request then the request is ignored (the
         *        operation is idempotent).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a version with this value already exists and the version of the <code>SecretString</code> and
         *        <code>SecretBinary</code> values are different from those in the request then the request fails
         *        because you cannot modify an existing secret version. You can only create new versions to store new
         *        secret values.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This value becomes the <code>VersionId</code> of the new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret. To use
         * this parameter in the command-line tools, we recommend that you store your binary data in a file and then use
         * the appropriate technique for your tool to pass the contents of the file as a parameter. Either
         * <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not both. They cannot both be
         * empty.
         * </p>
         * <p>
         * This parameter is not accessible if the secret using the Secrets Manager console.
         * </p>
         * <p/>
         * 
         * @param secretBinary
         *        (Optional) Specifies binary data that you want to encrypt and store in the new version of the secret.
         *        To use this parameter in the command-line tools, we recommend that you store your binary data in a
         *        file and then use the appropriate technique for your tool to pass the contents of the file as a
         *        parameter. Either <code>SecretBinary</code> or <code>SecretString</code> must have a value, but not
         *        both. They cannot both be empty.
         *        </p>
         *        <p>
         *        This parameter is not accessible if the secret using the Secrets Manager console.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretBinary(SdkBytes secretBinary);

        /**
         * <p>
         * (Optional) Specifies text data that you want to encrypt and store in this new version of the secret. Either
         * <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They cannot both be
         * empty.
         * </p>
         * <p>
         * If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected secret
         * text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the information as a
         * JSON structure of key/value pairs that the default Lambda rotation function knows how to parse.
         * </p>
         * <p>
         * For storing multiple values, we recommend that you use a JSON text string argument and specify key/value
         * pairs. For more information, see <a
         * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters.html">Specifying parameter values
         * for the Amazon Web Services CLI</a> in the Amazon Web Services CLI User Guide.
         * </p>
         * 
         * @param secretString
         *        (Optional) Specifies text data that you want to encrypt and store in this new version of the secret.
         *        Either <code>SecretString</code> or <code>SecretBinary</code> must have a value, but not both. They
         *        cannot both be empty.</p>
         *        <p>
         *        If you create this secret by using the Secrets Manager console then Secrets Manager puts the protected
         *        secret text in only the <code>SecretString</code> parameter. The Secrets Manager console stores the
         *        information as a JSON structure of key/value pairs that the default Lambda rotation function knows how
         *        to parse.
         *        </p>
         *        <p>
         *        For storing multiple values, we recommend that you use a JSON text string argument and specify
         *        key/value pairs. For more information, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-parameters.html">Specifying parameter
         *        values for the Amazon Web Services CLI</a> in the Amazon Web Services CLI User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretString(String secretString);

        /**
         * <p>
         * (Optional) Specifies a list of staging labels that are attached to this version of the secret. These staging
         * labels are used to track the versions through the rotation process by the Lambda rotation function.
         * </p>
         * <p>
         * A staging label must be unique to a single version of the secret. If you specify a staging label that's
         * already associated with a different version of the same secret then that staging label is automatically
         * removed from the other version and attached to this version.
         * </p>
         * <p>
         * If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves the
         * staging label <code>AWSCURRENT</code> to this new version.
         * </p>
         * 
         * @param versionStages
         *        (Optional) Specifies a list of staging labels that are attached to this version of the secret. These
         *        staging labels are used to track the versions through the rotation process by the Lambda rotation
         *        function.</p>
         *        <p>
         *        A staging label must be unique to a single version of the secret. If you specify a staging label
         *        that's already associated with a different version of the same secret then that staging label is
         *        automatically removed from the other version and attached to this version.
         *        </p>
         *        <p>
         *        If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves
         *        the staging label <code>AWSCURRENT</code> to this new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * (Optional) Specifies a list of staging labels that are attached to this version of the secret. These staging
         * labels are used to track the versions through the rotation process by the Lambda rotation function.
         * </p>
         * <p>
         * A staging label must be unique to a single version of the secret. If you specify a staging label that's
         * already associated with a different version of the same secret then that staging label is automatically
         * removed from the other version and attached to this version.
         * </p>
         * <p>
         * If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves the
         * staging label <code>AWSCURRENT</code> to this new version.
         * </p>
         * 
         * @param versionStages
         *        (Optional) Specifies a list of staging labels that are attached to this version of the secret. These
         *        staging labels are used to track the versions through the rotation process by the Lambda rotation
         *        function.</p>
         *        <p>
         *        A staging label must be unique to a single version of the secret. If you specify a staging label
         *        that's already associated with a different version of the same secret then that staging label is
         *        automatically removed from the other version and attached to this version.
         *        </p>
         *        <p>
         *        If you do not specify a value for <code>VersionStages</code> then Secrets Manager automatically moves
         *        the staging label <code>AWSCURRENT</code> to this new version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SecretsManagerRequest.BuilderImpl implements Builder {
        private String secretId;

        private String clientRequestToken;

        private SdkBytes secretBinary;

        private String secretString;

        private List<String> versionStages = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutSecretValueRequest model) {
            super(model);
            secretId(model.secretId);
            clientRequestToken(model.clientRequestToken);
            secretBinary(model.secretBinary);
            secretString(model.secretString);
            versionStages(model.versionStages);
        }

        public final String getSecretId() {
            return secretId;
        }

        public final void setSecretId(String secretId) {
            this.secretId = secretId;
        }

        @Override
        @Transient
        public final Builder secretId(String secretId) {
            this.secretId = secretId;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        @Transient
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final ByteBuffer getSecretBinary() {
            return secretBinary == null ? null : secretBinary.asByteBuffer();
        }

        public final void setSecretBinary(ByteBuffer secretBinary) {
            secretBinary(secretBinary == null ? null : SdkBytes.fromByteBuffer(secretBinary));
        }

        @Override
        @Transient
        public final Builder secretBinary(SdkBytes secretBinary) {
            this.secretBinary = secretBinary;
            return this;
        }

        public final String getSecretString() {
            return secretString;
        }

        public final void setSecretString(String secretString) {
            this.secretString = secretString;
        }

        @Override
        @Transient
        public final Builder secretString(String secretString) {
            this.secretString = secretString;
            return this;
        }

        public final Collection<String> getVersionStages() {
            if (versionStages instanceof SdkAutoConstructList) {
                return null;
            }
            return versionStages;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        @Override
        @Transient
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutSecretValueRequest build() {
            return new PutSecretValueRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
