/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains information about one version of a secret.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecretVersionsListEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<SecretVersionsListEntry.Builder, SecretVersionsListEntry> {
    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SecretVersionsListEntry::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionId").build()).build();

    private static final SdkField<List<String>> VERSION_STAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SecretVersionsListEntry::versionStages))
            .setter(setter(Builder::versionStages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_ACCESSED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(SecretVersionsListEntry::lastAccessedDate)).setter(setter(Builder::lastAccessedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastAccessedDate").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(SecretVersionsListEntry::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_ID_FIELD,
            VERSION_STAGES_FIELD, LAST_ACCESSED_DATE_FIELD, CREATED_DATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String versionId;

    private final List<String> versionStages;

    private final Instant lastAccessedDate;

    private final Instant createdDate;

    private SecretVersionsListEntry(BuilderImpl builder) {
        this.versionId = builder.versionId;
        this.versionStages = builder.versionStages;
        this.lastAccessedDate = builder.lastAccessedDate;
        this.createdDate = builder.createdDate;
    }

    /**
     * <p>
     * The unique version identifier of this version of the secret.
     * </p>
     * 
     * @return The unique version identifier of this version of the secret.
     */
    public String versionId() {
        return versionId;
    }

    /**
     * <p>
     * An array of staging labels that are currently associated with this version of the secret.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of staging labels that are currently associated with this version of the secret.
     */
    public List<String> versionStages() {
        return versionStages;
    }

    /**
     * <p>
     * The date that this version of the secret was last accessed. Note that the resolution of this field is at the date
     * level and does not include the time.
     * </p>
     * 
     * @return The date that this version of the secret was last accessed. Note that the resolution of this field is at
     *         the date level and does not include the time.
     */
    public Instant lastAccessedDate() {
        return lastAccessedDate;
    }

    /**
     * <p>
     * The date and time this version of the secret was created.
     * </p>
     * 
     * @return The date and time this version of the secret was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        hashCode = 31 * hashCode + Objects.hashCode(versionStages());
        hashCode = 31 * hashCode + Objects.hashCode(lastAccessedDate());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecretVersionsListEntry)) {
            return false;
        }
        SecretVersionsListEntry other = (SecretVersionsListEntry) obj;
        return Objects.equals(versionId(), other.versionId()) && Objects.equals(versionStages(), other.versionStages())
                && Objects.equals(lastAccessedDate(), other.lastAccessedDate())
                && Objects.equals(createdDate(), other.createdDate());
    }

    @Override
    public String toString() {
        return ToString.builder("SecretVersionsListEntry").add("VersionId", versionId()).add("VersionStages", versionStages())
                .add("LastAccessedDate", lastAccessedDate()).add("CreatedDate", createdDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VersionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        case "VersionStages":
            return Optional.ofNullable(clazz.cast(versionStages()));
        case "LastAccessedDate":
            return Optional.ofNullable(clazz.cast(lastAccessedDate()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SecretVersionsListEntry, T> g) {
        return obj -> g.apply((SecretVersionsListEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecretVersionsListEntry> {
        /**
         * <p>
         * The unique version identifier of this version of the secret.
         * </p>
         * 
         * @param versionId
         *        The unique version identifier of this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);

        /**
         * <p>
         * An array of staging labels that are currently associated with this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        An array of staging labels that are currently associated with this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(Collection<String> versionStages);

        /**
         * <p>
         * An array of staging labels that are currently associated with this version of the secret.
         * </p>
         * 
         * @param versionStages
         *        An array of staging labels that are currently associated with this version of the secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionStages(String... versionStages);

        /**
         * <p>
         * The date that this version of the secret was last accessed. Note that the resolution of this field is at the
         * date level and does not include the time.
         * </p>
         * 
         * @param lastAccessedDate
         *        The date that this version of the secret was last accessed. Note that the resolution of this field is
         *        at the date level and does not include the time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAccessedDate(Instant lastAccessedDate);

        /**
         * <p>
         * The date and time this version of the secret was created.
         * </p>
         * 
         * @param createdDate
         *        The date and time this version of the secret was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);
    }

    static final class BuilderImpl implements Builder {
        private String versionId;

        private List<String> versionStages = DefaultSdkAutoConstructList.getInstance();

        private Instant lastAccessedDate;

        private Instant createdDate;

        private BuilderImpl() {
        }

        private BuilderImpl(SecretVersionsListEntry model) {
            versionId(model.versionId);
            versionStages(model.versionStages);
            lastAccessedDate(model.lastAccessedDate);
            createdDate(model.createdDate);
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        public final Collection<String> getVersionStages() {
            return versionStages;
        }

        @Override
        public final Builder versionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionStages(String... versionStages) {
            versionStages(Arrays.asList(versionStages));
            return this;
        }

        public final void setVersionStages(Collection<String> versionStages) {
            this.versionStages = SecretVersionStagesTypeCopier.copy(versionStages);
        }

        public final Instant getLastAccessedDate() {
            return lastAccessedDate;
        }

        @Override
        public final Builder lastAccessedDate(Instant lastAccessedDate) {
            this.lastAccessedDate = lastAccessedDate;
            return this;
        }

        public final void setLastAccessedDate(Instant lastAccessedDate) {
            this.lastAccessedDate = lastAccessedDate;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        @Override
        public SecretVersionsListEntry build() {
            return new SecretVersionsListEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
