/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details for an individual security standard control.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StandardsControl implements SdkPojo, Serializable,
        ToCopyableBuilder<StandardsControl.Builder, StandardsControl> {
    private static final SdkField<String> STANDARDS_CONTROL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StandardsControlArn").getter(getter(StandardsControl::standardsControlArn))
            .setter(setter(Builder::standardsControlArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StandardsControlArn").build())
            .build();

    private static final SdkField<String> CONTROL_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ControlStatus").getter(getter(StandardsControl::controlStatusAsString))
            .setter(setter(Builder::controlStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ControlStatus").build()).build();

    private static final SdkField<String> DISABLED_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisabledReason").getter(getter(StandardsControl::disabledReason))
            .setter(setter(Builder::disabledReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisabledReason").build()).build();

    private static final SdkField<Instant> CONTROL_STATUS_UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ControlStatusUpdatedAt")
            .getter(getter(StandardsControl::controlStatusUpdatedAt))
            .setter(setter(Builder::controlStatusUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ControlStatusUpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CONTROL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ControlId").getter(getter(StandardsControl::controlId)).setter(setter(Builder::controlId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ControlId").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(StandardsControl::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(StandardsControl::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> REMEDIATION_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RemediationUrl").getter(getter(StandardsControl::remediationUrl))
            .setter(setter(Builder::remediationUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemediationUrl").build()).build();

    private static final SdkField<String> SEVERITY_RATING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SeverityRating").getter(getter(StandardsControl::severityRatingAsString))
            .setter(setter(Builder::severityRating))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SeverityRating").build()).build();

    private static final SdkField<List<String>> RELATED_REQUIREMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RelatedRequirements")
            .getter(getter(StandardsControl::relatedRequirements))
            .setter(setter(Builder::relatedRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STANDARDS_CONTROL_ARN_FIELD,
            CONTROL_STATUS_FIELD, DISABLED_REASON_FIELD, CONTROL_STATUS_UPDATED_AT_FIELD, CONTROL_ID_FIELD, TITLE_FIELD,
            DESCRIPTION_FIELD, REMEDIATION_URL_FIELD, SEVERITY_RATING_FIELD, RELATED_REQUIREMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String standardsControlArn;

    private final String controlStatus;

    private final String disabledReason;

    private final Instant controlStatusUpdatedAt;

    private final String controlId;

    private final String title;

    private final String description;

    private final String remediationUrl;

    private final String severityRating;

    private final List<String> relatedRequirements;

    private StandardsControl(BuilderImpl builder) {
        this.standardsControlArn = builder.standardsControlArn;
        this.controlStatus = builder.controlStatus;
        this.disabledReason = builder.disabledReason;
        this.controlStatusUpdatedAt = builder.controlStatusUpdatedAt;
        this.controlId = builder.controlId;
        this.title = builder.title;
        this.description = builder.description;
        this.remediationUrl = builder.remediationUrl;
        this.severityRating = builder.severityRating;
        this.relatedRequirements = builder.relatedRequirements;
    }

    /**
     * <p>
     * The ARN of the security standard control.
     * </p>
     * 
     * @return The ARN of the security standard control.
     */
    public final String standardsControlArn() {
        return standardsControlArn;
    }

    /**
     * <p>
     * The current status of the security standard control. Indicates whether the control is enabled or disabled.
     * Security Hub does not check against disabled controls.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #controlStatus}
     * will return {@link ControlStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #controlStatusAsString}.
     * </p>
     * 
     * @return The current status of the security standard control. Indicates whether the control is enabled or
     *         disabled. Security Hub does not check against disabled controls.
     * @see ControlStatus
     */
    public final ControlStatus controlStatus() {
        return ControlStatus.fromValue(controlStatus);
    }

    /**
     * <p>
     * The current status of the security standard control. Indicates whether the control is enabled or disabled.
     * Security Hub does not check against disabled controls.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #controlStatus}
     * will return {@link ControlStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #controlStatusAsString}.
     * </p>
     * 
     * @return The current status of the security standard control. Indicates whether the control is enabled or
     *         disabled. Security Hub does not check against disabled controls.
     * @see ControlStatus
     */
    public final String controlStatusAsString() {
        return controlStatus;
    }

    /**
     * <p>
     * The reason provided for the most recent change in status for the control.
     * </p>
     * 
     * @return The reason provided for the most recent change in status for the control.
     */
    public final String disabledReason() {
        return disabledReason;
    }

    /**
     * <p>
     * The date and time that the status of the security standard control was most recently updated.
     * </p>
     * 
     * @return The date and time that the status of the security standard control was most recently updated.
     */
    public final Instant controlStatusUpdatedAt() {
        return controlStatusUpdatedAt;
    }

    /**
     * <p>
     * The identifier of the security standard control.
     * </p>
     * 
     * @return The identifier of the security standard control.
     */
    public final String controlId() {
        return controlId;
    }

    /**
     * <p>
     * The title of the security standard control.
     * </p>
     * 
     * @return The title of the security standard control.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The longer description of the security standard control. Provides information about what the control is checking
     * for.
     * </p>
     * 
     * @return The longer description of the security standard control. Provides information about what the control is
     *         checking for.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A link to remediation information for the control in the Security Hub user documentation.
     * </p>
     * 
     * @return A link to remediation information for the control in the Security Hub user documentation.
     */
    public final String remediationUrl() {
        return remediationUrl;
    }

    /**
     * <p>
     * The severity of findings generated from this security standard control.
     * </p>
     * <p>
     * The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the issue
     * is detected.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of findings generated from this security standard control.</p>
     *         <p>
     *         The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the
     *         issue is detected.
     * @see SeverityRating
     */
    public final SeverityRating severityRating() {
        return SeverityRating.fromValue(severityRating);
    }

    /**
     * <p>
     * The severity of findings generated from this security standard control.
     * </p>
     * <p>
     * The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the issue
     * is detected.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of findings generated from this security standard control.</p>
     *         <p>
     *         The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the
     *         issue is detected.
     * @see SeverityRating
     */
    public final String severityRatingAsString() {
        return severityRating;
    }

    /**
     * Returns true if the RelatedRequirements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasRelatedRequirements() {
        return relatedRequirements != null && !(relatedRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of requirements that are related to this control.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRelatedRequirements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of requirements that are related to this control.
     */
    public final List<String> relatedRequirements() {
        return relatedRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(standardsControlArn());
        hashCode = 31 * hashCode + Objects.hashCode(controlStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(disabledReason());
        hashCode = 31 * hashCode + Objects.hashCode(controlStatusUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(controlId());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(remediationUrl());
        hashCode = 31 * hashCode + Objects.hashCode(severityRatingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedRequirements() ? relatedRequirements() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StandardsControl)) {
            return false;
        }
        StandardsControl other = (StandardsControl) obj;
        return Objects.equals(standardsControlArn(), other.standardsControlArn())
                && Objects.equals(controlStatusAsString(), other.controlStatusAsString())
                && Objects.equals(disabledReason(), other.disabledReason())
                && Objects.equals(controlStatusUpdatedAt(), other.controlStatusUpdatedAt())
                && Objects.equals(controlId(), other.controlId()) && Objects.equals(title(), other.title())
                && Objects.equals(description(), other.description()) && Objects.equals(remediationUrl(), other.remediationUrl())
                && Objects.equals(severityRatingAsString(), other.severityRatingAsString())
                && hasRelatedRequirements() == other.hasRelatedRequirements()
                && Objects.equals(relatedRequirements(), other.relatedRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StandardsControl").add("StandardsControlArn", standardsControlArn())
                .add("ControlStatus", controlStatusAsString()).add("DisabledReason", disabledReason())
                .add("ControlStatusUpdatedAt", controlStatusUpdatedAt()).add("ControlId", controlId()).add("Title", title())
                .add("Description", description()).add("RemediationUrl", remediationUrl())
                .add("SeverityRating", severityRatingAsString())
                .add("RelatedRequirements", hasRelatedRequirements() ? relatedRequirements() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StandardsControlArn":
            return Optional.ofNullable(clazz.cast(standardsControlArn()));
        case "ControlStatus":
            return Optional.ofNullable(clazz.cast(controlStatusAsString()));
        case "DisabledReason":
            return Optional.ofNullable(clazz.cast(disabledReason()));
        case "ControlStatusUpdatedAt":
            return Optional.ofNullable(clazz.cast(controlStatusUpdatedAt()));
        case "ControlId":
            return Optional.ofNullable(clazz.cast(controlId()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RemediationUrl":
            return Optional.ofNullable(clazz.cast(remediationUrl()));
        case "SeverityRating":
            return Optional.ofNullable(clazz.cast(severityRatingAsString()));
        case "RelatedRequirements":
            return Optional.ofNullable(clazz.cast(relatedRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StandardsControl, T> g) {
        return obj -> g.apply((StandardsControl) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StandardsControl> {
        /**
         * <p>
         * The ARN of the security standard control.
         * </p>
         * 
         * @param standardsControlArn
         *        The ARN of the security standard control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder standardsControlArn(String standardsControlArn);

        /**
         * <p>
         * The current status of the security standard control. Indicates whether the control is enabled or disabled.
         * Security Hub does not check against disabled controls.
         * </p>
         * 
         * @param controlStatus
         *        The current status of the security standard control. Indicates whether the control is enabled or
         *        disabled. Security Hub does not check against disabled controls.
         * @see ControlStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlStatus
         */
        Builder controlStatus(String controlStatus);

        /**
         * <p>
         * The current status of the security standard control. Indicates whether the control is enabled or disabled.
         * Security Hub does not check against disabled controls.
         * </p>
         * 
         * @param controlStatus
         *        The current status of the security standard control. Indicates whether the control is enabled or
         *        disabled. Security Hub does not check against disabled controls.
         * @see ControlStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlStatus
         */
        Builder controlStatus(ControlStatus controlStatus);

        /**
         * <p>
         * The reason provided for the most recent change in status for the control.
         * </p>
         * 
         * @param disabledReason
         *        The reason provided for the most recent change in status for the control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disabledReason(String disabledReason);

        /**
         * <p>
         * The date and time that the status of the security standard control was most recently updated.
         * </p>
         * 
         * @param controlStatusUpdatedAt
         *        The date and time that the status of the security standard control was most recently updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controlStatusUpdatedAt(Instant controlStatusUpdatedAt);

        /**
         * <p>
         * The identifier of the security standard control.
         * </p>
         * 
         * @param controlId
         *        The identifier of the security standard control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder controlId(String controlId);

        /**
         * <p>
         * The title of the security standard control.
         * </p>
         * 
         * @param title
         *        The title of the security standard control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The longer description of the security standard control. Provides information about what the control is
         * checking for.
         * </p>
         * 
         * @param description
         *        The longer description of the security standard control. Provides information about what the control
         *        is checking for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A link to remediation information for the control in the Security Hub user documentation.
         * </p>
         * 
         * @param remediationUrl
         *        A link to remediation information for the control in the Security Hub user documentation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediationUrl(String remediationUrl);

        /**
         * <p>
         * The severity of findings generated from this security standard control.
         * </p>
         * <p>
         * The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the
         * issue is detected.
         * </p>
         * 
         * @param severityRating
         *        The severity of findings generated from this security standard control.</p>
         *        <p>
         *        The finding severity is based on an assessment of how easy it would be to compromise AWS resources if
         *        the issue is detected.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(String severityRating);

        /**
         * <p>
         * The severity of findings generated from this security standard control.
         * </p>
         * <p>
         * The finding severity is based on an assessment of how easy it would be to compromise AWS resources if the
         * issue is detected.
         * </p>
         * 
         * @param severityRating
         *        The severity of findings generated from this security standard control.</p>
         *        <p>
         *        The finding severity is based on an assessment of how easy it would be to compromise AWS resources if
         *        the issue is detected.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(SeverityRating severityRating);

        /**
         * <p>
         * The list of requirements that are related to this control.
         * </p>
         * 
         * @param relatedRequirements
         *        The list of requirements that are related to this control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(Collection<String> relatedRequirements);

        /**
         * <p>
         * The list of requirements that are related to this control.
         * </p>
         * 
         * @param relatedRequirements
         *        The list of requirements that are related to this control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedRequirements(String... relatedRequirements);
    }

    static final class BuilderImpl implements Builder {
        private String standardsControlArn;

        private String controlStatus;

        private String disabledReason;

        private Instant controlStatusUpdatedAt;

        private String controlId;

        private String title;

        private String description;

        private String remediationUrl;

        private String severityRating;

        private List<String> relatedRequirements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StandardsControl model) {
            standardsControlArn(model.standardsControlArn);
            controlStatus(model.controlStatus);
            disabledReason(model.disabledReason);
            controlStatusUpdatedAt(model.controlStatusUpdatedAt);
            controlId(model.controlId);
            title(model.title);
            description(model.description);
            remediationUrl(model.remediationUrl);
            severityRating(model.severityRating);
            relatedRequirements(model.relatedRequirements);
        }

        public final String getStandardsControlArn() {
            return standardsControlArn;
        }

        @Override
        public final Builder standardsControlArn(String standardsControlArn) {
            this.standardsControlArn = standardsControlArn;
            return this;
        }

        public final void setStandardsControlArn(String standardsControlArn) {
            this.standardsControlArn = standardsControlArn;
        }

        public final String getControlStatus() {
            return controlStatus;
        }

        @Override
        public final Builder controlStatus(String controlStatus) {
            this.controlStatus = controlStatus;
            return this;
        }

        @Override
        public final Builder controlStatus(ControlStatus controlStatus) {
            this.controlStatus(controlStatus == null ? null : controlStatus.toString());
            return this;
        }

        public final void setControlStatus(String controlStatus) {
            this.controlStatus = controlStatus;
        }

        public final String getDisabledReason() {
            return disabledReason;
        }

        @Override
        public final Builder disabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
            return this;
        }

        public final void setDisabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
        }

        public final Instant getControlStatusUpdatedAt() {
            return controlStatusUpdatedAt;
        }

        @Override
        public final Builder controlStatusUpdatedAt(Instant controlStatusUpdatedAt) {
            this.controlStatusUpdatedAt = controlStatusUpdatedAt;
            return this;
        }

        public final void setControlStatusUpdatedAt(Instant controlStatusUpdatedAt) {
            this.controlStatusUpdatedAt = controlStatusUpdatedAt;
        }

        public final String getControlId() {
            return controlId;
        }

        @Override
        public final Builder controlId(String controlId) {
            this.controlId = controlId;
            return this;
        }

        public final void setControlId(String controlId) {
            this.controlId = controlId;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getRemediationUrl() {
            return remediationUrl;
        }

        @Override
        public final Builder remediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
            return this;
        }

        public final void setRemediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
        }

        public final String getSeverityRating() {
            return severityRating;
        }

        @Override
        public final Builder severityRating(String severityRating) {
            this.severityRating = severityRating;
            return this;
        }

        @Override
        public final Builder severityRating(SeverityRating severityRating) {
            this.severityRating(severityRating == null ? null : severityRating.toString());
            return this;
        }

        public final void setSeverityRating(String severityRating) {
            this.severityRating = severityRating;
        }

        public final Collection<String> getRelatedRequirements() {
            if (relatedRequirements instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedRequirements;
        }

        @Override
        public final Builder relatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedRequirements(String... relatedRequirements) {
            relatedRequirements(Arrays.asList(relatedRequirements));
            return this;
        }

        public final void setRelatedRequirements(Collection<String> relatedRequirements) {
            this.relatedRequirements = RelatedRequirementsListCopier.copy(relatedRequirements);
        }

        @Override
        public StandardsControl build() {
            return new StandardsControl(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
