/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.securityhub.model.AcceptAdministratorInvitationRequest;
import software.amazon.awssdk.services.securityhub.model.AcceptAdministratorInvitationResponse;
import software.amazon.awssdk.services.securityhub.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.securityhub.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.securityhub.model.AccessDeniedException;
import software.amazon.awssdk.services.securityhub.model.BatchDisableStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchDisableStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.BatchEnableStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchEnableStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.BatchImportFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchImportFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.BatchUpdateFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchUpdateFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.CreateActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.CreateActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.CreateFindingAggregatorRequest;
import software.amazon.awssdk.services.securityhub.model.CreateFindingAggregatorResponse;
import software.amazon.awssdk.services.securityhub.model.CreateInsightRequest;
import software.amazon.awssdk.services.securityhub.model.CreateInsightResponse;
import software.amazon.awssdk.services.securityhub.model.CreateMembersRequest;
import software.amazon.awssdk.services.securityhub.model.CreateMembersResponse;
import software.amazon.awssdk.services.securityhub.model.DeclineInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.DeclineInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteFindingAggregatorRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteFindingAggregatorResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteInsightRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteInsightResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteMembersRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteMembersResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeHubRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeHubResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeProductsResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.DisableImportFindingsForProductRequest;
import software.amazon.awssdk.services.securityhub.model.DisableImportFindingsForProductResponse;
import software.amazon.awssdk.services.securityhub.model.DisableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.securityhub.model.DisableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.securityhub.model.DisableSecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.DisableSecurityHubResponse;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromAdministratorAccountRequest;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromAdministratorAccountResponse;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromMasterAccountRequest;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromMasterAccountResponse;
import software.amazon.awssdk.services.securityhub.model.DisassociateMembersRequest;
import software.amazon.awssdk.services.securityhub.model.DisassociateMembersResponse;
import software.amazon.awssdk.services.securityhub.model.EnableImportFindingsForProductRequest;
import software.amazon.awssdk.services.securityhub.model.EnableImportFindingsForProductResponse;
import software.amazon.awssdk.services.securityhub.model.EnableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.securityhub.model.EnableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.securityhub.model.EnableSecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.EnableSecurityHubResponse;
import software.amazon.awssdk.services.securityhub.model.GetAdministratorAccountRequest;
import software.amazon.awssdk.services.securityhub.model.GetAdministratorAccountResponse;
import software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.GetFindingAggregatorRequest;
import software.amazon.awssdk.services.securityhub.model.GetFindingAggregatorResponse;
import software.amazon.awssdk.services.securityhub.model.GetFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.GetFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInsightResultsRequest;
import software.amazon.awssdk.services.securityhub.model.GetInsightResultsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInsightsRequest;
import software.amazon.awssdk.services.securityhub.model.GetInsightsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInvitationsCountRequest;
import software.amazon.awssdk.services.securityhub.model.GetInvitationsCountResponse;
import software.amazon.awssdk.services.securityhub.model.GetMasterAccountRequest;
import software.amazon.awssdk.services.securityhub.model.GetMasterAccountResponse;
import software.amazon.awssdk.services.securityhub.model.GetMembersRequest;
import software.amazon.awssdk.services.securityhub.model.GetMembersResponse;
import software.amazon.awssdk.services.securityhub.model.InternalException;
import software.amazon.awssdk.services.securityhub.model.InvalidAccessException;
import software.amazon.awssdk.services.securityhub.model.InvalidInputException;
import software.amazon.awssdk.services.securityhub.model.InviteMembersRequest;
import software.amazon.awssdk.services.securityhub.model.InviteMembersResponse;
import software.amazon.awssdk.services.securityhub.model.LimitExceededException;
import software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest;
import software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportResponse;
import software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsRequest;
import software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsResponse;
import software.amazon.awssdk.services.securityhub.model.ListInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.ListInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.ListMembersRequest;
import software.amazon.awssdk.services.securityhub.model.ListMembersResponse;
import software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsRequest;
import software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsResponse;
import software.amazon.awssdk.services.securityhub.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.securityhub.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.securityhub.model.ResourceConflictException;
import software.amazon.awssdk.services.securityhub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.securityhub.model.SecurityHubException;
import software.amazon.awssdk.services.securityhub.model.SecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.TagResourceRequest;
import software.amazon.awssdk.services.securityhub.model.TagResourceResponse;
import software.amazon.awssdk.services.securityhub.model.UntagResourceRequest;
import software.amazon.awssdk.services.securityhub.model.UntagResourceResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingAggregatorRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingAggregatorResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateInsightRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateInsightResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateSecurityHubConfigurationRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateSecurityHubConfigurationResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateStandardsControlRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateStandardsControlResponse;
import software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.DescribeProductsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsControlsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.GetEnabledStandardsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.GetFindingsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.GetInsightsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportPublisher;
import software.amazon.awssdk.services.securityhub.paginators.ListFindingAggregatorsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.ListInvitationsPublisher;
import software.amazon.awssdk.services.securityhub.paginators.ListMembersPublisher;
import software.amazon.awssdk.services.securityhub.paginators.ListOrganizationAdminAccountsPublisher;
import software.amazon.awssdk.services.securityhub.transform.AcceptAdministratorInvitationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchDisableStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchEnableStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchImportFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchUpdateFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateFindingAggregatorRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeclineInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteFindingAggregatorRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeActionTargetsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeProductsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeStandardsControlsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisableImportFindingsForProductRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisableSecurityHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisassociateFromAdministratorAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisassociateFromMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisassociateMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.EnableImportFindingsForProductRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.EnableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.EnableSecurityHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetAdministratorAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetEnabledStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetFindingAggregatorRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInsightResultsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInsightsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInvitationsCountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.InviteMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListEnabledProductsForImportRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListFindingAggregatorsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListOrganizationAdminAccountsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateFindingAggregatorRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateSecurityHubConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateStandardsControlRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SecurityHubAsyncClient}.
 *
 * @see SecurityHubAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecurityHubAsyncClient implements SecurityHubAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSecurityHubAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSecurityHubAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts the invitation to be a member account and be monitored by the Security Hub administrator account that the
     * invitation was sent from.
     * </p>
     * <p>
     * This operation is only used by member accounts that are not added through Organizations.
     * </p>
     * <p>
     * When the member account accepts the invitation, permission is granted to the administrator account to view
     * findings generated in the member account.
     * </p>
     *
     * @param acceptAdministratorInvitationRequest
     * @return A Java Future containing the result of the AcceptAdministratorInvitation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.AcceptAdministratorInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/AcceptAdministratorInvitation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptAdministratorInvitationResponse> acceptAdministratorInvitation(
            AcceptAdministratorInvitationRequest acceptAdministratorInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                acceptAdministratorInvitationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptAdministratorInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptAdministratorInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptAdministratorInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptAdministratorInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptAdministratorInvitationRequest, AcceptAdministratorInvitationResponse>()
                            .withOperationName("AcceptAdministratorInvitation")
                            .withMarshaller(new AcceptAdministratorInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptAdministratorInvitationRequest));
            CompletableFuture<AcceptAdministratorInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This method is deprecated. Instead, use <code>AcceptAdministratorInvitation</code>.
     * </p>
     * <p>
     * The Security Hub console continues to use <code>AcceptInvitation</code>. It will eventually change to use
     * <code>AcceptAdministratorInvitation</code>. Any IAM policies that specifically control access to this function
     * must continue to use <code>AcceptInvitation</code>. You should also add
     * <code>AcceptAdministratorInvitation</code> to your policies to ensure that the correct permissions are in place
     * after the console begins to use <code>AcceptAdministratorInvitation</code>.
     * </p>
     * <p>
     * Accepts the invitation to be a member account and be monitored by the Security Hub administrator account that the
     * invitation was sent from.
     * </p>
     * <p>
     * This operation is only used by member accounts that are not added through Organizations.
     * </p>
     * <p>
     * When the member account accepts the invitation, permission is granted to the administrator account to view
     * findings generated in the member account.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return A Java Future containing the result of the AcceptInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.AcceptInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/AcceptInvitation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptInvitationResponse> acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                            .withOperationName("AcceptInvitation")
                            .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptInvitationRequest));
            CompletableFuture<AcceptInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the standards specified by the provided <code>StandardsSubscriptionArns</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards.html">Security Standards</a>
     * section of the <i>Security Hub User Guide</i>.
     * </p>
     *
     * @param batchDisableStandardsRequest
     * @return A Java Future containing the result of the BatchDisableStandards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.BatchDisableStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchDisableStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDisableStandardsResponse> batchDisableStandards(
            BatchDisableStandardsRequest batchDisableStandardsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDisableStandardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDisableStandards");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDisableStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDisableStandardsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDisableStandardsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDisableStandardsRequest, BatchDisableStandardsResponse>()
                            .withOperationName("BatchDisableStandards")
                            .withMarshaller(new BatchDisableStandardsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchDisableStandardsRequest));
            CompletableFuture<BatchDisableStandardsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables the standards specified by the provided <code>StandardsArn</code>. To obtain the ARN for a standard, use
     * the <code>DescribeStandards</code> operation.
     * </p>
     * <p>
     * For more information, see the <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards.html">Security Standards</a>
     * section of the <i>Security Hub User Guide</i>.
     * </p>
     *
     * @param batchEnableStandardsRequest
     * @return A Java Future containing the result of the BatchEnableStandards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.BatchEnableStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchEnableStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEnableStandardsResponse> batchEnableStandards(
            BatchEnableStandardsRequest batchEnableStandardsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEnableStandardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEnableStandards");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEnableStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEnableStandardsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchEnableStandardsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEnableStandardsRequest, BatchEnableStandardsResponse>()
                            .withOperationName("BatchEnableStandards")
                            .withMarshaller(new BatchEnableStandardsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchEnableStandardsRequest));
            CompletableFuture<BatchEnableStandardsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports security findings generated by a finding provider into Security Hub. This action is requested by the
     * finding provider to import its findings into Security Hub.
     * </p>
     * <p>
     * <code>BatchImportFindings</code> must be called by one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The account that is associated with the findings. The identifier of the associated account is the value of the
     * <code>AwsAccountId</code> attribute for the finding.
     * </p>
     * </li>
     * <li>
     * <p>
     * An account that is allow-listed for an official Security Hub partner integration.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The maximum allowed size for a finding is 240 Kb. An error is returned for any finding larger than 240 Kb.
     * </p>
     * <p>
     * After a finding is created, <code>BatchImportFindings</code> cannot be used to update the following finding
     * fields and objects, which Security Hub customers use to manage their investigation workflow.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Note</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UserDefinedFields</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VerificationState</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Workflow</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Finding providers also should not use <code>BatchImportFindings</code> to update the following attributes.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Confidence</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Criticality</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RelatedFindings</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Severity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Types</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Instead, finding providers use <code>FindingProviderFields</code> to provide values for these attributes.
     * </p>
     *
     * @param batchImportFindingsRequest
     * @return A Java Future containing the result of the BatchImportFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.BatchImportFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchImportFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchImportFindingsResponse> batchImportFindings(
            BatchImportFindingsRequest batchImportFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchImportFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchImportFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchImportFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchImportFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchImportFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchImportFindingsRequest, BatchImportFindingsResponse>()
                            .withOperationName("BatchImportFindings")
                            .withMarshaller(new BatchImportFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchImportFindingsRequest));
            CompletableFuture<BatchImportFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used by Security Hub customers to update information about their investigation into a finding. Requested by
     * administrator accounts or member accounts. Administrator accounts can update findings for their account and their
     * member accounts. Member accounts can update findings for their account.
     * </p>
     * <p>
     * Updates from <code>BatchUpdateFindings</code> do not affect the value of <code>UpdatedAt</code> for a finding.
     * </p>
     * <p>
     * Administrator and member accounts can use <code>BatchUpdateFindings</code> to update the following finding fields
     * and objects.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Confidence</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Criticality</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Note</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RelatedFindings</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Severity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Types</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UserDefinedFields</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VerificationState</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Workflow</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can configure IAM policies to restrict access to fields and field values. For example, you might not want
     * member accounts to be able to suppress findings or change the finding severity. See <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/finding-update-batchupdatefindings.html#batchupdatefindings-configure-access"
     * >Configuring access to BatchUpdateFindings</a> in the <i>Security Hub User Guide</i>.
     * </p>
     *
     * @param batchUpdateFindingsRequest
     * @return A Java Future containing the result of the BatchUpdateFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.BatchUpdateFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchUpdateFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateFindingsResponse> batchUpdateFindings(
            BatchUpdateFindingsRequest batchUpdateFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchUpdateFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateFindingsRequest, BatchUpdateFindingsResponse>()
                            .withOperationName("BatchUpdateFindings")
                            .withMarshaller(new BatchUpdateFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchUpdateFindingsRequest));
            CompletableFuture<BatchUpdateFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a custom action target in Security Hub.
     * </p>
     * <p>
     * You can use custom actions on findings and insights in Security Hub to trigger target actions in Amazon
     * CloudWatch Events.
     * </p>
     *
     * @param createActionTargetRequest
     * @return A Java Future containing the result of the CreateActionTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.CreateActionTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateActionTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateActionTargetResponse> createActionTarget(CreateActionTargetRequest createActionTargetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createActionTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateActionTarget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateActionTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateActionTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateActionTargetRequest, CreateActionTargetResponse>()
                            .withOperationName("CreateActionTarget")
                            .withMarshaller(new CreateActionTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createActionTargetRequest));
            CompletableFuture<CreateActionTargetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to enable finding aggregation. Must be called from the aggregation Region.
     * </p>
     * <p>
     * For more details about cross-Region replication, see <a
     * href="securityhub/latest/userguide/finding-aggregation.html">Configuring finding aggregation</a> in the
     * <i>Security Hub User Guide</i>.
     * </p>
     *
     * @param createFindingAggregatorRequest
     * @return A Java Future containing the result of the CreateFindingAggregator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.CreateFindingAggregator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateFindingAggregator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFindingAggregatorResponse> createFindingAggregator(
            CreateFindingAggregatorRequest createFindingAggregatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFindingAggregatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFindingAggregator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFindingAggregatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFindingAggregatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFindingAggregatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFindingAggregatorRequest, CreateFindingAggregatorResponse>()
                            .withOperationName("CreateFindingAggregator")
                            .withMarshaller(new CreateFindingAggregatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFindingAggregatorRequest));
            CompletableFuture<CreateFindingAggregatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a custom insight in Security Hub. An insight is a consolidation of findings that relate to a security
     * issue that requires attention or remediation.
     * </p>
     * <p>
     * To group the related findings in the insight, use the <code>GroupByAttribute</code>.
     * </p>
     *
     * @param createInsightRequest
     * @return A Java Future containing the result of the CreateInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.CreateInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInsightResponse> createInsight(CreateInsightRequest createInsightRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInsightRequest, CreateInsightResponse>()
                            .withOperationName("CreateInsight")
                            .withMarshaller(new CreateInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createInsightRequest));
            CompletableFuture<CreateInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a member association in Security Hub between the specified accounts and the account used to make the
     * request, which is the administrator account. If you are integrated with Organizations, then the administrator
     * account is designated by the organization management account.
     * </p>
     * <p>
     * <code>CreateMembers</code> is always used to add accounts that are not organization members.
     * </p>
     * <p>
     * For accounts that are managed using Organizations, <code>CreateMembers</code> is only used in the following
     * cases:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Security Hub is not configured to automatically add new organization accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * The account was disassociated or deleted in Security Hub.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This action can only be used by an account that has Security Hub enabled. To enable Security Hub, you can use the
     * <code>EnableSecurityHub</code> operation.
     * </p>
     * <p>
     * For accounts that are not organization members, you create the account association and then send an invitation to
     * the member account. To send the invitation, you use the <code>InviteMembers</code> operation. If the account
     * owner accepts the invitation, the account becomes a member account in Security Hub.
     * </p>
     * <p>
     * Accounts that are managed using Organizations do not receive an invitation. They automatically become a member
     * account in Security Hub.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the organization account does not have Security Hub enabled, then Security Hub and the default standards are
     * automatically enabled. Note that Security Hub cannot be enabled automatically for the organization management
     * account. The organization management account must enable Security Hub before the administrator account enables it
     * as a member account.
     * </p>
     * </li>
     * <li>
     * <p>
     * For organization accounts that already have Security Hub enabled, Security Hub does not make any other changes to
     * those accounts. It does not change their enabled standards or controls.
     * </p>
     * </li>
     * </ul>
     * <p>
     * A permissions policy is added that permits the administrator account to view the findings generated in the member
     * account.
     * </p>
     * <p>
     * To remove the association between the administrator and member accounts, use the
     * <code>DisassociateFromMasterAccount</code> or <code>DisassociateMembers</code> operation.
     * </p>
     *
     * @param createMembersRequest
     * @return A Java Future containing the result of the CreateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.CreateMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMembersResponse> createMembers(CreateMembersRequest createMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                            .withOperationName("CreateMembers")
                            .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMembersRequest));
            CompletableFuture<CreateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Declines invitations to become a member account.
     * </p>
     * <p>
     * This operation is only used by accounts that are not part of an organization. Organization accounts do not
     * receive invitations.
     * </p>
     *
     * @param declineInvitationsRequest
     * @return A Java Future containing the result of the DeclineInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeclineInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeclineInvitations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeclineInvitationsResponse> declineInvitations(DeclineInvitationsRequest declineInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, declineInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeclineInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeclineInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeclineInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeclineInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeclineInvitationsRequest, DeclineInvitationsResponse>()
                            .withOperationName("DeclineInvitations")
                            .withMarshaller(new DeclineInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(declineInvitationsRequest));
            CompletableFuture<DeclineInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom action target from Security Hub.
     * </p>
     * <p>
     * Deleting a custom action target does not affect any findings or insights that were already sent to Amazon
     * CloudWatch Events using the custom action.
     * </p>
     *
     * @param deleteActionTargetRequest
     * @return A Java Future containing the result of the DeleteActionTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeleteActionTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteActionTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteActionTargetResponse> deleteActionTarget(DeleteActionTargetRequest deleteActionTargetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteActionTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteActionTarget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteActionTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteActionTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteActionTargetRequest, DeleteActionTargetResponse>()
                            .withOperationName("DeleteActionTarget")
                            .withMarshaller(new DeleteActionTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteActionTargetRequest));
            CompletableFuture<DeleteActionTargetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a finding aggregator. When you delete the finding aggregator, you stop finding aggregation.
     * </p>
     * <p>
     * When you stop finding aggregation, findings that were already aggregated to the aggregation Region are still
     * visible from the aggregation Region. New findings and finding updates are not aggregated.
     * </p>
     *
     * @param deleteFindingAggregatorRequest
     * @return A Java Future containing the result of the DeleteFindingAggregator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeleteFindingAggregator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteFindingAggregator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFindingAggregatorResponse> deleteFindingAggregator(
            DeleteFindingAggregatorRequest deleteFindingAggregatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFindingAggregatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFindingAggregator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFindingAggregatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFindingAggregatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFindingAggregatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFindingAggregatorRequest, DeleteFindingAggregatorResponse>()
                            .withOperationName("DeleteFindingAggregator")
                            .withMarshaller(new DeleteFindingAggregatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFindingAggregatorRequest));
            CompletableFuture<DeleteFindingAggregatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the insight specified by the <code>InsightArn</code>.
     * </p>
     *
     * @param deleteInsightRequest
     * @return A Java Future containing the result of the DeleteInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeleteInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInsightResponse> deleteInsight(DeleteInsightRequest deleteInsightRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInsightRequest, DeleteInsightResponse>()
                            .withOperationName("DeleteInsight")
                            .withMarshaller(new DeleteInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInsightRequest));
            CompletableFuture<DeleteInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes invitations received by the Amazon Web Services account to become a member account.
     * </p>
     * <p>
     * This operation is only used by accounts that are not part of an organization. Organization accounts do not
     * receive invitations.
     * </p>
     *
     * @param deleteInvitationsRequest
     * @return A Java Future containing the result of the DeleteInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeleteInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInvitationsResponse> deleteInvitations(DeleteInvitationsRequest deleteInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInvitationsRequest, DeleteInvitationsResponse>()
                            .withOperationName("DeleteInvitations")
                            .withMarshaller(new DeleteInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInvitationsRequest));
            CompletableFuture<DeleteInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified member accounts from Security Hub.
     * </p>
     * <p>
     * Can be used to delete member accounts that belong to an organization as well as member accounts that were invited
     * manually.
     * </p>
     *
     * @param deleteMembersRequest
     * @return A Java Future containing the result of the DeleteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DeleteMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMembersResponse> deleteMembers(DeleteMembersRequest deleteMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                            .withOperationName("DeleteMembers")
                            .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMembersRequest));
            CompletableFuture<DeleteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the custom action targets in Security Hub in your account.
     * </p>
     *
     * @param describeActionTargetsRequest
     * @return A Java Future containing the result of the DescribeActionTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeActionTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeActionTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeActionTargetsResponse> describeActionTargets(
            DescribeActionTargetsRequest describeActionTargetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeActionTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeActionTargets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeActionTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeActionTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeActionTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeActionTargetsRequest, DescribeActionTargetsResponse>()
                            .withOperationName("DescribeActionTargets")
                            .withMarshaller(new DescribeActionTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeActionTargetsRequest));
            CompletableFuture<DescribeActionTargetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the custom action targets in Security Hub in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeActionTargets(software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsPublisher publisher = client.describeActionTargetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsPublisher publisher = client.describeActionTargetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeActionTargets(software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeActionTargetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeActionTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeActionTargets"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeActionTargetsPublisher describeActionTargetsPaginator(DescribeActionTargetsRequest describeActionTargetsRequest) {
        return new DescribeActionTargetsPublisher(this, applyPaginatorUserAgent(describeActionTargetsRequest));
    }

    /**
     * <p>
     * Returns details about the Hub resource in your account, including the <code>HubArn</code> and the time when you
     * enabled Security Hub.
     * </p>
     *
     * @param describeHubRequest
     * @return A Java Future containing the result of the DescribeHub operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeHub
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeHub" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeHubResponse> describeHub(DescribeHubRequest describeHubRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHubRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHub");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeHubResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeHubResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeHubResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeHubRequest, DescribeHubResponse>()
                            .withOperationName("DescribeHub").withMarshaller(new DescribeHubRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeHubRequest));
            CompletableFuture<DescribeHubResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the Organizations configuration for Security Hub. Can only be called from a Security
     * Hub administrator account.
     * </p>
     *
     * @param describeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the DescribeOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationConfigurationResponse> describeOrganizationConfiguration(
            DescribeOrganizationConfigurationRequest describeOrganizationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationConfigurationRequest, DescribeOrganizationConfigurationResponse>()
                            .withOperationName("DescribeOrganizationConfiguration")
                            .withMarshaller(new DescribeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeOrganizationConfigurationRequest));
            CompletableFuture<DescribeOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about product integrations in Security Hub.
     * </p>
     * <p>
     * You can optionally provide an integration ARN. If you provide an integration ARN, then the results only include
     * that integration.
     * </p>
     * <p>
     * If you do not provide an integration ARN, then the results include all of the available product integrations.
     * </p>
     *
     * @param describeProductsRequest
     * @return A Java Future containing the result of the DescribeProducts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeProducts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeProducts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProductsResponse> describeProducts(DescribeProductsRequest describeProductsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProductsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProducts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProductsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProductsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProductsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProductsRequest, DescribeProductsResponse>()
                            .withOperationName("DescribeProducts")
                            .withMarshaller(new DescribeProductsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProductsRequest));
            CompletableFuture<DescribeProductsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about product integrations in Security Hub.
     * </p>
     * <p>
     * You can optionally provide an integration ARN. If you provide an integration ARN, then the results only include
     * that integration.
     * </p>
     * <p>
     * If you do not provide an integration ARN, then the results include all of the available product integrations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeProducts(software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeProductsPublisher publisher = client.describeProductsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeProductsPublisher publisher = client.describeProductsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.DescribeProductsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.DescribeProductsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeProducts(software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeProductsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeProducts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeProducts" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeProductsPublisher describeProductsPaginator(DescribeProductsRequest describeProductsRequest) {
        return new DescribeProductsPublisher(this, applyPaginatorUserAgent(describeProductsRequest));
    }

    /**
     * <p>
     * Returns a list of the available standards in Security Hub.
     * </p>
     * <p>
     * For each standard, the results include the standard ARN, the name, and a description.
     * </p>
     *
     * @param describeStandardsRequest
     * @return A Java Future containing the result of the DescribeStandards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeStandards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStandardsResponse> describeStandards(DescribeStandardsRequest describeStandardsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStandardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStandards");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStandardsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStandardsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStandardsRequest, DescribeStandardsResponse>()
                            .withOperationName("DescribeStandards")
                            .withMarshaller(new DescribeStandardsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeStandardsRequest));
            CompletableFuture<DescribeStandardsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of security standards controls.
     * </p>
     * <p>
     * For each control, the results include information about whether it is currently enabled, the severity, and a link
     * to remediation information.
     * </p>
     *
     * @param describeStandardsControlsRequest
     * @return A Java Future containing the result of the DescribeStandardsControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeStandardsControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeStandardsControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeStandardsControlsResponse> describeStandardsControls(
            DescribeStandardsControlsRequest describeStandardsControlsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStandardsControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStandardsControls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeStandardsControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeStandardsControlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeStandardsControlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeStandardsControlsRequest, DescribeStandardsControlsResponse>()
                            .withOperationName("DescribeStandardsControls")
                            .withMarshaller(new DescribeStandardsControlsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeStandardsControlsRequest));
            CompletableFuture<DescribeStandardsControlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of security standards controls.
     * </p>
     * <p>
     * For each control, the results include information about whether it is currently enabled, the severity, and a link
     * to remediation information.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeStandardsControls(software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsControlsPublisher publisher = client.describeStandardsControlsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsControlsPublisher publisher = client.describeStandardsControlsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeStandardsControls(software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeStandardsControlsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeStandardsControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeStandardsControls"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeStandardsControlsPublisher describeStandardsControlsPaginator(
            DescribeStandardsControlsRequest describeStandardsControlsRequest) {
        return new DescribeStandardsControlsPublisher(this, applyPaginatorUserAgent(describeStandardsControlsRequest));
    }

    /**
     * <p>
     * Returns a list of the available standards in Security Hub.
     * </p>
     * <p>
     * For each standard, the results include the standard ARN, the name, and a description.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeStandards(software.amazon.awssdk.services.securityhub.model.DescribeStandardsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsPublisher publisher = client.describeStandardsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeStandardsPublisher publisher = client.describeStandardsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.DescribeStandardsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.DescribeStandardsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeStandards(software.amazon.awssdk.services.securityhub.model.DescribeStandardsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeStandardsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DescribeStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeStandards" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeStandardsPublisher describeStandardsPaginator(DescribeStandardsRequest describeStandardsRequest) {
        return new DescribeStandardsPublisher(this, applyPaginatorUserAgent(describeStandardsRequest));
    }

    /**
     * <p>
     * Disables the integration of the specified product with Security Hub. After the integration is disabled, findings
     * from that product are no longer sent to Security Hub.
     * </p>
     *
     * @param disableImportFindingsForProductRequest
     * @return A Java Future containing the result of the DisableImportFindingsForProduct operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisableImportFindingsForProduct
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisableImportFindingsForProduct"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableImportFindingsForProductResponse> disableImportFindingsForProduct(
            DisableImportFindingsForProductRequest disableImportFindingsForProductRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableImportFindingsForProductRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableImportFindingsForProduct");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableImportFindingsForProductResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableImportFindingsForProductResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableImportFindingsForProductResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableImportFindingsForProductRequest, DisableImportFindingsForProductResponse>()
                            .withOperationName("DisableImportFindingsForProduct")
                            .withMarshaller(new DisableImportFindingsForProductRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableImportFindingsForProductRequest));
            CompletableFuture<DisableImportFindingsForProductResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables a Security Hub administrator account. Can only be called by the organization management account.
     * </p>
     *
     * @param disableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the DisableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableOrganizationAdminAccountResponse> disableOrganizationAdminAccount(
            DisableOrganizationAdminAccountRequest disableOrganizationAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableOrganizationAdminAccountRequest, DisableOrganizationAdminAccountResponse>()
                            .withOperationName("DisableOrganizationAdminAccount")
                            .withMarshaller(new DisableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableOrganizationAdminAccountRequest));
            CompletableFuture<DisableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables Security Hub in your account only in the current Region. To disable Security Hub in all Regions, you
     * must submit one request per Region where you have enabled Security Hub.
     * </p>
     * <p>
     * When you disable Security Hub for an administrator account, it doesn't disable Security Hub for any associated
     * member accounts.
     * </p>
     * <p>
     * When you disable Security Hub, your existing findings and insights and any Security Hub configuration settings
     * are deleted after 90 days and cannot be recovered. Any standards that were enabled are disabled, and your
     * administrator and member account associations are removed.
     * </p>
     * <p>
     * If you want to save your existing findings, you must export them before you disable Security Hub.
     * </p>
     *
     * @param disableSecurityHubRequest
     * @return A Java Future containing the result of the DisableSecurityHub operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisableSecurityHub
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisableSecurityHub"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableSecurityHubResponse> disableSecurityHub(DisableSecurityHubRequest disableSecurityHubRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableSecurityHubRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableSecurityHub");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableSecurityHubResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableSecurityHubResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableSecurityHubResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableSecurityHubRequest, DisableSecurityHubResponse>()
                            .withOperationName("DisableSecurityHub")
                            .withMarshaller(new DisableSecurityHubRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableSecurityHubRequest));
            CompletableFuture<DisableSecurityHubResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the current Security Hub member account from the associated administrator account.
     * </p>
     * <p>
     * This operation is only used by accounts that are not part of an organization. For organization accounts, only the
     * administrator account can disassociate a member account.
     * </p>
     *
     * @param disassociateFromAdministratorAccountRequest
     * @return A Java Future containing the result of the DisassociateFromAdministratorAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisassociateFromAdministratorAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisassociateFromAdministratorAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFromAdministratorAccountResponse> disassociateFromAdministratorAccount(
            DisassociateFromAdministratorAccountRequest disassociateFromAdministratorAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFromAdministratorAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFromAdministratorAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFromAdministratorAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateFromAdministratorAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateFromAdministratorAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFromAdministratorAccountRequest, DisassociateFromAdministratorAccountResponse>()
                            .withOperationName("DisassociateFromAdministratorAccount")
                            .withMarshaller(new DisassociateFromAdministratorAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateFromAdministratorAccountRequest));
            CompletableFuture<DisassociateFromAdministratorAccountResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This method is deprecated. Instead, use <code>DisassociateFromAdministratorAccount</code>.
     * </p>
     * <p>
     * The Security Hub console continues to use <code>DisassociateFromMasterAccount</code>. It will eventually change
     * to use <code>DisassociateFromAdministratorAccount</code>. Any IAM policies that specifically control access to
     * this function must continue to use <code>DisassociateFromMasterAccount</code>. You should also add
     * <code>DisassociateFromAdministratorAccount</code> to your policies to ensure that the correct permissions are in
     * place after the console begins to use <code>DisassociateFromAdministratorAccount</code>.
     * </p>
     * <p>
     * Disassociates the current Security Hub member account from the associated administrator account.
     * </p>
     * <p>
     * This operation is only used by accounts that are not part of an organization. For organization accounts, only the
     * administrator account can disassociate a member account.
     * </p>
     *
     * @param disassociateFromMasterAccountRequest
     * @return A Java Future containing the result of the DisassociateFromMasterAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisassociateFromMasterAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisassociateFromMasterAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFromMasterAccountResponse> disassociateFromMasterAccount(
            DisassociateFromMasterAccountRequest disassociateFromMasterAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFromMasterAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFromMasterAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFromMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateFromMasterAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateFromMasterAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFromMasterAccountRequest, DisassociateFromMasterAccountResponse>()
                            .withOperationName("DisassociateFromMasterAccount")
                            .withMarshaller(new DisassociateFromMasterAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateFromMasterAccountRequest));
            CompletableFuture<DisassociateFromMasterAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the specified member accounts from the associated administrator account.
     * </p>
     * <p>
     * Can be used to disassociate both accounts that are managed using Organizations and accounts that were invited
     * manually.
     * </p>
     *
     * @param disassociateMembersRequest
     * @return A Java Future containing the result of the DisassociateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.DisassociateMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisassociateMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateMembersResponse> disassociateMembers(
            DisassociateMembersRequest disassociateMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateMembersRequest, DisassociateMembersResponse>()
                            .withOperationName("DisassociateMembers")
                            .withMarshaller(new DisassociateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateMembersRequest));
            CompletableFuture<DisassociateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables the integration of a partner product with Security Hub. Integrated products send findings to Security
     * Hub.
     * </p>
     * <p>
     * When you enable a product integration, a permissions policy that grants permission for the product to send
     * findings to Security Hub is applied.
     * </p>
     *
     * @param enableImportFindingsForProductRequest
     * @return A Java Future containing the result of the EnableImportFindingsForProduct operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.EnableImportFindingsForProduct
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/EnableImportFindingsForProduct"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableImportFindingsForProductResponse> enableImportFindingsForProduct(
            EnableImportFindingsForProductRequest enableImportFindingsForProductRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableImportFindingsForProductRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableImportFindingsForProduct");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableImportFindingsForProductResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableImportFindingsForProductResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableImportFindingsForProductResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableImportFindingsForProductRequest, EnableImportFindingsForProductResponse>()
                            .withOperationName("EnableImportFindingsForProduct")
                            .withMarshaller(new EnableImportFindingsForProductRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableImportFindingsForProductRequest));
            CompletableFuture<EnableImportFindingsForProductResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the Security Hub administrator account for an organization. Can only be called by the organization
     * management account.
     * </p>
     *
     * @param enableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the EnableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.EnableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/EnableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableOrganizationAdminAccountResponse> enableOrganizationAdminAccount(
            EnableOrganizationAdminAccountRequest enableOrganizationAdminAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableOrganizationAdminAccountRequest, EnableOrganizationAdminAccountResponse>()
                            .withOperationName("EnableOrganizationAdminAccount")
                            .withMarshaller(new EnableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableOrganizationAdminAccountRequest));
            CompletableFuture<EnableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables Security Hub for your account in the current Region or the Region you specify in the request.
     * </p>
     * <p>
     * When you enable Security Hub, you grant to Security Hub the permissions necessary to gather findings from other
     * services that are integrated with Security Hub.
     * </p>
     * <p>
     * When you use the <code>EnableSecurityHub</code> operation to enable Security Hub, you also automatically enable
     * the following standards.
     * </p>
     * <ul>
     * <li>
     * <p>
     * CIS Amazon Web Services Foundations
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services Foundational Security Best Practices
     * </p>
     * </li>
     * </ul>
     * <p>
     * You do not enable the Payment Card Industry Data Security Standard (PCI DSS) standard.
     * </p>
     * <p>
     * To not enable the automatically enabled standards, set <code>EnableDefaultStandards</code> to <code>false</code>.
     * </p>
     * <p>
     * After you enable Security Hub, to enable a standard, use the <code>BatchEnableStandards</code> operation. To
     * disable a standard, use the <code>BatchDisableStandards</code> operation.
     * </p>
     * <p>
     * To learn more, see the <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-settingup.html">setup information</a>
     * in the <i>Security Hub User Guide</i>.
     * </p>
     *
     * @param enableSecurityHubRequest
     * @return A Java Future containing the result of the EnableSecurityHub operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.EnableSecurityHub
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/EnableSecurityHub" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableSecurityHubResponse> enableSecurityHub(EnableSecurityHubRequest enableSecurityHubRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableSecurityHubRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSecurityHub");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableSecurityHubResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableSecurityHubResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableSecurityHubResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableSecurityHubRequest, EnableSecurityHubResponse>()
                            .withOperationName("EnableSecurityHub")
                            .withMarshaller(new EnableSecurityHubRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableSecurityHubRequest));
            CompletableFuture<EnableSecurityHubResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides the details for the Security Hub administrator account for the current member account.
     * </p>
     * <p>
     * Can be used by both member accounts that are managed using Organizations and accounts that were invited manually.
     * </p>
     *
     * @param getAdministratorAccountRequest
     * @return A Java Future containing the result of the GetAdministratorAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetAdministratorAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetAdministratorAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAdministratorAccountResponse> getAdministratorAccount(
            GetAdministratorAccountRequest getAdministratorAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAdministratorAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAdministratorAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAdministratorAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAdministratorAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAdministratorAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAdministratorAccountRequest, GetAdministratorAccountResponse>()
                            .withOperationName("GetAdministratorAccount")
                            .withMarshaller(new GetAdministratorAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAdministratorAccountRequest));
            CompletableFuture<GetAdministratorAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the standards that are currently enabled.
     * </p>
     *
     * @param getEnabledStandardsRequest
     * @return A Java Future containing the result of the GetEnabledStandards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetEnabledStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetEnabledStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnabledStandardsResponse> getEnabledStandards(
            GetEnabledStandardsRequest getEnabledStandardsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnabledStandardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnabledStandards");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnabledStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnabledStandardsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnabledStandardsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnabledStandardsRequest, GetEnabledStandardsResponse>()
                            .withOperationName("GetEnabledStandards")
                            .withMarshaller(new GetEnabledStandardsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEnabledStandardsRequest));
            CompletableFuture<GetEnabledStandardsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the standards that are currently enabled.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getEnabledStandards(software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetEnabledStandardsPublisher publisher = client.getEnabledStandardsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetEnabledStandardsPublisher publisher = client.getEnabledStandardsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getEnabledStandards(software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getEnabledStandardsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetEnabledStandards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetEnabledStandards"
     *      target="_top">AWS API Documentation</a>
     */
    public GetEnabledStandardsPublisher getEnabledStandardsPaginator(GetEnabledStandardsRequest getEnabledStandardsRequest) {
        return new GetEnabledStandardsPublisher(this, applyPaginatorUserAgent(getEnabledStandardsRequest));
    }

    /**
     * <p>
     * Returns the current finding aggregation configuration.
     * </p>
     *
     * @param getFindingAggregatorRequest
     * @return A Java Future containing the result of the GetFindingAggregator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetFindingAggregator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetFindingAggregator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingAggregatorResponse> getFindingAggregator(
            GetFindingAggregatorRequest getFindingAggregatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingAggregatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFindingAggregator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingAggregatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetFindingAggregatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingAggregatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingAggregatorRequest, GetFindingAggregatorResponse>()
                            .withOperationName("GetFindingAggregator")
                            .withMarshaller(new GetFindingAggregatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFindingAggregatorRequest));
            CompletableFuture<GetFindingAggregatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of findings that match the specified criteria.
     * </p>
     * <p>
     * If finding aggregation is enabled, then when you call <code>GetFindings</code> from the aggregation Region, the
     * results include all of the matching findings from both the aggregation Region and the linked Regions.
     * </p>
     *
     * @param getFindingsRequest
     * @return A Java Future containing the result of the GetFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingsResponse> getFindings(GetFindingsRequest getFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingsRequest, GetFindingsResponse>()
                            .withOperationName("GetFindings").withMarshaller(new GetFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFindingsRequest));
            CompletableFuture<GetFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of findings that match the specified criteria.
     * </p>
     * <p>
     * If finding aggregation is enabled, then when you call <code>GetFindings</code> from the aggregation Region, the
     * results include all of the matching findings from both the aggregation Region and the linked Regions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getFindings(software.amazon.awssdk.services.securityhub.model.GetFindingsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetFindingsPublisher publisher = client.getFindingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetFindingsPublisher publisher = client.getFindingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.GetFindingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.GetFindingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFindings(software.amazon.awssdk.services.securityhub.model.GetFindingsRequest)} operation.</b>
     * </p>
     *
     * @param getFindingsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetFindings" target="_top">AWS API
     *      Documentation</a>
     */
    public GetFindingsPublisher getFindingsPaginator(GetFindingsRequest getFindingsRequest) {
        return new GetFindingsPublisher(this, applyPaginatorUserAgent(getFindingsRequest));
    }

    /**
     * <p>
     * Lists the results of the Security Hub insight specified by the insight ARN.
     * </p>
     *
     * @param getInsightResultsRequest
     * @return A Java Future containing the result of the GetInsightResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetInsightResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsightResults" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightResultsResponse> getInsightResults(GetInsightResultsRequest getInsightResultsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInsightResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightResultsRequest, GetInsightResultsResponse>()
                            .withOperationName("GetInsightResults")
                            .withMarshaller(new GetInsightResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInsightResultsRequest));
            CompletableFuture<GetInsightResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists and describes insights for the specified insight ARNs.
     * </p>
     *
     * @param getInsightsRequest
     * @return A Java Future containing the result of the GetInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightsResponse> getInsights(GetInsightsRequest getInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightsRequest, GetInsightsResponse>()
                            .withOperationName("GetInsights").withMarshaller(new GetInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInsightsRequest));
            CompletableFuture<GetInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists and describes insights for the specified insight ARNs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getInsights(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetInsightsPublisher publisher = client.getInsightsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetInsightsPublisher publisher = client.getInsightsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.GetInsightsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.GetInsightsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getInsights(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)} operation.</b>
     * </p>
     *
     * @param getInsightsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsights" target="_top">AWS API
     *      Documentation</a>
     */
    public GetInsightsPublisher getInsightsPaginator(GetInsightsRequest getInsightsRequest) {
        return new GetInsightsPublisher(this, applyPaginatorUserAgent(getInsightsRequest));
    }

    /**
     * <p>
     * Returns the count of all Security Hub membership invitations that were sent to the current member account, not
     * including the currently accepted invitation.
     * </p>
     *
     * @param getInvitationsCountRequest
     * @return A Java Future containing the result of the GetInvitationsCount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetInvitationsCount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInvitationsCount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInvitationsCountResponse> getInvitationsCount(
            GetInvitationsCountRequest getInvitationsCountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvitationsCountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvitationsCount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInvitationsCountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInvitationsCountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInvitationsCountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInvitationsCountRequest, GetInvitationsCountResponse>()
                            .withOperationName("GetInvitationsCount")
                            .withMarshaller(new GetInvitationsCountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInvitationsCountRequest));
            CompletableFuture<GetInvitationsCountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This method is deprecated. Instead, use <code>GetAdministratorAccount</code>.
     * </p>
     * <p>
     * The Security Hub console continues to use <code>GetMasterAccount</code>. It will eventually change to use
     * <code>GetAdministratorAccount</code>. Any IAM policies that specifically control access to this function must
     * continue to use <code>GetMasterAccount</code>. You should also add <code>GetAdministratorAccount</code> to your
     * policies to ensure that the correct permissions are in place after the console begins to use
     * <code>GetAdministratorAccount</code>.
     * </p>
     * <p>
     * Provides the details for the Security Hub administrator account for the current member account.
     * </p>
     * <p>
     * Can be used by both member accounts that are managed using Organizations and accounts that were invited manually.
     * </p>
     *
     * @param getMasterAccountRequest
     * @return A Java Future containing the result of the GetMasterAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetMasterAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetMasterAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMasterAccountResponse> getMasterAccount(GetMasterAccountRequest getMasterAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMasterAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMasterAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMasterAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMasterAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMasterAccountRequest, GetMasterAccountResponse>()
                            .withOperationName("GetMasterAccount")
                            .withMarshaller(new GetMasterAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMasterAccountRequest));
            CompletableFuture<GetMasterAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details for the Security Hub member accounts for the specified account IDs.
     * </p>
     * <p>
     * An administrator account can be either the delegated Security Hub administrator account for an organization or an
     * administrator account that enabled Security Hub manually.
     * </p>
     * <p>
     * The results include both member accounts that are managed using Organizations and accounts that were invited
     * manually.
     * </p>
     *
     * @param getMembersRequest
     * @return A Java Future containing the result of the GetMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.GetMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMembersResponse> getMembers(GetMembersRequest getMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>().withOperationName("GetMembers")
                            .withMarshaller(new GetMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMembersRequest));
            CompletableFuture<GetMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Invites other Amazon Web Services accounts to become member accounts for the Security Hub administrator account
     * that the invitation is sent from.
     * </p>
     * <p>
     * This operation is only used to invite accounts that do not belong to an organization. Organization accounts do
     * not receive invitations.
     * </p>
     * <p>
     * Before you can use this action to invite a member, you must first use the <code>CreateMembers</code> action to
     * create the member account in Security Hub.
     * </p>
     * <p>
     * When the account owner enables Security Hub and accepts the invitation to become a member account, the
     * administrator account can view the findings generated from the member account.
     * </p>
     *
     * @param inviteMembersRequest
     * @return A Java Future containing the result of the InviteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.InviteMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/InviteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<InviteMembersResponse> inviteMembers(InviteMembersRequest inviteMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, inviteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InviteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InviteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InviteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InviteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InviteMembersRequest, InviteMembersResponse>()
                            .withOperationName("InviteMembers")
                            .withMarshaller(new InviteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(inviteMembersRequest));
            CompletableFuture<InviteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all findings-generating solutions (products) that you are subscribed to receive findings from in Security
     * Hub.
     * </p>
     *
     * @param listEnabledProductsForImportRequest
     * @return A Java Future containing the result of the ListEnabledProductsForImport operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListEnabledProductsForImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListEnabledProductsForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnabledProductsForImportResponse> listEnabledProductsForImport(
            ListEnabledProductsForImportRequest listEnabledProductsForImportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnabledProductsForImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnabledProductsForImport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnabledProductsForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnabledProductsForImportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnabledProductsForImportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnabledProductsForImportRequest, ListEnabledProductsForImportResponse>()
                            .withOperationName("ListEnabledProductsForImport")
                            .withMarshaller(new ListEnabledProductsForImportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEnabledProductsForImportRequest));
            CompletableFuture<ListEnabledProductsForImportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all findings-generating solutions (products) that you are subscribed to receive findings from in Security
     * Hub.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnabledProductsForImport(software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportPublisher publisher = client.listEnabledProductsForImportPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportPublisher publisher = client.listEnabledProductsForImportPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnabledProductsForImport(software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnabledProductsForImportRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListEnabledProductsForImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListEnabledProductsForImport"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEnabledProductsForImportPublisher listEnabledProductsForImportPaginator(
            ListEnabledProductsForImportRequest listEnabledProductsForImportRequest) {
        return new ListEnabledProductsForImportPublisher(this, applyPaginatorUserAgent(listEnabledProductsForImportRequest));
    }

    /**
     * <p>
     * If finding aggregation is enabled, then <code>ListFindingAggregators</code> returns the ARN of the finding
     * aggregator. You can run this operation from any Region.
     * </p>
     *
     * @param listFindingAggregatorsRequest
     * @return A Java Future containing the result of the ListFindingAggregators operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListFindingAggregators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListFindingAggregators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFindingAggregatorsResponse> listFindingAggregators(
            ListFindingAggregatorsRequest listFindingAggregatorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFindingAggregatorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFindingAggregators");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFindingAggregatorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFindingAggregatorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFindingAggregatorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFindingAggregatorsRequest, ListFindingAggregatorsResponse>()
                            .withOperationName("ListFindingAggregators")
                            .withMarshaller(new ListFindingAggregatorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFindingAggregatorsRequest));
            CompletableFuture<ListFindingAggregatorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * If finding aggregation is enabled, then <code>ListFindingAggregators</code> returns the ARN of the finding
     * aggregator. You can run this operation from any Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFindingAggregators(software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListFindingAggregatorsPublisher publisher = client.listFindingAggregatorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListFindingAggregatorsPublisher publisher = client.listFindingAggregatorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFindingAggregators(software.amazon.awssdk.services.securityhub.model.ListFindingAggregatorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listFindingAggregatorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListFindingAggregators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListFindingAggregators"
     *      target="_top">AWS API Documentation</a>
     */
    public ListFindingAggregatorsPublisher listFindingAggregatorsPaginator(
            ListFindingAggregatorsRequest listFindingAggregatorsRequest) {
        return new ListFindingAggregatorsPublisher(this, applyPaginatorUserAgent(listFindingAggregatorsRequest));
    }

    /**
     * <p>
     * Lists all Security Hub membership invitations that were sent to the current Amazon Web Services account.
     * </p>
     * <p>
     * This operation is only used by accounts that are managed by invitation. Accounts that are managed using the
     * integration with Organizations do not receive invitations.
     * </p>
     *
     * @param listInvitationsRequest
     * @return A Java Future containing the result of the ListInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvitationsResponse> listInvitations(ListInvitationsRequest listInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                            .withOperationName("ListInvitations")
                            .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInvitationsRequest));
            CompletableFuture<ListInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Security Hub membership invitations that were sent to the current Amazon Web Services account.
     * </p>
     * <p>
     * This operation is only used by accounts that are managed by invitation. Accounts that are managed using the
     * integration with Organizations do not receive invitations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.securityhub.model.ListInvitationsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.ListInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.ListInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.securityhub.model.ListInvitationsRequest)} operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    public ListInvitationsPublisher listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest) {
        return new ListInvitationsPublisher(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Lists details about all member accounts for the current Security Hub administrator account.
     * </p>
     * <p>
     * The results include both member accounts that belong to an organization and member accounts that were invited
     * manually.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembersRequest));
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists details about all member accounts for the current Security Hub administrator account.
     * </p>
     * <p>
     * The results include both member accounts that belong to an organization and member accounts that were invited
     * manually.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.securityhub.model.ListMembersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.ListMembersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.ListMembersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.securityhub.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMembersPublisher listMembersPaginator(ListMembersRequest listMembersRequest) {
        return new ListMembersPublisher(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Lists the Security Hub administrator accounts. Can only be called by the organization management account.
     * </p>
     *
     * @param listOrganizationAdminAccountsRequest
     * @return A Java Future containing the result of the ListOrganizationAdminAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListOrganizationAdminAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListOrganizationAdminAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationAdminAccountsResponse> listOrganizationAdminAccounts(
            ListOrganizationAdminAccountsRequest listOrganizationAdminAccountsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationAdminAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationAdminAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationAdminAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOrganizationAdminAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationAdminAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationAdminAccountsRequest, ListOrganizationAdminAccountsResponse>()
                            .withOperationName("ListOrganizationAdminAccounts")
                            .withMarshaller(new ListOrganizationAdminAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOrganizationAdminAccountsRequest));
            CompletableFuture<ListOrganizationAdminAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Security Hub administrator accounts. Can only be called by the organization management account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOrganizationAdminAccounts(software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListOrganizationAdminAccountsPublisher publisher = client.listOrganizationAdminAccountsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListOrganizationAdminAccountsPublisher publisher = client.listOrganizationAdminAccountsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrganizationAdminAccounts(software.amazon.awssdk.services.securityhub.model.ListOrganizationAdminAccountsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOrganizationAdminAccountsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListOrganizationAdminAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListOrganizationAdminAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    public ListOrganizationAdminAccountsPublisher listOrganizationAdminAccountsPaginator(
            ListOrganizationAdminAccountsRequest listOrganizationAdminAccountsRequest) {
        return new ListOrganizationAdminAccountsPublisher(this, applyPaginatorUserAgent(listOrganizationAdminAccountsRequest));
    }

    /**
     * <p>
     * Returns a list of tags associated with a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name and description of a custom action target in Security Hub.
     * </p>
     *
     * @param updateActionTargetRequest
     * @return A Java Future containing the result of the UpdateActionTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateActionTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateActionTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateActionTargetResponse> updateActionTarget(UpdateActionTargetRequest updateActionTargetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateActionTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateActionTarget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateActionTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateActionTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateActionTargetRequest, UpdateActionTargetResponse>()
                            .withOperationName("UpdateActionTarget")
                            .withMarshaller(new UpdateActionTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateActionTargetRequest));
            CompletableFuture<UpdateActionTargetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the finding aggregation configuration. Used to update the Region linking mode and the list of included or
     * excluded Regions. You cannot use <code>UpdateFindingAggregator</code> to change the aggregation Region.
     * </p>
     * <p>
     * You must run <code>UpdateFindingAggregator</code> from the current aggregation Region.
     * </p>
     *
     * @param updateFindingAggregatorRequest
     * @return A Java Future containing the result of the UpdateFindingAggregator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action specified in the request.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateFindingAggregator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateFindingAggregator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFindingAggregatorResponse> updateFindingAggregator(
            UpdateFindingAggregatorRequest updateFindingAggregatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFindingAggregatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFindingAggregator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFindingAggregatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFindingAggregatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFindingAggregatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFindingAggregatorRequest, UpdateFindingAggregatorResponse>()
                            .withOperationName("UpdateFindingAggregator")
                            .withMarshaller(new UpdateFindingAggregatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFindingAggregatorRequest));
            CompletableFuture<UpdateFindingAggregatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>UpdateFindings</code> is deprecated. Instead of <code>UpdateFindings</code>, use
     * <code>BatchUpdateFindings</code>.
     * </p>
     * <p>
     * Updates the <code>Note</code> and <code>RecordState</code> of the Security Hub-aggregated findings that the
     * filter attributes specify. Any member account that can view the finding also sees the update to the finding.
     * </p>
     *
     * @param updateFindingsRequest
     * @return A Java Future containing the result of the UpdateFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFindingsResponse> updateFindings(UpdateFindingsRequest updateFindingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFindingsRequest, UpdateFindingsResponse>()
                            .withOperationName("UpdateFindings")
                            .withMarshaller(new UpdateFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFindingsRequest));
            CompletableFuture<UpdateFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the Security Hub insight identified by the specified insight ARN.
     * </p>
     *
     * @param updateInsightRequest
     * @return A Java Future containing the result of the UpdateInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateInsight
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInsightResponse> updateInsight(UpdateInsightRequest updateInsightRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInsightRequest, UpdateInsightResponse>()
                            .withOperationName("UpdateInsight")
                            .withMarshaller(new UpdateInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateInsightRequest));
            CompletableFuture<UpdateInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to update the configuration related to Organizations. Can only be called from a Security Hub administrator
     * account.
     * </p>
     *
     * @param updateOrganizationConfigurationRequest
     * @return A Java Future containing the result of the UpdateOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOrganizationConfigurationResponse> updateOrganizationConfiguration(
            UpdateOrganizationConfigurationRequest updateOrganizationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOrganizationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOrganizationConfigurationRequest, UpdateOrganizationConfigurationResponse>()
                            .withOperationName("UpdateOrganizationConfiguration")
                            .withMarshaller(new UpdateOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateOrganizationConfigurationRequest));
            CompletableFuture<UpdateOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates configuration options for Security Hub.
     * </p>
     *
     * @param updateSecurityHubConfigurationRequest
     * @return A Java Future containing the result of the UpdateSecurityHubConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current Amazon Web Services account or throttling limits. The error code describes the limit exceeded.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateSecurityHubConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateSecurityHubConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSecurityHubConfigurationResponse> updateSecurityHubConfiguration(
            UpdateSecurityHubConfigurationRequest updateSecurityHubConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSecurityHubConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSecurityHubConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSecurityHubConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSecurityHubConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSecurityHubConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSecurityHubConfigurationRequest, UpdateSecurityHubConfigurationResponse>()
                            .withOperationName("UpdateSecurityHubConfiguration")
                            .withMarshaller(new UpdateSecurityHubConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSecurityHubConfigurationRequest));
            CompletableFuture<UpdateSecurityHubConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to control whether an individual security standard control is enabled or disabled.
     * </p>
     *
     * @param updateStandardsControlRequest
     * @return A Java Future containing the result of the UpdateStandardsControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because you supplied an invalid or out-of-range value
     *         for an input parameter.</li>
     *         <li>InvalidAccessException There is an issue with the account used to make the request. Either Security
     *         Hub is not enabled for the account, or the account does not have permission to perform this action.</li>
     *         <li>ResourceNotFoundException The request was rejected because we can't find the specified resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityHubException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityHubAsyncClient.UpdateStandardsControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateStandardsControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStandardsControlResponse> updateStandardsControl(
            UpdateStandardsControlRequest updateStandardsControlRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStandardsControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStandardsControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStandardsControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateStandardsControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStandardsControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStandardsControlRequest, UpdateStandardsControlResponse>()
                            .withOperationName("UpdateStandardsControl")
                            .withMarshaller(new UpdateStandardsControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateStandardsControlRequest));
            CompletableFuture<UpdateStandardsControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecurityHubException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAccessException")
                                .exceptionBuilderSupplier(InvalidAccessException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SecurityHubRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
