/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of events that changed the specified finding during the specified time period. Each record represents a single
 * finding change event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FindingHistoryRecord implements SdkPojo, Serializable,
        ToCopyableBuilder<FindingHistoryRecord.Builder, FindingHistoryRecord> {
    private static final SdkField<AwsSecurityFindingIdentifier> FINDING_IDENTIFIER_FIELD = SdkField
            .<AwsSecurityFindingIdentifier> builder(MarshallingType.SDK_POJO).memberName("FindingIdentifier")
            .getter(getter(FindingHistoryRecord::findingIdentifier)).setter(setter(Builder::findingIdentifier))
            .constructor(AwsSecurityFindingIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FindingIdentifier").build()).build();

    private static final SdkField<Instant> UPDATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateTime")
            .getter(getter(FindingHistoryRecord::updateTime))
            .setter(setter(Builder::updateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Boolean> FINDING_CREATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("FindingCreated").getter(getter(FindingHistoryRecord::findingCreated))
            .setter(setter(Builder::findingCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FindingCreated").build()).build();

    private static final SdkField<FindingHistoryUpdateSource> UPDATE_SOURCE_FIELD = SdkField
            .<FindingHistoryUpdateSource> builder(MarshallingType.SDK_POJO).memberName("UpdateSource")
            .getter(getter(FindingHistoryRecord::updateSource)).setter(setter(Builder::updateSource))
            .constructor(FindingHistoryUpdateSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateSource").build()).build();

    private static final SdkField<List<FindingHistoryUpdate>> UPDATES_FIELD = SdkField
            .<List<FindingHistoryUpdate>> builder(MarshallingType.LIST)
            .memberName("Updates")
            .getter(getter(FindingHistoryRecord::updates))
            .setter(setter(Builder::updates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Updates").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FindingHistoryUpdate> builder(MarshallingType.SDK_POJO)
                                            .constructor(FindingHistoryUpdate::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(FindingHistoryRecord::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FINDING_IDENTIFIER_FIELD,
            UPDATE_TIME_FIELD, FINDING_CREATED_FIELD, UPDATE_SOURCE_FIELD, UPDATES_FIELD, NEXT_TOKEN_FIELD));

    private static final long serialVersionUID = 1L;

    private final AwsSecurityFindingIdentifier findingIdentifier;

    private final Instant updateTime;

    private final Boolean findingCreated;

    private final FindingHistoryUpdateSource updateSource;

    private final List<FindingHistoryUpdate> updates;

    private final String nextToken;

    private FindingHistoryRecord(BuilderImpl builder) {
        this.findingIdentifier = builder.findingIdentifier;
        this.updateTime = builder.updateTime;
        this.findingCreated = builder.findingCreated;
        this.updateSource = builder.updateSource;
        this.updates = builder.updates;
        this.nextToken = builder.nextToken;
    }

    /**
     * Returns the value of the FindingIdentifier property for this object.
     * 
     * @return The value of the FindingIdentifier property for this object.
     */
    public final AwsSecurityFindingIdentifier findingIdentifier() {
        return findingIdentifier;
    }

    /**
     * <p>
     * A timestamp that indicates when Security Hub processed the updated finding record.
     * </p>
     * <p>
     * This field accepts only the specified formats. Timestamps can end with <code>Z</code> or
     * <code>("+" / "-") time-hour [":" time-minute]</code>. The time-secfrac after seconds is limited to a maximum of 9
     * digits. The offset is bounded by +/-18:00. Here are valid timestamp formats with examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>YYYY-MM-DDTHH:MM:SSZ</code> (for example, <code>2019-01-31T23:00:00Z</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmmZ</code> (for example, <code>2019-01-31T23:00:00.123456789Z</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>YYYY-MM-DDTHH:MM:SS+HH:MM</code> (for example, <code>2024-01-04T15:25:10+17:59</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>YYYY-MM-DDTHH:MM:SS-HHMM</code> (for example, <code>2024-01-04T15:25:10-1759</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmm+HH:MM</code> (for example, <code>2024-01-04T15:25:10.123456789+17:59</code>)
     * </p>
     * </li>
     * </ul>
     * 
     * @return A timestamp that indicates when Security Hub processed the updated finding record.</p>
     *         <p>
     *         This field accepts only the specified formats. Timestamps can end with <code>Z</code> or
     *         <code>("+" / "-") time-hour [":" time-minute]</code>. The time-secfrac after seconds is limited to a
     *         maximum of 9 digits. The offset is bounded by +/-18:00. Here are valid timestamp formats with examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>YYYY-MM-DDTHH:MM:SSZ</code> (for example, <code>2019-01-31T23:00:00Z</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmmZ</code> (for example, <code>2019-01-31T23:00:00.123456789Z</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>YYYY-MM-DDTHH:MM:SS+HH:MM</code> (for example, <code>2024-01-04T15:25:10+17:59</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>YYYY-MM-DDTHH:MM:SS-HHMM</code> (for example, <code>2024-01-04T15:25:10-1759</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmm+HH:MM</code> (for example,
     *         <code>2024-01-04T15:25:10.123456789+17:59</code>)
     *         </p>
     *         </li>
     */
    public final Instant updateTime() {
        return updateTime;
    }

    /**
     * <p>
     * Identifies whether the event marks the creation of a new finding. A value of <code>True</code> means that the
     * finding is newly created. A value of <code>False</code> means that the finding isn’t newly created.
     * </p>
     * 
     * @return Identifies whether the event marks the creation of a new finding. A value of <code>True</code> means that
     *         the finding is newly created. A value of <code>False</code> means that the finding isn’t newly created.
     */
    public final Boolean findingCreated() {
        return findingCreated;
    }

    /**
     * <p>
     * Identifies the source of the event that changed the finding. For example, an integrated Amazon Web Service or
     * third-party partner integration may call <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
     * <code>BatchImportFindings</code> </a>, or an Security Hub customer may call <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
     * <code>BatchUpdateFindings</code> </a>.
     * </p>
     * 
     * @return Identifies the source of the event that changed the finding. For example, an integrated Amazon Web
     *         Service or third-party partner integration may call <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
     *         <code>BatchImportFindings</code> </a>, or an Security Hub customer may call <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
     *         <code>BatchUpdateFindings</code> </a>.
     */
    public final FindingHistoryUpdateSource updateSource() {
        return updateSource;
    }

    /**
     * For responses, this returns true if the service returned a value for the Updates property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUpdates() {
        return updates != null && !(updates instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that provides details about the finding change event, including the Amazon Web Services
     * Security Finding Format (ASFF) field that changed, the value of the field before the change, and the value of the
     * field after the change.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUpdates} method.
     * </p>
     * 
     * @return An array of objects that provides details about the finding change event, including the Amazon Web
     *         Services Security Finding Format (ASFF) field that changed, the value of the field before the change, and
     *         the value of the field after the change.
     */
    public final List<FindingHistoryUpdate> updates() {
        return updates;
    }

    /**
     * <p>
     * A token for pagination purposes. Provide this token in the subsequent request to <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingsHistory.html">
     * <code>GetFindingsHistory</code> </a> to get up to an additional 100 results of history for the same finding that
     * you specified in your initial request.
     * </p>
     * 
     * @return A token for pagination purposes. Provide this token in the subsequent request to <a
     *         href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingsHistory.html">
     *         <code>GetFindingsHistory</code> </a> to get up to an additional 100 results of history for the same
     *         finding that you specified in your initial request.
     */
    public final String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(findingIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(updateTime());
        hashCode = 31 * hashCode + Objects.hashCode(findingCreated());
        hashCode = 31 * hashCode + Objects.hashCode(updateSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasUpdates() ? updates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FindingHistoryRecord)) {
            return false;
        }
        FindingHistoryRecord other = (FindingHistoryRecord) obj;
        return Objects.equals(findingIdentifier(), other.findingIdentifier()) && Objects.equals(updateTime(), other.updateTime())
                && Objects.equals(findingCreated(), other.findingCreated())
                && Objects.equals(updateSource(), other.updateSource()) && hasUpdates() == other.hasUpdates()
                && Objects.equals(updates(), other.updates()) && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FindingHistoryRecord").add("FindingIdentifier", findingIdentifier())
                .add("UpdateTime", updateTime()).add("FindingCreated", findingCreated()).add("UpdateSource", updateSource())
                .add("Updates", hasUpdates() ? updates() : null).add("NextToken", nextToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FindingIdentifier":
            return Optional.ofNullable(clazz.cast(findingIdentifier()));
        case "UpdateTime":
            return Optional.ofNullable(clazz.cast(updateTime()));
        case "FindingCreated":
            return Optional.ofNullable(clazz.cast(findingCreated()));
        case "UpdateSource":
            return Optional.ofNullable(clazz.cast(updateSource()));
        case "Updates":
            return Optional.ofNullable(clazz.cast(updates()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FindingHistoryRecord, T> g) {
        return obj -> g.apply((FindingHistoryRecord) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FindingHistoryRecord> {
        /**
         * Sets the value of the FindingIdentifier property for this object.
         *
         * @param findingIdentifier
         *        The new value for the FindingIdentifier property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingIdentifier(AwsSecurityFindingIdentifier findingIdentifier);

        /**
         * Sets the value of the FindingIdentifier property for this object.
         *
         * This is a convenience method that creates an instance of the {@link AwsSecurityFindingIdentifier.Builder}
         * avoiding the need to create one manually via {@link AwsSecurityFindingIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AwsSecurityFindingIdentifier.Builder#build()} is called
         * immediately and its result is passed to {@link #findingIdentifier(AwsSecurityFindingIdentifier)}.
         * 
         * @param findingIdentifier
         *        a consumer that will call methods on {@link AwsSecurityFindingIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #findingIdentifier(AwsSecurityFindingIdentifier)
         */
        default Builder findingIdentifier(Consumer<AwsSecurityFindingIdentifier.Builder> findingIdentifier) {
            return findingIdentifier(AwsSecurityFindingIdentifier.builder().applyMutation(findingIdentifier).build());
        }

        /**
         * <p>
         * A timestamp that indicates when Security Hub processed the updated finding record.
         * </p>
         * <p>
         * This field accepts only the specified formats. Timestamps can end with <code>Z</code> or
         * <code>("+" / "-") time-hour [":" time-minute]</code>. The time-secfrac after seconds is limited to a maximum
         * of 9 digits. The offset is bounded by +/-18:00. Here are valid timestamp formats with examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>YYYY-MM-DDTHH:MM:SSZ</code> (for example, <code>2019-01-31T23:00:00Z</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmmZ</code> (for example, <code>2019-01-31T23:00:00.123456789Z</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>YYYY-MM-DDTHH:MM:SS+HH:MM</code> (for example, <code>2024-01-04T15:25:10+17:59</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>YYYY-MM-DDTHH:MM:SS-HHMM</code> (for example, <code>2024-01-04T15:25:10-1759</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmm+HH:MM</code> (for example,
         * <code>2024-01-04T15:25:10.123456789+17:59</code>)
         * </p>
         * </li>
         * </ul>
         * 
         * @param updateTime
         *        A timestamp that indicates when Security Hub processed the updated finding record.</p>
         *        <p>
         *        This field accepts only the specified formats. Timestamps can end with <code>Z</code> or
         *        <code>("+" / "-") time-hour [":" time-minute]</code>. The time-secfrac after seconds is limited to a
         *        maximum of 9 digits. The offset is bounded by +/-18:00. Here are valid timestamp formats with
         *        examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>YYYY-MM-DDTHH:MM:SSZ</code> (for example, <code>2019-01-31T23:00:00Z</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmmZ</code> (for example, <code>2019-01-31T23:00:00.123456789Z</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>YYYY-MM-DDTHH:MM:SS+HH:MM</code> (for example, <code>2024-01-04T15:25:10+17:59</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>YYYY-MM-DDTHH:MM:SS-HHMM</code> (for example, <code>2024-01-04T15:25:10-1759</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>YYYY-MM-DDTHH:MM:SS.mmmmmmmmm+HH:MM</code> (for example,
         *        <code>2024-01-04T15:25:10.123456789+17:59</code>)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateTime(Instant updateTime);

        /**
         * <p>
         * Identifies whether the event marks the creation of a new finding. A value of <code>True</code> means that the
         * finding is newly created. A value of <code>False</code> means that the finding isn’t newly created.
         * </p>
         * 
         * @param findingCreated
         *        Identifies whether the event marks the creation of a new finding. A value of <code>True</code> means
         *        that the finding is newly created. A value of <code>False</code> means that the finding isn’t newly
         *        created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingCreated(Boolean findingCreated);

        /**
         * <p>
         * Identifies the source of the event that changed the finding. For example, an integrated Amazon Web Service or
         * third-party partner integration may call <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
         * <code>BatchImportFindings</code> </a>, or an Security Hub customer may call <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
         * <code>BatchUpdateFindings</code> </a>.
         * </p>
         * 
         * @param updateSource
         *        Identifies the source of the event that changed the finding. For example, an integrated Amazon Web
         *        Service or third-party partner integration may call <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
         *        <code>BatchImportFindings</code> </a>, or an Security Hub customer may call <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
         *        <code>BatchUpdateFindings</code> </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateSource(FindingHistoryUpdateSource updateSource);

        /**
         * <p>
         * Identifies the source of the event that changed the finding. For example, an integrated Amazon Web Service or
         * third-party partner integration may call <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
         * <code>BatchImportFindings</code> </a>, or an Security Hub customer may call <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
         * <code>BatchUpdateFindings</code> </a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link FindingHistoryUpdateSource.Builder}
         * avoiding the need to create one manually via {@link FindingHistoryUpdateSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FindingHistoryUpdateSource.Builder#build()} is called immediately
         * and its result is passed to {@link #updateSource(FindingHistoryUpdateSource)}.
         * 
         * @param updateSource
         *        a consumer that will call methods on {@link FindingHistoryUpdateSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #updateSource(FindingHistoryUpdateSource)
         */
        default Builder updateSource(Consumer<FindingHistoryUpdateSource.Builder> updateSource) {
            return updateSource(FindingHistoryUpdateSource.builder().applyMutation(updateSource).build());
        }

        /**
         * <p>
         * An array of objects that provides details about the finding change event, including the Amazon Web Services
         * Security Finding Format (ASFF) field that changed, the value of the field before the change, and the value of
         * the field after the change.
         * </p>
         * 
         * @param updates
         *        An array of objects that provides details about the finding change event, including the Amazon Web
         *        Services Security Finding Format (ASFF) field that changed, the value of the field before the change,
         *        and the value of the field after the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updates(Collection<FindingHistoryUpdate> updates);

        /**
         * <p>
         * An array of objects that provides details about the finding change event, including the Amazon Web Services
         * Security Finding Format (ASFF) field that changed, the value of the field before the change, and the value of
         * the field after the change.
         * </p>
         * 
         * @param updates
         *        An array of objects that provides details about the finding change event, including the Amazon Web
         *        Services Security Finding Format (ASFF) field that changed, the value of the field before the change,
         *        and the value of the field after the change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updates(FindingHistoryUpdate... updates);

        /**
         * <p>
         * An array of objects that provides details about the finding change event, including the Amazon Web Services
         * Security Finding Format (ASFF) field that changed, the value of the field before the change, and the value of
         * the field after the change.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.FindingHistoryUpdate.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.securityhub.model.FindingHistoryUpdate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.FindingHistoryUpdate.Builder#build()} is called
         * immediately and its result is passed to {@link #updates(List<FindingHistoryUpdate>)}.
         * 
         * @param updates
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.FindingHistoryUpdate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #updates(java.util.Collection<FindingHistoryUpdate>)
         */
        Builder updates(Consumer<FindingHistoryUpdate.Builder>... updates);

        /**
         * <p>
         * A token for pagination purposes. Provide this token in the subsequent request to <a
         * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingsHistory.html">
         * <code>GetFindingsHistory</code> </a> to get up to an additional 100 results of history for the same finding
         * that you specified in your initial request.
         * </p>
         * 
         * @param nextToken
         *        A token for pagination purposes. Provide this token in the subsequent request to <a
         *        href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingsHistory.html">
         *        <code>GetFindingsHistory</code> </a> to get up to an additional 100 results of history for the same
         *        finding that you specified in your initial request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);
    }

    static final class BuilderImpl implements Builder {
        private AwsSecurityFindingIdentifier findingIdentifier;

        private Instant updateTime;

        private Boolean findingCreated;

        private FindingHistoryUpdateSource updateSource;

        private List<FindingHistoryUpdate> updates = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(FindingHistoryRecord model) {
            findingIdentifier(model.findingIdentifier);
            updateTime(model.updateTime);
            findingCreated(model.findingCreated);
            updateSource(model.updateSource);
            updates(model.updates);
            nextToken(model.nextToken);
        }

        public final AwsSecurityFindingIdentifier.Builder getFindingIdentifier() {
            return findingIdentifier != null ? findingIdentifier.toBuilder() : null;
        }

        public final void setFindingIdentifier(AwsSecurityFindingIdentifier.BuilderImpl findingIdentifier) {
            this.findingIdentifier = findingIdentifier != null ? findingIdentifier.build() : null;
        }

        @Override
        public final Builder findingIdentifier(AwsSecurityFindingIdentifier findingIdentifier) {
            this.findingIdentifier = findingIdentifier;
            return this;
        }

        public final Instant getUpdateTime() {
            return updateTime;
        }

        public final void setUpdateTime(Instant updateTime) {
            this.updateTime = updateTime;
        }

        @Override
        public final Builder updateTime(Instant updateTime) {
            this.updateTime = updateTime;
            return this;
        }

        public final Boolean getFindingCreated() {
            return findingCreated;
        }

        public final void setFindingCreated(Boolean findingCreated) {
            this.findingCreated = findingCreated;
        }

        @Override
        public final Builder findingCreated(Boolean findingCreated) {
            this.findingCreated = findingCreated;
            return this;
        }

        public final FindingHistoryUpdateSource.Builder getUpdateSource() {
            return updateSource != null ? updateSource.toBuilder() : null;
        }

        public final void setUpdateSource(FindingHistoryUpdateSource.BuilderImpl updateSource) {
            this.updateSource = updateSource != null ? updateSource.build() : null;
        }

        @Override
        public final Builder updateSource(FindingHistoryUpdateSource updateSource) {
            this.updateSource = updateSource;
            return this;
        }

        public final List<FindingHistoryUpdate.Builder> getUpdates() {
            List<FindingHistoryUpdate.Builder> result = FindingHistoryUpdatesListCopier.copyToBuilder(this.updates);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUpdates(Collection<FindingHistoryUpdate.BuilderImpl> updates) {
            this.updates = FindingHistoryUpdatesListCopier.copyFromBuilder(updates);
        }

        @Override
        public final Builder updates(Collection<FindingHistoryUpdate> updates) {
            this.updates = FindingHistoryUpdatesListCopier.copy(updates);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder updates(FindingHistoryUpdate... updates) {
            updates(Arrays.asList(updates));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder updates(Consumer<FindingHistoryUpdate.Builder>... updates) {
            updates(Stream.of(updates).map(c -> FindingHistoryUpdate.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        @Override
        public FindingHistoryRecord build() {
            return new FindingHistoryRecord(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
