/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CVSS scores from the advisory related to the vulnerability.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cvss implements SdkPojo, Serializable, ToCopyableBuilder<Cvss.Builder, Cvss> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(Cvss::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<Double> BASE_SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("BaseScore").getter(getter(Cvss::baseScore)).setter(setter(Builder::baseScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseScore").build()).build();

    private static final SdkField<String> BASE_VECTOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseVector").getter(getter(Cvss::baseVector)).setter(setter(Builder::baseVector))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseVector").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(Cvss::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<List<Adjustment>> ADJUSTMENTS_FIELD = SdkField
            .<List<Adjustment>> builder(MarshallingType.LIST)
            .memberName("Adjustments")
            .getter(getter(Cvss::adjustments))
            .setter(setter(Builder::adjustments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Adjustments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Adjustment> builder(MarshallingType.SDK_POJO)
                                            .constructor(Adjustment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD,
            BASE_SCORE_FIELD, BASE_VECTOR_FIELD, SOURCE_FIELD, ADJUSTMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String version;

    private final Double baseScore;

    private final String baseVector;

    private final String source;

    private final List<Adjustment> adjustments;

    private Cvss(BuilderImpl builder) {
        this.version = builder.version;
        this.baseScore = builder.baseScore;
        this.baseVector = builder.baseVector;
        this.source = builder.source;
        this.adjustments = builder.adjustments;
    }

    /**
     * <p>
     * The version of CVSS for the CVSS score.
     * </p>
     * 
     * @return The version of CVSS for the CVSS score.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The base CVSS score.
     * </p>
     * 
     * @return The base CVSS score.
     */
    public final Double baseScore() {
        return baseScore;
    }

    /**
     * <p>
     * The base scoring vector for the CVSS score.
     * </p>
     * 
     * @return The base scoring vector for the CVSS score.
     */
    public final String baseVector() {
        return baseVector;
    }

    /**
     * <p>
     * The origin of the original CVSS score and vector.
     * </p>
     * 
     * @return The origin of the original CVSS score and vector.
     */
    public final String source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the Adjustments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdjustments() {
        return adjustments != null && !(adjustments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Adjustments to the CVSS metrics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdjustments} method.
     * </p>
     * 
     * @return Adjustments to the CVSS metrics.
     */
    public final List<Adjustment> adjustments() {
        return adjustments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(baseScore());
        hashCode = 31 * hashCode + Objects.hashCode(baseVector());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdjustments() ? adjustments() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cvss)) {
            return false;
        }
        Cvss other = (Cvss) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(baseScore(), other.baseScore())
                && Objects.equals(baseVector(), other.baseVector()) && Objects.equals(source(), other.source())
                && hasAdjustments() == other.hasAdjustments() && Objects.equals(adjustments(), other.adjustments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Cvss").add("Version", version()).add("BaseScore", baseScore()).add("BaseVector", baseVector())
                .add("Source", source()).add("Adjustments", hasAdjustments() ? adjustments() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "BaseScore":
            return Optional.ofNullable(clazz.cast(baseScore()));
        case "BaseVector":
            return Optional.ofNullable(clazz.cast(baseVector()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Adjustments":
            return Optional.ofNullable(clazz.cast(adjustments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Cvss, T> g) {
        return obj -> g.apply((Cvss) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cvss> {
        /**
         * <p>
         * The version of CVSS for the CVSS score.
         * </p>
         * 
         * @param version
         *        The version of CVSS for the CVSS score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The base CVSS score.
         * </p>
         * 
         * @param baseScore
         *        The base CVSS score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseScore(Double baseScore);

        /**
         * <p>
         * The base scoring vector for the CVSS score.
         * </p>
         * 
         * @param baseVector
         *        The base scoring vector for the CVSS score.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseVector(String baseVector);

        /**
         * <p>
         * The origin of the original CVSS score and vector.
         * </p>
         * 
         * @param source
         *        The origin of the original CVSS score and vector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * Adjustments to the CVSS metrics.
         * </p>
         * 
         * @param adjustments
         *        Adjustments to the CVSS metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adjustments(Collection<Adjustment> adjustments);

        /**
         * <p>
         * Adjustments to the CVSS metrics.
         * </p>
         * 
         * @param adjustments
         *        Adjustments to the CVSS metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adjustments(Adjustment... adjustments);

        /**
         * <p>
         * Adjustments to the CVSS metrics.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.Adjustment.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.Adjustment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.Adjustment.Builder#build()} is called immediately
         * and its result is passed to {@link #adjustments(List<Adjustment>)}.
         * 
         * @param adjustments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.Adjustment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #adjustments(java.util.Collection<Adjustment>)
         */
        Builder adjustments(Consumer<Adjustment.Builder>... adjustments);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private Double baseScore;

        private String baseVector;

        private String source;

        private List<Adjustment> adjustments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Cvss model) {
            version(model.version);
            baseScore(model.baseScore);
            baseVector(model.baseVector);
            source(model.source);
            adjustments(model.adjustments);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Double getBaseScore() {
            return baseScore;
        }

        public final void setBaseScore(Double baseScore) {
            this.baseScore = baseScore;
        }

        @Override
        public final Builder baseScore(Double baseScore) {
            this.baseScore = baseScore;
            return this;
        }

        public final String getBaseVector() {
            return baseVector;
        }

        public final void setBaseVector(String baseVector) {
            this.baseVector = baseVector;
        }

        @Override
        public final Builder baseVector(String baseVector) {
            this.baseVector = baseVector;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final List<Adjustment.Builder> getAdjustments() {
            List<Adjustment.Builder> result = AdjustmentListCopier.copyToBuilder(this.adjustments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAdjustments(Collection<Adjustment.BuilderImpl> adjustments) {
            this.adjustments = AdjustmentListCopier.copyFromBuilder(adjustments);
        }

        @Override
        public final Builder adjustments(Collection<Adjustment> adjustments) {
            this.adjustments = AdjustmentListCopier.copy(adjustments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder adjustments(Adjustment... adjustments) {
            adjustments(Arrays.asList(adjustments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder adjustments(Consumer<Adjustment.Builder>... adjustments) {
            adjustments(Stream.of(adjustments).map(c -> Adjustment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Cvss build() {
            return new Cvss(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
