/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In a <a href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html">
 * <code>BatchImportFindings</code> </a> request, finding providers use <code>FindingProviderFields</code> to provide
 * and update values for the following fields:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>Confidence</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Criticality</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RelatedFindings</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Severity</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>Types</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * The preceding fields are nested under the <code>FindingProviderFields</code> object, but also have analogues of the
 * same name as top-level ASFF fields. When a new finding is sent to Security Hub by a finding provider, Security Hub
 * populates the <code>FindingProviderFields</code> object automatically, if it is empty, based on the corresponding
 * top-level fields.
 * </p>
 * <p>
 * Finding providers can update <code>FindingProviderFields</code> only by using the <code>BatchImportFindings</code>
 * operation. Finding providers can't update this object with the <a
 * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html">
 * <code>BatchUpdateFindings</code> </a> operation. Customers can update the top-level fields by using the
 * <code>BatchUpdateFindings</code> operation. Customers can't update <code>FindingProviderFields</code>.
 * </p>
 * <p>
 * For information about how Security Hub handles updates from <code>BatchImportFindings</code> to
 * <code>FindingProviderFields</code> and to the corresponding top-level attributes, see <a href=
 * "https://docs.aws.amazon.com/securityhub/latest/userguide/finding-update-batchimportfindings.html#batchimportfindings-findingproviderfields"
 * >Using <code>FindingProviderFields</code> </a> in the <i>Security Hub User Guide</i>.
 * </p>
 * <p/>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FindingProviderFields implements SdkPojo, Serializable,
        ToCopyableBuilder<FindingProviderFields.Builder, FindingProviderFields> {
    private static final SdkField<Integer> CONFIDENCE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Confidence").getter(getter(FindingProviderFields::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<Integer> CRITICALITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Criticality").getter(getter(FindingProviderFields::criticality)).setter(setter(Builder::criticality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Criticality").build()).build();

    private static final SdkField<List<RelatedFinding>> RELATED_FINDINGS_FIELD = SdkField
            .<List<RelatedFinding>> builder(MarshallingType.LIST)
            .memberName("RelatedFindings")
            .getter(getter(FindingProviderFields::relatedFindings))
            .setter(setter(Builder::relatedFindings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedFindings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedFinding> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedFinding::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FindingProviderSeverity> SEVERITY_FIELD = SdkField
            .<FindingProviderSeverity> builder(MarshallingType.SDK_POJO).memberName("Severity")
            .getter(getter(FindingProviderFields::severity)).setter(setter(Builder::severity))
            .constructor(FindingProviderSeverity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<List<String>> TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Types")
            .getter(getter(FindingProviderFields::types))
            .setter(setter(Builder::types))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Types").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIDENCE_FIELD,
            CRITICALITY_FIELD, RELATED_FINDINGS_FIELD, SEVERITY_FIELD, TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer confidence;

    private final Integer criticality;

    private final List<RelatedFinding> relatedFindings;

    private final FindingProviderSeverity severity;

    private final List<String> types;

    private FindingProviderFields(BuilderImpl builder) {
        this.confidence = builder.confidence;
        this.criticality = builder.criticality;
        this.relatedFindings = builder.relatedFindings;
        this.severity = builder.severity;
        this.types = builder.types;
    }

    /**
     * <p>
     * A finding's confidence. Confidence is defined as the likelihood that a finding accurately identifies the behavior
     * or issue that it was intended to identify.
     * </p>
     * <p>
     * Confidence is scored on a 0-100 basis using a ratio scale, where 0 means zero percent confidence and 100 means
     * 100 percent confidence.
     * </p>
     * 
     * @return A finding's confidence. Confidence is defined as the likelihood that a finding accurately identifies the
     *         behavior or issue that it was intended to identify.</p>
     *         <p>
     *         Confidence is scored on a 0-100 basis using a ratio scale, where 0 means zero percent confidence and 100
     *         means 100 percent confidence.
     */
    public final Integer confidence() {
        return confidence;
    }

    /**
     * <p>
     * The level of importance assigned to the resources associated with the finding.
     * </p>
     * <p>
     * A score of 0 means that the underlying resources have no criticality, and a score of 100 is reserved for the most
     * critical resources.
     * </p>
     * 
     * @return The level of importance assigned to the resources associated with the finding.</p>
     *         <p>
     *         A score of 0 means that the underlying resources have no criticality, and a score of 100 is reserved for
     *         the most critical resources.
     */
    public final Integer criticality() {
        return criticality;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedFindings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedFindings() {
        return relatedFindings != null && !(relatedFindings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of findings that are related to the current finding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedFindings} method.
     * </p>
     * 
     * @return A list of findings that are related to the current finding.
     */
    public final List<RelatedFinding> relatedFindings() {
        return relatedFindings;
    }

    /**
     * <p>
     * The severity of a finding.
     * </p>
     * 
     * @return The severity of a finding.
     */
    public final FindingProviderSeverity severity() {
        return severity;
    }

    /**
     * For responses, this returns true if the service returned a value for the Types property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTypes() {
        return types != null && !(types instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more finding types in the format of <code>namespace/category/classifier</code> that classify a finding.
     * </p>
     * <p>
     * Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors | Sensitive
     * Data Identifications
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTypes} method.
     * </p>
     * 
     * @return One or more finding types in the format of <code>namespace/category/classifier</code> that classify a
     *         finding.</p>
     *         <p>
     *         Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors |
     *         Sensitive Data Identifications
     */
    public final List<String> types() {
        return types;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(criticality());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedFindings() ? relatedFindings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(hasTypes() ? types() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FindingProviderFields)) {
            return false;
        }
        FindingProviderFields other = (FindingProviderFields) obj;
        return Objects.equals(confidence(), other.confidence()) && Objects.equals(criticality(), other.criticality())
                && hasRelatedFindings() == other.hasRelatedFindings()
                && Objects.equals(relatedFindings(), other.relatedFindings()) && Objects.equals(severity(), other.severity())
                && hasTypes() == other.hasTypes() && Objects.equals(types(), other.types());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FindingProviderFields").add("Confidence", confidence()).add("Criticality", criticality())
                .add("RelatedFindings", hasRelatedFindings() ? relatedFindings() : null).add("Severity", severity())
                .add("Types", hasTypes() ? types() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Criticality":
            return Optional.ofNullable(clazz.cast(criticality()));
        case "RelatedFindings":
            return Optional.ofNullable(clazz.cast(relatedFindings()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "Types":
            return Optional.ofNullable(clazz.cast(types()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FindingProviderFields, T> g) {
        return obj -> g.apply((FindingProviderFields) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FindingProviderFields> {
        /**
         * <p>
         * A finding's confidence. Confidence is defined as the likelihood that a finding accurately identifies the
         * behavior or issue that it was intended to identify.
         * </p>
         * <p>
         * Confidence is scored on a 0-100 basis using a ratio scale, where 0 means zero percent confidence and 100
         * means 100 percent confidence.
         * </p>
         * 
         * @param confidence
         *        A finding's confidence. Confidence is defined as the likelihood that a finding accurately identifies
         *        the behavior or issue that it was intended to identify.</p>
         *        <p>
         *        Confidence is scored on a 0-100 basis using a ratio scale, where 0 means zero percent confidence and
         *        100 means 100 percent confidence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Integer confidence);

        /**
         * <p>
         * The level of importance assigned to the resources associated with the finding.
         * </p>
         * <p>
         * A score of 0 means that the underlying resources have no criticality, and a score of 100 is reserved for the
         * most critical resources.
         * </p>
         * 
         * @param criticality
         *        The level of importance assigned to the resources associated with the finding.</p>
         *        <p>
         *        A score of 0 means that the underlying resources have no criticality, and a score of 100 is reserved
         *        for the most critical resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder criticality(Integer criticality);

        /**
         * <p>
         * A list of findings that are related to the current finding.
         * </p>
         * 
         * @param relatedFindings
         *        A list of findings that are related to the current finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedFindings(Collection<RelatedFinding> relatedFindings);

        /**
         * <p>
         * A list of findings that are related to the current finding.
         * </p>
         * 
         * @param relatedFindings
         *        A list of findings that are related to the current finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedFindings(RelatedFinding... relatedFindings);

        /**
         * <p>
         * A list of findings that are related to the current finding.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.RelatedFinding.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.securityhub.model.RelatedFinding#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.RelatedFinding.Builder#build()} is called
         * immediately and its result is passed to {@link #relatedFindings(List<RelatedFinding>)}.
         * 
         * @param relatedFindings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.RelatedFinding.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedFindings(java.util.Collection<RelatedFinding>)
         */
        Builder relatedFindings(Consumer<RelatedFinding.Builder>... relatedFindings);

        /**
         * <p>
         * The severity of a finding.
         * </p>
         * 
         * @param severity
         *        The severity of a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(FindingProviderSeverity severity);

        /**
         * <p>
         * The severity of a finding.
         * </p>
         * This is a convenience method that creates an instance of the {@link FindingProviderSeverity.Builder} avoiding
         * the need to create one manually via {@link FindingProviderSeverity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FindingProviderSeverity.Builder#build()} is called immediately
         * and its result is passed to {@link #severity(FindingProviderSeverity)}.
         * 
         * @param severity
         *        a consumer that will call methods on {@link FindingProviderSeverity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #severity(FindingProviderSeverity)
         */
        default Builder severity(Consumer<FindingProviderSeverity.Builder> severity) {
            return severity(FindingProviderSeverity.builder().applyMutation(severity).build());
        }

        /**
         * <p>
         * One or more finding types in the format of <code>namespace/category/classifier</code> that classify a
         * finding.
         * </p>
         * <p>
         * Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors |
         * Sensitive Data Identifications
         * </p>
         * 
         * @param types
         *        One or more finding types in the format of <code>namespace/category/classifier</code> that classify a
         *        finding.</p>
         *        <p>
         *        Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors |
         *        Sensitive Data Identifications
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder types(Collection<String> types);

        /**
         * <p>
         * One or more finding types in the format of <code>namespace/category/classifier</code> that classify a
         * finding.
         * </p>
         * <p>
         * Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors |
         * Sensitive Data Identifications
         * </p>
         * 
         * @param types
         *        One or more finding types in the format of <code>namespace/category/classifier</code> that classify a
         *        finding.</p>
         *        <p>
         *        Valid namespace values are: Software and Configuration Checks | TTPs | Effects | Unusual Behaviors |
         *        Sensitive Data Identifications
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder types(String... types);
    }

    static final class BuilderImpl implements Builder {
        private Integer confidence;

        private Integer criticality;

        private List<RelatedFinding> relatedFindings = DefaultSdkAutoConstructList.getInstance();

        private FindingProviderSeverity severity;

        private List<String> types = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FindingProviderFields model) {
            confidence(model.confidence);
            criticality(model.criticality);
            relatedFindings(model.relatedFindings);
            severity(model.severity);
            types(model.types);
        }

        public final Integer getConfidence() {
            return confidence;
        }

        public final void setConfidence(Integer confidence) {
            this.confidence = confidence;
        }

        @Override
        public final Builder confidence(Integer confidence) {
            this.confidence = confidence;
            return this;
        }

        public final Integer getCriticality() {
            return criticality;
        }

        public final void setCriticality(Integer criticality) {
            this.criticality = criticality;
        }

        @Override
        public final Builder criticality(Integer criticality) {
            this.criticality = criticality;
            return this;
        }

        public final List<RelatedFinding.Builder> getRelatedFindings() {
            List<RelatedFinding.Builder> result = RelatedFindingListCopier.copyToBuilder(this.relatedFindings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelatedFindings(Collection<RelatedFinding.BuilderImpl> relatedFindings) {
            this.relatedFindings = RelatedFindingListCopier.copyFromBuilder(relatedFindings);
        }

        @Override
        public final Builder relatedFindings(Collection<RelatedFinding> relatedFindings) {
            this.relatedFindings = RelatedFindingListCopier.copy(relatedFindings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedFindings(RelatedFinding... relatedFindings) {
            relatedFindings(Arrays.asList(relatedFindings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedFindings(Consumer<RelatedFinding.Builder>... relatedFindings) {
            relatedFindings(Stream.of(relatedFindings).map(c -> RelatedFinding.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final FindingProviderSeverity.Builder getSeverity() {
            return severity != null ? severity.toBuilder() : null;
        }

        public final void setSeverity(FindingProviderSeverity.BuilderImpl severity) {
            this.severity = severity != null ? severity.build() : null;
        }

        @Override
        public final Builder severity(FindingProviderSeverity severity) {
            this.severity = severity;
            return this;
        }

        public final Collection<String> getTypes() {
            if (types instanceof SdkAutoConstructList) {
                return null;
            }
            return types;
        }

        public final void setTypes(Collection<String> types) {
            this.types = TypeListCopier.copy(types);
        }

        @Override
        public final Builder types(Collection<String> types) {
            this.types = TypeListCopier.copy(types);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder types(String... types) {
            types(Arrays.asList(types));
            return this;
        }

        @Override
        public FindingProviderFields build() {
            return new FindingProviderFields(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
