/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A security control in Security Hub describes a security best practice related to a specific resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecurityControl implements SdkPojo, Serializable, ToCopyableBuilder<SecurityControl.Builder, SecurityControl> {
    private static final SdkField<String> SECURITY_CONTROL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlId").getter(getter(SecurityControl::securityControlId))
            .setter(setter(Builder::securityControlId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlId").build()).build();

    private static final SdkField<String> SECURITY_CONTROL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlArn").getter(getter(SecurityControl::securityControlArn))
            .setter(setter(Builder::securityControlArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlArn").build())
            .build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(SecurityControl::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(SecurityControl::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> REMEDIATION_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RemediationUrl").getter(getter(SecurityControl::remediationUrl)).setter(setter(Builder::remediationUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemediationUrl").build()).build();

    private static final SdkField<String> SEVERITY_RATING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SeverityRating").getter(getter(SecurityControl::severityRatingAsString))
            .setter(setter(Builder::severityRating))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SeverityRating").build()).build();

    private static final SdkField<String> SECURITY_CONTROL_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlStatus").getter(getter(SecurityControl::securityControlStatusAsString))
            .setter(setter(Builder::securityControlStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlStatus").build())
            .build();

    private static final SdkField<String> UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpdateStatus").getter(getter(SecurityControl::updateStatusAsString))
            .setter(setter(Builder::updateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateStatus").build()).build();

    private static final SdkField<Map<String, ParameterConfiguration>> PARAMETERS_FIELD = SdkField
            .<Map<String, ParameterConfiguration>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(SecurityControl::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ParameterConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> LAST_UPDATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastUpdateReason").getter(getter(SecurityControl::lastUpdateReason))
            .setter(setter(Builder::lastUpdateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SECURITY_CONTROL_ID_FIELD,
            SECURITY_CONTROL_ARN_FIELD, TITLE_FIELD, DESCRIPTION_FIELD, REMEDIATION_URL_FIELD, SEVERITY_RATING_FIELD,
            SECURITY_CONTROL_STATUS_FIELD, UPDATE_STATUS_FIELD, PARAMETERS_FIELD, LAST_UPDATE_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String securityControlId;

    private final String securityControlArn;

    private final String title;

    private final String description;

    private final String remediationUrl;

    private final String severityRating;

    private final String securityControlStatus;

    private final String updateStatus;

    private final Map<String, ParameterConfiguration> parameters;

    private final String lastUpdateReason;

    private SecurityControl(BuilderImpl builder) {
        this.securityControlId = builder.securityControlId;
        this.securityControlArn = builder.securityControlArn;
        this.title = builder.title;
        this.description = builder.description;
        this.remediationUrl = builder.remediationUrl;
        this.severityRating = builder.severityRating;
        this.securityControlStatus = builder.securityControlStatus;
        this.updateStatus = builder.updateStatus;
        this.parameters = builder.parameters;
        this.lastUpdateReason = builder.lastUpdateReason;
    }

    /**
     * <p>
     * The unique identifier of a security control across standards. Values for this field typically consist of an
     * Amazon Web Services service name and a number, such as APIGateway.3.
     * </p>
     * 
     * @return The unique identifier of a security control across standards. Values for this field typically consist of
     *         an Amazon Web Services service name and a number, such as APIGateway.3.
     */
    public final String securityControlId() {
        return securityControlId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for a security control across standards, such as
     * <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't mention
     * a specific standard.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for a security control across standards, such as
     *         <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't
     *         mention a specific standard.
     */
    public final String securityControlArn() {
        return securityControlArn;
    }

    /**
     * <p>
     * The title of a security control.
     * </p>
     * 
     * @return The title of a security control.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The description of a security control across standards. This typically summarizes how Security Hub evaluates the
     * control and the conditions under which it produces a failed finding. This parameter doesn't reference a specific
     * standard.
     * </p>
     * 
     * @return The description of a security control across standards. This typically summarizes how Security Hub
     *         evaluates the control and the conditions under which it produces a failed finding. This parameter doesn't
     *         reference a specific standard.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A link to Security Hub documentation that explains how to remediate a failed finding for a security control.
     * </p>
     * 
     * @return A link to Security Hub documentation that explains how to remediate a failed finding for a security
     *         control.
     */
    public final String remediationUrl() {
        return remediationUrl;
    }

    /**
     * <p>
     * The severity of a security control. For more information about how Security Hub determines control severity, see
     * <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of a security control. For more information about how Security Hub determines control
     *         severity, see <a href=
     *         "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     *         >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * @see SeverityRating
     */
    public final SeverityRating severityRating() {
        return SeverityRating.fromValue(severityRating);
    }

    /**
     * <p>
     * The severity of a security control. For more information about how Security Hub determines control severity, see
     * <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of a security control. For more information about how Security Hub determines control
     *         severity, see <a href=
     *         "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     *         >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * @see SeverityRating
     */
    public final String severityRatingAsString() {
        return severityRating;
    }

    /**
     * <p>
     * The enablement status of a security control in a specific standard.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #securityControlStatus} will return {@link ControlStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #securityControlStatusAsString}.
     * </p>
     * 
     * @return The enablement status of a security control in a specific standard.
     * @see ControlStatus
     */
    public final ControlStatus securityControlStatus() {
        return ControlStatus.fromValue(securityControlStatus);
    }

    /**
     * <p>
     * The enablement status of a security control in a specific standard.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #securityControlStatus} will return {@link ControlStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #securityControlStatusAsString}.
     * </p>
     * 
     * @return The enablement status of a security control in a specific standard.
     * @see ControlStatus
     */
    public final String securityControlStatusAsString() {
        return securityControlStatus;
    }

    /**
     * <p>
     * Identifies whether customizable properties of a security control are reflected in Security Hub findings. A status
     * of <code>READY</code> indicates that Security Hub uses the current control parameter values when running security
     * checks of the control. A status of <code>UPDATING</code> indicates that all security checks might not use the
     * current parameter values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link UpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #updateStatusAsString}.
     * </p>
     * 
     * @return Identifies whether customizable properties of a security control are reflected in Security Hub findings.
     *         A status of <code>READY</code> indicates that Security Hub uses the current control parameter values when
     *         running security checks of the control. A status of <code>UPDATING</code> indicates that all security
     *         checks might not use the current parameter values.
     * @see UpdateStatus
     */
    public final UpdateStatus updateStatus() {
        return UpdateStatus.fromValue(updateStatus);
    }

    /**
     * <p>
     * Identifies whether customizable properties of a security control are reflected in Security Hub findings. A status
     * of <code>READY</code> indicates that Security Hub uses the current control parameter values when running security
     * checks of the control. A status of <code>UPDATING</code> indicates that all security checks might not use the
     * current parameter values.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link UpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #updateStatusAsString}.
     * </p>
     * 
     * @return Identifies whether customizable properties of a security control are reflected in Security Hub findings.
     *         A status of <code>READY</code> indicates that Security Hub uses the current control parameter values when
     *         running security checks of the control. A status of <code>UPDATING</code> indicates that all security
     *         checks might not use the current parameter values.
     * @see UpdateStatus
     */
    public final String updateStatusAsString() {
        return updateStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An object that identifies the name of a control parameter, its current value, and whether it has been customized.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return An object that identifies the name of a control parameter, its current value, and whether it has been
     *         customized.
     */
    public final Map<String, ParameterConfiguration> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The most recent reason for updating the customizable properties of a security control. This differs from the
     * <code>UpdateReason</code> field of the <a href=
     * "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html">
     * <code>BatchUpdateStandardsControlAssociations</code> </a> API, which tracks the reason for updating the
     * enablement status of a control. This field accepts alphanumeric characters in addition to white spaces, dashes,
     * and underscores.
     * </p>
     * 
     * @return The most recent reason for updating the customizable properties of a security control. This differs from
     *         the <code>UpdateReason</code> field of the <a href=
     *         "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html"
     *         > <code>BatchUpdateStandardsControlAssociations</code> </a> API, which tracks the reason for updating the
     *         enablement status of a control. This field accepts alphanumeric characters in addition to white spaces,
     *         dashes, and underscores.
     */
    public final String lastUpdateReason() {
        return lastUpdateReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(securityControlId());
        hashCode = 31 * hashCode + Objects.hashCode(securityControlArn());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(remediationUrl());
        hashCode = 31 * hashCode + Objects.hashCode(severityRatingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(securityControlStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updateStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecurityControl)) {
            return false;
        }
        SecurityControl other = (SecurityControl) obj;
        return Objects.equals(securityControlId(), other.securityControlId())
                && Objects.equals(securityControlArn(), other.securityControlArn()) && Objects.equals(title(), other.title())
                && Objects.equals(description(), other.description()) && Objects.equals(remediationUrl(), other.remediationUrl())
                && Objects.equals(severityRatingAsString(), other.severityRatingAsString())
                && Objects.equals(securityControlStatusAsString(), other.securityControlStatusAsString())
                && Objects.equals(updateStatusAsString(), other.updateStatusAsString())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(lastUpdateReason(), other.lastUpdateReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecurityControl").add("SecurityControlId", securityControlId())
                .add("SecurityControlArn", securityControlArn()).add("Title", title()).add("Description", description())
                .add("RemediationUrl", remediationUrl()).add("SeverityRating", severityRatingAsString())
                .add("SecurityControlStatus", securityControlStatusAsString()).add("UpdateStatus", updateStatusAsString())
                .add("Parameters", hasParameters() ? parameters() : null).add("LastUpdateReason", lastUpdateReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SecurityControlId":
            return Optional.ofNullable(clazz.cast(securityControlId()));
        case "SecurityControlArn":
            return Optional.ofNullable(clazz.cast(securityControlArn()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RemediationUrl":
            return Optional.ofNullable(clazz.cast(remediationUrl()));
        case "SeverityRating":
            return Optional.ofNullable(clazz.cast(severityRatingAsString()));
        case "SecurityControlStatus":
            return Optional.ofNullable(clazz.cast(securityControlStatusAsString()));
        case "UpdateStatus":
            return Optional.ofNullable(clazz.cast(updateStatusAsString()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "LastUpdateReason":
            return Optional.ofNullable(clazz.cast(lastUpdateReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SecurityControl, T> g) {
        return obj -> g.apply((SecurityControl) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecurityControl> {
        /**
         * <p>
         * The unique identifier of a security control across standards. Values for this field typically consist of an
         * Amazon Web Services service name and a number, such as APIGateway.3.
         * </p>
         * 
         * @param securityControlId
         *        The unique identifier of a security control across standards. Values for this field typically consist
         *        of an Amazon Web Services service name and a number, such as APIGateway.3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlId(String securityControlId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for a security control across standards, such as
         * <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter doesn't
         * mention a specific standard.
         * </p>
         * 
         * @param securityControlArn
         *        The Amazon Resource Name (ARN) for a security control across standards, such as
         *        <code>arn:aws:securityhub:eu-central-1:123456789012:security-control/S3.1</code>. This parameter
         *        doesn't mention a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlArn(String securityControlArn);

        /**
         * <p>
         * The title of a security control.
         * </p>
         * 
         * @param title
         *        The title of a security control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The description of a security control across standards. This typically summarizes how Security Hub evaluates
         * the control and the conditions under which it produces a failed finding. This parameter doesn't reference a
         * specific standard.
         * </p>
         * 
         * @param description
         *        The description of a security control across standards. This typically summarizes how Security Hub
         *        evaluates the control and the conditions under which it produces a failed finding. This parameter
         *        doesn't reference a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A link to Security Hub documentation that explains how to remediate a failed finding for a security control.
         * </p>
         * 
         * @param remediationUrl
         *        A link to Security Hub documentation that explains how to remediate a failed finding for a security
         *        control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediationUrl(String remediationUrl);

        /**
         * <p>
         * The severity of a security control. For more information about how Security Hub determines control severity,
         * see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param severityRating
         *        The severity of a security control. For more information about how Security Hub determines control
         *        severity, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         *        >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(String severityRating);

        /**
         * <p>
         * The severity of a security control. For more information about how Security Hub determines control severity,
         * see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param severityRating
         *        The severity of a security control. For more information about how Security Hub determines control
         *        severity, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         *        >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(SeverityRating severityRating);

        /**
         * <p>
         * The enablement status of a security control in a specific standard.
         * </p>
         * 
         * @param securityControlStatus
         *        The enablement status of a security control in a specific standard.
         * @see ControlStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlStatus
         */
        Builder securityControlStatus(String securityControlStatus);

        /**
         * <p>
         * The enablement status of a security control in a specific standard.
         * </p>
         * 
         * @param securityControlStatus
         *        The enablement status of a security control in a specific standard.
         * @see ControlStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlStatus
         */
        Builder securityControlStatus(ControlStatus securityControlStatus);

        /**
         * <p>
         * Identifies whether customizable properties of a security control are reflected in Security Hub findings. A
         * status of <code>READY</code> indicates that Security Hub uses the current control parameter values when
         * running security checks of the control. A status of <code>UPDATING</code> indicates that all security checks
         * might not use the current parameter values.
         * </p>
         * 
         * @param updateStatus
         *        Identifies whether customizable properties of a security control are reflected in Security Hub
         *        findings. A status of <code>READY</code> indicates that Security Hub uses the current control
         *        parameter values when running security checks of the control. A status of <code>UPDATING</code>
         *        indicates that all security checks might not use the current parameter values.
         * @see UpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateStatus
         */
        Builder updateStatus(String updateStatus);

        /**
         * <p>
         * Identifies whether customizable properties of a security control are reflected in Security Hub findings. A
         * status of <code>READY</code> indicates that Security Hub uses the current control parameter values when
         * running security checks of the control. A status of <code>UPDATING</code> indicates that all security checks
         * might not use the current parameter values.
         * </p>
         * 
         * @param updateStatus
         *        Identifies whether customizable properties of a security control are reflected in Security Hub
         *        findings. A status of <code>READY</code> indicates that Security Hub uses the current control
         *        parameter values when running security checks of the control. A status of <code>UPDATING</code>
         *        indicates that all security checks might not use the current parameter values.
         * @see UpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateStatus
         */
        Builder updateStatus(UpdateStatus updateStatus);

        /**
         * <p>
         * An object that identifies the name of a control parameter, its current value, and whether it has been
         * customized.
         * </p>
         * 
         * @param parameters
         *        An object that identifies the name of a control parameter, its current value, and whether it has been
         *        customized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ParameterConfiguration> parameters);

        /**
         * <p>
         * The most recent reason for updating the customizable properties of a security control. This differs from the
         * <code>UpdateReason</code> field of the <a href=
         * "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html">
         * <code>BatchUpdateStandardsControlAssociations</code> </a> API, which tracks the reason for updating the
         * enablement status of a control. This field accepts alphanumeric characters in addition to white spaces,
         * dashes, and underscores.
         * </p>
         * 
         * @param lastUpdateReason
         *        The most recent reason for updating the customizable properties of a security control. This differs
         *        from the <code>UpdateReason</code> field of the <a href=
         *        "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html"
         *        > <code>BatchUpdateStandardsControlAssociations</code> </a> API, which tracks the reason for updating
         *        the enablement status of a control. This field accepts alphanumeric characters in addition to white
         *        spaces, dashes, and underscores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateReason(String lastUpdateReason);
    }

    static final class BuilderImpl implements Builder {
        private String securityControlId;

        private String securityControlArn;

        private String title;

        private String description;

        private String remediationUrl;

        private String severityRating;

        private String securityControlStatus;

        private String updateStatus;

        private Map<String, ParameterConfiguration> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String lastUpdateReason;

        private BuilderImpl() {
        }

        private BuilderImpl(SecurityControl model) {
            securityControlId(model.securityControlId);
            securityControlArn(model.securityControlArn);
            title(model.title);
            description(model.description);
            remediationUrl(model.remediationUrl);
            severityRating(model.severityRating);
            securityControlStatus(model.securityControlStatus);
            updateStatus(model.updateStatus);
            parameters(model.parameters);
            lastUpdateReason(model.lastUpdateReason);
        }

        public final String getSecurityControlId() {
            return securityControlId;
        }

        public final void setSecurityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
        }

        @Override
        public final Builder securityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
            return this;
        }

        public final String getSecurityControlArn() {
            return securityControlArn;
        }

        public final void setSecurityControlArn(String securityControlArn) {
            this.securityControlArn = securityControlArn;
        }

        @Override
        public final Builder securityControlArn(String securityControlArn) {
            this.securityControlArn = securityControlArn;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRemediationUrl() {
            return remediationUrl;
        }

        public final void setRemediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
        }

        @Override
        public final Builder remediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
            return this;
        }

        public final String getSeverityRating() {
            return severityRating;
        }

        public final void setSeverityRating(String severityRating) {
            this.severityRating = severityRating;
        }

        @Override
        public final Builder severityRating(String severityRating) {
            this.severityRating = severityRating;
            return this;
        }

        @Override
        public final Builder severityRating(SeverityRating severityRating) {
            this.severityRating(severityRating == null ? null : severityRating.toString());
            return this;
        }

        public final String getSecurityControlStatus() {
            return securityControlStatus;
        }

        public final void setSecurityControlStatus(String securityControlStatus) {
            this.securityControlStatus = securityControlStatus;
        }

        @Override
        public final Builder securityControlStatus(String securityControlStatus) {
            this.securityControlStatus = securityControlStatus;
            return this;
        }

        @Override
        public final Builder securityControlStatus(ControlStatus securityControlStatus) {
            this.securityControlStatus(securityControlStatus == null ? null : securityControlStatus.toString());
            return this;
        }

        public final String getUpdateStatus() {
            return updateStatus;
        }

        public final void setUpdateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
        }

        @Override
        public final Builder updateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
            return this;
        }

        @Override
        public final Builder updateStatus(UpdateStatus updateStatus) {
            this.updateStatus(updateStatus == null ? null : updateStatus.toString());
            return this;
        }

        public final Map<String, ParameterConfiguration.Builder> getParameters() {
            Map<String, ParameterConfiguration.Builder> result = ParametersCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setParameters(Map<String, ParameterConfiguration.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Map<String, ParameterConfiguration> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        public final String getLastUpdateReason() {
            return lastUpdateReason;
        }

        public final void setLastUpdateReason(String lastUpdateReason) {
            this.lastUpdateReason = lastUpdateReason;
        }

        @Override
        public final Builder lastUpdateReason(String lastUpdateReason) {
            this.lastUpdateReason = lastUpdateReason;
            return this;
        }

        @Override
        public SecurityControl build() {
            return new SecurityControl(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
