/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides metadata for a security control, including its unique standard-agnostic identifier, title, description,
 * severity, availability in Amazon Web Services Regions, and a link to remediation steps.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecurityControlDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<SecurityControlDefinition.Builder, SecurityControlDefinition> {
    private static final SdkField<String> SECURITY_CONTROL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityControlId").getter(getter(SecurityControlDefinition::securityControlId))
            .setter(setter(Builder::securityControlId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlId").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(SecurityControlDefinition::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(SecurityControlDefinition::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> REMEDIATION_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RemediationUrl").getter(getter(SecurityControlDefinition::remediationUrl))
            .setter(setter(Builder::remediationUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemediationUrl").build()).build();

    private static final SdkField<String> SEVERITY_RATING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SeverityRating").getter(getter(SecurityControlDefinition::severityRatingAsString))
            .setter(setter(Builder::severityRating))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SeverityRating").build()).build();

    private static final SdkField<String> CURRENT_REGION_AVAILABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CurrentRegionAvailability").getter(getter(SecurityControlDefinition::currentRegionAvailabilityAsString))
            .setter(setter(Builder::currentRegionAvailability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentRegionAvailability").build())
            .build();

    private static final SdkField<List<String>> CUSTOMIZABLE_PROPERTIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomizableProperties")
            .getter(getter(SecurityControlDefinition::customizablePropertiesAsStrings))
            .setter(setter(Builder::customizablePropertiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomizableProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, ParameterDefinition>> PARAMETER_DEFINITIONS_FIELD = SdkField
            .<Map<String, ParameterDefinition>> builder(MarshallingType.MAP)
            .memberName("ParameterDefinitions")
            .getter(getter(SecurityControlDefinition::parameterDefinitions))
            .setter(setter(Builder::parameterDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterDefinitions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ParameterDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SECURITY_CONTROL_ID_FIELD,
            TITLE_FIELD, DESCRIPTION_FIELD, REMEDIATION_URL_FIELD, SEVERITY_RATING_FIELD, CURRENT_REGION_AVAILABILITY_FIELD,
            CUSTOMIZABLE_PROPERTIES_FIELD, PARAMETER_DEFINITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String securityControlId;

    private final String title;

    private final String description;

    private final String remediationUrl;

    private final String severityRating;

    private final String currentRegionAvailability;

    private final List<String> customizableProperties;

    private final Map<String, ParameterDefinition> parameterDefinitions;

    private SecurityControlDefinition(BuilderImpl builder) {
        this.securityControlId = builder.securityControlId;
        this.title = builder.title;
        this.description = builder.description;
        this.remediationUrl = builder.remediationUrl;
        this.severityRating = builder.severityRating;
        this.currentRegionAvailability = builder.currentRegionAvailability;
        this.customizableProperties = builder.customizableProperties;
        this.parameterDefinitions = builder.parameterDefinitions;
    }

    /**
     * <p>
     * The unique identifier of a security control across standards. Values for this field typically consist of an
     * Amazon Web Services service name and a number (for example, APIGateway.3). This parameter differs from
     * <code>SecurityControlArn</code>, which is a unique Amazon Resource Name (ARN) assigned to a control. The ARN
     * references the security control ID (for example,
     * arn:aws:securityhub:eu-central-1:123456789012:security-control/APIGateway.3).
     * </p>
     * 
     * @return The unique identifier of a security control across standards. Values for this field typically consist of
     *         an Amazon Web Services service name and a number (for example, APIGateway.3). This parameter differs from
     *         <code>SecurityControlArn</code>, which is a unique Amazon Resource Name (ARN) assigned to a control. The
     *         ARN references the security control ID (for example,
     *         arn:aws:securityhub:eu-central-1:123456789012:security-control/APIGateway.3).
     */
    public final String securityControlId() {
        return securityControlId;
    }

    /**
     * <p>
     * The title of a security control.
     * </p>
     * 
     * @return The title of a security control.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The description of a security control across standards. This typically summarizes how Security Hub evaluates the
     * control and the conditions under which it produces a failed finding. This parameter doesn't reference a specific
     * standard.
     * </p>
     * 
     * @return The description of a security control across standards. This typically summarizes how Security Hub
     *         evaluates the control and the conditions under which it produces a failed finding. This parameter doesn't
     *         reference a specific standard.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A link to Security Hub documentation that explains how to remediate a failed finding for a security control.
     * </p>
     * 
     * @return A link to Security Hub documentation that explains how to remediate a failed finding for a security
     *         control.
     */
    public final String remediationUrl() {
        return remediationUrl;
    }

    /**
     * <p>
     * The severity of a security control. For more information about how Security Hub determines control severity, see
     * <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of a security control. For more information about how Security Hub determines control
     *         severity, see <a href=
     *         "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     *         >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * @see SeverityRating
     */
    public final SeverityRating severityRating() {
        return SeverityRating.fromValue(severityRating);
    }

    /**
     * <p>
     * The severity of a security control. For more information about how Security Hub determines control severity, see
     * <a href=
     * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityRating}
     * will return {@link SeverityRating#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #severityRatingAsString}.
     * </p>
     * 
     * @return The severity of a security control. For more information about how Security Hub determines control
     *         severity, see <a href=
     *         "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
     *         >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
     * @see SeverityRating
     */
    public final String severityRatingAsString() {
        return severityRating;
    }

    /**
     * <p>
     * Specifies whether a security control is available in the current Amazon Web Services Region.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentRegionAvailability} will return {@link RegionAvailabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #currentRegionAvailabilityAsString}.
     * </p>
     * 
     * @return Specifies whether a security control is available in the current Amazon Web Services Region.
     * @see RegionAvailabilityStatus
     */
    public final RegionAvailabilityStatus currentRegionAvailability() {
        return RegionAvailabilityStatus.fromValue(currentRegionAvailability);
    }

    /**
     * <p>
     * Specifies whether a security control is available in the current Amazon Web Services Region.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentRegionAvailability} will return {@link RegionAvailabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #currentRegionAvailabilityAsString}.
     * </p>
     * 
     * @return Specifies whether a security control is available in the current Amazon Web Services Region.
     * @see RegionAvailabilityStatus
     */
    public final String currentRegionAvailabilityAsString() {
        return currentRegionAvailability;
    }

    /**
     * <p>
     * Security control properties that you can customize. Currently, only parameter customization is supported for
     * select controls. An empty array is returned for controls that don’t support custom properties.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomizableProperties} method.
     * </p>
     * 
     * @return Security control properties that you can customize. Currently, only parameter customization is supported
     *         for select controls. An empty array is returned for controls that don’t support custom properties.
     */
    public final List<SecurityControlProperty> customizableProperties() {
        return CustomizablePropertiesCopier.copyStringToEnum(customizableProperties);
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomizableProperties property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCustomizableProperties() {
        return customizableProperties != null && !(customizableProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Security control properties that you can customize. Currently, only parameter customization is supported for
     * select controls. An empty array is returned for controls that don’t support custom properties.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomizableProperties} method.
     * </p>
     * 
     * @return Security control properties that you can customize. Currently, only parameter customization is supported
     *         for select controls. An empty array is returned for controls that don’t support custom properties.
     */
    public final List<String> customizablePropertiesAsStrings() {
        return customizableProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the ParameterDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasParameterDefinitions() {
        return parameterDefinitions != null && !(parameterDefinitions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An object that provides a security control parameter name, description, and the options for customizing it. This
     * object is excluded for a control that doesn't support custom parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameterDefinitions} method.
     * </p>
     * 
     * @return An object that provides a security control parameter name, description, and the options for customizing
     *         it. This object is excluded for a control that doesn't support custom parameters.
     */
    public final Map<String, ParameterDefinition> parameterDefinitions() {
        return parameterDefinitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(securityControlId());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(remediationUrl());
        hashCode = 31 * hashCode + Objects.hashCode(severityRatingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(currentRegionAvailabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomizableProperties() ? customizablePropertiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasParameterDefinitions() ? parameterDefinitions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecurityControlDefinition)) {
            return false;
        }
        SecurityControlDefinition other = (SecurityControlDefinition) obj;
        return Objects.equals(securityControlId(), other.securityControlId()) && Objects.equals(title(), other.title())
                && Objects.equals(description(), other.description()) && Objects.equals(remediationUrl(), other.remediationUrl())
                && Objects.equals(severityRatingAsString(), other.severityRatingAsString())
                && Objects.equals(currentRegionAvailabilityAsString(), other.currentRegionAvailabilityAsString())
                && hasCustomizableProperties() == other.hasCustomizableProperties()
                && Objects.equals(customizablePropertiesAsStrings(), other.customizablePropertiesAsStrings())
                && hasParameterDefinitions() == other.hasParameterDefinitions()
                && Objects.equals(parameterDefinitions(), other.parameterDefinitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecurityControlDefinition").add("SecurityControlId", securityControlId()).add("Title", title())
                .add("Description", description()).add("RemediationUrl", remediationUrl())
                .add("SeverityRating", severityRatingAsString())
                .add("CurrentRegionAvailability", currentRegionAvailabilityAsString())
                .add("CustomizableProperties", hasCustomizableProperties() ? customizablePropertiesAsStrings() : null)
                .add("ParameterDefinitions", hasParameterDefinitions() ? parameterDefinitions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SecurityControlId":
            return Optional.ofNullable(clazz.cast(securityControlId()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RemediationUrl":
            return Optional.ofNullable(clazz.cast(remediationUrl()));
        case "SeverityRating":
            return Optional.ofNullable(clazz.cast(severityRatingAsString()));
        case "CurrentRegionAvailability":
            return Optional.ofNullable(clazz.cast(currentRegionAvailabilityAsString()));
        case "CustomizableProperties":
            return Optional.ofNullable(clazz.cast(customizablePropertiesAsStrings()));
        case "ParameterDefinitions":
            return Optional.ofNullable(clazz.cast(parameterDefinitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SecurityControlDefinition, T> g) {
        return obj -> g.apply((SecurityControlDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecurityControlDefinition> {
        /**
         * <p>
         * The unique identifier of a security control across standards. Values for this field typically consist of an
         * Amazon Web Services service name and a number (for example, APIGateway.3). This parameter differs from
         * <code>SecurityControlArn</code>, which is a unique Amazon Resource Name (ARN) assigned to a control. The ARN
         * references the security control ID (for example,
         * arn:aws:securityhub:eu-central-1:123456789012:security-control/APIGateway.3).
         * </p>
         * 
         * @param securityControlId
         *        The unique identifier of a security control across standards. Values for this field typically consist
         *        of an Amazon Web Services service name and a number (for example, APIGateway.3). This parameter
         *        differs from <code>SecurityControlArn</code>, which is a unique Amazon Resource Name (ARN) assigned to
         *        a control. The ARN references the security control ID (for example,
         *        arn:aws:securityhub:eu-central-1:123456789012:security-control/APIGateway.3).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlId(String securityControlId);

        /**
         * <p>
         * The title of a security control.
         * </p>
         * 
         * @param title
         *        The title of a security control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The description of a security control across standards. This typically summarizes how Security Hub evaluates
         * the control and the conditions under which it produces a failed finding. This parameter doesn't reference a
         * specific standard.
         * </p>
         * 
         * @param description
         *        The description of a security control across standards. This typically summarizes how Security Hub
         *        evaluates the control and the conditions under which it produces a failed finding. This parameter
         *        doesn't reference a specific standard.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A link to Security Hub documentation that explains how to remediate a failed finding for a security control.
         * </p>
         * 
         * @param remediationUrl
         *        A link to Security Hub documentation that explains how to remediate a failed finding for a security
         *        control.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediationUrl(String remediationUrl);

        /**
         * <p>
         * The severity of a security control. For more information about how Security Hub determines control severity,
         * see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param severityRating
         *        The severity of a security control. For more information about how Security Hub determines control
         *        severity, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         *        >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(String severityRating);

        /**
         * <p>
         * The severity of a security control. For more information about how Security Hub determines control severity,
         * see <a href=
         * "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         * >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * </p>
         * 
         * @param severityRating
         *        The severity of a security control. For more information about how Security Hub determines control
         *        severity, see <a href=
         *        "https://docs.aws.amazon.com/securityhub/latest/userguide/controls-findings-create-update.html#control-findings-severity"
         *        >Assigning severity to control findings</a> in the <i>Security Hub User Guide</i>.
         * @see SeverityRating
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityRating
         */
        Builder severityRating(SeverityRating severityRating);

        /**
         * <p>
         * Specifies whether a security control is available in the current Amazon Web Services Region.
         * </p>
         * 
         * @param currentRegionAvailability
         *        Specifies whether a security control is available in the current Amazon Web Services Region.
         * @see RegionAvailabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionAvailabilityStatus
         */
        Builder currentRegionAvailability(String currentRegionAvailability);

        /**
         * <p>
         * Specifies whether a security control is available in the current Amazon Web Services Region.
         * </p>
         * 
         * @param currentRegionAvailability
         *        Specifies whether a security control is available in the current Amazon Web Services Region.
         * @see RegionAvailabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionAvailabilityStatus
         */
        Builder currentRegionAvailability(RegionAvailabilityStatus currentRegionAvailability);

        /**
         * <p>
         * Security control properties that you can customize. Currently, only parameter customization is supported for
         * select controls. An empty array is returned for controls that don’t support custom properties.
         * </p>
         * 
         * @param customizableProperties
         *        Security control properties that you can customize. Currently, only parameter customization is
         *        supported for select controls. An empty array is returned for controls that don’t support custom
         *        properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizablePropertiesWithStrings(Collection<String> customizableProperties);

        /**
         * <p>
         * Security control properties that you can customize. Currently, only parameter customization is supported for
         * select controls. An empty array is returned for controls that don’t support custom properties.
         * </p>
         * 
         * @param customizableProperties
         *        Security control properties that you can customize. Currently, only parameter customization is
         *        supported for select controls. An empty array is returned for controls that don’t support custom
         *        properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizablePropertiesWithStrings(String... customizableProperties);

        /**
         * <p>
         * Security control properties that you can customize. Currently, only parameter customization is supported for
         * select controls. An empty array is returned for controls that don’t support custom properties.
         * </p>
         * 
         * @param customizableProperties
         *        Security control properties that you can customize. Currently, only parameter customization is
         *        supported for select controls. An empty array is returned for controls that don’t support custom
         *        properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizableProperties(Collection<SecurityControlProperty> customizableProperties);

        /**
         * <p>
         * Security control properties that you can customize. Currently, only parameter customization is supported for
         * select controls. An empty array is returned for controls that don’t support custom properties.
         * </p>
         * 
         * @param customizableProperties
         *        Security control properties that you can customize. Currently, only parameter customization is
         *        supported for select controls. An empty array is returned for controls that don’t support custom
         *        properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customizableProperties(SecurityControlProperty... customizableProperties);

        /**
         * <p>
         * An object that provides a security control parameter name, description, and the options for customizing it.
         * This object is excluded for a control that doesn't support custom parameters.
         * </p>
         * 
         * @param parameterDefinitions
         *        An object that provides a security control parameter name, description, and the options for
         *        customizing it. This object is excluded for a control that doesn't support custom parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterDefinitions(Map<String, ParameterDefinition> parameterDefinitions);
    }

    static final class BuilderImpl implements Builder {
        private String securityControlId;

        private String title;

        private String description;

        private String remediationUrl;

        private String severityRating;

        private String currentRegionAvailability;

        private List<String> customizableProperties = DefaultSdkAutoConstructList.getInstance();

        private Map<String, ParameterDefinition> parameterDefinitions = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SecurityControlDefinition model) {
            securityControlId(model.securityControlId);
            title(model.title);
            description(model.description);
            remediationUrl(model.remediationUrl);
            severityRating(model.severityRating);
            currentRegionAvailability(model.currentRegionAvailability);
            customizablePropertiesWithStrings(model.customizableProperties);
            parameterDefinitions(model.parameterDefinitions);
        }

        public final String getSecurityControlId() {
            return securityControlId;
        }

        public final void setSecurityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
        }

        @Override
        public final Builder securityControlId(String securityControlId) {
            this.securityControlId = securityControlId;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRemediationUrl() {
            return remediationUrl;
        }

        public final void setRemediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
        }

        @Override
        public final Builder remediationUrl(String remediationUrl) {
            this.remediationUrl = remediationUrl;
            return this;
        }

        public final String getSeverityRating() {
            return severityRating;
        }

        public final void setSeverityRating(String severityRating) {
            this.severityRating = severityRating;
        }

        @Override
        public final Builder severityRating(String severityRating) {
            this.severityRating = severityRating;
            return this;
        }

        @Override
        public final Builder severityRating(SeverityRating severityRating) {
            this.severityRating(severityRating == null ? null : severityRating.toString());
            return this;
        }

        public final String getCurrentRegionAvailability() {
            return currentRegionAvailability;
        }

        public final void setCurrentRegionAvailability(String currentRegionAvailability) {
            this.currentRegionAvailability = currentRegionAvailability;
        }

        @Override
        public final Builder currentRegionAvailability(String currentRegionAvailability) {
            this.currentRegionAvailability = currentRegionAvailability;
            return this;
        }

        @Override
        public final Builder currentRegionAvailability(RegionAvailabilityStatus currentRegionAvailability) {
            this.currentRegionAvailability(currentRegionAvailability == null ? null : currentRegionAvailability.toString());
            return this;
        }

        public final Collection<String> getCustomizableProperties() {
            if (customizableProperties instanceof SdkAutoConstructList) {
                return null;
            }
            return customizableProperties;
        }

        public final void setCustomizableProperties(Collection<String> customizableProperties) {
            this.customizableProperties = CustomizablePropertiesCopier.copy(customizableProperties);
        }

        @Override
        public final Builder customizablePropertiesWithStrings(Collection<String> customizableProperties) {
            this.customizableProperties = CustomizablePropertiesCopier.copy(customizableProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customizablePropertiesWithStrings(String... customizableProperties) {
            customizablePropertiesWithStrings(Arrays.asList(customizableProperties));
            return this;
        }

        @Override
        public final Builder customizableProperties(Collection<SecurityControlProperty> customizableProperties) {
            this.customizableProperties = CustomizablePropertiesCopier.copyEnumToString(customizableProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customizableProperties(SecurityControlProperty... customizableProperties) {
            customizableProperties(Arrays.asList(customizableProperties));
            return this;
        }

        public final Map<String, ParameterDefinition.Builder> getParameterDefinitions() {
            Map<String, ParameterDefinition.Builder> result = ParameterDefinitionsCopier.copyToBuilder(this.parameterDefinitions);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setParameterDefinitions(Map<String, ParameterDefinition.BuilderImpl> parameterDefinitions) {
            this.parameterDefinitions = ParameterDefinitionsCopier.copyFromBuilder(parameterDefinitions);
        }

        @Override
        public final Builder parameterDefinitions(Map<String, ParameterDefinition> parameterDefinitions) {
            this.parameterDefinitions = ParameterDefinitionsCopier.copy(parameterDefinitions);
            return this;
        }

        @Override
        public SecurityControlDefinition build() {
            return new SecurityControlDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
