/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Security Lake can collect logs and events from supported Amazon Web Services services and custom sources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccountSources implements SdkPojo, Serializable, ToCopyableBuilder<AccountSources.Builder, AccountSources> {
    private static final SdkField<String> ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("account")
            .getter(getter(AccountSources::account)).setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("account").build()).build();

    private static final SdkField<String> EVENT_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventClass").getter(getter(AccountSources::eventClassAsString)).setter(setter(Builder::eventClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventClass").build()).build();

    private static final SdkField<List<LogsStatus>> LOGS_STATUS_FIELD = SdkField
            .<List<LogsStatus>> builder(MarshallingType.LIST)
            .memberName("logsStatus")
            .getter(getter(AccountSources::logsStatus))
            .setter(setter(Builder::logsStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logsStatus").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LogsStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogsStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceType").getter(getter(AccountSources::sourceType)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_FIELD,
            EVENT_CLASS_FIELD, LOGS_STATUS_FIELD, SOURCE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String account;

    private final String eventClass;

    private final List<LogsStatus> logsStatus;

    private final String sourceType;

    private AccountSources(BuilderImpl builder) {
        this.account = builder.account;
        this.eventClass = builder.eventClass;
        this.logsStatus = builder.logsStatus;
        this.sourceType = builder.sourceType;
    }

    /**
     * <p>
     * Account ID of the Security Lake account for which logs are collected.
     * </p>
     * 
     * @return Account ID of the Security Lake account for which logs are collected.
     */
    public final String account() {
        return account;
    }

    /**
     * <p>
     * Initializes a new instance of the Event class.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventClass} will
     * return {@link OcsfEventClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventClassAsString}.
     * </p>
     * 
     * @return Initializes a new instance of the Event class.
     * @see OcsfEventClass
     */
    public final OcsfEventClass eventClass() {
        return OcsfEventClass.fromValue(eventClass);
    }

    /**
     * <p>
     * Initializes a new instance of the Event class.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventClass} will
     * return {@link OcsfEventClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventClassAsString}.
     * </p>
     * 
     * @return Initializes a new instance of the Event class.
     * @see OcsfEventClass
     */
    public final String eventClassAsString() {
        return eventClass;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogsStatus property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLogsStatus() {
        return logsStatus != null && !(logsStatus instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Log status for the Security Lake account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogsStatus} method.
     * </p>
     * 
     * @return Log status for the Security Lake account.
     */
    public final List<LogsStatus> logsStatus() {
        return logsStatus;
    }

    /**
     * <p>
     * The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
     * supports logs and events collection for natively-supported Amazon Web Services services. For more information,
     * see the Amazon Security Lake User Guide.
     * </p>
     * 
     * @return The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
     *         supports logs and events collection for natively-supported Amazon Web Services services. For more
     *         information, see the Amazon Security Lake User Guide.
     */
    public final String sourceType() {
        return sourceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(eventClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogsStatus() ? logsStatus() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccountSources)) {
            return false;
        }
        AccountSources other = (AccountSources) obj;
        return Objects.equals(account(), other.account()) && Objects.equals(eventClassAsString(), other.eventClassAsString())
                && hasLogsStatus() == other.hasLogsStatus() && Objects.equals(logsStatus(), other.logsStatus())
                && Objects.equals(sourceType(), other.sourceType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccountSources").add("Account", account()).add("EventClass", eventClassAsString())
                .add("LogsStatus", hasLogsStatus() ? logsStatus() : null).add("SourceType", sourceType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "account":
            return Optional.ofNullable(clazz.cast(account()));
        case "eventClass":
            return Optional.ofNullable(clazz.cast(eventClassAsString()));
        case "logsStatus":
            return Optional.ofNullable(clazz.cast(logsStatus()));
        case "sourceType":
            return Optional.ofNullable(clazz.cast(sourceType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AccountSources, T> g) {
        return obj -> g.apply((AccountSources) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccountSources> {
        /**
         * <p>
         * Account ID of the Security Lake account for which logs are collected.
         * </p>
         * 
         * @param account
         *        Account ID of the Security Lake account for which logs are collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * Initializes a new instance of the Event class.
         * </p>
         * 
         * @param eventClass
         *        Initializes a new instance of the Event class.
         * @see OcsfEventClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OcsfEventClass
         */
        Builder eventClass(String eventClass);

        /**
         * <p>
         * Initializes a new instance of the Event class.
         * </p>
         * 
         * @param eventClass
         *        Initializes a new instance of the Event class.
         * @see OcsfEventClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OcsfEventClass
         */
        Builder eventClass(OcsfEventClass eventClass);

        /**
         * <p>
         * Log status for the Security Lake account.
         * </p>
         * 
         * @param logsStatus
         *        Log status for the Security Lake account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logsStatus(Collection<LogsStatus> logsStatus);

        /**
         * <p>
         * Log status for the Security Lake account.
         * </p>
         * 
         * @param logsStatus
         *        Log status for the Security Lake account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logsStatus(LogsStatus... logsStatus);

        /**
         * <p>
         * Log status for the Security Lake account.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securitylake.model.LogsStatus.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securitylake.model.LogsStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securitylake.model.LogsStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #logsStatus(List<LogsStatus>)}.
         * 
         * @param logsStatus
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securitylake.model.LogsStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logsStatus(java.util.Collection<LogsStatus>)
         */
        Builder logsStatus(Consumer<LogsStatus.Builder>... logsStatus);

        /**
         * <p>
         * The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
         * supports logs and events collection for natively-supported Amazon Web Services services. For more
         * information, see the Amazon Security Lake User Guide.
         * </p>
         * 
         * @param sourceType
         *        The supported Amazon Web Services services from which logs and events are collected. Amazon Security
         *        Lake supports logs and events collection for natively-supported Amazon Web Services services. For more
         *        information, see the Amazon Security Lake User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceType(String sourceType);
    }

    static final class BuilderImpl implements Builder {
        private String account;

        private String eventClass;

        private List<LogsStatus> logsStatus = DefaultSdkAutoConstructList.getInstance();

        private String sourceType;

        private BuilderImpl() {
        }

        private BuilderImpl(AccountSources model) {
            account(model.account);
            eventClass(model.eventClass);
            logsStatus(model.logsStatus);
            sourceType(model.sourceType);
        }

        public final String getAccount() {
            return account;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final String getEventClass() {
            return eventClass;
        }

        public final void setEventClass(String eventClass) {
            this.eventClass = eventClass;
        }

        @Override
        public final Builder eventClass(String eventClass) {
            this.eventClass = eventClass;
            return this;
        }

        @Override
        public final Builder eventClass(OcsfEventClass eventClass) {
            this.eventClass(eventClass == null ? null : eventClass.toString());
            return this;
        }

        public final List<LogsStatus.Builder> getLogsStatus() {
            List<LogsStatus.Builder> result = LogsStatusListCopier.copyToBuilder(this.logsStatus);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLogsStatus(Collection<LogsStatus.BuilderImpl> logsStatus) {
            this.logsStatus = LogsStatusListCopier.copyFromBuilder(logsStatus);
        }

        @Override
        public final Builder logsStatus(Collection<LogsStatus> logsStatus) {
            this.logsStatus = LogsStatusListCopier.copy(logsStatus);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logsStatus(LogsStatus... logsStatus) {
            logsStatus(Arrays.asList(logsStatus));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logsStatus(Consumer<LogsStatus.Builder>... logsStatus) {
            logsStatus(Stream.of(logsStatus).map(c -> LogsStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public AccountSources build() {
            return new AccountSources(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
