/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.securitylake.model.AccessDeniedException;
import software.amazon.awssdk.services.securitylake.model.AccountNotFoundException;
import software.amazon.awssdk.services.securitylake.model.BucketNotFoundException;
import software.amazon.awssdk.services.securitylake.model.ConcurrentModificationException;
import software.amazon.awssdk.services.securitylake.model.ConflictException;
import software.amazon.awssdk.services.securitylake.model.ConflictSourceNamesException;
import software.amazon.awssdk.services.securitylake.model.ConflictSubscriptionException;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeAutoEnableRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeAutoEnableResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeDelegatedAdminRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeDelegatedAdminResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeExceptionsSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeExceptionsSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDatalakeResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriptionNotificationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriptionNotificationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeAutoEnableRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeAutoEnableResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeDelegatedAdminRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeDelegatedAdminResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeExceptionsSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeExceptionsSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDatalakeResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriptionNotificationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriptionNotificationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.EventBridgeException;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeAutoEnableRequest;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeAutoEnableResponse;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeExceptionsExpiryRequest;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeExceptionsExpiryResponse;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeExceptionsSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeExceptionsSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeRequest;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeResponse;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusRequest;
import software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusResponse;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.InternalServerException;
import software.amazon.awssdk.services.securitylake.model.InvalidInputException;
import software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsRequest;
import software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsResponse;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesRequest;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesResponse;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersRequest;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersResponse;
import software.amazon.awssdk.services.securitylake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.securitylake.model.S3Exception;
import software.amazon.awssdk.services.securitylake.model.SecurityLakeException;
import software.amazon.awssdk.services.securitylake.model.SecurityLakeRequest;
import software.amazon.awssdk.services.securitylake.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.securitylake.model.ThrottlingException;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeExceptionsExpiryRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeExceptionsExpiryResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeExceptionsSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeExceptionsSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDatalakeResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriptionNotificationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriptionNotificationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.ValidationException;
import software.amazon.awssdk.services.securitylake.paginators.GetDatalakeStatusPublisher;
import software.amazon.awssdk.services.securitylake.paginators.ListDatalakeExceptionsPublisher;
import software.amazon.awssdk.services.securitylake.paginators.ListLogSourcesPublisher;
import software.amazon.awssdk.services.securitylake.paginators.ListSubscribersPublisher;
import software.amazon.awssdk.services.securitylake.transform.CreateAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDatalakeAutoEnableRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDatalakeDelegatedAdminRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDatalakeExceptionsSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDatalakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriptionNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDatalakeAutoEnableRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDatalakeDelegatedAdminRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDatalakeExceptionsSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDatalakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriptionNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDatalakeAutoEnableRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDatalakeExceptionsExpiryRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDatalakeExceptionsSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDatalakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDatalakeStatusRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListDatalakeExceptionsRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListLogSourcesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListSubscribersRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDatalakeExceptionsExpiryRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDatalakeExceptionsSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDatalakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriptionNotificationConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SecurityLakeAsyncClient}.
 *
 * @see SecurityLakeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecurityLakeAsyncClient implements SecurityLakeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSecurityLakeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSecurityLakeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a natively supported Amazon Web Service as an Amazon Security Lake source. Enables source types for member
     * accounts in required Amazon Web Services Regions, based on the parameters you specify. You can choose any source
     * type in any Region for either accounts that are part of a trusted organization or standalone accounts. At least
     * one of the three dimensions is a mandatory input to this API. However, you can supply any combination of the
     * three dimensions to this API.
     * </p>
     * <p>
     * By default, a dimension refers to the entire set. When you don't provide a dimension, Security Lake assumes that
     * the missing dimension refers to the entire set. This is overridden when you supply any one of the inputs. For
     * instance, when you do not specify members, the API enables all Security Lake member accounts for all sources.
     * Similarly, when you do not specify Regions, Security Lake is enabled for all the Regions where Security Lake is
     * available as a service.
     * </p>
     * <p>
     * You can use this API only to enable natively supported Amazon Web Services as a source. Use
     * <code>CreateCustomLogSource</code> to enable data collection from a custom source.
     * </p>
     *
     * @param createAwsLogSourceRequest
     * @return A Java Future containing the result of the CreateAwsLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>S3Exception Provides an extension of the AmazonServiceException for errors reported by Amazon S3
     *         while processing a request. In particular, this class provides access to the Amazon S3 extended request
     *         ID. If Amazon S3 is incorrectly handling a request and you need to contact Amazon, this extended request
     *         ID may provide useful debugging information.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAwsLogSourceResponse> createAwsLogSource(CreateAwsLogSourceRequest createAwsLogSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAwsLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAwsLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAwsLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAwsLogSourceRequest, CreateAwsLogSourceResponse>()
                            .withOperationName("CreateAwsLogSource")
                            .withMarshaller(new CreateAwsLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAwsLogSourceRequest));
            CompletableFuture<CreateAwsLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a third-party custom source in Amazon Security Lake, from the Amazon Web Services Region where you want to
     * create a custom source. Security Lake can collect logs and events from third-party custom sources. After creating
     * the appropriate IAM role to invoke Glue crawler, use this API to add a custom source name in Security Lake. This
     * operation creates a partition in the Amazon S3 bucket for Security Lake as the target location for log files from
     * the custom source in addition to an associated Glue table and an Glue crawler.
     * </p>
     *
     * @param createCustomLogSourceRequest
     * @return A Java Future containing the result of the CreateCustomLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ConflictSourceNamesException There was a conflict when you attempted to modify a Security Lake source
     *         name.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>BucketNotFoundException Amazon Security Lake generally returns 404 errors if the requested object is
     *         missing from the bucket.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomLogSourceResponse> createCustomLogSource(
            CreateCustomLogSourceRequest createCustomLogSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCustomLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomLogSourceRequest, CreateCustomLogSourceResponse>()
                            .withOperationName("CreateCustomLogSource")
                            .withMarshaller(new CreateCustomLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCustomLogSourceRequest));
            CompletableFuture<CreateCustomLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initializes an Amazon Security Lake instance with the provided (or default) configuration. You can enable
     * Security Lake in Amazon Web Services Regions with customized settings before enabling log collection in Regions.
     * You can either use the <code>enableAll</code> parameter to specify all Regions or specify the Regions where you
     * want to enable Security Lake. To specify particular Regions, use the <code>Regions</code> parameter and then
     * configure these Regions using the <code>configurations</code> parameter. If you have already enabled Security
     * Lake in a Region when you call this command, the command will update the Region if you provide new configuration
     * parameters. If you have not already enabled Security Lake in the Region when you call this API, it will set up
     * the data lake in the Region with the specified configurations.
     * </p>
     * <p>
     * When you enable Security Lake, it starts ingesting security data after the <code>CreateAwsLogSource</code> call.
     * This includes ingesting security data from sources, storing data, and making data accessible to subscribers.
     * Security Lake also enables all the existing settings and resources that it stores or maintains for your Amazon
     * Web Services account in the current Region, including security log and event data. For more information, see the
     * <a href="https://docs.aws.amazon.com/security-lake/latest/userguide/what-is-security-lake.html">Amazon Security
     * Lake User Guide</a>.
     * </p>
     *
     * @param createDatalakeRequest
     * @return A Java Future containing the result of the CreateDatalake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded your service quota. To perform the requested action,
     *         remove some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDatalake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDatalake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatalakeResponse> createDatalake(CreateDatalakeRequest createDatalakeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatalakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatalake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatalakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatalakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatalakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatalakeRequest, CreateDatalakeResponse>()
                            .withOperationName("CreateDatalake")
                            .withMarshaller(new CreateDatalakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatalakeRequest));
            CompletableFuture<CreateDatalakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Automatically enables Amazon Security Lake for new member accounts in your organization. Security Lake is not
     * automatically enabled for any existing member accounts in your organization.
     * </p>
     *
     * @param createDatalakeAutoEnableRequest
     * @return A Java Future containing the result of the CreateDatalakeAutoEnable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDatalakeAutoEnable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDatalakeAutoEnable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatalakeAutoEnableResponse> createDatalakeAutoEnable(
            CreateDatalakeAutoEnableRequest createDatalakeAutoEnableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatalakeAutoEnableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatalakeAutoEnable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatalakeAutoEnableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatalakeAutoEnableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatalakeAutoEnableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatalakeAutoEnableRequest, CreateDatalakeAutoEnableResponse>()
                            .withOperationName("CreateDatalakeAutoEnable")
                            .withMarshaller(new CreateDatalakeAutoEnableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatalakeAutoEnableRequest));
            CompletableFuture<CreateDatalakeAutoEnableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param createDatalakeDelegatedAdminRequest
     * @return A Java Future containing the result of the CreateDatalakeDelegatedAdmin operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDatalakeDelegatedAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDatalakeDelegatedAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatalakeDelegatedAdminResponse> createDatalakeDelegatedAdmin(
            CreateDatalakeDelegatedAdminRequest createDatalakeDelegatedAdminRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatalakeDelegatedAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatalakeDelegatedAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatalakeDelegatedAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatalakeDelegatedAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatalakeDelegatedAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatalakeDelegatedAdminRequest, CreateDatalakeDelegatedAdminResponse>()
                            .withOperationName("CreateDatalakeDelegatedAdmin")
                            .withMarshaller(new CreateDatalakeDelegatedAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatalakeDelegatedAdminRequest));
            CompletableFuture<CreateDatalakeDelegatedAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param createDatalakeExceptionsSubscriptionRequest
     * @return A Java Future containing the result of the CreateDatalakeExceptionsSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDatalakeExceptionsSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDatalakeExceptionsSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatalakeExceptionsSubscriptionResponse> createDatalakeExceptionsSubscription(
            CreateDatalakeExceptionsSubscriptionRequest createDatalakeExceptionsSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDatalakeExceptionsSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatalakeExceptionsSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatalakeExceptionsSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateDatalakeExceptionsSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatalakeExceptionsSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatalakeExceptionsSubscriptionRequest, CreateDatalakeExceptionsSubscriptionResponse>()
                            .withOperationName("CreateDatalakeExceptionsSubscription")
                            .withMarshaller(new CreateDatalakeExceptionsSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatalakeExceptionsSubscriptionRequest));
            CompletableFuture<CreateDatalakeExceptionsSubscriptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subscription permission for accounts that are already enabled in Amazon Security Lake. You can create a
     * subscriber with access to data in the current Amazon Web Services Region.
     * </p>
     *
     * @param createSubscriberRequest
     * @return A Java Future containing the result of the CreateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictSubscriptionException A conflicting subscription exception operation is in progress.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>BucketNotFoundException Amazon Security Lake generally returns 404 errors if the requested object is
     *         missing from the bucket.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriberResponse> createSubscriber(CreateSubscriberRequest createSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                            .withOperationName("CreateSubscriber")
                            .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSubscriberRequest));
            CompletableFuture<CreateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Notifies the subscriber when new data is written to the data lake for the sources that the subscriber consumes in
     * Security Lake.
     * </p>
     *
     * @param createSubscriptionNotificationConfigurationRequest
     * @return A Java Future containing the result of the CreateSubscriptionNotificationConfiguration operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException More than one process tried to modify a resource at the same time.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateSubscriptionNotificationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriptionNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriptionNotificationConfigurationResponse> createSubscriptionNotificationConfiguration(
            CreateSubscriptionNotificationConfigurationRequest createSubscriptionNotificationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createSubscriptionNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriptionNotificationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriptionNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateSubscriptionNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriptionNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriptionNotificationConfigurationRequest, CreateSubscriptionNotificationConfigurationResponse>()
                            .withOperationName("CreateSubscriptionNotificationConfiguration")
                            .withMarshaller(new CreateSubscriptionNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(createSubscriptionNotificationConfigurationRequest));
            CompletableFuture<CreateSubscriptionNotificationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a natively supported Amazon Web Service as an Amazon Security Lake source. When you remove the source,
     * Security Lake stops collecting data from that source, and subscribers can no longer consume new data from the
     * source. Subscribers can still consume data that Security Lake collected from the source before disablement.
     * </p>
     * <p>
     * You can choose any source type in any Amazon Web Services Region for either accounts that are part of a trusted
     * organization or standalone accounts. At least one of the three dimensions is a mandatory input to this API.
     * However, you can supply any combination of the three dimensions to this API.
     * </p>
     * <p>
     * By default, a dimension refers to the entire set. This is overridden when you supply any one of the inputs. For
     * instance, when you do not specify members, the API disables all Security Lake member accounts for sources.
     * Similarly, when you do not specify Regions, Security Lake is disabled for all the Regions where Security Lake is
     * available as a service.
     * </p>
     * <p>
     * When you don't provide a dimension, Security Lake assumes that the missing dimension refers to the entire set.
     * For example, if you don't provide specific accounts, the API applies to the entire set of accounts in your
     * organization.
     * </p>
     *
     * @param deleteAwsLogSourceRequest
     * @return A Java Future containing the result of the DeleteAwsLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAwsLogSourceResponse> deleteAwsLogSource(DeleteAwsLogSourceRequest deleteAwsLogSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAwsLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAwsLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAwsLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAwsLogSourceRequest, DeleteAwsLogSourceResponse>()
                            .withOperationName("DeleteAwsLogSource")
                            .withMarshaller(new DeleteAwsLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAwsLogSourceRequest));
            CompletableFuture<DeleteAwsLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a custom log source from Amazon Security Lake.
     * </p>
     *
     * @param deleteCustomLogSourceRequest
     * @return A Java Future containing the result of the DeleteCustomLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ConflictSourceNamesException There was a conflict when you attempted to modify a Security Lake source
     *         name.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>BucketNotFoundException Amazon Security Lake generally returns 404 errors if the requested object is
     *         missing from the bucket.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomLogSourceResponse> deleteCustomLogSource(
            DeleteCustomLogSourceRequest deleteCustomLogSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomLogSourceRequest, DeleteCustomLogSourceResponse>()
                            .withOperationName("DeleteCustomLogSource")
                            .withMarshaller(new DeleteCustomLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCustomLogSourceRequest));
            CompletableFuture<DeleteCustomLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you delete Amazon Security Lake from your account, Security Lake is disabled in all Amazon Web Services
     * Regions. Also, this API automatically takes steps to remove the account from Security Lake .
     * </p>
     * <p>
     * This operation disables security data collection from sources, deletes data stored, and stops making data
     * accessible to subscribers. Security Lake also deletes all the existing settings and resources that it stores or
     * maintains for your Amazon Web Services account in the current Region, including security log and event data. The
     * <code>DeleteDatalake</code> operation does not delete the Amazon S3 bucket, which is owned by your Amazon Web
     * Services account. For more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/disable-security-lake.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param deleteDatalakeRequest
     * @return A Java Future containing the result of the DeleteDatalake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You have exceeded your service quota. To perform the requested action,
     *         remove some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDatalake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDatalake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatalakeResponse> deleteDatalake(DeleteDatalakeRequest deleteDatalakeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatalakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatalake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatalakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatalakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatalakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatalakeRequest, DeleteDatalakeResponse>()
                            .withOperationName("DeleteDatalake")
                            .withMarshaller(new DeleteDatalakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatalakeRequest));
            CompletableFuture<DeleteDatalakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Automatically deletes Amazon Security Lake to stop collecting security data. When you delete Amazon Security Lake
     * from your account, Security Lake is disabled in all Regions. Also, this API automatically takes steps to remove
     * the account from Security Lake .
     * </p>
     * <p>
     * This operation disables security data collection from sources, deletes data stored, and stops making data
     * accessible to subscribers. Security Lake also deletes all the existing settings and resources that it stores or
     * maintains for your Amazon Web Services account in the current Region, including security log and event data. The
     * <code>DeleteDatalake</code> operation does not delete the Amazon S3 bucket, which is owned by your Amazon Web
     * Services account. For more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/disable-security-lake.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param deleteDatalakeAutoEnableRequest
     * @return A Java Future containing the result of the DeleteDatalakeAutoEnable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDatalakeAutoEnable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDatalakeAutoEnable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatalakeAutoEnableResponse> deleteDatalakeAutoEnable(
            DeleteDatalakeAutoEnableRequest deleteDatalakeAutoEnableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatalakeAutoEnableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatalakeAutoEnable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatalakeAutoEnableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatalakeAutoEnableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatalakeAutoEnableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatalakeAutoEnableRequest, DeleteDatalakeAutoEnableResponse>()
                            .withOperationName("DeleteDatalakeAutoEnable")
                            .withMarshaller(new DeleteDatalakeAutoEnableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatalakeAutoEnableRequest));
            CompletableFuture<DeleteDatalakeAutoEnableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param deleteDatalakeDelegatedAdminRequest
     * @return A Java Future containing the result of the DeleteDatalakeDelegatedAdmin operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDatalakeDelegatedAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDatalakeDelegatedAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatalakeDelegatedAdminResponse> deleteDatalakeDelegatedAdmin(
            DeleteDatalakeDelegatedAdminRequest deleteDatalakeDelegatedAdminRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatalakeDelegatedAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatalakeDelegatedAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatalakeDelegatedAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatalakeDelegatedAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatalakeDelegatedAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatalakeDelegatedAdminRequest, DeleteDatalakeDelegatedAdminResponse>()
                            .withOperationName("DeleteDatalakeDelegatedAdmin")
                            .withMarshaller(new DeleteDatalakeDelegatedAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatalakeDelegatedAdminRequest));
            CompletableFuture<DeleteDatalakeDelegatedAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteDatalakeExceptionsSubscriptionRequest
     * @return A Java Future containing the result of the DeleteDatalakeExceptionsSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDatalakeExceptionsSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDatalakeExceptionsSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatalakeExceptionsSubscriptionResponse> deleteDatalakeExceptionsSubscription(
            DeleteDatalakeExceptionsSubscriptionRequest deleteDatalakeExceptionsSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDatalakeExceptionsSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatalakeExceptionsSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatalakeExceptionsSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteDatalakeExceptionsSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatalakeExceptionsSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatalakeExceptionsSubscriptionRequest, DeleteDatalakeExceptionsSubscriptionResponse>()
                            .withOperationName("DeleteDatalakeExceptionsSubscription")
                            .withMarshaller(new DeleteDatalakeExceptionsSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatalakeExceptionsSubscriptionRequest));
            CompletableFuture<DeleteDatalakeExceptionsSubscriptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the subscription permission for accounts that are already enabled in Amazon Security Lake. You can delete
     * a subscriber and remove access to data in the current Amazon Web Services Region.
     * </p>
     *
     * @param deleteSubscriberRequest
     * @return A Java Future containing the result of the DeleteSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException More than one process tried to modify a resource at the same time.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>BucketNotFoundException Amazon Security Lake generally returns 404 errors if the requested object is
     *         missing from the bucket.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubscriberResponse> deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                            .withOperationName("DeleteSubscriber")
                            .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSubscriberRequest));
            CompletableFuture<DeleteSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteSubscriptionNotificationConfigurationRequest
     * @return A Java Future containing the result of the DeleteSubscriptionNotificationConfiguration operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException More than one process tried to modify a resource at the same time.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteSubscriptionNotificationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriptionNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubscriptionNotificationConfigurationResponse> deleteSubscriptionNotificationConfiguration(
            DeleteSubscriptionNotificationConfigurationRequest deleteSubscriptionNotificationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSubscriptionNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriptionNotificationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriptionNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteSubscriptionNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriptionNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriptionNotificationConfigurationRequest, DeleteSubscriptionNotificationConfigurationResponse>()
                            .withOperationName("DeleteSubscriptionNotificationConfiguration")
                            .withMarshaller(new DeleteSubscriptionNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSubscriptionNotificationConfigurationRequest));
            CompletableFuture<DeleteSubscriptionNotificationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Amazon Security Lake configuration object for the specified Amazon Web Services account ID. You can
     * use the <code>GetDatalake</code> API to know whether Security Lake is enabled for the current Region. This API
     * does not take input parameters.
     * </p>
     *
     * @param getDatalakeRequest
     * @return A Java Future containing the result of the GetDatalake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalake" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatalakeResponse> getDatalake(GetDatalakeRequest getDatalakeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatalakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatalake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatalakeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDatalakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatalakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatalakeRequest, GetDatalakeResponse>()
                            .withOperationName("GetDatalake").withMarshaller(new GetDatalakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDatalakeRequest));
            CompletableFuture<GetDatalakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the configuration that will be automatically set up for accounts added to the organization after the
     * organization has onboarded to Amazon Security Lake. This API does not take input parameters.
     * </p>
     *
     * @param getDatalakeAutoEnableRequest
     * @return A Java Future containing the result of the GetDatalakeAutoEnable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalakeAutoEnable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalakeAutoEnable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatalakeAutoEnableResponse> getDatalakeAutoEnable(
            GetDatalakeAutoEnableRequest getDatalakeAutoEnableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatalakeAutoEnableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatalakeAutoEnable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatalakeAutoEnableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDatalakeAutoEnableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatalakeAutoEnableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatalakeAutoEnableRequest, GetDatalakeAutoEnableResponse>()
                            .withOperationName("GetDatalakeAutoEnable")
                            .withMarshaller(new GetDatalakeAutoEnableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDatalakeAutoEnableRequest));
            CompletableFuture<GetDatalakeAutoEnableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the expiration period and time-to-live (TTL) for which the exception message will remain. Exceptions
     * are stored by default, for 2 weeks from when a record was created in Amazon Security Lake. This API does not take
     * input parameters.
     * </p>
     *
     * @param getDatalakeExceptionsExpiryRequest
     * @return A Java Future containing the result of the GetDatalakeExceptionsExpiry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalakeExceptionsExpiry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalakeExceptionsExpiry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatalakeExceptionsExpiryResponse> getDatalakeExceptionsExpiry(
            GetDatalakeExceptionsExpiryRequest getDatalakeExceptionsExpiryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatalakeExceptionsExpiryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatalakeExceptionsExpiry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatalakeExceptionsExpiryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDatalakeExceptionsExpiryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatalakeExceptionsExpiryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatalakeExceptionsExpiryRequest, GetDatalakeExceptionsExpiryResponse>()
                            .withOperationName("GetDatalakeExceptionsExpiry")
                            .withMarshaller(new GetDatalakeExceptionsExpiryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDatalakeExceptionsExpiryRequest));
            CompletableFuture<GetDatalakeExceptionsExpiryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of exception notifications for the account in Amazon Security Lake.
     * </p>
     *
     * @param getDatalakeExceptionsSubscriptionRequest
     * @return A Java Future containing the result of the GetDatalakeExceptionsSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalakeExceptionsSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalakeExceptionsSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatalakeExceptionsSubscriptionResponse> getDatalakeExceptionsSubscription(
            GetDatalakeExceptionsSubscriptionRequest getDatalakeExceptionsSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDatalakeExceptionsSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatalakeExceptionsSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatalakeExceptionsSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetDatalakeExceptionsSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatalakeExceptionsSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatalakeExceptionsSubscriptionRequest, GetDatalakeExceptionsSubscriptionResponse>()
                            .withOperationName("GetDatalakeExceptionsSubscription")
                            .withMarshaller(new GetDatalakeExceptionsSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDatalakeExceptionsSubscriptionRequest));
            CompletableFuture<GetDatalakeExceptionsSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a snapshot of the current Region, including whether Amazon Security Lake is enabled for those accounts
     * and which sources Security Lake is collecting data from.
     * </p>
     *
     * @param getDatalakeStatusRequest
     * @return A Java Future containing the result of the GetDatalakeStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalakeStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalakeStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatalakeStatusResponse> getDatalakeStatus(GetDatalakeStatusRequest getDatalakeStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatalakeStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatalakeStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatalakeStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDatalakeStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatalakeStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatalakeStatusRequest, GetDatalakeStatusResponse>()
                            .withOperationName("GetDatalakeStatus")
                            .withMarshaller(new GetDatalakeStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDatalakeStatusRequest));
            CompletableFuture<GetDatalakeStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a snapshot of the current Region, including whether Amazon Security Lake is enabled for those accounts
     * and which sources Security Lake is collecting data from.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getDatalakeStatus(software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.GetDatalakeStatusPublisher publisher = client.getDatalakeStatusPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.GetDatalakeStatusPublisher publisher = client.getDatalakeStatusPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxAccountResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getDatalakeStatus(software.amazon.awssdk.services.securitylake.model.GetDatalakeStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getDatalakeStatusRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDatalakeStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDatalakeStatus"
     *      target="_top">AWS API Documentation</a>
     */
    public GetDatalakeStatusPublisher getDatalakeStatusPaginator(GetDatalakeStatusRequest getDatalakeStatusRequest) {
        return new GetDatalakeStatusPublisher(this, applyPaginatorUserAgent(getDatalakeStatusRequest));
    }

    /**
     * <p>
     * Retrieves the subscription information for the specified subscription ID. You can get information about a
     * specific subscriber.
     * </p>
     *
     * @param getSubscriberRequest
     * @return A Java Future containing the result of the GetSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriberResponse> getSubscriber(GetSubscriberRequest getSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriberRequest, GetSubscriberResponse>()
                            .withOperationName("GetSubscriber")
                            .withMarshaller(new GetSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSubscriberRequest));
            CompletableFuture<GetSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Security Lake exceptions that you can use to find the source of problems and fix them.
     * </p>
     *
     * @param listDatalakeExceptionsRequest
     * @return A Java Future containing the result of the ListDatalakeExceptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListDatalakeExceptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDatalakeExceptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatalakeExceptionsResponse> listDatalakeExceptions(
            ListDatalakeExceptionsRequest listDatalakeExceptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatalakeExceptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatalakeExceptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatalakeExceptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatalakeExceptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatalakeExceptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatalakeExceptionsRequest, ListDatalakeExceptionsResponse>()
                            .withOperationName("ListDatalakeExceptions")
                            .withMarshaller(new ListDatalakeExceptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatalakeExceptionsRequest));
            CompletableFuture<ListDatalakeExceptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Security Lake exceptions that you can use to find the source of problems and fix them.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatalakeExceptions(software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListDatalakeExceptionsPublisher publisher = client.listDatalakeExceptionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListDatalakeExceptionsPublisher publisher = client.listDatalakeExceptionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxFailures won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatalakeExceptions(software.amazon.awssdk.services.securitylake.model.ListDatalakeExceptionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatalakeExceptionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListDatalakeExceptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDatalakeExceptions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDatalakeExceptionsPublisher listDatalakeExceptionsPaginator(
            ListDatalakeExceptionsRequest listDatalakeExceptionsRequest) {
        return new ListDatalakeExceptionsPublisher(this, applyPaginatorUserAgent(listDatalakeExceptionsRequest));
    }

    /**
     * <p>
     * Retrieves the log sources in the current Amazon Web Services Region.
     * </p>
     *
     * @param listLogSourcesRequest
     * @return A Java Future containing the result of the ListLogSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListLogSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListLogSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLogSourcesResponse> listLogSources(ListLogSourcesRequest listLogSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLogSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLogSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLogSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLogSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLogSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLogSourcesRequest, ListLogSourcesResponse>()
                            .withOperationName("ListLogSources")
                            .withMarshaller(new ListLogSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLogSourcesRequest));
            CompletableFuture<ListLogSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the log sources in the current Amazon Web Services Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLogSources(software.amazon.awssdk.services.securitylake.model.ListLogSourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListLogSourcesPublisher publisher = client.listLogSourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListLogSourcesPublisher publisher = client.listLogSourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securitylake.model.ListLogSourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securitylake.model.ListLogSourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLogSources(software.amazon.awssdk.services.securitylake.model.ListLogSourcesRequest)} operation.</b>
     * </p>
     *
     * @param listLogSourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListLogSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListLogSources" target="_top">AWS
     *      API Documentation</a>
     */
    public ListLogSourcesPublisher listLogSourcesPaginator(ListLogSourcesRequest listLogSourcesRequest) {
        return new ListLogSourcesPublisher(this, applyPaginatorUserAgent(listLogSourcesRequest));
    }

    /**
     * <p>
     * List all subscribers for the specific Amazon Security Lake account ID. You can retrieve a list of subscriptions
     * associated with a specific organization or Amazon Web Services account.
     * </p>
     *
     * @param listSubscribersRequest
     * @return A Java Future containing the result of the ListSubscribers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListSubscribers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListSubscribers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscribersResponse> listSubscribers(ListSubscribersRequest listSubscribersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubscribersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubscribers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSubscribersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSubscribersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSubscribersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscribersRequest, ListSubscribersResponse>()
                            .withOperationName("ListSubscribers")
                            .withMarshaller(new ListSubscribersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSubscribersRequest));
            CompletableFuture<ListSubscribersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all subscribers for the specific Amazon Security Lake account ID. You can retrieve a list of subscriptions
     * associated with a specific organization or Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubscribers(software.amazon.awssdk.services.securitylake.model.ListSubscribersRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListSubscribersPublisher publisher = client.listSubscribersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securitylake.paginators.ListSubscribersPublisher publisher = client.listSubscribersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.securitylake.model.ListSubscribersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.securitylake.model.ListSubscribersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubscribers(software.amazon.awssdk.services.securitylake.model.ListSubscribersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSubscribersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListSubscribers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListSubscribers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSubscribersPublisher listSubscribersPaginator(ListSubscribersRequest listSubscribersRequest) {
        return new ListSubscribersPublisher(this, applyPaginatorUserAgent(listSubscribersRequest));
    }

    /**
     * <p>
     * Specifies where to store your security data and for how long. You can add a rollup Region to consolidate data
     * from multiple Amazon Web Services Regions.
     * </p>
     *
     * @param updateDatalakeRequest
     * @return A Java Future containing the result of the UpdateDatalake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventBridgeException Represents an error interacting with the Amazon EventBridge service.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateDatalake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDatalake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatalakeResponse> updateDatalake(UpdateDatalakeRequest updateDatalakeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatalakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatalake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatalakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatalakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatalakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatalakeRequest, UpdateDatalakeResponse>()
                            .withOperationName("UpdateDatalake")
                            .withMarshaller(new UpdateDatalakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDatalakeRequest));
            CompletableFuture<UpdateDatalakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the expiration period for the exception message to your preferred time, and control the time-to-live (TTL)
     * for the exception message to remain. Exceptions are stored by default for 2 weeks from when a record was created
     * in Amazon Security Lake.
     * </p>
     *
     * @param updateDatalakeExceptionsExpiryRequest
     * @return A Java Future containing the result of the UpdateDatalakeExceptionsExpiry operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateDatalakeExceptionsExpiry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDatalakeExceptionsExpiry"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatalakeExceptionsExpiryResponse> updateDatalakeExceptionsExpiry(
            UpdateDatalakeExceptionsExpiryRequest updateDatalakeExceptionsExpiryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDatalakeExceptionsExpiryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatalakeExceptionsExpiry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatalakeExceptionsExpiryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatalakeExceptionsExpiryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatalakeExceptionsExpiryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatalakeExceptionsExpiryRequest, UpdateDatalakeExceptionsExpiryResponse>()
                            .withOperationName("UpdateDatalakeExceptionsExpiry")
                            .withMarshaller(new UpdateDatalakeExceptionsExpiryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDatalakeExceptionsExpiryRequest));
            CompletableFuture<UpdateDatalakeExceptionsExpiryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param updateDatalakeExceptionsSubscriptionRequest
     * @return A Java Future containing the result of the UpdateDatalakeExceptionsSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateDatalakeExceptionsSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDatalakeExceptionsSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatalakeExceptionsSubscriptionResponse> updateDatalakeExceptionsSubscription(
            UpdateDatalakeExceptionsSubscriptionRequest updateDatalakeExceptionsSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDatalakeExceptionsSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatalakeExceptionsSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatalakeExceptionsSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateDatalakeExceptionsSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatalakeExceptionsSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatalakeExceptionsSubscriptionRequest, UpdateDatalakeExceptionsSubscriptionResponse>()
                            .withOperationName("UpdateDatalakeExceptionsSubscription")
                            .withMarshaller(new UpdateDatalakeExceptionsSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDatalakeExceptionsSubscriptionRequest));
            CompletableFuture<UpdateDatalakeExceptionsSubscriptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing subscription for the given Amazon Security Lake account ID. You can update a subscriber by
     * changing the sources that the subscriber consumes data from.
     * </p>
     *
     * @param updateSubscriberRequest
     * @return A Java Future containing the result of the UpdateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictSubscriptionException A conflicting subscription exception operation is in progress.</li>
     *         <li>ConcurrentModificationException More than one process tried to modify a resource at the same time.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriberResponse> updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                            .withOperationName("UpdateSubscriber")
                            .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSubscriberRequest));
            CompletableFuture<UpdateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new subscription notification or adds the existing subscription notification setting for the specified
     * subscription ID.
     * </p>
     *
     * @param updateSubscriptionNotificationConfigurationRequest
     * @return A Java Future containing the result of the UpdateSubscriptionNotificationConfiguration operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException More than one process tried to modify a resource at the same time.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>ValidationException Your signing certificate could not be validated.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>AccountNotFoundException Amazon Security Lake cannot find an Amazon Web Services account with the
     *         accountID that you specified, or the account whose credentials you used to make this request isn't a
     *         member of an organization.</li>
     *         <li>InvalidInputException The request was rejected because a value that's not valid or is out of range
     *         was supplied for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateSubscriptionNotificationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriptionNotificationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriptionNotificationConfigurationResponse> updateSubscriptionNotificationConfiguration(
            UpdateSubscriptionNotificationConfigurationRequest updateSubscriptionNotificationConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSubscriptionNotificationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriptionNotificationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriptionNotificationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateSubscriptionNotificationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriptionNotificationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriptionNotificationConfigurationRequest, UpdateSubscriptionNotificationConfigurationResponse>()
                            .withOperationName("UpdateSubscriptionNotificationConfiguration")
                            .withMarshaller(new UpdateSubscriptionNotificationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSubscriptionNotificationConfigurationRequest));
            CompletableFuture<UpdateSubscriptionNotificationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecurityLakeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictSourceNamesException")
                                .exceptionBuilderSupplier(ConflictSourceNamesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("S3Exception").exceptionBuilderSupplier(S3Exception::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BucketNotFoundException")
                                .exceptionBuilderSupplier(BucketNotFoundException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictSubscriptionException")
                                .exceptionBuilderSupplier(ConflictSubscriptionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountNotFoundException")
                                .exceptionBuilderSupplier(AccountNotFoundException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventBridgeException")
                                .exceptionBuilderSupplier(EventBridgeException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SecurityLakeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
