/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.securitylake.model.AccessDeniedException;
import software.amazon.awssdk.services.securitylake.model.BadRequestException;
import software.amazon.awssdk.services.securitylake.model.ConflictException;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.DeregisterDataLakeDelegatedAdministratorRequest;
import software.amazon.awssdk.services.securitylake.model.DeregisterDataLakeDelegatedAdministratorResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeSourcesRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeSourcesResponse;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.InternalServerException;
import software.amazon.awssdk.services.securitylake.model.ListDataLakeExceptionsRequest;
import software.amazon.awssdk.services.securitylake.model.ListDataLakeExceptionsResponse;
import software.amazon.awssdk.services.securitylake.model.ListDataLakesRequest;
import software.amazon.awssdk.services.securitylake.model.ListDataLakesResponse;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesRequest;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesResponse;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersRequest;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersResponse;
import software.amazon.awssdk.services.securitylake.model.RegisterDataLakeDelegatedAdministratorRequest;
import software.amazon.awssdk.services.securitylake.model.RegisterDataLakeDelegatedAdministratorResponse;
import software.amazon.awssdk.services.securitylake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.securitylake.model.SecurityLakeException;
import software.amazon.awssdk.services.securitylake.model.ThrottlingException;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.transform.CreateAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeregisterDataLakeDelegatedAdministratorRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeSourcesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListDataLakeExceptionsRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListDataLakesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListLogSourcesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListSubscribersRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.RegisterDataLakeDelegatedAdministratorRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SecurityLakeClient}.
 *
 * @see SecurityLakeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecurityLakeClient implements SecurityLakeClient {
    private static final Logger log = Logger.loggerFor(DefaultSecurityLakeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SecurityLakeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSecurityLakeClient(SecurityLakeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds a natively supported Amazon Web Service as an Amazon Security Lake source. Enables source types for member
     * accounts in required Amazon Web Services Regions, based on the parameters you specify. You can choose any source
     * type in any Region for either accounts that are part of a trusted organization or standalone accounts. Once you
     * add an Amazon Web Service as a source, Security Lake starts collecting logs and events from it,
     * </p>
     * <p>
     * You can use this API only to enable natively supported Amazon Web Services as a source. Use
     * <code>CreateCustomLogSource</code> to enable data collection from a custom source.
     * </p>
     *
     * @param createAwsLogSourceRequest
     * @return Result of the CreateAwsLogSource operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAwsLogSourceResponse createAwsLogSource(CreateAwsLogSourceRequest createAwsLogSourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAwsLogSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAwsLogSource");

            return clientHandler.execute(new ClientExecutionParams<CreateAwsLogSourceRequest, CreateAwsLogSourceResponse>()
                    .withOperationName("CreateAwsLogSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAwsLogSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAwsLogSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a third-party custom source in Amazon Security Lake, from the Amazon Web Services Region where you want to
     * create a custom source. Security Lake can collect logs and events from third-party custom sources. After creating
     * the appropriate IAM role to invoke Glue crawler, use this API to add a custom source name in Security Lake. This
     * operation creates a partition in the Amazon S3 bucket for Security Lake as the target location for log files from
     * the custom source. In addition, this operation also creates an associated Glue table and an Glue crawler.
     * </p>
     *
     * @param createCustomLogSourceRequest
     * @return Result of the CreateCustomLogSource operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCustomLogSourceResponse createCustomLogSource(CreateCustomLogSourceRequest createCustomLogSourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCustomLogSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomLogSource");

            return clientHandler.execute(new ClientExecutionParams<CreateCustomLogSourceRequest, CreateCustomLogSourceResponse>()
                    .withOperationName("CreateCustomLogSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createCustomLogSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCustomLogSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initializes an Amazon Security Lake instance with the provided (or default) configuration. You can enable
     * Security Lake in Amazon Web Services Regions with customized settings before enabling log collection in Regions.
     * By default, the <code>CreateDataLake</code> Security Lake in all Regions. To specify particular Regions,
     * configure these Regions using the <code>configurations</code> parameter. If you have already enabled Security
     * Lake in a Region when you call this command, the command will update the Region if you provide new configuration
     * parameters. If you have not already enabled Security Lake in the Region when you call this API, it will set up
     * the data lake in the Region with the specified configurations.
     * </p>
     * <p>
     * When you enable Security Lake, it starts ingesting security data after the <code>CreateAwsLogSource</code> call.
     * This includes ingesting security data from sources, storing data, and making data accessible to subscribers.
     * Security Lake also enables all the existing settings and resources that it stores or maintains for your Amazon
     * Web Services account in the current Region, including security log and event data. For more information, see the
     * <a href="https://docs.aws.amazon.com/security-lake/latest/userguide/what-is-security-lake.html">Amazon Security
     * Lake User Guide</a>.
     * </p>
     *
     * @param createDataLakeRequest
     * @return Result of the CreateDataLake operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDataLakeResponse createDataLake(CreateDataLakeRequest createDataLakeRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDataLakeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLake");

            return clientHandler.execute(new ClientExecutionParams<CreateDataLakeRequest, CreateDataLakeResponse>()
                    .withOperationName("CreateDataLake").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDataLakeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDataLakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param createDataLakeExceptionSubscriptionRequest
     * @return Result of the CreateDataLakeExceptionSubscription operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataLakeExceptionSubscriptionResponse createDataLakeExceptionSubscription(
            CreateDataLakeExceptionSubscriptionRequest createDataLakeExceptionSubscriptionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataLakeExceptionSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLakeExceptionSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeExceptionSubscriptionRequest, CreateDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("CreateDataLakeExceptionSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDataLakeExceptionSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Automatically enables Amazon Security Lake for new member accounts in your organization. Security Lake is not
     * automatically enabled for any existing member accounts in your organization.
     * </p>
     *
     * @param createDataLakeOrganizationConfigurationRequest
     * @return Result of the CreateDataLakeOrganizationConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataLakeOrganizationConfigurationResponse createDataLakeOrganizationConfiguration(
            CreateDataLakeOrganizationConfigurationRequest createDataLakeOrganizationConfigurationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, CreateDataLakeOrganizationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLakeOrganizationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeOrganizationConfigurationRequest, CreateDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("CreateDataLakeOrganizationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDataLakeOrganizationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a subscription permission for accounts that are already enabled in Amazon Security Lake. You can create a
     * subscriber with access to data in the current Amazon Web Services Region.
     * </p>
     *
     * @param createSubscriberRequest
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateSubscriberResponse createSubscriber(CreateSubscriberRequest createSubscriberRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSubscriberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriber");

            return clientHandler.execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                    .withOperationName("CreateSubscriber").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Notifies the subscriber when new data is written to the data lake for the sources that the subscriber consumes in
     * Security Lake. You can create only one subscriber notification per subscriber.
     * </p>
     *
     * @param createSubscriberNotificationRequest
     * @return Result of the CreateSubscriberNotification operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.CreateSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSubscriberNotificationResponse createSubscriberNotification(
            CreateSubscriberNotificationRequest createSubscriberNotificationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSubscriberNotificationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriberNotification");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriberNotificationRequest, CreateSubscriberNotificationResponse>()
                            .withOperationName("CreateSubscriberNotification").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createSubscriberNotificationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSubscriberNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a natively supported Amazon Web Service as an Amazon Security Lake source. You can remove a source for
     * one or more Regions. When you remove the source, Security Lake stops collecting data from that source in the
     * specified Regions and accounts, and subscribers can no longer consume new data from the source. However,
     * subscribers can still consume data that Security Lake collected from the source before removal.
     * </p>
     * <p>
     * You can choose any source type in any Amazon Web Services Region for either accounts that are part of a trusted
     * organization or standalone accounts.
     * </p>
     *
     * @param deleteAwsLogSourceRequest
     * @return Result of the DeleteAwsLogSource operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAwsLogSourceResponse deleteAwsLogSource(DeleteAwsLogSourceRequest deleteAwsLogSourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAwsLogSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAwsLogSource");

            return clientHandler.execute(new ClientExecutionParams<DeleteAwsLogSourceRequest, DeleteAwsLogSourceResponse>()
                    .withOperationName("DeleteAwsLogSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAwsLogSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAwsLogSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a custom log source from Amazon Security Lake, to stop sending data from the custom source to Security
     * Lake.
     * </p>
     *
     * @param deleteCustomLogSourceRequest
     * @return Result of the DeleteCustomLogSource operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCustomLogSourceResponse deleteCustomLogSource(DeleteCustomLogSourceRequest deleteCustomLogSourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCustomLogSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomLogSource");

            return clientHandler.execute(new ClientExecutionParams<DeleteCustomLogSourceRequest, DeleteCustomLogSourceResponse>()
                    .withOperationName("DeleteCustomLogSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteCustomLogSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCustomLogSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When you disable Amazon Security Lake from your account, Security Lake is disabled in all Amazon Web Services
     * Regions and it stops collecting data from your sources. Also, this API automatically takes steps to remove the
     * account from Security Lake. However, Security Lake retains all of your existing settings and the resources that
     * it created in your Amazon Web Services account in the current Amazon Web Services Region.
     * </p>
     * <p>
     * The <code>DeleteDataLake</code> operation does not delete the data that is stored in your Amazon S3 bucket, which
     * is owned by your Amazon Web Services account. For more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/disable-security-lake.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param deleteDataLakeRequest
     * @return Result of the DeleteDataLake operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDataLakeResponse deleteDataLake(DeleteDataLakeRequest deleteDataLakeRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDataLakeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLake");

            return clientHandler.execute(new ClientExecutionParams<DeleteDataLakeRequest, DeleteDataLakeResponse>()
                    .withOperationName("DeleteDataLake").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataLakeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDataLakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteDataLakeExceptionSubscriptionRequest
     * @return Result of the DeleteDataLakeExceptionSubscription operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataLakeExceptionSubscriptionResponse deleteDataLakeExceptionSubscription(
            DeleteDataLakeExceptionSubscriptionRequest deleteDataLakeExceptionSubscriptionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataLakeExceptionSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLakeExceptionSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeExceptionSubscriptionRequest, DeleteDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("DeleteDataLakeExceptionSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataLakeExceptionSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes automatic the enablement of configuration settings for new member accounts (but retains the settings for
     * the delegated administrator) from Amazon Security Lake. You must run this API using the credentials of the
     * delegated administrator. When you run this API, new member accounts that are added after the organization enables
     * Security Lake won't contribute to the data lake.
     * </p>
     *
     * @param deleteDataLakeOrganizationConfigurationRequest
     * @return Result of the DeleteDataLakeOrganizationConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataLakeOrganizationConfigurationResponse deleteDataLakeOrganizationConfiguration(
            DeleteDataLakeOrganizationConfigurationRequest deleteDataLakeOrganizationConfigurationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DeleteDataLakeOrganizationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLakeOrganizationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeOrganizationConfigurationRequest, DeleteDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("DeleteDataLakeOrganizationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDataLakeOrganizationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the subscription permission and all notification settings for accounts that are already enabled in Amazon
     * Security Lake. When you run <code>DeleteSubscriber</code>, the subscriber will no longer consume data from
     * Security Lake and the subscriber is removed. This operation deletes the subscriber and removes access to data in
     * the current Amazon Web Services Region.
     * </p>
     *
     * @param deleteSubscriberRequest
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSubscriberResponse deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSubscriberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriber");

            return clientHandler.execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                    .withOperationName("DeleteSubscriber").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteSubscriberNotificationRequest
     * @return Result of the DeleteSubscriberNotification operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeleteSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSubscriberNotificationResponse deleteSubscriberNotification(
            DeleteSubscriberNotificationRequest deleteSubscriberNotificationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSubscriberNotificationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriberNotification");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriberNotificationRequest, DeleteSubscriberNotificationResponse>()
                            .withOperationName("DeleteSubscriberNotification").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteSubscriberNotificationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSubscriberNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param deregisterDataLakeDelegatedAdministratorRequest
     * @return Result of the DeregisterDataLakeDelegatedAdministrator operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.DeregisterDataLakeDelegatedAdministrator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeregisterDataLakeDelegatedAdministrator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterDataLakeDelegatedAdministratorResponse deregisterDataLakeDelegatedAdministrator(
            DeregisterDataLakeDelegatedAdministratorRequest deregisterDataLakeDelegatedAdministratorRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterDataLakeDelegatedAdministratorResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DeregisterDataLakeDelegatedAdministratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterDataLakeDelegatedAdministratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterDataLakeDelegatedAdministrator");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterDataLakeDelegatedAdministratorRequest, DeregisterDataLakeDelegatedAdministratorResponse>()
                            .withOperationName("DeregisterDataLakeDelegatedAdministrator").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(deregisterDataLakeDelegatedAdministratorRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterDataLakeDelegatedAdministratorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details of exception notifications for the account in Amazon Security Lake.
     * </p>
     *
     * @param getDataLakeExceptionSubscriptionRequest
     * @return Result of the GetDataLakeExceptionSubscription operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.GetDataLakeExceptionSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataLakeExceptionSubscriptionResponse getDataLakeExceptionSubscription(
            GetDataLakeExceptionSubscriptionRequest getDataLakeExceptionSubscriptionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataLakeExceptionSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeExceptionSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeExceptionSubscriptionRequest, GetDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("GetDataLakeExceptionSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getDataLakeExceptionSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the configuration that will be automatically set up for accounts added to the organization after the
     * organization has onboarded to Amazon Security Lake. This API does not take input parameters.
     * </p>
     *
     * @param getDataLakeOrganizationConfigurationRequest
     * @return Result of the GetDataLakeOrganizationConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.GetDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataLakeOrganizationConfigurationResponse getDataLakeOrganizationConfiguration(
            GetDataLakeOrganizationConfigurationRequest getDataLakeOrganizationConfigurationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetDataLakeOrganizationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeOrganizationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeOrganizationConfigurationRequest, GetDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("GetDataLakeOrganizationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDataLakeOrganizationConfigurationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a snapshot of the current Region, including whether Amazon Security Lake is enabled for those accounts
     * and which sources Security Lake is collecting data from.
     * </p>
     *
     * @param getDataLakeSourcesRequest
     * @return Result of the GetDataLakeSources operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.GetDataLakeSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataLakeSourcesResponse getDataLakeSources(GetDataLakeSourcesRequest getDataLakeSourcesRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataLakeSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataLakeSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeSources");

            return clientHandler.execute(new ClientExecutionParams<GetDataLakeSourcesRequest, GetDataLakeSourcesResponse>()
                    .withOperationName("GetDataLakeSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataLakeSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDataLakeSourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the subscription information for the specified subscription ID. You can get information about a
     * specific subscriber.
     * </p>
     *
     * @param getSubscriberRequest
     * @return Result of the GetSubscriber operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.GetSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSubscriberResponse getSubscriber(GetSubscriberRequest getSubscriberRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSubscriberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriber");

            return clientHandler.execute(new ClientExecutionParams<GetSubscriberRequest, GetSubscriberResponse>()
                    .withOperationName("GetSubscriber").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Security Lake exceptions that you can use to find the source of problems and fix them.
     * </p>
     *
     * @param listDataLakeExceptionsRequest
     * @return Result of the ListDataLakeExceptions operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.ListDataLakeExceptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDataLakeExceptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataLakeExceptionsResponse listDataLakeExceptions(ListDataLakeExceptionsRequest listDataLakeExceptionsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataLakeExceptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataLakeExceptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataLakeExceptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataLakeExceptions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataLakeExceptionsRequest, ListDataLakeExceptionsResponse>()
                            .withOperationName("ListDataLakeExceptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDataLakeExceptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataLakeExceptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the Amazon Security Lake configuration object for the specified Amazon Web Services account ID. You can
     * use the <code>ListDataLakes</code> API to know whether Security Lake is enabled for any region.
     * </p>
     *
     * @param listDataLakesRequest
     * @return Result of the ListDataLakes operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.ListDataLakes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDataLakes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDataLakesResponse listDataLakes(ListDataLakesRequest listDataLakesRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataLakesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDataLakesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataLakesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataLakes");

            return clientHandler.execute(new ClientExecutionParams<ListDataLakesRequest, ListDataLakesResponse>()
                    .withOperationName("ListDataLakes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataLakesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataLakesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the log sources in the current Amazon Web Services Region.
     * </p>
     *
     * @param listLogSourcesRequest
     * @return Result of the ListLogSources operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.ListLogSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListLogSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLogSourcesResponse listLogSources(ListLogSourcesRequest listLogSourcesRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLogSourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLogSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLogSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLogSources");

            return clientHandler.execute(new ClientExecutionParams<ListLogSourcesRequest, ListLogSourcesResponse>()
                    .withOperationName("ListLogSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLogSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLogSourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all subscribers for the specific Amazon Security Lake account ID. You can retrieve a list of subscriptions
     * associated with a specific organization or Amazon Web Services account.
     * </p>
     *
     * @param listSubscribersRequest
     * @return Result of the ListSubscribers operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.ListSubscribers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListSubscribers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSubscribersResponse listSubscribers(ListSubscribersRequest listSubscribersRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSubscribersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSubscribersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubscribersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubscribers");

            return clientHandler.execute(new ClientExecutionParams<ListSubscribersRequest, ListSubscribersResponse>()
                    .withOperationName("ListSubscribers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSubscribersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSubscribersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Designates the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param registerDataLakeDelegatedAdministratorRequest
     * @return Result of the RegisterDataLakeDelegatedAdministrator operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.RegisterDataLakeDelegatedAdministrator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/RegisterDataLakeDelegatedAdministrator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterDataLakeDelegatedAdministratorResponse registerDataLakeDelegatedAdministrator(
            RegisterDataLakeDelegatedAdministratorRequest registerDataLakeDelegatedAdministratorRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AccessDeniedException,
            ConflictException, ThrottlingException, AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterDataLakeDelegatedAdministratorResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, RegisterDataLakeDelegatedAdministratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerDataLakeDelegatedAdministratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterDataLakeDelegatedAdministrator");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterDataLakeDelegatedAdministratorRequest, RegisterDataLakeDelegatedAdministratorResponse>()
                            .withOperationName("RegisterDataLakeDelegatedAdministrator").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(registerDataLakeDelegatedAdministratorRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterDataLakeDelegatedAdministratorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Specifies where to store your security data and for how long. You can add a rollup Region to consolidate data
     * from multiple Amazon Web Services Regions.
     * </p>
     *
     * @param updateDataLakeRequest
     * @return Result of the UpdateDataLake operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.UpdateDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDataLakeResponse updateDataLake(UpdateDataLakeRequest updateDataLakeRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDataLakeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataLake");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataLakeRequest, UpdateDataLakeResponse>()
                    .withOperationName("UpdateDataLake").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataLakeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataLakeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param updateDataLakeExceptionSubscriptionRequest
     * @return Result of the UpdateDataLakeExceptionSubscription operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.UpdateDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataLakeExceptionSubscriptionResponse updateDataLakeExceptionSubscription(
            UpdateDataLakeExceptionSubscriptionRequest updateDataLakeExceptionSubscriptionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataLakeExceptionSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataLakeExceptionSubscription");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDataLakeExceptionSubscriptionRequest, UpdateDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("UpdateDataLakeExceptionSubscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDataLakeExceptionSubscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing subscription for the given Amazon Security Lake account ID. You can update a subscriber by
     * changing the sources that the subscriber consumes data from.
     * </p>
     *
     * @param updateSubscriberRequest
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.UpdateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateSubscriberResponse updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSubscriberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriber");

            return clientHandler.execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                    .withOperationName("UpdateSubscriber").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing notification method for the subscription (SQS or HTTPs endpoint) or switches the notification
     * subscription endpoint for a subscriber.
     * </p>
     *
     * @param updateSubscriberNotificationRequest
     * @return Result of the UpdateSubscriberNotification operation returned by the service.
     * @throws BadRequestException
     *         The request is malformed or contains an error such as an invalid parameter value or a missing required
     *         parameter.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws InternalServerException
     *         Internal service exceptions are sometimes caused by transient issues. Before you start troubleshooting,
     *         perform the operation again.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. Access denied errors appear when Amazon
     *         Security Lake explicitly or implicitly denies an authorization request. An explicit denial occurs when a
     *         policy contains a Deny statement for the specific Amazon Web Services action. An implicit denial occurs
     *         when there is no applicable Deny statement and also no applicable Allow statement.
     * @throws ConflictException
     *         Occurs when a conflict with a previous successful write is detected. This generally occurs when the
     *         previous write did not have time to propagate to the host serving the current request. A retry (with
     *         appropriate backoff logic) is the recommended response to this exception.
     * @throws ThrottlingException
     *         The limit on the number of requests per second was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityLakeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityLakeClient.UpdateSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSubscriberNotificationResponse updateSubscriberNotification(
            UpdateSubscriberNotificationRequest updateSubscriberNotificationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, SecurityLakeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSubscriberNotificationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriberNotification");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriberNotificationRequest, UpdateSubscriberNotificationResponse>()
                            .withOperationName("UpdateSubscriberNotification").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSubscriberNotificationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSubscriberNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecurityLakeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public final SecurityLakeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
