/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides details about the Amazon Security Lake account subscription. Subscribers are notified of new objects for a
 * source as the data is written to your Amazon S3 bucket for Security Lake.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SubscriberResource implements SdkPojo, Serializable,
        ToCopyableBuilder<SubscriberResource.Builder, SubscriberResource> {
    private static final SdkField<List<String>> ACCESS_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("accessTypes")
            .getter(getter(SubscriberResource::accessTypesAsStrings))
            .setter(setter(Builder::accessTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(SubscriberResource::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(SubscriberResource::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> EXTERNAL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("externalId").getter(getter(SubscriberResource::externalId)).setter(setter(Builder::externalId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalId").build()).build();

    private static final SdkField<String> RESOURCE_SHARE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceShareArn").getter(getter(SubscriberResource::resourceShareArn))
            .setter(setter(Builder::resourceShareArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareArn").build()).build();

    private static final SdkField<String> RESOURCE_SHARE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceShareName").getter(getter(SubscriberResource::resourceShareName))
            .setter(setter(Builder::resourceShareName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareName").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(SubscriberResource::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> S3_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("s3BucketArn").getter(getter(SubscriberResource::s3BucketArn)).setter(setter(Builder::s3BucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3BucketArn").build()).build();

    private static final SdkField<String> SNS_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("snsArn")
            .getter(getter(SubscriberResource::snsArn)).setter(setter(Builder::snsArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("snsArn").build()).build();

    private static final SdkField<List<SourceType>> SOURCE_TYPES_FIELD = SdkField
            .<List<SourceType>> builder(MarshallingType.LIST)
            .memberName("sourceTypes")
            .getter(getter(SubscriberResource::sourceTypes))
            .setter(setter(Builder::sourceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceType> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SUBSCRIBER_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberDescription").getter(getter(SubscriberResource::subscriberDescription))
            .setter(setter(Builder::subscriberDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberDescription").build())
            .build();

    private static final SdkField<String> SUBSCRIBER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberName").getter(getter(SubscriberResource::subscriberName))
            .setter(setter(Builder::subscriberName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberName").build()).build();

    private static final SdkField<String> SUBSCRIPTION_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriptionEndpoint").getter(getter(SubscriberResource::subscriptionEndpoint))
            .setter(setter(Builder::subscriptionEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriptionEndpoint").build())
            .build();

    private static final SdkField<String> SUBSCRIPTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriptionId").getter(getter(SubscriberResource::subscriptionId))
            .setter(setter(Builder::subscriptionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriptionId").build()).build();

    private static final SdkField<String> SUBSCRIPTION_PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriptionProtocol").getter(getter(SubscriberResource::subscriptionProtocolAsString))
            .setter(setter(Builder::subscriptionProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriptionProtocol").build())
            .build();

    private static final SdkField<String> SUBSCRIPTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriptionStatus").getter(getter(SubscriberResource::subscriptionStatusAsString))
            .setter(setter(Builder::subscriptionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriptionStatus").build())
            .build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(SubscriberResource::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESS_TYPES_FIELD,
            ACCOUNT_ID_FIELD, CREATED_AT_FIELD, EXTERNAL_ID_FIELD, RESOURCE_SHARE_ARN_FIELD, RESOURCE_SHARE_NAME_FIELD,
            ROLE_ARN_FIELD, S3_BUCKET_ARN_FIELD, SNS_ARN_FIELD, SOURCE_TYPES_FIELD, SUBSCRIBER_DESCRIPTION_FIELD,
            SUBSCRIBER_NAME_FIELD, SUBSCRIPTION_ENDPOINT_FIELD, SUBSCRIPTION_ID_FIELD, SUBSCRIPTION_PROTOCOL_FIELD,
            SUBSCRIPTION_STATUS_FIELD, UPDATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> accessTypes;

    private final String accountId;

    private final Instant createdAt;

    private final String externalId;

    private final String resourceShareArn;

    private final String resourceShareName;

    private final String roleArn;

    private final String s3BucketArn;

    private final String snsArn;

    private final List<SourceType> sourceTypes;

    private final String subscriberDescription;

    private final String subscriberName;

    private final String subscriptionEndpoint;

    private final String subscriptionId;

    private final String subscriptionProtocol;

    private final String subscriptionStatus;

    private final Instant updatedAt;

    private SubscriberResource(BuilderImpl builder) {
        this.accessTypes = builder.accessTypes;
        this.accountId = builder.accountId;
        this.createdAt = builder.createdAt;
        this.externalId = builder.externalId;
        this.resourceShareArn = builder.resourceShareArn;
        this.resourceShareName = builder.resourceShareName;
        this.roleArn = builder.roleArn;
        this.s3BucketArn = builder.s3BucketArn;
        this.snsArn = builder.snsArn;
        this.sourceTypes = builder.sourceTypes;
        this.subscriberDescription = builder.subscriberDescription;
        this.subscriberName = builder.subscriberName;
        this.subscriptionEndpoint = builder.subscriptionEndpoint;
        this.subscriptionId = builder.subscriptionId;
        this.subscriptionProtocol = builder.subscriptionProtocol;
        this.subscriptionStatus = builder.subscriptionStatus;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
     * through messaging to an HTTPS endpoint provided by the subscriber.
     * </p>
     * <p>
     * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through services
     * like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessTypes} method.
     * </p>
     * 
     * @return You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
     *         queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
     *         <p>
     *         Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
     *         services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     */
    public final List<AccessType> accessTypes() {
        return AccessTypeListCopier.copyStringToEnum(accessTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessTypes() {
        return accessTypes != null && !(accessTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
     * through messaging to an HTTPS endpoint provided by the subscriber.
     * </p>
     * <p>
     * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through services
     * like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessTypes} method.
     * </p>
     * 
     * @return You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
     *         queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
     *         <p>
     *         Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
     *         services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     */
    public final List<String> accessTypesAsStrings() {
        return accessTypes;
    }

    /**
     * <p>
     * The Amazon Web Services account ID you are using to create your Amazon Security Lake account.
     * </p>
     * 
     * @return The Amazon Web Services account ID you are using to create your Amazon Security Lake account.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The date and time when the subscription was created.
     * </p>
     * 
     * @return The date and time when the subscription was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The external ID of the subscriber. The external ID lets the user that is assuming the role assert the
     * circumstances in which they are operating. It also provides a way for the account owner to permit the role to be
     * assumed only under specific circumstances.
     * </p>
     * 
     * @return The external ID of the subscriber. The external ID lets the user that is assuming the role assert the
     *         circumstances in which they are operating. It also provides a way for the account owner to permit the
     *         role to be assumed only under specific circumstances.
     */
    public final String externalId() {
        return externalId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) which uniquely defines the AWS RAM resource share. Before accepting the RAM
     * resource share invitation, you can view details related to the RAM resource share.
     * </p>
     * <p>
     * This field is available only for Lake Formation subscribers created after March 8, 2023.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) which uniquely defines the AWS RAM resource share. Before accepting the
     *         RAM resource share invitation, you can view details related to the RAM resource share.</p>
     *         <p>
     *         This field is available only for Lake Formation subscribers created after March 8, 2023.
     */
    public final String resourceShareArn() {
        return resourceShareArn;
    }

    /**
     * <p>
     * The name of the resource share.
     * </p>
     * 
     * @return The name of the resource share.
     */
    public final String resourceShareName() {
        return resourceShareName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) specifying the role of the subscriber.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) specifying the role of the subscriber.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The ARN for the Amazon S3 bucket.
     * </p>
     * 
     * @return The ARN for the Amazon S3 bucket.
     */
    public final String s3BucketArn() {
        return s3BucketArn;
    }

    /**
     * <p>
     * The ARN for the Amazon Simple Notification Service.
     * </p>
     * 
     * @return The ARN for the Amazon Simple Notification Service.
     */
    public final String snsArn() {
        return snsArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceTypes() {
        return sourceTypes != null && !(sourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For more
     * information, see the Amazon Security Lake User Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceTypes} method.
     * </p>
     * 
     * @return Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For
     *         more information, see the Amazon Security Lake User Guide.
     */
    public final List<SourceType> sourceTypes() {
        return sourceTypes;
    }

    /**
     * <p>
     * The subscriber descriptions for a subscriber account. The description for a subscriber includes
     * <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and <code>subscriptionId</code>.
     * </p>
     * 
     * @return The subscriber descriptions for a subscriber account. The description for a subscriber includes
     *         <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and
     *         <code>subscriptionId</code>.
     */
    public final String subscriberDescription() {
        return subscriberDescription;
    }

    /**
     * <p>
     * The name of your Amazon Security Lake subscriber account.
     * </p>
     * 
     * @return The name of your Amazon Security Lake subscriber account.
     */
    public final String subscriberName() {
        return subscriberName;
    }

    /**
     * <p>
     * The subscription endpoint to which exception messages are posted.
     * </p>
     * 
     * @return The subscription endpoint to which exception messages are posted.
     */
    public final String subscriptionEndpoint() {
        return subscriptionEndpoint;
    }

    /**
     * <p>
     * The subscription ID of the Amazon Security Lake subscriber account.
     * </p>
     * 
     * @return The subscription ID of the Amazon Security Lake subscriber account.
     */
    public final String subscriptionId() {
        return subscriptionId;
    }

    /**
     * <p>
     * The subscription protocol to which exception messages are posted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #subscriptionProtocol} will return {@link EndpointProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #subscriptionProtocolAsString}.
     * </p>
     * 
     * @return The subscription protocol to which exception messages are posted.
     * @see EndpointProtocol
     */
    public final EndpointProtocol subscriptionProtocol() {
        return EndpointProtocol.fromValue(subscriptionProtocol);
    }

    /**
     * <p>
     * The subscription protocol to which exception messages are posted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #subscriptionProtocol} will return {@link EndpointProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #subscriptionProtocolAsString}.
     * </p>
     * 
     * @return The subscription protocol to which exception messages are posted.
     * @see EndpointProtocol
     */
    public final String subscriptionProtocolAsString() {
        return subscriptionProtocol;
    }

    /**
     * <p>
     * The subscription status of the Amazon Security Lake subscriber account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #subscriptionStatus} will return {@link SubscriptionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #subscriptionStatusAsString}.
     * </p>
     * 
     * @return The subscription status of the Amazon Security Lake subscriber account.
     * @see SubscriptionStatus
     */
    public final SubscriptionStatus subscriptionStatus() {
        return SubscriptionStatus.fromValue(subscriptionStatus);
    }

    /**
     * <p>
     * The subscription status of the Amazon Security Lake subscriber account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #subscriptionStatus} will return {@link SubscriptionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #subscriptionStatusAsString}.
     * </p>
     * 
     * @return The subscription status of the Amazon Security Lake subscriber account.
     * @see SubscriptionStatus
     */
    public final String subscriptionStatusAsString() {
        return subscriptionStatus;
    }

    /**
     * <p>
     * The date and time when the subscription was created.
     * </p>
     * 
     * @return The date and time when the subscription was created.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessTypes() ? accessTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(externalId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceShareArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceShareName());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(snsArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceTypes() ? sourceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(subscriberDescription());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberName());
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionId());
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SubscriberResource)) {
            return false;
        }
        SubscriberResource other = (SubscriberResource) obj;
        return hasAccessTypes() == other.hasAccessTypes() && Objects.equals(accessTypesAsStrings(), other.accessTypesAsStrings())
                && Objects.equals(accountId(), other.accountId()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(externalId(), other.externalId())
                && Objects.equals(resourceShareArn(), other.resourceShareArn())
                && Objects.equals(resourceShareName(), other.resourceShareName()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(s3BucketArn(), other.s3BucketArn()) && Objects.equals(snsArn(), other.snsArn())
                && hasSourceTypes() == other.hasSourceTypes() && Objects.equals(sourceTypes(), other.sourceTypes())
                && Objects.equals(subscriberDescription(), other.subscriberDescription())
                && Objects.equals(subscriberName(), other.subscriberName())
                && Objects.equals(subscriptionEndpoint(), other.subscriptionEndpoint())
                && Objects.equals(subscriptionId(), other.subscriptionId())
                && Objects.equals(subscriptionProtocolAsString(), other.subscriptionProtocolAsString())
                && Objects.equals(subscriptionStatusAsString(), other.subscriptionStatusAsString())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SubscriberResource").add("AccessTypes", hasAccessTypes() ? accessTypesAsStrings() : null)
                .add("AccountId", accountId()).add("CreatedAt", createdAt()).add("ExternalId", externalId())
                .add("ResourceShareArn", resourceShareArn()).add("ResourceShareName", resourceShareName())
                .add("RoleArn", roleArn()).add("S3BucketArn", s3BucketArn()).add("SnsArn", snsArn())
                .add("SourceTypes", hasSourceTypes() ? sourceTypes() : null)
                .add("SubscriberDescription", subscriberDescription()).add("SubscriberName", subscriberName())
                .add("SubscriptionEndpoint", subscriptionEndpoint()).add("SubscriptionId", subscriptionId())
                .add("SubscriptionProtocol", subscriptionProtocolAsString())
                .add("SubscriptionStatus", subscriptionStatusAsString()).add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accessTypes":
            return Optional.ofNullable(clazz.cast(accessTypesAsStrings()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "externalId":
            return Optional.ofNullable(clazz.cast(externalId()));
        case "resourceShareArn":
            return Optional.ofNullable(clazz.cast(resourceShareArn()));
        case "resourceShareName":
            return Optional.ofNullable(clazz.cast(resourceShareName()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "s3BucketArn":
            return Optional.ofNullable(clazz.cast(s3BucketArn()));
        case "snsArn":
            return Optional.ofNullable(clazz.cast(snsArn()));
        case "sourceTypes":
            return Optional.ofNullable(clazz.cast(sourceTypes()));
        case "subscriberDescription":
            return Optional.ofNullable(clazz.cast(subscriberDescription()));
        case "subscriberName":
            return Optional.ofNullable(clazz.cast(subscriberName()));
        case "subscriptionEndpoint":
            return Optional.ofNullable(clazz.cast(subscriptionEndpoint()));
        case "subscriptionId":
            return Optional.ofNullable(clazz.cast(subscriptionId()));
        case "subscriptionProtocol":
            return Optional.ofNullable(clazz.cast(subscriptionProtocolAsString()));
        case "subscriptionStatus":
            return Optional.ofNullable(clazz.cast(subscriptionStatusAsString()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SubscriberResource, T> g) {
        return obj -> g.apply((SubscriberResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SubscriberResource> {
        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypesWithStrings(Collection<String> accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypesWithStrings(String... accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypes(Collection<AccessType> accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber. </p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypes(AccessType... accessTypes);

        /**
         * <p>
         * The Amazon Web Services account ID you are using to create your Amazon Security Lake account.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID you are using to create your Amazon Security Lake account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The date and time when the subscription was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the subscription was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The external ID of the subscriber. The external ID lets the user that is assuming the role assert the
         * circumstances in which they are operating. It also provides a way for the account owner to permit the role to
         * be assumed only under specific circumstances.
         * </p>
         * 
         * @param externalId
         *        The external ID of the subscriber. The external ID lets the user that is assuming the role assert the
         *        circumstances in which they are operating. It also provides a way for the account owner to permit the
         *        role to be assumed only under specific circumstances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalId(String externalId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) which uniquely defines the AWS RAM resource share. Before accepting the RAM
         * resource share invitation, you can view details related to the RAM resource share.
         * </p>
         * <p>
         * This field is available only for Lake Formation subscribers created after March 8, 2023.
         * </p>
         * 
         * @param resourceShareArn
         *        The Amazon Resource Name (ARN) which uniquely defines the AWS RAM resource share. Before accepting the
         *        RAM resource share invitation, you can view details related to the RAM resource share.</p>
         *        <p>
         *        This field is available only for Lake Formation subscribers created after March 8, 2023.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareArn(String resourceShareArn);

        /**
         * <p>
         * The name of the resource share.
         * </p>
         * 
         * @param resourceShareName
         *        The name of the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareName(String resourceShareName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) specifying the role of the subscriber.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) specifying the role of the subscriber.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The ARN for the Amazon S3 bucket.
         * </p>
         * 
         * @param s3BucketArn
         *        The ARN for the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketArn(String s3BucketArn);

        /**
         * <p>
         * The ARN for the Amazon Simple Notification Service.
         * </p>
         * 
         * @param snsArn
         *        The ARN for the Amazon Simple Notification Service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsArn(String snsArn);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For more
         * information, see the Amazon Security Lake User Guide.
         * </p>
         * 
         * @param sourceTypes
         *        Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For
         *        more information, see the Amazon Security Lake User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceTypes(Collection<SourceType> sourceTypes);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For more
         * information, see the Amazon Security Lake User Guide.
         * </p>
         * 
         * @param sourceTypes
         *        Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For
         *        more information, see the Amazon Security Lake User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceTypes(SourceType... sourceTypes);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services. For more
         * information, see the Amazon Security Lake User Guide.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securitylake.model.SourceType.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securitylake.model.SourceType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securitylake.model.SourceType.Builder#build()} is called immediately
         * and its result is passed to {@link #sourceTypes(List<SourceType>)}.
         * 
         * @param sourceTypes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securitylake.model.SourceType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceTypes(java.util.Collection<SourceType>)
         */
        Builder sourceTypes(Consumer<SourceType.Builder>... sourceTypes);

        /**
         * <p>
         * The subscriber descriptions for a subscriber account. The description for a subscriber includes
         * <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and <code>subscriptionId</code>
         * .
         * </p>
         * 
         * @param subscriberDescription
         *        The subscriber descriptions for a subscriber account. The description for a subscriber includes
         *        <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and
         *        <code>subscriptionId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberDescription(String subscriberDescription);

        /**
         * <p>
         * The name of your Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberName
         *        The name of your Amazon Security Lake subscriber account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberName(String subscriberName);

        /**
         * <p>
         * The subscription endpoint to which exception messages are posted.
         * </p>
         * 
         * @param subscriptionEndpoint
         *        The subscription endpoint to which exception messages are posted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriptionEndpoint(String subscriptionEndpoint);

        /**
         * <p>
         * The subscription ID of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriptionId
         *        The subscription ID of the Amazon Security Lake subscriber account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriptionId(String subscriptionId);

        /**
         * <p>
         * The subscription protocol to which exception messages are posted.
         * </p>
         * 
         * @param subscriptionProtocol
         *        The subscription protocol to which exception messages are posted.
         * @see EndpointProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointProtocol
         */
        Builder subscriptionProtocol(String subscriptionProtocol);

        /**
         * <p>
         * The subscription protocol to which exception messages are posted.
         * </p>
         * 
         * @param subscriptionProtocol
         *        The subscription protocol to which exception messages are posted.
         * @see EndpointProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointProtocol
         */
        Builder subscriptionProtocol(EndpointProtocol subscriptionProtocol);

        /**
         * <p>
         * The subscription status of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriptionStatus
         *        The subscription status of the Amazon Security Lake subscriber account.
         * @see SubscriptionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionStatus
         */
        Builder subscriptionStatus(String subscriptionStatus);

        /**
         * <p>
         * The subscription status of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriptionStatus
         *        The subscription status of the Amazon Security Lake subscriber account.
         * @see SubscriptionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionStatus
         */
        Builder subscriptionStatus(SubscriptionStatus subscriptionStatus);

        /**
         * <p>
         * The date and time when the subscription was created.
         * </p>
         * 
         * @param updatedAt
         *        The date and time when the subscription was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private List<String> accessTypes = DefaultSdkAutoConstructList.getInstance();

        private String accountId;

        private Instant createdAt;

        private String externalId;

        private String resourceShareArn;

        private String resourceShareName;

        private String roleArn;

        private String s3BucketArn;

        private String snsArn;

        private List<SourceType> sourceTypes = DefaultSdkAutoConstructList.getInstance();

        private String subscriberDescription;

        private String subscriberName;

        private String subscriptionEndpoint;

        private String subscriptionId;

        private String subscriptionProtocol;

        private String subscriptionStatus;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(SubscriberResource model) {
            accessTypesWithStrings(model.accessTypes);
            accountId(model.accountId);
            createdAt(model.createdAt);
            externalId(model.externalId);
            resourceShareArn(model.resourceShareArn);
            resourceShareName(model.resourceShareName);
            roleArn(model.roleArn);
            s3BucketArn(model.s3BucketArn);
            snsArn(model.snsArn);
            sourceTypes(model.sourceTypes);
            subscriberDescription(model.subscriberDescription);
            subscriberName(model.subscriberName);
            subscriptionEndpoint(model.subscriptionEndpoint);
            subscriptionId(model.subscriptionId);
            subscriptionProtocol(model.subscriptionProtocol);
            subscriptionStatus(model.subscriptionStatus);
            updatedAt(model.updatedAt);
        }

        public final Collection<String> getAccessTypes() {
            if (accessTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return accessTypes;
        }

        public final void setAccessTypes(Collection<String> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copy(accessTypes);
        }

        @Override
        public final Builder accessTypesWithStrings(Collection<String> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copy(accessTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessTypesWithStrings(String... accessTypes) {
            accessTypesWithStrings(Arrays.asList(accessTypes));
            return this;
        }

        @Override
        public final Builder accessTypes(Collection<AccessType> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copyEnumToString(accessTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessTypes(AccessType... accessTypes) {
            accessTypes(Arrays.asList(accessTypes));
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getExternalId() {
            return externalId;
        }

        public final void setExternalId(String externalId) {
            this.externalId = externalId;
        }

        @Override
        public final Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public final String getResourceShareArn() {
            return resourceShareArn;
        }

        public final void setResourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
        }

        @Override
        public final Builder resourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
            return this;
        }

        public final String getResourceShareName() {
            return resourceShareName;
        }

        public final void setResourceShareName(String resourceShareName) {
            this.resourceShareName = resourceShareName;
        }

        @Override
        public final Builder resourceShareName(String resourceShareName) {
            this.resourceShareName = resourceShareName;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getS3BucketArn() {
            return s3BucketArn;
        }

        public final void setS3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
        }

        @Override
        public final Builder s3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
            return this;
        }

        public final String getSnsArn() {
            return snsArn;
        }

        public final void setSnsArn(String snsArn) {
            this.snsArn = snsArn;
        }

        @Override
        public final Builder snsArn(String snsArn) {
            this.snsArn = snsArn;
            return this;
        }

        public final List<SourceType.Builder> getSourceTypes() {
            List<SourceType.Builder> result = SourceTypeListCopier.copyToBuilder(this.sourceTypes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceTypes(Collection<SourceType.BuilderImpl> sourceTypes) {
            this.sourceTypes = SourceTypeListCopier.copyFromBuilder(sourceTypes);
        }

        @Override
        public final Builder sourceTypes(Collection<SourceType> sourceTypes) {
            this.sourceTypes = SourceTypeListCopier.copy(sourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceTypes(SourceType... sourceTypes) {
            sourceTypes(Arrays.asList(sourceTypes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceTypes(Consumer<SourceType.Builder>... sourceTypes) {
            sourceTypes(Stream.of(sourceTypes).map(c -> SourceType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSubscriberDescription() {
            return subscriberDescription;
        }

        public final void setSubscriberDescription(String subscriberDescription) {
            this.subscriberDescription = subscriberDescription;
        }

        @Override
        public final Builder subscriberDescription(String subscriberDescription) {
            this.subscriberDescription = subscriberDescription;
            return this;
        }

        public final String getSubscriberName() {
            return subscriberName;
        }

        public final void setSubscriberName(String subscriberName) {
            this.subscriberName = subscriberName;
        }

        @Override
        public final Builder subscriberName(String subscriberName) {
            this.subscriberName = subscriberName;
            return this;
        }

        public final String getSubscriptionEndpoint() {
            return subscriptionEndpoint;
        }

        public final void setSubscriptionEndpoint(String subscriptionEndpoint) {
            this.subscriptionEndpoint = subscriptionEndpoint;
        }

        @Override
        public final Builder subscriptionEndpoint(String subscriptionEndpoint) {
            this.subscriptionEndpoint = subscriptionEndpoint;
            return this;
        }

        public final String getSubscriptionId() {
            return subscriptionId;
        }

        public final void setSubscriptionId(String subscriptionId) {
            this.subscriptionId = subscriptionId;
        }

        @Override
        public final Builder subscriptionId(String subscriptionId) {
            this.subscriptionId = subscriptionId;
            return this;
        }

        public final String getSubscriptionProtocol() {
            return subscriptionProtocol;
        }

        public final void setSubscriptionProtocol(String subscriptionProtocol) {
            this.subscriptionProtocol = subscriptionProtocol;
        }

        @Override
        public final Builder subscriptionProtocol(String subscriptionProtocol) {
            this.subscriptionProtocol = subscriptionProtocol;
            return this;
        }

        @Override
        public final Builder subscriptionProtocol(EndpointProtocol subscriptionProtocol) {
            this.subscriptionProtocol(subscriptionProtocol == null ? null : subscriptionProtocol.toString());
            return this;
        }

        public final String getSubscriptionStatus() {
            return subscriptionStatus;
        }

        public final void setSubscriptionStatus(String subscriptionStatus) {
            this.subscriptionStatus = subscriptionStatus;
        }

        @Override
        public final Builder subscriptionStatus(String subscriptionStatus) {
            this.subscriptionStatus = subscriptionStatus;
            return this;
        }

        @Override
        public final Builder subscriptionStatus(SubscriptionStatus subscriptionStatus) {
            this.subscriptionStatus(subscriptionStatus == null ? null : subscriptionStatus.toString());
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public SubscriberResource build() {
            return new SubscriberResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
