/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.securitylake.internal.SecurityLakeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.securitylake.model.AccessDeniedException;
import software.amazon.awssdk.services.securitylake.model.BadRequestException;
import software.amazon.awssdk.services.securitylake.model.ConflictException;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.CreateCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.CreateDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteAwsLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteCustomLogSourceResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.DeregisterDataLakeDelegatedAdministratorRequest;
import software.amazon.awssdk.services.securitylake.model.DeregisterDataLakeDelegatedAdministratorResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeSourcesRequest;
import software.amazon.awssdk.services.securitylake.model.GetDataLakeSourcesResponse;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.GetSubscriberResponse;
import software.amazon.awssdk.services.securitylake.model.InternalServerException;
import software.amazon.awssdk.services.securitylake.model.ListDataLakeExceptionsRequest;
import software.amazon.awssdk.services.securitylake.model.ListDataLakeExceptionsResponse;
import software.amazon.awssdk.services.securitylake.model.ListDataLakesRequest;
import software.amazon.awssdk.services.securitylake.model.ListDataLakesResponse;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesRequest;
import software.amazon.awssdk.services.securitylake.model.ListLogSourcesResponse;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersRequest;
import software.amazon.awssdk.services.securitylake.model.ListSubscribersResponse;
import software.amazon.awssdk.services.securitylake.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.securitylake.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.securitylake.model.RegisterDataLakeDelegatedAdministratorRequest;
import software.amazon.awssdk.services.securitylake.model.RegisterDataLakeDelegatedAdministratorResponse;
import software.amazon.awssdk.services.securitylake.model.ResourceNotFoundException;
import software.amazon.awssdk.services.securitylake.model.SecurityLakeException;
import software.amazon.awssdk.services.securitylake.model.TagResourceRequest;
import software.amazon.awssdk.services.securitylake.model.TagResourceResponse;
import software.amazon.awssdk.services.securitylake.model.ThrottlingException;
import software.amazon.awssdk.services.securitylake.model.UntagResourceRequest;
import software.amazon.awssdk.services.securitylake.model.UntagResourceResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeExceptionSubscriptionRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeExceptionSubscriptionResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateDataLakeResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberNotificationRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberNotificationResponse;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.securitylake.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.securitylake.transform.CreateAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteAwsLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteCustomLogSourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.DeregisterDataLakeDelegatedAdministratorRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetDataLakeSourcesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.GetSubscriberRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListDataLakeExceptionsRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListDataLakesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListLogSourcesRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListSubscribersRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.RegisterDataLakeDelegatedAdministratorRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDataLakeExceptionSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateDataLakeRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriberNotificationRequestMarshaller;
import software.amazon.awssdk.services.securitylake.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SecurityLakeAsyncClient}.
 *
 * @see SecurityLakeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecurityLakeAsyncClient implements SecurityLakeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSecurityLakeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSecurityLakeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds a natively supported Amazon Web Services service as an Amazon Security Lake source. Enables source types for
     * member accounts in required Amazon Web Services Regions, based on the parameters you specify. You can choose any
     * source type in any Region for either accounts that are part of a trusted organization or standalone accounts.
     * Once you add an Amazon Web Services service as a source, Security Lake starts collecting logs and events from it.
     * </p>
     * <p>
     * You can use this API only to enable natively supported Amazon Web Services services as a source. Use
     * <code>CreateCustomLogSource</code> to enable data collection from a custom source.
     * </p>
     *
     * @param createAwsLogSourceRequest
     * @return A Java Future containing the result of the CreateAwsLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAwsLogSourceResponse> createAwsLogSource(CreateAwsLogSourceRequest createAwsLogSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAwsLogSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAwsLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAwsLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAwsLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAwsLogSourceRequest, CreateAwsLogSourceResponse>()
                            .withOperationName("CreateAwsLogSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAwsLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAwsLogSourceRequest));
            CompletableFuture<CreateAwsLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a third-party custom source in Amazon Security Lake, from the Amazon Web Services Region where you want to
     * create a custom source. Security Lake can collect logs and events from third-party custom sources. After creating
     * the appropriate IAM role to invoke Glue crawler, use this API to add a custom source name in Security Lake. This
     * operation creates a partition in the Amazon S3 bucket for Security Lake as the target location for log files from
     * the custom source. In addition, this operation also creates an associated Glue table and an Glue crawler.
     * </p>
     *
     * @param createCustomLogSourceRequest
     * @return A Java Future containing the result of the CreateCustomLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomLogSourceResponse> createCustomLogSource(
            CreateCustomLogSourceRequest createCustomLogSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCustomLogSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCustomLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomLogSourceRequest, CreateCustomLogSourceResponse>()
                            .withOperationName("CreateCustomLogSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCustomLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCustomLogSourceRequest));
            CompletableFuture<CreateCustomLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initializes an Amazon Security Lake instance with the provided (or default) configuration. You can enable
     * Security Lake in Amazon Web Services Regions with customized settings before enabling log collection in Regions.
     * To specify particular Regions, configure these Regions using the <code>configurations</code> parameter. If you
     * have already enabled Security Lake in a Region when you call this command, the command will update the Region if
     * you provide new configuration parameters. If you have not already enabled Security Lake in the Region when you
     * call this API, it will set up the data lake in the Region with the specified configurations.
     * </p>
     * <p>
     * When you enable Security Lake, it starts ingesting security data after the <code>CreateAwsLogSource</code> call
     * and after you create subscribers using the <code>CreateSubscriber</code> API. This includes ingesting security
     * data from sources, storing data, and making data accessible to subscribers. Security Lake also enables all the
     * existing settings and resources that it stores or maintains for your Amazon Web Services account in the current
     * Region, including security log and event data. For more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/what-is-security-lake.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param createDataLakeRequest
     * @return A Java Future containing the result of the CreateDataLake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataLakeResponse> createDataLake(CreateDataLakeRequest createDataLakeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataLakeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataLakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataLakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeRequest, CreateDataLakeResponse>()
                            .withOperationName("CreateDataLake").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataLakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataLakeRequest));
            CompletableFuture<CreateDataLakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates the specified notification subscription in Amazon Security Lake for the organization you specify. The
     * notification subscription is created for exceptions that cannot be resolved by Security Lake automatically.
     * </p>
     *
     * @param createDataLakeExceptionSubscriptionRequest
     * @return A Java Future containing the result of the CreateDataLakeExceptionSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataLakeExceptionSubscriptionResponse> createDataLakeExceptionSubscription(
            CreateDataLakeExceptionSubscriptionRequest createDataLakeExceptionSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataLakeExceptionSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLakeExceptionSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateDataLakeExceptionSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataLakeExceptionSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeExceptionSubscriptionRequest, CreateDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("CreateDataLakeExceptionSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataLakeExceptionSubscriptionRequest));
            CompletableFuture<CreateDataLakeExceptionSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Automatically enables Amazon Security Lake for new member accounts in your organization. Security Lake is not
     * automatically enabled for any existing member accounts in your organization.
     * </p>
     * <p>
     * This operation merges the new data lake organization configuration with the existing configuration for Security
     * Lake in your organization. If you want to create a new data lake organization configuration, you must delete the
     * existing one using <a href=
     * "https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLakeOrganizationConfiguration.html"
     * >DeleteDataLakeOrganizationConfiguration</a>.
     * </p>
     *
     * @param createDataLakeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the CreateDataLakeOrganizationConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataLakeOrganizationConfigurationResponse> createDataLakeOrganizationConfiguration(
            CreateDataLakeOrganizationConfigurationRequest createDataLakeOrganizationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataLakeOrganizationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataLakeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateDataLakeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataLakeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataLakeOrganizationConfigurationRequest, CreateDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("CreateDataLakeOrganizationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataLakeOrganizationConfigurationRequest));
            CompletableFuture<CreateDataLakeOrganizationConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subscriber for accounts that are already enabled in Amazon Security Lake. You can create a subscriber
     * with access to data in the current Amazon Web Services Region.
     * </p>
     *
     * @param createSubscriberRequest
     * @return A Java Future containing the result of the CreateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriberResponse> createSubscriber(CreateSubscriberRequest createSubscriberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                            .withOperationName("CreateSubscriber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSubscriberRequest));
            CompletableFuture<CreateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Notifies the subscriber when new data is written to the data lake for the sources that the subscriber consumes in
     * Security Lake. You can create only one subscriber notification per subscriber.
     * </p>
     *
     * @param createSubscriberNotificationRequest
     * @return A Java Future containing the result of the CreateSubscriberNotification operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.CreateSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/CreateSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriberNotificationResponse> createSubscriberNotification(
            CreateSubscriberNotificationRequest createSubscriberNotificationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSubscriberNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriberNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriberNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriberNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriberNotificationRequest, CreateSubscriberNotificationResponse>()
                            .withOperationName("CreateSubscriberNotification").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSubscriberNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSubscriberNotificationRequest));
            CompletableFuture<CreateSubscriberNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a natively supported Amazon Web Services service as an Amazon Security Lake source. You can remove a
     * source for one or more Regions. When you remove the source, Security Lake stops collecting data from that source
     * in the specified Regions and accounts, and subscribers can no longer consume new data from the source. However,
     * subscribers can still consume data that Security Lake collected from the source before removal.
     * </p>
     * <p>
     * You can choose any source type in any Amazon Web Services Region for either accounts that are part of a trusted
     * organization or standalone accounts.
     * </p>
     *
     * @param deleteAwsLogSourceRequest
     * @return A Java Future containing the result of the DeleteAwsLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteAwsLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteAwsLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAwsLogSourceResponse> deleteAwsLogSource(DeleteAwsLogSourceRequest deleteAwsLogSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAwsLogSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAwsLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAwsLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAwsLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAwsLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAwsLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAwsLogSourceRequest, DeleteAwsLogSourceResponse>()
                            .withOperationName("DeleteAwsLogSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAwsLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAwsLogSourceRequest));
            CompletableFuture<DeleteAwsLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a custom log source from Amazon Security Lake, to stop sending data from the custom source to Security
     * Lake.
     * </p>
     *
     * @param deleteCustomLogSourceRequest
     * @return A Java Future containing the result of the DeleteCustomLogSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteCustomLogSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteCustomLogSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomLogSourceResponse> deleteCustomLogSource(
            DeleteCustomLogSourceRequest deleteCustomLogSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomLogSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomLogSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomLogSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomLogSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomLogSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomLogSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomLogSourceRequest, DeleteCustomLogSourceResponse>()
                            .withOperationName("DeleteCustomLogSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCustomLogSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCustomLogSourceRequest));
            CompletableFuture<DeleteCustomLogSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you disable Amazon Security Lake from your account, Security Lake is disabled in all Amazon Web Services
     * Regions and it stops collecting data from your sources. Also, this API automatically takes steps to remove the
     * account from Security Lake. However, Security Lake retains all of your existing settings and the resources that
     * it created in your Amazon Web Services account in the current Amazon Web Services Region.
     * </p>
     * <p>
     * The <code>DeleteDataLake</code> operation does not delete the data that is stored in your Amazon S3 bucket, which
     * is owned by your Amazon Web Services account. For more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/disable-security-lake.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param deleteDataLakeRequest
     * @return A Java Future containing the result of the DeleteDataLake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataLakeResponse> deleteDataLake(DeleteDataLakeRequest deleteDataLakeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataLakeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataLakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataLakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeRequest, DeleteDataLakeResponse>()
                            .withOperationName("DeleteDataLake").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataLakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataLakeRequest));
            CompletableFuture<DeleteDataLakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteDataLakeExceptionSubscriptionRequest
     * @return A Java Future containing the result of the DeleteDataLakeExceptionSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataLakeExceptionSubscriptionResponse> deleteDataLakeExceptionSubscription(
            DeleteDataLakeExceptionSubscriptionRequest deleteDataLakeExceptionSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataLakeExceptionSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLakeExceptionSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteDataLakeExceptionSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataLakeExceptionSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeExceptionSubscriptionRequest, DeleteDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("DeleteDataLakeExceptionSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataLakeExceptionSubscriptionRequest));
            CompletableFuture<DeleteDataLakeExceptionSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Turns off automatic enablement of Amazon Security Lake for member accounts that are added to an organization in
     * Organizations. Only the delegated Security Lake administrator for an organization can perform this operation. If
     * the delegated Security Lake administrator performs this operation, new member accounts won't automatically
     * contribute data to the data lake.
     * </p>
     *
     * @param deleteDataLakeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the DeleteDataLakeOrganizationConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataLakeOrganizationConfigurationResponse> deleteDataLakeOrganizationConfiguration(
            DeleteDataLakeOrganizationConfigurationRequest deleteDataLakeOrganizationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataLakeOrganizationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataLakeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteDataLakeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataLakeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataLakeOrganizationConfigurationRequest, DeleteDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("DeleteDataLakeOrganizationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataLakeOrganizationConfigurationRequest));
            CompletableFuture<DeleteDataLakeOrganizationConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the subscription permission and all notification settings for accounts that are already enabled in Amazon
     * Security Lake. When you run <code>DeleteSubscriber</code>, the subscriber will no longer consume data from
     * Security Lake and the subscriber is removed. This operation deletes the subscriber and removes access to data in
     * the current Amazon Web Services Region.
     * </p>
     *
     * @param deleteSubscriberRequest
     * @return A Java Future containing the result of the DeleteSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubscriberResponse> deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                            .withOperationName("DeleteSubscriber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSubscriberRequest));
            CompletableFuture<DeleteSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified subscription notification in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param deleteSubscriberNotificationRequest
     * @return A Java Future containing the result of the DeleteSubscriberNotification operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeleteSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeleteSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubscriberNotificationResponse> deleteSubscriberNotification(
            DeleteSubscriberNotificationRequest deleteSubscriberNotificationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSubscriberNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriberNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubscriberNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriberNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriberNotificationRequest, DeleteSubscriberNotificationResponse>()
                            .withOperationName("DeleteSubscriberNotification").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSubscriberNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSubscriberNotificationRequest));
            CompletableFuture<DeleteSubscriberNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param deregisterDataLakeDelegatedAdministratorRequest
     * @return A Java Future containing the result of the DeregisterDataLakeDelegatedAdministrator operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.DeregisterDataLakeDelegatedAdministrator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/DeregisterDataLakeDelegatedAdministrator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterDataLakeDelegatedAdministratorResponse> deregisterDataLakeDelegatedAdministrator(
            DeregisterDataLakeDelegatedAdministratorRequest deregisterDataLakeDelegatedAdministratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deregisterDataLakeDelegatedAdministratorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterDataLakeDelegatedAdministratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterDataLakeDelegatedAdministrator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterDataLakeDelegatedAdministratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeregisterDataLakeDelegatedAdministratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterDataLakeDelegatedAdministratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterDataLakeDelegatedAdministratorRequest, DeregisterDataLakeDelegatedAdministratorResponse>()
                            .withOperationName("DeregisterDataLakeDelegatedAdministrator").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterDataLakeDelegatedAdministratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterDataLakeDelegatedAdministratorRequest));
            CompletableFuture<DeregisterDataLakeDelegatedAdministratorResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the protocol and endpoint that were provided when subscribing to Amazon SNS topics for exception
     * notifications.
     * </p>
     *
     * @param getDataLakeExceptionSubscriptionRequest
     * @return A Java Future containing the result of the GetDataLakeExceptionSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDataLakeExceptionSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeExceptionSubscriptionResponse> getDataLakeExceptionSubscription(
            GetDataLakeExceptionSubscriptionRequest getDataLakeExceptionSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataLakeExceptionSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeExceptionSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetDataLakeExceptionSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeExceptionSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeExceptionSubscriptionRequest, GetDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("GetDataLakeExceptionSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataLakeExceptionSubscriptionRequest));
            CompletableFuture<GetDataLakeExceptionSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the configuration that will be automatically set up for accounts added to the organization after the
     * organization has onboarded to Amazon Security Lake. This API does not take input parameters.
     * </p>
     *
     * @param getDataLakeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the GetDataLakeOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDataLakeOrganizationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeOrganizationConfigurationResponse> getDataLakeOrganizationConfiguration(
            GetDataLakeOrganizationConfigurationRequest getDataLakeOrganizationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataLakeOrganizationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getDataLakeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetDataLakeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeOrganizationConfigurationRequest, GetDataLakeOrganizationConfigurationResponse>()
                            .withOperationName("GetDataLakeOrganizationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataLakeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataLakeOrganizationConfigurationRequest));
            CompletableFuture<GetDataLakeOrganizationConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a snapshot of the current Region, including whether Amazon Security Lake is enabled for those accounts
     * and which sources Security Lake is collecting data from.
     * </p>
     *
     * @param getDataLakeSourcesRequest
     * @return A Java Future containing the result of the GetDataLakeSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetDataLakeSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetDataLakeSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataLakeSourcesResponse> getDataLakeSources(GetDataLakeSourcesRequest getDataLakeSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataLakeSourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataLakeSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataLakeSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataLakeSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataLakeSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataLakeSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataLakeSourcesRequest, GetDataLakeSourcesResponse>()
                            .withOperationName("GetDataLakeSources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataLakeSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataLakeSourcesRequest));
            CompletableFuture<GetDataLakeSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the subscription information for the specified subscription ID. You can get information about a
     * specific subscriber.
     * </p>
     *
     * @param getSubscriberRequest
     * @return A Java Future containing the result of the GetSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.GetSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/GetSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriberResponse> getSubscriber(GetSubscriberRequest getSubscriberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSubscriberRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriberRequest, GetSubscriberResponse>()
                            .withOperationName("GetSubscriber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSubscriberRequest));
            CompletableFuture<GetSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Security Lake exceptions that you can use to find the source of problems and fix them.
     * </p>
     *
     * @param listDataLakeExceptionsRequest
     * @return A Java Future containing the result of the ListDataLakeExceptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListDataLakeExceptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDataLakeExceptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataLakeExceptionsResponse> listDataLakeExceptions(
            ListDataLakeExceptionsRequest listDataLakeExceptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataLakeExceptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataLakeExceptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataLakeExceptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataLakeExceptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataLakeExceptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataLakeExceptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataLakeExceptionsRequest, ListDataLakeExceptionsResponse>()
                            .withOperationName("ListDataLakeExceptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataLakeExceptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataLakeExceptionsRequest));
            CompletableFuture<ListDataLakeExceptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Amazon Security Lake configuration object for the specified Amazon Web Services Regions. You can
     * use this operation to determine whether Security Lake is enabled for a Region.
     * </p>
     *
     * @param listDataLakesRequest
     * @return A Java Future containing the result of the ListDataLakes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListDataLakes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListDataLakes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataLakesResponse> listDataLakes(ListDataLakesRequest listDataLakesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataLakesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataLakesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataLakes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataLakesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDataLakesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataLakesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataLakesRequest, ListDataLakesResponse>()
                            .withOperationName("ListDataLakes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataLakesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataLakesRequest));
            CompletableFuture<ListDataLakesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the log sources.
     * </p>
     *
     * @param listLogSourcesRequest
     * @return A Java Future containing the result of the ListLogSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListLogSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListLogSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLogSourcesResponse> listLogSources(ListLogSourcesRequest listLogSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLogSourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLogSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLogSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLogSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLogSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLogSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLogSourcesRequest, ListLogSourcesResponse>()
                            .withOperationName("ListLogSources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLogSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLogSourcesRequest));
            CompletableFuture<ListLogSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all subscribers for the specific Amazon Security Lake account ID. You can retrieve a list of subscriptions
     * associated with a specific organization or Amazon Web Services account.
     * </p>
     *
     * @param listSubscribersRequest
     * @return A Java Future containing the result of the ListSubscribers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListSubscribers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListSubscribers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubscribersResponse> listSubscribers(ListSubscribersRequest listSubscribersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSubscribersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubscribersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubscribers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSubscribersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSubscribersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSubscribersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubscribersRequest, ListSubscribersResponse>()
                            .withOperationName("ListSubscribers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSubscribersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSubscribersRequest));
            CompletableFuture<ListSubscribersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tags (keys and values) that are associated with an Amazon Security Lake resource: a subscriber, or
     * the data lake configuration for your Amazon Web Services account in a particular Amazon Web Services Region.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the Amazon Security Lake delegated administrator account for the organization. This API can only be
     * called by the organization management account. The organization management account cannot be the delegated
     * administrator account.
     * </p>
     *
     * @param registerDataLakeDelegatedAdministratorRequest
     * @return A Java Future containing the result of the RegisterDataLakeDelegatedAdministrator operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.RegisterDataLakeDelegatedAdministrator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/RegisterDataLakeDelegatedAdministrator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterDataLakeDelegatedAdministratorResponse> registerDataLakeDelegatedAdministrator(
            RegisterDataLakeDelegatedAdministratorRequest registerDataLakeDelegatedAdministratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerDataLakeDelegatedAdministratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerDataLakeDelegatedAdministratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterDataLakeDelegatedAdministrator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterDataLakeDelegatedAdministratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, RegisterDataLakeDelegatedAdministratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterDataLakeDelegatedAdministratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterDataLakeDelegatedAdministratorRequest, RegisterDataLakeDelegatedAdministratorResponse>()
                            .withOperationName("RegisterDataLakeDelegatedAdministrator").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterDataLakeDelegatedAdministratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerDataLakeDelegatedAdministratorRequest));
            CompletableFuture<RegisterDataLakeDelegatedAdministratorResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates one or more tags that are associated with an Amazon Security Lake resource: a subscriber, or the
     * data lake configuration for your Amazon Web Services account in a particular Amazon Web Services Region. A
     * <i>tag</i> is a label that you can define and associate with Amazon Web Services resources. Each tag consists of
     * a required <i>tag key</i> and an associated <i>tag value</i>. A <i>tag key</i> is a general label that acts as a
     * category for a more specific tag value. A <i>tag value</i> acts as a descriptor for a tag key. Tags can help you
     * identify, categorize, and manage resources in different ways, such as by owner, environment, or other criteria.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/tagging-resources.html">Tagging Amazon Security
     * Lake resources</a> in the <i>Amazon Security Lake User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags (keys and values) from an Amazon Security Lake resource: a subscriber, or the data lake
     * configuration for your Amazon Web Services account in a particular Amazon Web Services Region.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use <code>UpdateDataLake</code> to specify where to store your security data, how it should be encrypted
     * at rest and for how long. You can add a <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/manage-regions.html#add-rollup-region">Rollup
     * Region</a> to consolidate data from multiple Amazon Web Services Regions, replace default encryption (SSE-S3)
     * with <a href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#customer-cmk">Customer Manged
     * Key</a>, or specify transition and expiration actions through storage <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/lifecycle-management.html">Lifecycle
     * management</a>. The <code>UpdateDataLake</code> API works as an "upsert" operation that performs an insert if the
     * specified item or record does not exist, or an update if it already exists. Security Lake securely stores your
     * data at rest using Amazon Web Services encryption solutions. For more details, see <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/data-protection.html">Data protection in Amazon
     * Security Lake</a>.
     * </p>
     * <p>
     * For example, omitting the key <code>encryptionConfiguration</code> from a Region that is included in an update
     * call that currently uses KMS will leave that Region's KMS key in place, but specifying
     * <code>encryptionConfiguration: {kmsKeyId: 'S3_MANAGED_KEY'}</code> for that same Region will reset the key to
     * <code>S3-managed</code>.
     * </p>
     * <p>
     * For more details about lifecycle management and how to update retention settings for one or more Regions after
     * enabling Security Lake, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/lifecycle-management.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     *
     * @param updateDataLakeRequest
     * @return A Java Future containing the result of the UpdateDataLake operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateDataLake
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDataLake" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataLakeResponse> updateDataLake(UpdateDataLakeRequest updateDataLakeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataLakeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataLakeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataLake");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataLakeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataLakeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataLakeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataLakeRequest, UpdateDataLakeResponse>()
                            .withOperationName("UpdateDataLake").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataLakeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataLakeRequest));
            CompletableFuture<UpdateDataLakeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified notification subscription in Amazon Security Lake for the organization you specify.
     * </p>
     *
     * @param updateDataLakeExceptionSubscriptionRequest
     * @return A Java Future containing the result of the UpdateDataLakeExceptionSubscription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateDataLakeExceptionSubscription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateDataLakeExceptionSubscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataLakeExceptionSubscriptionResponse> updateDataLakeExceptionSubscription(
            UpdateDataLakeExceptionSubscriptionRequest updateDataLakeExceptionSubscriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataLakeExceptionSubscriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDataLakeExceptionSubscriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataLakeExceptionSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataLakeExceptionSubscriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateDataLakeExceptionSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataLakeExceptionSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataLakeExceptionSubscriptionRequest, UpdateDataLakeExceptionSubscriptionResponse>()
                            .withOperationName("UpdateDataLakeExceptionSubscription").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataLakeExceptionSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataLakeExceptionSubscriptionRequest));
            CompletableFuture<UpdateDataLakeExceptionSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing subscription for the given Amazon Security Lake account ID. You can update a subscriber by
     * changing the sources that the subscriber consumes data from.
     * </p>
     *
     * @param updateSubscriberRequest
     * @return A Java Future containing the result of the UpdateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateSubscriber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriberResponse> updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                            .withOperationName("UpdateSubscriber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSubscriberRequest));
            CompletableFuture<UpdateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing notification method for the subscription (SQS or HTTPs endpoint) or switches the notification
     * subscription endpoint for a subscriber.
     * </p>
     *
     * @param updateSubscriberNotificationRequest
     * @return A Java Future containing the result of the UpdateSubscriberNotification operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The request is malformed or contains an error such as an invalid parameter value
     *         or a missing required parameter.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>InternalServerException Internal service exceptions are sometimes caused by transient issues. Before
     *         you start troubleshooting, perform the operation again.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action. Access denied errors
     *         appear when Amazon Security Lake explicitly or implicitly denies an authorization request. An explicit
     *         denial occurs when a policy contains a Deny statement for the specific Amazon Web Services action. An
     *         implicit denial occurs when there is no applicable Deny statement and also no applicable Allow statement.
     *         </li>
     *         <li>ConflictException Occurs when a conflict with a previous successful write is detected. This generally
     *         occurs when the previous write did not have time to propagate to the host serving the current request. A
     *         retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>ThrottlingException The limit on the number of requests per second was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecurityLakeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample SecurityLakeAsyncClient.UpdateSubscriberNotification
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/securitylake-2018-05-10/UpdateSubscriberNotification"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriberNotificationResponse> updateSubscriberNotification(
            UpdateSubscriberNotificationRequest updateSubscriberNotificationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSubscriberNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "SecurityLake");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriberNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriberNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriberNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriberNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriberNotificationRequest, UpdateSubscriberNotificationResponse>()
                            .withOperationName("UpdateSubscriberNotification").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSubscriberNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSubscriberNotificationRequest));
            CompletableFuture<UpdateSubscriberNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final SecurityLakeServiceClientConfiguration serviceClientConfiguration() {
        return new SecurityLakeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecurityLakeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        SecurityLakeServiceClientConfigurationBuilder serviceConfigBuilder = new SecurityLakeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
