/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides details about the Amazon Security Lake account subscription. Subscribers are notified of new objects for a
 * source as the data is written to your Amazon S3 bucket for Security Lake.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SubscriberResource implements SdkPojo, Serializable,
        ToCopyableBuilder<SubscriberResource.Builder, SubscriberResource> {
    private static final SdkField<List<String>> ACCESS_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("accessTypes")
            .getter(getter(SubscriberResource::accessTypesAsStrings))
            .setter(setter(Builder::accessTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(SubscriberResource::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> RESOURCE_SHARE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceShareArn").getter(getter(SubscriberResource::resourceShareArn))
            .setter(setter(Builder::resourceShareArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareArn").build()).build();

    private static final SdkField<String> RESOURCE_SHARE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceShareName").getter(getter(SubscriberResource::resourceShareName))
            .setter(setter(Builder::resourceShareName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceShareName").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(SubscriberResource::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> S3_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("s3BucketArn").getter(getter(SubscriberResource::s3BucketArn)).setter(setter(Builder::s3BucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3BucketArn").build()).build();

    private static final SdkField<List<LogSourceResource>> SOURCES_FIELD = SdkField
            .<List<LogSourceResource>> builder(MarshallingType.LIST)
            .memberName("sources")
            .getter(getter(SubscriberResource::sources))
            .setter(setter(Builder::sources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LogSourceResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogSourceResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SUBSCRIBER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberArn").getter(getter(SubscriberResource::subscriberArn)).setter(setter(Builder::subscriberArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberArn").build()).build();

    private static final SdkField<String> SUBSCRIBER_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberDescription").getter(getter(SubscriberResource::subscriberDescription))
            .setter(setter(Builder::subscriberDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberDescription").build())
            .build();

    private static final SdkField<String> SUBSCRIBER_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberEndpoint").getter(getter(SubscriberResource::subscriberEndpoint))
            .setter(setter(Builder::subscriberEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberEndpoint").build())
            .build();

    private static final SdkField<String> SUBSCRIBER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberId").getter(getter(SubscriberResource::subscriberId)).setter(setter(Builder::subscriberId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberId").build()).build();

    private static final SdkField<AwsIdentity> SUBSCRIBER_IDENTITY_FIELD = SdkField
            .<AwsIdentity> builder(MarshallingType.SDK_POJO).memberName("subscriberIdentity")
            .getter(getter(SubscriberResource::subscriberIdentity)).setter(setter(Builder::subscriberIdentity))
            .constructor(AwsIdentity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberIdentity").build())
            .build();

    private static final SdkField<String> SUBSCRIBER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberName").getter(getter(SubscriberResource::subscriberName))
            .setter(setter(Builder::subscriberName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberName").build()).build();

    private static final SdkField<String> SUBSCRIBER_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subscriberStatus").getter(getter(SubscriberResource::subscriberStatusAsString))
            .setter(setter(Builder::subscriberStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subscriberStatus").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(SubscriberResource::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCESS_TYPES_FIELD,
            CREATED_AT_FIELD, RESOURCE_SHARE_ARN_FIELD, RESOURCE_SHARE_NAME_FIELD, ROLE_ARN_FIELD, S3_BUCKET_ARN_FIELD,
            SOURCES_FIELD, SUBSCRIBER_ARN_FIELD, SUBSCRIBER_DESCRIPTION_FIELD, SUBSCRIBER_ENDPOINT_FIELD, SUBSCRIBER_ID_FIELD,
            SUBSCRIBER_IDENTITY_FIELD, SUBSCRIBER_NAME_FIELD, SUBSCRIBER_STATUS_FIELD, UPDATED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> accessTypes;

    private final Instant createdAt;

    private final String resourceShareArn;

    private final String resourceShareName;

    private final String roleArn;

    private final String s3BucketArn;

    private final List<LogSourceResource> sources;

    private final String subscriberArn;

    private final String subscriberDescription;

    private final String subscriberEndpoint;

    private final String subscriberId;

    private final AwsIdentity subscriberIdentity;

    private final String subscriberName;

    private final String subscriberStatus;

    private final Instant updatedAt;

    private SubscriberResource(BuilderImpl builder) {
        this.accessTypes = builder.accessTypes;
        this.createdAt = builder.createdAt;
        this.resourceShareArn = builder.resourceShareArn;
        this.resourceShareName = builder.resourceShareName;
        this.roleArn = builder.roleArn;
        this.s3BucketArn = builder.s3BucketArn;
        this.sources = builder.sources;
        this.subscriberArn = builder.subscriberArn;
        this.subscriberDescription = builder.subscriberDescription;
        this.subscriberEndpoint = builder.subscriberEndpoint;
        this.subscriberId = builder.subscriberId;
        this.subscriberIdentity = builder.subscriberIdentity;
        this.subscriberName = builder.subscriberName;
        this.subscriberStatus = builder.subscriberStatus;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
     * through messaging to an HTTPS endpoint provided by the subscriber.
     * </p>
     * <p>
     * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through services
     * like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessTypes} method.
     * </p>
     * 
     * @return You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
     *         queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
     *         <p>
     *         Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
     *         services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     */
    public final List<AccessType> accessTypes() {
        return AccessTypeListCopier.copyStringToEnum(accessTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessTypes() {
        return accessTypes != null && !(accessTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
     * through messaging to an HTTPS endpoint provided by the subscriber.
     * </p>
     * <p>
     * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through services
     * like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessTypes} method.
     * </p>
     * 
     * @return You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
     *         queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
     *         <p>
     *         Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
     *         services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
     */
    public final List<String> accessTypesAsStrings() {
        return accessTypes;
    }

    /**
     * <p>
     * The date and time when the subscriber was created.
     * </p>
     * 
     * @return The date and time when the subscriber was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) which uniquely defines the Amazon Web Services RAM resource share. Before
     * accepting the RAM resource share invitation, you can view details related to the RAM resource share.
     * </p>
     * <p>
     * This field is available only for Lake Formation subscribers created after March 8, 2023.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) which uniquely defines the Amazon Web Services RAM resource share. Before
     *         accepting the RAM resource share invitation, you can view details related to the RAM resource share.</p>
     *         <p>
     *         This field is available only for Lake Formation subscribers created after March 8, 2023.
     */
    public final String resourceShareArn() {
        return resourceShareArn;
    }

    /**
     * <p>
     * The name of the resource share.
     * </p>
     * 
     * @return The name of the resource share.
     */
    public final String resourceShareName() {
        return resourceShareName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) specifying the role of the subscriber.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) specifying the role of the subscriber.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The ARN for the Amazon S3 bucket.
     * </p>
     * 
     * @return The ARN for the Amazon S3 bucket.
     */
    public final String s3BucketArn() {
        return s3BucketArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Sources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSources() {
        return sources != null && !(sources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services services. For
     * more information, see the <a
     * href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon Security Lake
     * User Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSources} method.
     * </p>
     * 
     * @return Amazon Security Lake supports log and event collection for natively supported Amazon Web Services
     *         services. For more information, see the <a
     *         href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon Security
     *         Lake User Guide</a>.
     */
    public final List<LogSourceResource> sources() {
        return sources;
    }

    /**
     * <p>
     * The subscriber ARN of the Amazon Security Lake subscriber account.
     * </p>
     * 
     * @return The subscriber ARN of the Amazon Security Lake subscriber account.
     */
    public final String subscriberArn() {
        return subscriberArn;
    }

    /**
     * <p>
     * The subscriber descriptions for a subscriber account. The description for a subscriber includes
     * <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and <code>subscriberId</code>.
     * </p>
     * 
     * @return The subscriber descriptions for a subscriber account. The description for a subscriber includes
     *         <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and
     *         <code>subscriberId</code>.
     */
    public final String subscriberDescription() {
        return subscriberDescription;
    }

    /**
     * <p>
     * The subscriber endpoint to which exception messages are posted.
     * </p>
     * 
     * @return The subscriber endpoint to which exception messages are posted.
     */
    public final String subscriberEndpoint() {
        return subscriberEndpoint;
    }

    /**
     * <p>
     * The subscriber ID of the Amazon Security Lake subscriber account.
     * </p>
     * 
     * @return The subscriber ID of the Amazon Security Lake subscriber account.
     */
    public final String subscriberId() {
        return subscriberId;
    }

    /**
     * <p>
     * The Amazon Web Services identity used to access your data.
     * </p>
     * 
     * @return The Amazon Web Services identity used to access your data.
     */
    public final AwsIdentity subscriberIdentity() {
        return subscriberIdentity;
    }

    /**
     * <p>
     * The name of your Amazon Security Lake subscriber account.
     * </p>
     * 
     * @return The name of your Amazon Security Lake subscriber account.
     */
    public final String subscriberName() {
        return subscriberName;
    }

    /**
     * <p>
     * The subscriber status of the Amazon Security Lake subscriber account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriberStatus}
     * will return {@link SubscriberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriberStatusAsString}.
     * </p>
     * 
     * @return The subscriber status of the Amazon Security Lake subscriber account.
     * @see SubscriberStatus
     */
    public final SubscriberStatus subscriberStatus() {
        return SubscriberStatus.fromValue(subscriberStatus);
    }

    /**
     * <p>
     * The subscriber status of the Amazon Security Lake subscriber account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriberStatus}
     * will return {@link SubscriberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriberStatusAsString}.
     * </p>
     * 
     * @return The subscriber status of the Amazon Security Lake subscriber account.
     * @see SubscriberStatus
     */
    public final String subscriberStatusAsString() {
        return subscriberStatus;
    }

    /**
     * <p>
     * The date and time when the subscriber was last updated.
     * </p>
     * 
     * @return The date and time when the subscriber was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessTypes() ? accessTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(resourceShareArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceShareName());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasSources() ? sources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(subscriberArn());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberDescription());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberId());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberIdentity());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberName());
        hashCode = 31 * hashCode + Objects.hashCode(subscriberStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SubscriberResource)) {
            return false;
        }
        SubscriberResource other = (SubscriberResource) obj;
        return hasAccessTypes() == other.hasAccessTypes() && Objects.equals(accessTypesAsStrings(), other.accessTypesAsStrings())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(resourceShareArn(), other.resourceShareArn())
                && Objects.equals(resourceShareName(), other.resourceShareName()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(s3BucketArn(), other.s3BucketArn()) && hasSources() == other.hasSources()
                && Objects.equals(sources(), other.sources()) && Objects.equals(subscriberArn(), other.subscriberArn())
                && Objects.equals(subscriberDescription(), other.subscriberDescription())
                && Objects.equals(subscriberEndpoint(), other.subscriberEndpoint())
                && Objects.equals(subscriberId(), other.subscriberId())
                && Objects.equals(subscriberIdentity(), other.subscriberIdentity())
                && Objects.equals(subscriberName(), other.subscriberName())
                && Objects.equals(subscriberStatusAsString(), other.subscriberStatusAsString())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SubscriberResource").add("AccessTypes", hasAccessTypes() ? accessTypesAsStrings() : null)
                .add("CreatedAt", createdAt()).add("ResourceShareArn", resourceShareArn())
                .add("ResourceShareName", resourceShareName()).add("RoleArn", roleArn()).add("S3BucketArn", s3BucketArn())
                .add("Sources", hasSources() ? sources() : null).add("SubscriberArn", subscriberArn())
                .add("SubscriberDescription", subscriberDescription()).add("SubscriberEndpoint", subscriberEndpoint())
                .add("SubscriberId", subscriberId()).add("SubscriberIdentity", subscriberIdentity())
                .add("SubscriberName", subscriberName()).add("SubscriberStatus", subscriberStatusAsString())
                .add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accessTypes":
            return Optional.ofNullable(clazz.cast(accessTypesAsStrings()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "resourceShareArn":
            return Optional.ofNullable(clazz.cast(resourceShareArn()));
        case "resourceShareName":
            return Optional.ofNullable(clazz.cast(resourceShareName()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "s3BucketArn":
            return Optional.ofNullable(clazz.cast(s3BucketArn()));
        case "sources":
            return Optional.ofNullable(clazz.cast(sources()));
        case "subscriberArn":
            return Optional.ofNullable(clazz.cast(subscriberArn()));
        case "subscriberDescription":
            return Optional.ofNullable(clazz.cast(subscriberDescription()));
        case "subscriberEndpoint":
            return Optional.ofNullable(clazz.cast(subscriberEndpoint()));
        case "subscriberId":
            return Optional.ofNullable(clazz.cast(subscriberId()));
        case "subscriberIdentity":
            return Optional.ofNullable(clazz.cast(subscriberIdentity()));
        case "subscriberName":
            return Optional.ofNullable(clazz.cast(subscriberName()));
        case "subscriberStatus":
            return Optional.ofNullable(clazz.cast(subscriberStatusAsString()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("accessTypes", ACCESS_TYPES_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("resourceShareArn", RESOURCE_SHARE_ARN_FIELD);
        map.put("resourceShareName", RESOURCE_SHARE_NAME_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("s3BucketArn", S3_BUCKET_ARN_FIELD);
        map.put("sources", SOURCES_FIELD);
        map.put("subscriberArn", SUBSCRIBER_ARN_FIELD);
        map.put("subscriberDescription", SUBSCRIBER_DESCRIPTION_FIELD);
        map.put("subscriberEndpoint", SUBSCRIBER_ENDPOINT_FIELD);
        map.put("subscriberId", SUBSCRIBER_ID_FIELD);
        map.put("subscriberIdentity", SUBSCRIBER_IDENTITY_FIELD);
        map.put("subscriberName", SUBSCRIBER_NAME_FIELD);
        map.put("subscriberStatus", SUBSCRIBER_STATUS_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SubscriberResource, T> g) {
        return obj -> g.apply((SubscriberResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SubscriberResource> {
        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypesWithStrings(Collection<String> accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypesWithStrings(String... accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypes(Collection<AccessType> accessTypes);

        /**
         * <p>
         * You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS) queue or
         * through messaging to an HTTPS endpoint provided by the subscriber.
         * </p>
         * <p>
         * Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket through
         * services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * </p>
         * 
         * @param accessTypes
         *        You can choose to notify subscribers of new objects with an Amazon Simple Queue Service (Amazon SQS)
         *        queue or through messaging to an HTTPS endpoint provided by the subscriber.</p>
         *        <p>
         *        Subscribers can consume data by directly querying Lake Formation tables in your Amazon S3 bucket
         *        through services like Amazon Athena. This subscription type is defined as <code>LAKEFORMATION</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessTypes(AccessType... accessTypes);

        /**
         * <p>
         * The date and time when the subscriber was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the subscriber was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Amazon Resource Name (ARN) which uniquely defines the Amazon Web Services RAM resource share. Before
         * accepting the RAM resource share invitation, you can view details related to the RAM resource share.
         * </p>
         * <p>
         * This field is available only for Lake Formation subscribers created after March 8, 2023.
         * </p>
         * 
         * @param resourceShareArn
         *        The Amazon Resource Name (ARN) which uniquely defines the Amazon Web Services RAM resource share.
         *        Before accepting the RAM resource share invitation, you can view details related to the RAM resource
         *        share.</p>
         *        <p>
         *        This field is available only for Lake Formation subscribers created after March 8, 2023.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareArn(String resourceShareArn);

        /**
         * <p>
         * The name of the resource share.
         * </p>
         * 
         * @param resourceShareName
         *        The name of the resource share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceShareName(String resourceShareName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) specifying the role of the subscriber.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) specifying the role of the subscriber.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The ARN for the Amazon S3 bucket.
         * </p>
         * 
         * @param s3BucketArn
         *        The ARN for the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketArn(String s3BucketArn);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services services.
         * For more information, see the <a
         * href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon Security Lake
         * User Guide</a>.
         * </p>
         * 
         * @param sources
         *        Amazon Security Lake supports log and event collection for natively supported Amazon Web Services
         *        services. For more information, see the <a
         *        href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon
         *        Security Lake User Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(Collection<LogSourceResource> sources);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services services.
         * For more information, see the <a
         * href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon Security Lake
         * User Guide</a>.
         * </p>
         * 
         * @param sources
         *        Amazon Security Lake supports log and event collection for natively supported Amazon Web Services
         *        services. For more information, see the <a
         *        href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon
         *        Security Lake User Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(LogSourceResource... sources);

        /**
         * <p>
         * Amazon Security Lake supports log and event collection for natively supported Amazon Web Services services.
         * For more information, see the <a
         * href="https://docs.aws.amazon.com/security-lake/latest/userguide/source-management.html">Amazon Security Lake
         * User Guide</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securitylake.model.LogSourceResource.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.securitylake.model.LogSourceResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securitylake.model.LogSourceResource.Builder#build()} is called
         * immediately and its result is passed to {@link #sources(List<LogSourceResource>)}.
         * 
         * @param sources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securitylake.model.LogSourceResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sources(java.util.Collection<LogSourceResource>)
         */
        Builder sources(Consumer<LogSourceResource.Builder>... sources);

        /**
         * <p>
         * The subscriber ARN of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberArn
         *        The subscriber ARN of the Amazon Security Lake subscriber account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberArn(String subscriberArn);

        /**
         * <p>
         * The subscriber descriptions for a subscriber account. The description for a subscriber includes
         * <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and <code>subscriberId</code>.
         * </p>
         * 
         * @param subscriberDescription
         *        The subscriber descriptions for a subscriber account. The description for a subscriber includes
         *        <code>subscriberName</code>, <code>accountID</code>, <code>externalID</code>, and
         *        <code>subscriberId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberDescription(String subscriberDescription);

        /**
         * <p>
         * The subscriber endpoint to which exception messages are posted.
         * </p>
         * 
         * @param subscriberEndpoint
         *        The subscriber endpoint to which exception messages are posted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberEndpoint(String subscriberEndpoint);

        /**
         * <p>
         * The subscriber ID of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberId
         *        The subscriber ID of the Amazon Security Lake subscriber account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberId(String subscriberId);

        /**
         * <p>
         * The Amazon Web Services identity used to access your data.
         * </p>
         * 
         * @param subscriberIdentity
         *        The Amazon Web Services identity used to access your data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberIdentity(AwsIdentity subscriberIdentity);

        /**
         * <p>
         * The Amazon Web Services identity used to access your data.
         * </p>
         * This is a convenience method that creates an instance of the {@link AwsIdentity.Builder} avoiding the need to
         * create one manually via {@link AwsIdentity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AwsIdentity.Builder#build()} is called immediately and its result
         * is passed to {@link #subscriberIdentity(AwsIdentity)}.
         * 
         * @param subscriberIdentity
         *        a consumer that will call methods on {@link AwsIdentity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subscriberIdentity(AwsIdentity)
         */
        default Builder subscriberIdentity(Consumer<AwsIdentity.Builder> subscriberIdentity) {
            return subscriberIdentity(AwsIdentity.builder().applyMutation(subscriberIdentity).build());
        }

        /**
         * <p>
         * The name of your Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberName
         *        The name of your Amazon Security Lake subscriber account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscriberName(String subscriberName);

        /**
         * <p>
         * The subscriber status of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberStatus
         *        The subscriber status of the Amazon Security Lake subscriber account.
         * @see SubscriberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriberStatus
         */
        Builder subscriberStatus(String subscriberStatus);

        /**
         * <p>
         * The subscriber status of the Amazon Security Lake subscriber account.
         * </p>
         * 
         * @param subscriberStatus
         *        The subscriber status of the Amazon Security Lake subscriber account.
         * @see SubscriberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriberStatus
         */
        Builder subscriberStatus(SubscriberStatus subscriberStatus);

        /**
         * <p>
         * The date and time when the subscriber was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The date and time when the subscriber was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private List<String> accessTypes = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private String resourceShareArn;

        private String resourceShareName;

        private String roleArn;

        private String s3BucketArn;

        private List<LogSourceResource> sources = DefaultSdkAutoConstructList.getInstance();

        private String subscriberArn;

        private String subscriberDescription;

        private String subscriberEndpoint;

        private String subscriberId;

        private AwsIdentity subscriberIdentity;

        private String subscriberName;

        private String subscriberStatus;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(SubscriberResource model) {
            accessTypesWithStrings(model.accessTypes);
            createdAt(model.createdAt);
            resourceShareArn(model.resourceShareArn);
            resourceShareName(model.resourceShareName);
            roleArn(model.roleArn);
            s3BucketArn(model.s3BucketArn);
            sources(model.sources);
            subscriberArn(model.subscriberArn);
            subscriberDescription(model.subscriberDescription);
            subscriberEndpoint(model.subscriberEndpoint);
            subscriberId(model.subscriberId);
            subscriberIdentity(model.subscriberIdentity);
            subscriberName(model.subscriberName);
            subscriberStatus(model.subscriberStatus);
            updatedAt(model.updatedAt);
        }

        public final Collection<String> getAccessTypes() {
            if (accessTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return accessTypes;
        }

        public final void setAccessTypes(Collection<String> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copy(accessTypes);
        }

        @Override
        public final Builder accessTypesWithStrings(Collection<String> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copy(accessTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessTypesWithStrings(String... accessTypes) {
            accessTypesWithStrings(Arrays.asList(accessTypes));
            return this;
        }

        @Override
        public final Builder accessTypes(Collection<AccessType> accessTypes) {
            this.accessTypes = AccessTypeListCopier.copyEnumToString(accessTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessTypes(AccessType... accessTypes) {
            accessTypes(Arrays.asList(accessTypes));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getResourceShareArn() {
            return resourceShareArn;
        }

        public final void setResourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
        }

        @Override
        public final Builder resourceShareArn(String resourceShareArn) {
            this.resourceShareArn = resourceShareArn;
            return this;
        }

        public final String getResourceShareName() {
            return resourceShareName;
        }

        public final void setResourceShareName(String resourceShareName) {
            this.resourceShareName = resourceShareName;
        }

        @Override
        public final Builder resourceShareName(String resourceShareName) {
            this.resourceShareName = resourceShareName;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getS3BucketArn() {
            return s3BucketArn;
        }

        public final void setS3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
        }

        @Override
        public final Builder s3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
            return this;
        }

        public final List<LogSourceResource.Builder> getSources() {
            List<LogSourceResource.Builder> result = LogSourceResourceListCopier.copyToBuilder(this.sources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSources(Collection<LogSourceResource.BuilderImpl> sources) {
            this.sources = LogSourceResourceListCopier.copyFromBuilder(sources);
        }

        @Override
        public final Builder sources(Collection<LogSourceResource> sources) {
            this.sources = LogSourceResourceListCopier.copy(sources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sources(LogSourceResource... sources) {
            sources(Arrays.asList(sources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sources(Consumer<LogSourceResource.Builder>... sources) {
            sources(Stream.of(sources).map(c -> LogSourceResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSubscriberArn() {
            return subscriberArn;
        }

        public final void setSubscriberArn(String subscriberArn) {
            this.subscriberArn = subscriberArn;
        }

        @Override
        public final Builder subscriberArn(String subscriberArn) {
            this.subscriberArn = subscriberArn;
            return this;
        }

        public final String getSubscriberDescription() {
            return subscriberDescription;
        }

        public final void setSubscriberDescription(String subscriberDescription) {
            this.subscriberDescription = subscriberDescription;
        }

        @Override
        public final Builder subscriberDescription(String subscriberDescription) {
            this.subscriberDescription = subscriberDescription;
            return this;
        }

        public final String getSubscriberEndpoint() {
            return subscriberEndpoint;
        }

        public final void setSubscriberEndpoint(String subscriberEndpoint) {
            this.subscriberEndpoint = subscriberEndpoint;
        }

        @Override
        public final Builder subscriberEndpoint(String subscriberEndpoint) {
            this.subscriberEndpoint = subscriberEndpoint;
            return this;
        }

        public final String getSubscriberId() {
            return subscriberId;
        }

        public final void setSubscriberId(String subscriberId) {
            this.subscriberId = subscriberId;
        }

        @Override
        public final Builder subscriberId(String subscriberId) {
            this.subscriberId = subscriberId;
            return this;
        }

        public final AwsIdentity.Builder getSubscriberIdentity() {
            return subscriberIdentity != null ? subscriberIdentity.toBuilder() : null;
        }

        public final void setSubscriberIdentity(AwsIdentity.BuilderImpl subscriberIdentity) {
            this.subscriberIdentity = subscriberIdentity != null ? subscriberIdentity.build() : null;
        }

        @Override
        public final Builder subscriberIdentity(AwsIdentity subscriberIdentity) {
            this.subscriberIdentity = subscriberIdentity;
            return this;
        }

        public final String getSubscriberName() {
            return subscriberName;
        }

        public final void setSubscriberName(String subscriberName) {
            this.subscriberName = subscriberName;
        }

        @Override
        public final Builder subscriberName(String subscriberName) {
            this.subscriberName = subscriberName;
            return this;
        }

        public final String getSubscriberStatus() {
            return subscriberStatus;
        }

        public final void setSubscriberStatus(String subscriberStatus) {
            this.subscriberStatus = subscriberStatus;
        }

        @Override
        public final Builder subscriberStatus(String subscriberStatus) {
            this.subscriberStatus = subscriberStatus;
            return this;
        }

        @Override
        public final Builder subscriberStatus(SubscriberStatus subscriberStatus) {
            this.subscriberStatus(subscriberStatus == null ? null : subscriberStatus.toString());
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public SubscriberResource build() {
            return new SubscriberResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
