/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securitylake.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Amazon Security Lake collects logs and events from supported Amazon Web Services services and custom sources. For the
 * list of supported Amazon Web Services services, see the <a
 * href="https://docs.aws.amazon.com/security-lake/latest/userguide/internal-sources.html">Amazon Security Lake User
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataLakeSource implements SdkPojo, Serializable, ToCopyableBuilder<DataLakeSource.Builder, DataLakeSource> {
    private static final SdkField<String> ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("account")
            .getter(getter(DataLakeSource::account)).setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("account").build()).build();

    private static final SdkField<List<String>> EVENT_CLASSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("eventClasses")
            .getter(getter(DataLakeSource::eventClasses))
            .setter(setter(Builder::eventClasses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventClasses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceName").getter(getter(DataLakeSource::sourceName)).setter(setter(Builder::sourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceName").build()).build();

    private static final SdkField<List<DataLakeSourceStatus>> SOURCE_STATUSES_FIELD = SdkField
            .<List<DataLakeSourceStatus>> builder(MarshallingType.LIST)
            .memberName("sourceStatuses")
            .getter(getter(DataLakeSource::sourceStatuses))
            .setter(setter(Builder::sourceStatuses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceStatuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataLakeSourceStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataLakeSourceStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_FIELD,
            EVENT_CLASSES_FIELD, SOURCE_NAME_FIELD, SOURCE_STATUSES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String account;

    private final List<String> eventClasses;

    private final String sourceName;

    private final List<DataLakeSourceStatus> sourceStatuses;

    private DataLakeSource(BuilderImpl builder) {
        this.account = builder.account;
        this.eventClasses = builder.eventClasses;
        this.sourceName = builder.sourceName;
        this.sourceStatuses = builder.sourceStatuses;
    }

    /**
     * <p>
     * The ID of the Security Lake account for which logs are collected.
     * </p>
     * 
     * @return The ID of the Security Lake account for which logs are collected.
     */
    public final String account() {
        return account;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventClasses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventClasses() {
        return eventClasses != null && !(eventClasses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the custom source
     * will send to Security Lake. For the list of supported event classes, see <a href=
     * "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
     * >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventClasses} method.
     * </p>
     * 
     * @return The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the custom
     *         source will send to Security Lake. For the list of supported event classes, see <a href=
     *         "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
     *         >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
     */
    public final List<String> eventClasses() {
        return eventClasses;
    }

    /**
     * <p>
     * The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
     * supports log and event collection for natively supported Amazon Web Services services.
     * </p>
     * 
     * @return The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
     *         supports log and event collection for natively supported Amazon Web Services services.
     */
    public final String sourceName() {
        return sourceName;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceStatuses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceStatuses() {
        return sourceStatuses != null && !(sourceStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The log status for the Security Lake account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceStatuses} method.
     * </p>
     * 
     * @return The log status for the Security Lake account.
     */
    public final List<DataLakeSourceStatus> sourceStatuses() {
        return sourceStatuses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(hasEventClasses() ? eventClasses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceName());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceStatuses() ? sourceStatuses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataLakeSource)) {
            return false;
        }
        DataLakeSource other = (DataLakeSource) obj;
        return Objects.equals(account(), other.account()) && hasEventClasses() == other.hasEventClasses()
                && Objects.equals(eventClasses(), other.eventClasses()) && Objects.equals(sourceName(), other.sourceName())
                && hasSourceStatuses() == other.hasSourceStatuses() && Objects.equals(sourceStatuses(), other.sourceStatuses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataLakeSource").add("Account", account())
                .add("EventClasses", hasEventClasses() ? eventClasses() : null).add("SourceName", sourceName())
                .add("SourceStatuses", hasSourceStatuses() ? sourceStatuses() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "account":
            return Optional.ofNullable(clazz.cast(account()));
        case "eventClasses":
            return Optional.ofNullable(clazz.cast(eventClasses()));
        case "sourceName":
            return Optional.ofNullable(clazz.cast(sourceName()));
        case "sourceStatuses":
            return Optional.ofNullable(clazz.cast(sourceStatuses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("account", ACCOUNT_FIELD);
        map.put("eventClasses", EVENT_CLASSES_FIELD);
        map.put("sourceName", SOURCE_NAME_FIELD);
        map.put("sourceStatuses", SOURCE_STATUSES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataLakeSource, T> g) {
        return obj -> g.apply((DataLakeSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataLakeSource> {
        /**
         * <p>
         * The ID of the Security Lake account for which logs are collected.
         * </p>
         * 
         * @param account
         *        The ID of the Security Lake account for which logs are collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the custom
         * source will send to Security Lake. For the list of supported event classes, see <a href=
         * "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
         * >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
         * </p>
         * 
         * @param eventClasses
         *        The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the
         *        custom source will send to Security Lake. For the list of supported event classes, see <a href=
         *        "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
         *        >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventClasses(Collection<String> eventClasses);

        /**
         * <p>
         * The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the custom
         * source will send to Security Lake. For the list of supported event classes, see <a href=
         * "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
         * >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
         * </p>
         * 
         * @param eventClasses
         *        The Open Cybersecurity Schema Framework (OCSF) event classes describes the type of data that the
         *        custom source will send to Security Lake. For the list of supported event classes, see <a href=
         *        "https://docs.aws.amazon.com/security-lake/latest/userguide/adding-custom-sources.html#ocsf-eventclass.html"
         *        >Supported OCSF Event classes</a> in the Amazon Security Lake User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventClasses(String... eventClasses);

        /**
         * <p>
         * The supported Amazon Web Services services from which logs and events are collected. Amazon Security Lake
         * supports log and event collection for natively supported Amazon Web Services services.
         * </p>
         * 
         * @param sourceName
         *        The supported Amazon Web Services services from which logs and events are collected. Amazon Security
         *        Lake supports log and event collection for natively supported Amazon Web Services services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceName(String sourceName);

        /**
         * <p>
         * The log status for the Security Lake account.
         * </p>
         * 
         * @param sourceStatuses
         *        The log status for the Security Lake account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceStatuses(Collection<DataLakeSourceStatus> sourceStatuses);

        /**
         * <p>
         * The log status for the Security Lake account.
         * </p>
         * 
         * @param sourceStatuses
         *        The log status for the Security Lake account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceStatuses(DataLakeSourceStatus... sourceStatuses);

        /**
         * <p>
         * The log status for the Security Lake account.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securitylake.model.DataLakeSourceStatus.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.securitylake.model.DataLakeSourceStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securitylake.model.DataLakeSourceStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceStatuses(List<DataLakeSourceStatus>)}.
         * 
         * @param sourceStatuses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securitylake.model.DataLakeSourceStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceStatuses(java.util.Collection<DataLakeSourceStatus>)
         */
        Builder sourceStatuses(Consumer<DataLakeSourceStatus.Builder>... sourceStatuses);
    }

    static final class BuilderImpl implements Builder {
        private String account;

        private List<String> eventClasses = DefaultSdkAutoConstructList.getInstance();

        private String sourceName;

        private List<DataLakeSourceStatus> sourceStatuses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataLakeSource model) {
            account(model.account);
            eventClasses(model.eventClasses);
            sourceName(model.sourceName);
            sourceStatuses(model.sourceStatuses);
        }

        public final String getAccount() {
            return account;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final Collection<String> getEventClasses() {
            if (eventClasses instanceof SdkAutoConstructList) {
                return null;
            }
            return eventClasses;
        }

        public final void setEventClasses(Collection<String> eventClasses) {
            this.eventClasses = OcsfEventClassListCopier.copy(eventClasses);
        }

        @Override
        public final Builder eventClasses(Collection<String> eventClasses) {
            this.eventClasses = OcsfEventClassListCopier.copy(eventClasses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventClasses(String... eventClasses) {
            eventClasses(Arrays.asList(eventClasses));
            return this;
        }

        public final String getSourceName() {
            return sourceName;
        }

        public final void setSourceName(String sourceName) {
            this.sourceName = sourceName;
        }

        @Override
        public final Builder sourceName(String sourceName) {
            this.sourceName = sourceName;
            return this;
        }

        public final List<DataLakeSourceStatus.Builder> getSourceStatuses() {
            List<DataLakeSourceStatus.Builder> result = DataLakeSourceStatusListCopier.copyToBuilder(this.sourceStatuses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceStatuses(Collection<DataLakeSourceStatus.BuilderImpl> sourceStatuses) {
            this.sourceStatuses = DataLakeSourceStatusListCopier.copyFromBuilder(sourceStatuses);
        }

        @Override
        public final Builder sourceStatuses(Collection<DataLakeSourceStatus> sourceStatuses) {
            this.sourceStatuses = DataLakeSourceStatusListCopier.copy(sourceStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceStatuses(DataLakeSourceStatus... sourceStatuses) {
            sourceStatuses(Arrays.asList(sourceStatuses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceStatuses(Consumer<DataLakeSourceStatus.Builder>... sourceStatuses) {
            sourceStatuses(Stream.of(sourceStatuses).map(c -> DataLakeSourceStatus.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public DataLakeSource build() {
            return new DataLakeSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
