/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicecatalog.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-defined preferences that will be applied when updating a provisioned product. Not all preferences are
 * applicable to all provisioned product type
 * </p>
 * <p>
 * One or more Amazon Web Services accounts that will have access to the provisioned product.
 * </p>
 * <p>
 * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
 * </p>
 * <p>
 * The Amazon Web Services accounts specified should be within the list of accounts in the <code>STACKSET</code>
 * constraint. To get the list of accounts in the <code>STACKSET</code> constraint, use the
 * <code>DescribeProvisioningParameters</code> operation.
 * </p>
 * <p>
 * If no values are specified, the default value is all accounts from the <code>STACKSET</code> constraint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProvisioningPreferences implements SdkPojo, Serializable,
        ToCopyableBuilder<ProvisioningPreferences.Builder, ProvisioningPreferences> {
    private static final SdkField<List<String>> STACK_SET_ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StackSetAccounts")
            .getter(getter(ProvisioningPreferences::stackSetAccounts))
            .setter(setter(Builder::stackSetAccounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetAccounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> STACK_SET_REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StackSetRegions")
            .getter(getter(ProvisioningPreferences::stackSetRegions))
            .setter(setter(Builder::stackSetRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> STACK_SET_FAILURE_TOLERANCE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StackSetFailureToleranceCount")
            .getter(getter(ProvisioningPreferences::stackSetFailureToleranceCount))
            .setter(setter(Builder::stackSetFailureToleranceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetFailureToleranceCount")
                    .build()).build();

    private static final SdkField<Integer> STACK_SET_FAILURE_TOLERANCE_PERCENTAGE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StackSetFailureTolerancePercentage")
            .getter(getter(ProvisioningPreferences::stackSetFailureTolerancePercentage))
            .setter(setter(Builder::stackSetFailureTolerancePercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetFailureTolerancePercentage")
                    .build()).build();

    private static final SdkField<Integer> STACK_SET_MAX_CONCURRENCY_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StackSetMaxConcurrencyCount")
            .getter(getter(ProvisioningPreferences::stackSetMaxConcurrencyCount))
            .setter(setter(Builder::stackSetMaxConcurrencyCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetMaxConcurrencyCount")
                    .build()).build();

    private static final SdkField<Integer> STACK_SET_MAX_CONCURRENCY_PERCENTAGE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StackSetMaxConcurrencyPercentage")
            .getter(getter(ProvisioningPreferences::stackSetMaxConcurrencyPercentage))
            .setter(setter(Builder::stackSetMaxConcurrencyPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetMaxConcurrencyPercentage")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_SET_ACCOUNTS_FIELD,
            STACK_SET_REGIONS_FIELD, STACK_SET_FAILURE_TOLERANCE_COUNT_FIELD, STACK_SET_FAILURE_TOLERANCE_PERCENTAGE_FIELD,
            STACK_SET_MAX_CONCURRENCY_COUNT_FIELD, STACK_SET_MAX_CONCURRENCY_PERCENTAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> stackSetAccounts;

    private final List<String> stackSetRegions;

    private final Integer stackSetFailureToleranceCount;

    private final Integer stackSetFailureTolerancePercentage;

    private final Integer stackSetMaxConcurrencyCount;

    private final Integer stackSetMaxConcurrencyPercentage;

    private ProvisioningPreferences(BuilderImpl builder) {
        this.stackSetAccounts = builder.stackSetAccounts;
        this.stackSetRegions = builder.stackSetRegions;
        this.stackSetFailureToleranceCount = builder.stackSetFailureToleranceCount;
        this.stackSetFailureTolerancePercentage = builder.stackSetFailureTolerancePercentage;
        this.stackSetMaxConcurrencyCount = builder.stackSetMaxConcurrencyCount;
        this.stackSetMaxConcurrencyPercentage = builder.stackSetMaxConcurrencyPercentage;
    }

    /**
     * For responses, this returns true if the service returned a value for the StackSetAccounts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStackSetAccounts() {
        return stackSetAccounts != null && !(stackSetAccounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more Amazon Web Services accounts where the provisioned product will be available.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * The specified accounts should be within the list of accounts from the <code>STACKSET</code> constraint. To get
     * the list of accounts in the <code>STACKSET</code> constraint, use the <code>DescribeProvisioningParameters</code>
     * operation.
     * </p>
     * <p>
     * If no values are specified, the default value is all acounts from the <code>STACKSET</code> constraint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStackSetAccounts} method.
     * </p>
     * 
     * @return One or more Amazon Web Services accounts where the provisioned product will be available.</p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         The specified accounts should be within the list of accounts from the <code>STACKSET</code> constraint.
     *         To get the list of accounts in the <code>STACKSET</code> constraint, use the
     *         <code>DescribeProvisioningParameters</code> operation.
     *         </p>
     *         <p>
     *         If no values are specified, the default value is all acounts from the <code>STACKSET</code> constraint.
     */
    public final List<String> stackSetAccounts() {
        return stackSetAccounts;
    }

    /**
     * For responses, this returns true if the service returned a value for the StackSetRegions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStackSetRegions() {
        return stackSetRegions != null && !(stackSetRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more Amazon Web Services Regions where the provisioned product will be available.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint. To get the
     * list of Regions in the <code>STACKSET</code> constraint, use the <code>DescribeProvisioningParameters</code>
     * operation.
     * </p>
     * <p>
     * If no values are specified, the default value is all Regions from the <code>STACKSET</code> constraint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStackSetRegions} method.
     * </p>
     * 
     * @return One or more Amazon Web Services Regions where the provisioned product will be available.</p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint. To
     *         get the list of Regions in the <code>STACKSET</code> constraint, use the
     *         <code>DescribeProvisioningParameters</code> operation.
     *         </p>
     *         <p>
     *         If no values are specified, the default value is all Regions from the <code>STACKSET</code> constraint.
     */
    public final List<String> stackSetRegions() {
        return stackSetRegions;
    }

    /**
     * <p>
     * The number of accounts, per Region, for which this operation can fail before Service Catalog stops the operation
     * in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the operation in any
     * subsequent Regions.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
     * <code>StackSetFailureTolerancePercentage</code>, but not both.
     * </p>
     * <p>
     * The default value is <code>0</code> if no value is specified.
     * </p>
     * 
     * @return The number of accounts, per Region, for which this operation can fail before Service Catalog stops the
     *         operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the
     *         operation in any subsequent Regions.</p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
     *         <code>StackSetFailureTolerancePercentage</code>, but not both.
     *         </p>
     *         <p>
     *         The default value is <code>0</code> if no value is specified.
     */
    public final Integer stackSetFailureToleranceCount() {
        return stackSetFailureToleranceCount;
    }

    /**
     * <p>
     * The percentage of accounts, per Region, for which this stack operation can fail before Service Catalog stops the
     * operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the operation
     * in any subsequent Regions.
     * </p>
     * <p>
     * When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to the
     * next whole number.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
     * <code>StackSetFailureTolerancePercentage</code>, but not both.
     * </p>
     * 
     * @return The percentage of accounts, per Region, for which this stack operation can fail before Service Catalog
     *         stops the operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't
     *         attempt the operation in any subsequent Regions.</p>
     *         <p>
     *         When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to
     *         the next whole number.
     *         </p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
     *         <code>StackSetFailureTolerancePercentage</code>, but not both.
     */
    public final Integer stackSetFailureTolerancePercentage() {
        return stackSetFailureTolerancePercentage;
    }

    /**
     * <p>
     * The maximum number of accounts in which to perform this operation at one time. This is dependent on the value of
     * <code>StackSetFailureToleranceCount</code>. <code>StackSetMaxConcurrentCount</code> is at most one more than the
     * <code>StackSetFailureToleranceCount</code>.
     * </p>
     * <p>
     * Note that this setting lets you specify the maximum for operations. For large deployments, under certain
     * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
     * <code>StackSetMaxConcurrentPercentage</code>, but not both.
     * </p>
     * 
     * @return The maximum number of accounts in which to perform this operation at one time. This is dependent on the
     *         value of <code>StackSetFailureToleranceCount</code>. <code>StackSetMaxConcurrentCount</code> is at most
     *         one more than the <code>StackSetFailureToleranceCount</code>.</p>
     *         <p>
     *         Note that this setting lets you specify the maximum for operations. For large deployments, under certain
     *         circumstances the actual number of accounts acted upon concurrently may be lower due to service
     *         throttling.
     *         </p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
     *         <code>StackSetMaxConcurrentPercentage</code>, but not both.
     */
    public final Integer stackSetMaxConcurrencyCount() {
        return stackSetMaxConcurrencyCount;
    }

    /**
     * <p>
     * The maximum percentage of accounts in which to perform this operation at one time.
     * </p>
     * <p>
     * When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to the
     * next whole number. This is true except in cases where rounding down would result is zero. In this case, Service
     * Catalog sets the number as <code>1</code> instead.
     * </p>
     * <p>
     * Note that this setting lets you specify the maximum for operations. For large deployments, under certain
     * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
     * </p>
     * <p>
     * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     * </p>
     * <p>
     * Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
     * <code>StackSetMaxConcurrentPercentage</code>, but not both.
     * </p>
     * 
     * @return The maximum percentage of accounts in which to perform this operation at one time.</p>
     *         <p>
     *         When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to
     *         the next whole number. This is true except in cases where rounding down would result is zero. In this
     *         case, Service Catalog sets the number as <code>1</code> instead.
     *         </p>
     *         <p>
     *         Note that this setting lets you specify the maximum for operations. For large deployments, under certain
     *         circumstances the actual number of accounts acted upon concurrently may be lower due to service
     *         throttling.
     *         </p>
     *         <p>
     *         Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
     *         <code>StackSetMaxConcurrentPercentage</code>, but not both.
     */
    public final Integer stackSetMaxConcurrencyPercentage() {
        return stackSetMaxConcurrencyPercentage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasStackSetAccounts() ? stackSetAccounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStackSetRegions() ? stackSetRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stackSetFailureToleranceCount());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetFailureTolerancePercentage());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetMaxConcurrencyCount());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetMaxConcurrencyPercentage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProvisioningPreferences)) {
            return false;
        }
        ProvisioningPreferences other = (ProvisioningPreferences) obj;
        return hasStackSetAccounts() == other.hasStackSetAccounts()
                && Objects.equals(stackSetAccounts(), other.stackSetAccounts())
                && hasStackSetRegions() == other.hasStackSetRegions()
                && Objects.equals(stackSetRegions(), other.stackSetRegions())
                && Objects.equals(stackSetFailureToleranceCount(), other.stackSetFailureToleranceCount())
                && Objects.equals(stackSetFailureTolerancePercentage(), other.stackSetFailureTolerancePercentage())
                && Objects.equals(stackSetMaxConcurrencyCount(), other.stackSetMaxConcurrencyCount())
                && Objects.equals(stackSetMaxConcurrencyPercentage(), other.stackSetMaxConcurrencyPercentage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProvisioningPreferences")
                .add("StackSetAccounts", hasStackSetAccounts() ? stackSetAccounts() : null)
                .add("StackSetRegions", hasStackSetRegions() ? stackSetRegions() : null)
                .add("StackSetFailureToleranceCount", stackSetFailureToleranceCount())
                .add("StackSetFailureTolerancePercentage", stackSetFailureTolerancePercentage())
                .add("StackSetMaxConcurrencyCount", stackSetMaxConcurrencyCount())
                .add("StackSetMaxConcurrencyPercentage", stackSetMaxConcurrencyPercentage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetAccounts":
            return Optional.ofNullable(clazz.cast(stackSetAccounts()));
        case "StackSetRegions":
            return Optional.ofNullable(clazz.cast(stackSetRegions()));
        case "StackSetFailureToleranceCount":
            return Optional.ofNullable(clazz.cast(stackSetFailureToleranceCount()));
        case "StackSetFailureTolerancePercentage":
            return Optional.ofNullable(clazz.cast(stackSetFailureTolerancePercentage()));
        case "StackSetMaxConcurrencyCount":
            return Optional.ofNullable(clazz.cast(stackSetMaxConcurrencyCount()));
        case "StackSetMaxConcurrencyPercentage":
            return Optional.ofNullable(clazz.cast(stackSetMaxConcurrencyPercentage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StackSetAccounts", STACK_SET_ACCOUNTS_FIELD);
        map.put("StackSetRegions", STACK_SET_REGIONS_FIELD);
        map.put("StackSetFailureToleranceCount", STACK_SET_FAILURE_TOLERANCE_COUNT_FIELD);
        map.put("StackSetFailureTolerancePercentage", STACK_SET_FAILURE_TOLERANCE_PERCENTAGE_FIELD);
        map.put("StackSetMaxConcurrencyCount", STACK_SET_MAX_CONCURRENCY_COUNT_FIELD);
        map.put("StackSetMaxConcurrencyPercentage", STACK_SET_MAX_CONCURRENCY_PERCENTAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProvisioningPreferences, T> g) {
        return obj -> g.apply((ProvisioningPreferences) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProvisioningPreferences> {
        /**
         * <p>
         * One or more Amazon Web Services accounts where the provisioned product will be available.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * The specified accounts should be within the list of accounts from the <code>STACKSET</code> constraint. To
         * get the list of accounts in the <code>STACKSET</code> constraint, use the
         * <code>DescribeProvisioningParameters</code> operation.
         * </p>
         * <p>
         * If no values are specified, the default value is all acounts from the <code>STACKSET</code> constraint.
         * </p>
         * 
         * @param stackSetAccounts
         *        One or more Amazon Web Services accounts where the provisioned product will be available.</p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        The specified accounts should be within the list of accounts from the <code>STACKSET</code>
         *        constraint. To get the list of accounts in the <code>STACKSET</code> constraint, use the
         *        <code>DescribeProvisioningParameters</code> operation.
         *        </p>
         *        <p>
         *        If no values are specified, the default value is all acounts from the <code>STACKSET</code>
         *        constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetAccounts(Collection<String> stackSetAccounts);

        /**
         * <p>
         * One or more Amazon Web Services accounts where the provisioned product will be available.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * The specified accounts should be within the list of accounts from the <code>STACKSET</code> constraint. To
         * get the list of accounts in the <code>STACKSET</code> constraint, use the
         * <code>DescribeProvisioningParameters</code> operation.
         * </p>
         * <p>
         * If no values are specified, the default value is all acounts from the <code>STACKSET</code> constraint.
         * </p>
         * 
         * @param stackSetAccounts
         *        One or more Amazon Web Services accounts where the provisioned product will be available.</p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        The specified accounts should be within the list of accounts from the <code>STACKSET</code>
         *        constraint. To get the list of accounts in the <code>STACKSET</code> constraint, use the
         *        <code>DescribeProvisioningParameters</code> operation.
         *        </p>
         *        <p>
         *        If no values are specified, the default value is all acounts from the <code>STACKSET</code>
         *        constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetAccounts(String... stackSetAccounts);

        /**
         * <p>
         * One or more Amazon Web Services Regions where the provisioned product will be available.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint. To get
         * the list of Regions in the <code>STACKSET</code> constraint, use the
         * <code>DescribeProvisioningParameters</code> operation.
         * </p>
         * <p>
         * If no values are specified, the default value is all Regions from the <code>STACKSET</code> constraint.
         * </p>
         * 
         * @param stackSetRegions
         *        One or more Amazon Web Services Regions where the provisioned product will be available.</p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint.
         *        To get the list of Regions in the <code>STACKSET</code> constraint, use the
         *        <code>DescribeProvisioningParameters</code> operation.
         *        </p>
         *        <p>
         *        If no values are specified, the default value is all Regions from the <code>STACKSET</code>
         *        constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetRegions(Collection<String> stackSetRegions);

        /**
         * <p>
         * One or more Amazon Web Services Regions where the provisioned product will be available.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint. To get
         * the list of Regions in the <code>STACKSET</code> constraint, use the
         * <code>DescribeProvisioningParameters</code> operation.
         * </p>
         * <p>
         * If no values are specified, the default value is all Regions from the <code>STACKSET</code> constraint.
         * </p>
         * 
         * @param stackSetRegions
         *        One or more Amazon Web Services Regions where the provisioned product will be available.</p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        The specified Regions should be within the list of Regions from the <code>STACKSET</code> constraint.
         *        To get the list of Regions in the <code>STACKSET</code> constraint, use the
         *        <code>DescribeProvisioningParameters</code> operation.
         *        </p>
         *        <p>
         *        If no values are specified, the default value is all Regions from the <code>STACKSET</code>
         *        constraint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetRegions(String... stackSetRegions);

        /**
         * <p>
         * The number of accounts, per Region, for which this operation can fail before Service Catalog stops the
         * operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the
         * operation in any subsequent Regions.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
         * <code>StackSetFailureTolerancePercentage</code>, but not both.
         * </p>
         * <p>
         * The default value is <code>0</code> if no value is specified.
         * </p>
         * 
         * @param stackSetFailureToleranceCount
         *        The number of accounts, per Region, for which this operation can fail before Service Catalog stops the
         *        operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the
         *        operation in any subsequent Regions.</p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
         *        <code>StackSetFailureTolerancePercentage</code>, but not both.
         *        </p>
         *        <p>
         *        The default value is <code>0</code> if no value is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetFailureToleranceCount(Integer stackSetFailureToleranceCount);

        /**
         * <p>
         * The percentage of accounts, per Region, for which this stack operation can fail before Service Catalog stops
         * the operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't attempt the
         * operation in any subsequent Regions.
         * </p>
         * <p>
         * When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to the
         * next whole number.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
         * <code>StackSetFailureTolerancePercentage</code>, but not both.
         * </p>
         * 
         * @param stackSetFailureTolerancePercentage
         *        The percentage of accounts, per Region, for which this stack operation can fail before Service Catalog
         *        stops the operation in that Region. If the operation is stopped in a Region, Service Catalog doesn't
         *        attempt the operation in any subsequent Regions.</p>
         *        <p>
         *        When calculating the number of accounts based on the specified percentage, Service Catalog rounds down
         *        to the next whole number.
         *        </p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>StackSetFailureToleranceCount</code> or
         *        <code>StackSetFailureTolerancePercentage</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetFailureTolerancePercentage(Integer stackSetFailureTolerancePercentage);

        /**
         * <p>
         * The maximum number of accounts in which to perform this operation at one time. This is dependent on the value
         * of <code>StackSetFailureToleranceCount</code>. <code>StackSetMaxConcurrentCount</code> is at most one more
         * than the <code>StackSetFailureToleranceCount</code>.
         * </p>
         * <p>
         * Note that this setting lets you specify the maximum for operations. For large deployments, under certain
         * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
         * <code>StackSetMaxConcurrentPercentage</code>, but not both.
         * </p>
         * 
         * @param stackSetMaxConcurrencyCount
         *        The maximum number of accounts in which to perform this operation at one time. This is dependent on
         *        the value of <code>StackSetFailureToleranceCount</code>. <code>StackSetMaxConcurrentCount</code> is at
         *        most one more than the <code>StackSetFailureToleranceCount</code>.</p>
         *        <p>
         *        Note that this setting lets you specify the maximum for operations. For large deployments, under
         *        certain circumstances the actual number of accounts acted upon concurrently may be lower due to
         *        service throttling.
         *        </p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
         *        <code>StackSetMaxConcurrentPercentage</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetMaxConcurrencyCount(Integer stackSetMaxConcurrencyCount);

        /**
         * <p>
         * The maximum percentage of accounts in which to perform this operation at one time.
         * </p>
         * <p>
         * When calculating the number of accounts based on the specified percentage, Service Catalog rounds down to the
         * next whole number. This is true except in cases where rounding down would result is zero. In this case,
         * Service Catalog sets the number as <code>1</code> instead.
         * </p>
         * <p>
         * Note that this setting lets you specify the maximum for operations. For large deployments, under certain
         * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
         * </p>
         * <p>
         * Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         * </p>
         * <p>
         * Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
         * <code>StackSetMaxConcurrentPercentage</code>, but not both.
         * </p>
         * 
         * @param stackSetMaxConcurrencyPercentage
         *        The maximum percentage of accounts in which to perform this operation at one time.</p>
         *        <p>
         *        When calculating the number of accounts based on the specified percentage, Service Catalog rounds down
         *        to the next whole number. This is true except in cases where rounding down would result is zero. In
         *        this case, Service Catalog sets the number as <code>1</code> instead.
         *        </p>
         *        <p>
         *        Note that this setting lets you specify the maximum for operations. For large deployments, under
         *        certain circumstances the actual number of accounts acted upon concurrently may be lower due to
         *        service throttling.
         *        </p>
         *        <p>
         *        Applicable only to a <code>CFN_STACKSET</code> provisioned product type.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>StackSetMaxConcurrentCount</code> or
         *        <code>StackSetMaxConcurrentPercentage</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetMaxConcurrencyPercentage(Integer stackSetMaxConcurrencyPercentage);
    }

    static final class BuilderImpl implements Builder {
        private List<String> stackSetAccounts = DefaultSdkAutoConstructList.getInstance();

        private List<String> stackSetRegions = DefaultSdkAutoConstructList.getInstance();

        private Integer stackSetFailureToleranceCount;

        private Integer stackSetFailureTolerancePercentage;

        private Integer stackSetMaxConcurrencyCount;

        private Integer stackSetMaxConcurrencyPercentage;

        private BuilderImpl() {
        }

        private BuilderImpl(ProvisioningPreferences model) {
            stackSetAccounts(model.stackSetAccounts);
            stackSetRegions(model.stackSetRegions);
            stackSetFailureToleranceCount(model.stackSetFailureToleranceCount);
            stackSetFailureTolerancePercentage(model.stackSetFailureTolerancePercentage);
            stackSetMaxConcurrencyCount(model.stackSetMaxConcurrencyCount);
            stackSetMaxConcurrencyPercentage(model.stackSetMaxConcurrencyPercentage);
        }

        public final Collection<String> getStackSetAccounts() {
            if (stackSetAccounts instanceof SdkAutoConstructList) {
                return null;
            }
            return stackSetAccounts;
        }

        public final void setStackSetAccounts(Collection<String> stackSetAccounts) {
            this.stackSetAccounts = StackSetAccountsCopier.copy(stackSetAccounts);
        }

        @Override
        public final Builder stackSetAccounts(Collection<String> stackSetAccounts) {
            this.stackSetAccounts = StackSetAccountsCopier.copy(stackSetAccounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stackSetAccounts(String... stackSetAccounts) {
            stackSetAccounts(Arrays.asList(stackSetAccounts));
            return this;
        }

        public final Collection<String> getStackSetRegions() {
            if (stackSetRegions instanceof SdkAutoConstructList) {
                return null;
            }
            return stackSetRegions;
        }

        public final void setStackSetRegions(Collection<String> stackSetRegions) {
            this.stackSetRegions = StackSetRegionsCopier.copy(stackSetRegions);
        }

        @Override
        public final Builder stackSetRegions(Collection<String> stackSetRegions) {
            this.stackSetRegions = StackSetRegionsCopier.copy(stackSetRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stackSetRegions(String... stackSetRegions) {
            stackSetRegions(Arrays.asList(stackSetRegions));
            return this;
        }

        public final Integer getStackSetFailureToleranceCount() {
            return stackSetFailureToleranceCount;
        }

        public final void setStackSetFailureToleranceCount(Integer stackSetFailureToleranceCount) {
            this.stackSetFailureToleranceCount = stackSetFailureToleranceCount;
        }

        @Override
        public final Builder stackSetFailureToleranceCount(Integer stackSetFailureToleranceCount) {
            this.stackSetFailureToleranceCount = stackSetFailureToleranceCount;
            return this;
        }

        public final Integer getStackSetFailureTolerancePercentage() {
            return stackSetFailureTolerancePercentage;
        }

        public final void setStackSetFailureTolerancePercentage(Integer stackSetFailureTolerancePercentage) {
            this.stackSetFailureTolerancePercentage = stackSetFailureTolerancePercentage;
        }

        @Override
        public final Builder stackSetFailureTolerancePercentage(Integer stackSetFailureTolerancePercentage) {
            this.stackSetFailureTolerancePercentage = stackSetFailureTolerancePercentage;
            return this;
        }

        public final Integer getStackSetMaxConcurrencyCount() {
            return stackSetMaxConcurrencyCount;
        }

        public final void setStackSetMaxConcurrencyCount(Integer stackSetMaxConcurrencyCount) {
            this.stackSetMaxConcurrencyCount = stackSetMaxConcurrencyCount;
        }

        @Override
        public final Builder stackSetMaxConcurrencyCount(Integer stackSetMaxConcurrencyCount) {
            this.stackSetMaxConcurrencyCount = stackSetMaxConcurrencyCount;
            return this;
        }

        public final Integer getStackSetMaxConcurrencyPercentage() {
            return stackSetMaxConcurrencyPercentage;
        }

        public final void setStackSetMaxConcurrencyPercentage(Integer stackSetMaxConcurrencyPercentage) {
            this.stackSetMaxConcurrencyPercentage = stackSetMaxConcurrencyPercentage;
        }

        @Override
        public final Builder stackSetMaxConcurrencyPercentage(Integer stackSetMaxConcurrencyPercentage) {
            this.stackSetMaxConcurrencyPercentage = stackSetMaxConcurrencyPercentage;
            return this;
        }

        @Override
        public ProvisioningPreferences build() {
            return new ProvisioningPreferences(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
