/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicecatalogappregistry;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicecatalogappregistry.model.AssociateAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.AssociateAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.AssociateResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.AssociateResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ConflictException;
import software.amazon.awssdk.services.servicecatalogappregistry.model.CreateApplicationRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.CreateApplicationResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.CreateAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.CreateAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DeleteAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DeleteAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DisassociateAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DisassociateAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DisassociateResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.DisassociateResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetApplicationRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetApplicationResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetAssociatedResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetAssociatedResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.GetAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.InternalServerException;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListApplicationsRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListApplicationsResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedAttributeGroupsRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedAttributeGroupsResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedResourcesRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedResourcesResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsForApplicationRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsForApplicationResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ResourceNotFoundException;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ServiceCatalogAppRegistryException;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ServiceCatalogAppRegistryRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.servicecatalogappregistry.model.SyncResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.SyncResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.TagResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.TagResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UntagResourceRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UntagResourceResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UpdateAttributeGroupRequest;
import software.amazon.awssdk.services.servicecatalogappregistry.model.UpdateAttributeGroupResponse;
import software.amazon.awssdk.services.servicecatalogappregistry.model.ValidationException;
import software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListApplicationsIterable;
import software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedAttributeGroupsIterable;
import software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedResourcesIterable;
import software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsForApplicationIterable;
import software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsIterable;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.AssociateAttributeGroupRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.AssociateResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.CreateAttributeGroupRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.DeleteAttributeGroupRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.DisassociateAttributeGroupRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.DisassociateResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.GetAssociatedResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.GetAttributeGroupRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListAssociatedAttributeGroupsRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListAssociatedResourcesRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListAttributeGroupsForApplicationRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListAttributeGroupsRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.SyncResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.servicecatalogappregistry.transform.UpdateAttributeGroupRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ServiceCatalogAppRegistryClient}.
 *
 * @see ServiceCatalogAppRegistryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceCatalogAppRegistryClient implements ServiceCatalogAppRegistryClient {
    private static final Logger log = Logger.loggerFor(DefaultServiceCatalogAppRegistryClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceCatalogAppRegistryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates an attribute group with an application to augment the application's metadata with the group's
     * attributes. This feature enables applications to be described with user-defined details that are
     * machine-readable, such as third-party integrations.
     * </p>
     *
     * @param associateAttributeGroupRequest
     * @return Result of the AssociateAttributeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ServiceQuotaExceededException
     *         The maximum number of resources per account has been reached.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.AssociateAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/AssociateAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateAttributeGroupResponse associateAttributeGroup(AssociateAttributeGroupRequest associateAttributeGroupRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateAttributeGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateAttributeGroupRequest, AssociateAttributeGroupResponse>()
                            .withOperationName("AssociateAttributeGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateAttributeGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a resource with an application. Both the resource and the application can be specified either by ID or
     * name.
     * </p>
     *
     * @param associateResourceRequest
     * @return Result of the AssociateResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ServiceQuotaExceededException
     *         The maximum number of resources per account has been reached.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.AssociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/AssociateResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResourceResponse associateResource(AssociateResourceRequest associateResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ServiceQuotaExceededException, ConflictException,
            ValidationException, AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResource");

            return clientHandler.execute(new ClientExecutionParams<AssociateResourceRequest, AssociateResourceResponse>()
                    .withOperationName("AssociateResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new application that is the top-level node in a hierarchy of related cloud resource abstractions.
     * </p>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The maximum number of resources per account has been reached.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws ServiceQuotaExceededException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");

            return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                    .withOperationName("CreateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new attribute group as a container for user-defined attributes. This feature enables users to have full
     * control over their cloud application's metadata in a rich machine-readable format to facilitate integration with
     * automated workflows and third-party tools.
     * </p>
     *
     * @param createAttributeGroupRequest
     * @return Result of the CreateAttributeGroup operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The maximum number of resources per account has been reached.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.CreateAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/CreateAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAttributeGroupResponse createAttributeGroup(CreateAttributeGroupRequest createAttributeGroupRequest)
            throws ServiceQuotaExceededException, ConflictException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAttributeGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateAttributeGroupRequest, CreateAttributeGroupResponse>()
                    .withOperationName("CreateAttributeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAttributeGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an application that is specified either by its application ID or name. All associated attribute groups
     * and resources must be disassociated from it before deleting an application.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return Result of the DeleteApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteApplicationResponse deleteApplication(DeleteApplicationRequest deleteApplicationRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");

            return clientHandler.execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                    .withOperationName("DeleteApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an attribute group, specified either by its attribute group ID or name.
     * </p>
     *
     * @param deleteAttributeGroupRequest
     * @return Result of the DeleteAttributeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.DeleteAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/DeleteAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAttributeGroupResponse deleteAttributeGroup(DeleteAttributeGroupRequest deleteAttributeGroupRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAttributeGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteAttributeGroupRequest, DeleteAttributeGroupResponse>()
                    .withOperationName("DeleteAttributeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAttributeGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates an attribute group from an application to remove the extra attributes contained in the attribute
     * group from the application's metadata. This operation reverts <code>AssociateAttributeGroup</code>.
     * </p>
     *
     * @param disassociateAttributeGroupRequest
     * @return Result of the DisassociateAttributeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.DisassociateAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/DisassociateAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateAttributeGroupResponse disassociateAttributeGroup(
            DisassociateAttributeGroupRequest disassociateAttributeGroupRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateAttributeGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateAttributeGroupRequest, DisassociateAttributeGroupResponse>()
                            .withOperationName("DisassociateAttributeGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateAttributeGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a resource from application. Both the resource and the application can be specified either by ID or
     * name.
     * </p>
     *
     * @param disassociateResourceRequest
     * @return Result of the DisassociateResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.DisassociateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/DisassociateResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResourceResponse disassociateResource(DisassociateResourceRequest disassociateResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResource");

            return clientHandler.execute(new ClientExecutionParams<DisassociateResourceRequest, DisassociateResourceResponse>()
                    .withOperationName("DisassociateResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves metadata information about one of your applications. The application can be specified either by its
     * unique ID or by its name (which is unique within one account in one region at a given point in time). Specify by
     * ID in automated workflows if you want to make sure that the exact same application is returned or a
     * <code>ResourceNotFoundException</code> is thrown, avoiding the ABA addressing problem.
     * </p>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/GetApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                    .withOperationName("GetApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the resource associated with the application.
     * </p>
     *
     * @param getAssociatedResourceRequest
     * @return Result of the GetAssociatedResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.GetAssociatedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/GetAssociatedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAssociatedResourceResponse getAssociatedResource(GetAssociatedResourceRequest getAssociatedResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssociatedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAssociatedResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssociatedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssociatedResource");

            return clientHandler.execute(new ClientExecutionParams<GetAssociatedResourceRequest, GetAssociatedResourceResponse>()
                    .withOperationName("GetAssociatedResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAssociatedResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAssociatedResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves an attribute group, either by its name or its ID. The attribute group can be specified either by its
     * unique ID or by its name.
     * </p>
     *
     * @param getAttributeGroupRequest
     * @return Result of the GetAttributeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.GetAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/GetAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAttributeGroupResponse getAttributeGroup(GetAttributeGroupRequest getAttributeGroupRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttributeGroup");

            return clientHandler.execute(new ClientExecutionParams<GetAttributeGroupRequest, GetAttributeGroupResponse>()
                    .withOperationName("GetAttributeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAttributeGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all of your applications. Results are paginated.
     * </p>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all of your applications. Results are paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.servicecatalogappregistry.model.ListApplicationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListApplicationsIterable responses = client
     *             .listApplicationsPaginator(request);
     *     for (software.amazon.awssdk.services.servicecatalogappregistry.model.ListApplicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.servicecatalogappregistry.model.ListApplicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListApplicationsIterable listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        return new ListApplicationsIterable(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p>
     * Lists all attribute groups that are associated with specified application. Results are paginated.
     * </p>
     *
     * @param listAssociatedAttributeGroupsRequest
     * @return Result of the ListAssociatedAttributeGroups operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAssociatedAttributeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAssociatedAttributeGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedAttributeGroupsResponse listAssociatedAttributeGroups(
            ListAssociatedAttributeGroupsRequest listAssociatedAttributeGroupsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssociatedAttributeGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssociatedAttributeGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociatedAttributeGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedAttributeGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedAttributeGroupsRequest, ListAssociatedAttributeGroupsResponse>()
                            .withOperationName("ListAssociatedAttributeGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAssociatedAttributeGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAssociatedAttributeGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all attribute groups that are associated with specified application. Results are paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAssociatedAttributeGroups(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedAttributeGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedAttributeGroupsIterable responses = client.listAssociatedAttributeGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedAttributeGroupsIterable responses = client
     *             .listAssociatedAttributeGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedAttributeGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedAttributeGroupsIterable responses = client.listAssociatedAttributeGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssociatedAttributeGroups(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedAttributeGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAssociatedAttributeGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAssociatedAttributeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAssociatedAttributeGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedAttributeGroupsIterable listAssociatedAttributeGroupsPaginator(
            ListAssociatedAttributeGroupsRequest listAssociatedAttributeGroupsRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        return new ListAssociatedAttributeGroupsIterable(this, applyPaginatorUserAgent(listAssociatedAttributeGroupsRequest));
    }

    /**
     * <p>
     * Lists all resources that are associated with specified application. Results are paginated.
     * </p>
     *
     * @param listAssociatedResourcesRequest
     * @return Result of the ListAssociatedResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAssociatedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAssociatedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedResourcesResponse listAssociatedResources(ListAssociatedResourcesRequest listAssociatedResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAssociatedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAssociatedResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssociatedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedResourcesRequest, ListAssociatedResourcesResponse>()
                            .withOperationName("ListAssociatedResources").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAssociatedResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAssociatedResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all resources that are associated with specified application. Results are paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAssociatedResources(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedResourcesIterable responses = client.listAssociatedResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedResourcesIterable responses = client
     *             .listAssociatedResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAssociatedResourcesIterable responses = client.listAssociatedResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAssociatedResources(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAssociatedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAssociatedResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAssociatedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAssociatedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAssociatedResourcesIterable listAssociatedResourcesPaginator(
            ListAssociatedResourcesRequest listAssociatedResourcesRequest) throws ResourceNotFoundException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        return new ListAssociatedResourcesIterable(this, applyPaginatorUserAgent(listAssociatedResourcesRequest));
    }

    /**
     * <p>
     * Lists all attribute groups which you have access to. Results are paginated.
     * </p>
     *
     * @param listAttributeGroupsRequest
     * @return Result of the ListAttributeGroups operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAttributeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAttributeGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttributeGroupsResponse listAttributeGroups(ListAttributeGroupsRequest listAttributeGroupsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttributeGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAttributeGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttributeGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttributeGroups");

            return clientHandler.execute(new ClientExecutionParams<ListAttributeGroupsRequest, ListAttributeGroupsResponse>()
                    .withOperationName("ListAttributeGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAttributeGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAttributeGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all attribute groups which you have access to. Results are paginated.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttributeGroups(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsIterable responses = client.listAttributeGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsIterable responses = client
     *             .listAttributeGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsIterable responses = client.listAttributeGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttributeGroups(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAttributeGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAttributeGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAttributeGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttributeGroupsIterable listAttributeGroupsPaginator(ListAttributeGroupsRequest listAttributeGroupsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        return new ListAttributeGroupsIterable(this, applyPaginatorUserAgent(listAttributeGroupsRequest));
    }

    /**
     * <p>
     * Lists the details of all attribute groups associated with a specific application. The results display in pages.
     * </p>
     *
     * @param listAttributeGroupsForApplicationRequest
     * @return Result of the ListAttributeGroupsForApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAttributeGroupsForApplication
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAttributeGroupsForApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttributeGroupsForApplicationResponse listAttributeGroupsForApplication(
            ListAttributeGroupsForApplicationRequest listAttributeGroupsForApplicationRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttributeGroupsForApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAttributeGroupsForApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAttributeGroupsForApplicationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttributeGroupsForApplication");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAttributeGroupsForApplicationRequest, ListAttributeGroupsForApplicationResponse>()
                            .withOperationName("ListAttributeGroupsForApplication").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listAttributeGroupsForApplicationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAttributeGroupsForApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the details of all attribute groups associated with a specific application. The results display in pages.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttributeGroupsForApplication(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsForApplicationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsForApplicationIterable responses = client.listAttributeGroupsForApplicationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsForApplicationIterable responses = client
     *             .listAttributeGroupsForApplicationPaginator(request);
     *     for (software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsForApplicationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicecatalogappregistry.paginators.ListAttributeGroupsForApplicationIterable responses = client.listAttributeGroupsForApplicationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttributeGroupsForApplication(software.amazon.awssdk.services.servicecatalogappregistry.model.ListAttributeGroupsForApplicationRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAttributeGroupsForApplicationRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListAttributeGroupsForApplication
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListAttributeGroupsForApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttributeGroupsForApplicationIterable listAttributeGroupsForApplicationPaginator(
            ListAttributeGroupsForApplicationRequest listAttributeGroupsForApplicationRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        return new ListAttributeGroupsForApplicationIterable(this,
                applyPaginatorUserAgent(listAttributeGroupsForApplicationRequest));
    }

    /**
     * <p>
     * Lists all of the tags on the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Syncs the resource with current AppRegistry records.
     * </p>
     * <p>
     * Specifically, the resource’s AppRegistry system tags sync with its associated application. We remove the
     * resource's AppRegistry system tags if it does not associate with the application. The caller must have
     * permissions to read and update the resource.
     * </p>
     *
     * @param syncResourceRequest
     * @return Result of the SyncResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.SyncResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/SyncResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SyncResourceResponse syncResource(SyncResourceRequest syncResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SyncResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SyncResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, syncResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SyncResource");

            return clientHandler.execute(new ClientExecutionParams<SyncResourceRequest, SyncResourceResponse>()
                    .withOperationName("SyncResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(syncResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SyncResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     * <p>
     * Each tag consists of a key and an optional value. If a tag with the same key is already associated with the
     * resource, this action updates its value.
     * </p>
     * <p>
     * This operation returns an empty response if the call was successful.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a resource.
     * </p>
     * <p>
     * This operation returns an empty response if the call was successful.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing application with new attributes.
     * </p>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws ResourceNotFoundException, ConflictException, InternalServerException, ValidationException,
            AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");

            return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                    .withOperationName("UpdateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing attribute group with new details.
     * </p>
     *
     * @param updateAttributeGroupRequest
     * @return Result of the UpdateAttributeGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a resource with the given name already
     *         exists within the account).
     * @throws InternalServerException
     *         The service is experiencing internal problems.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceCatalogAppRegistryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceCatalogAppRegistryClient.UpdateAttributeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWS242AppRegistry-2020-06-24/UpdateAttributeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAttributeGroupResponse updateAttributeGroup(UpdateAttributeGroupRequest updateAttributeGroupRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, ServiceCatalogAppRegistryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAttributeGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAttributeGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAttributeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Catalog AppRegistry");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAttributeGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateAttributeGroupRequest, UpdateAttributeGroupResponse>()
                    .withOperationName("UpdateAttributeGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAttributeGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAttributeGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceCatalogAppRegistryException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ServiceCatalogAppRegistryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
