/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.servicediscovery.transform.DnsConfigMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the resource record sets that you want Amazon Route 53 to create when
 * you register an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DnsConfig implements StructuredPojo, ToCopyableBuilder<DnsConfig.Builder, DnsConfig> {
    private final String namespaceId;

    private final List<DnsRecord> dnsRecords;

    private DnsConfig(BuilderImpl builder) {
        this.namespaceId = builder.namespaceId;
        this.dnsRecords = builder.dnsRecords;
    }

    /**
     * <p>
     * The ID of the namespace to use for DNS configuration.
     * </p>
     * 
     * @return The ID of the namespace to use for DNS configuration.
     */
    public String namespaceId() {
        return namespaceId;
    }

    /**
     * <p>
     * An array that contains one <code>DnsRecord</code> object for each resource record set that you want Amazon Route
     * 53 to create when you register an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array that contains one <code>DnsRecord</code> object for each resource record set that you want
     *         Amazon Route 53 to create when you register an instance.
     */
    public List<DnsRecord> dnsRecords() {
        return dnsRecords;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespaceId());
        hashCode = 31 * hashCode + Objects.hashCode(dnsRecords());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DnsConfig)) {
            return false;
        }
        DnsConfig other = (DnsConfig) obj;
        return Objects.equals(namespaceId(), other.namespaceId()) && Objects.equals(dnsRecords(), other.dnsRecords());
    }

    @Override
    public String toString() {
        return ToString.builder("DnsConfig").add("NamespaceId", namespaceId()).add("DnsRecords", dnsRecords()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NamespaceId":
            return Optional.ofNullable(clazz.cast(namespaceId()));
        case "DnsRecords":
            return Optional.ofNullable(clazz.cast(dnsRecords()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DnsConfigMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DnsConfig> {
        /**
         * <p>
         * The ID of the namespace to use for DNS configuration.
         * </p>
         * 
         * @param namespaceId
         *        The ID of the namespace to use for DNS configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaceId(String namespaceId);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each resource record set that you want Amazon
         * Route 53 to create when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each resource record set that you want
         *        Amazon Route 53 to create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(Collection<DnsRecord> dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each resource record set that you want Amazon
         * Route 53 to create when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each resource record set that you want
         *        Amazon Route 53 to create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(DnsRecord... dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each resource record set that you want Amazon
         * Route 53 to create when you register an instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DnsRecord>.Builder} avoiding the need to
         * create one manually via {@link List<DnsRecord>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DnsRecord>.Builder#build()} is called immediately and its
         * result is passed to {@link #dnsRecords(List<DnsRecord>)}.
         * 
         * @param dnsRecords
         *        a consumer that will call methods on {@link List<DnsRecord>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dnsRecords(List<DnsRecord>)
         */
        Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords);
    }

    static final class BuilderImpl implements Builder {
        private String namespaceId;

        private List<DnsRecord> dnsRecords = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DnsConfig model) {
            namespaceId(model.namespaceId);
            dnsRecords(model.dnsRecords);
        }

        public final String getNamespaceId() {
            return namespaceId;
        }

        @Override
        public final Builder namespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
            return this;
        }

        public final void setNamespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
        }

        public final Collection<DnsRecord.Builder> getDnsRecords() {
            return dnsRecords != null ? dnsRecords.stream().map(DnsRecord::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dnsRecords(Collection<DnsRecord> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copy(dnsRecords);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(DnsRecord... dnsRecords) {
            dnsRecords(Arrays.asList(dnsRecords));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords) {
            dnsRecords(Stream.of(dnsRecords).map(c -> DnsRecord.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDnsRecords(Collection<DnsRecord.BuilderImpl> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copyFromBuilder(dnsRecords);
        }

        @Override
        public DnsConfig build() {
            return new DnsConfig(this);
        }
    }
}
