/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.servicediscovery.transform.DnsConfigMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the records that you want Amazon Route 53 to create when you register
 * an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DnsConfig implements StructuredPojo, ToCopyableBuilder<DnsConfig.Builder, DnsConfig> {
    private final String namespaceId;

    private final String routingPolicy;

    private final List<DnsRecord> dnsRecords;

    private DnsConfig(BuilderImpl builder) {
        this.namespaceId = builder.namespaceId;
        this.routingPolicy = builder.routingPolicy;
        this.dnsRecords = builder.dnsRecords;
    }

    /**
     * <p>
     * The ID of the namespace to use for DNS configuration.
     * </p>
     * 
     * @return The ID of the namespace to use for DNS configuration.
     */
    public String namespaceId() {
        return namespaceId;
    }

    /**
     * <p>
     * The routing policy that you want to apply to all records that Route 53 creates when you register an instance and
     * specify this service.
     * </p>
     * <note>
     * <p>
     * If you want to use this service to register instances that create alias records, specify <code>WEIGHTED</code>
     * for the routing policy.
     * </p>
     * </note>
     * <p>
     * You can specify the following values:
     * </p>
     * <p>
     * <b>MULTIVALUE</b>
     * </p>
     * <p>
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
     * value for up to eight instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
     * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP addresses
     * for all of the healthy instances.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the values for up to eight instances.
     * </p>
     * <p>
     * For more information about the multivalue routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * <b>WEIGHTED</b>
     * </p>
     * <p>
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service. Currently, all records have the same weight, so you can't route more or less
     * traffic to any instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly selected
     * instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS queries as if
     * all of the instances were healthy.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the applicable value for one randomly selected instance.
     * </p>
     * <p>
     * For more information about the weighted routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingPolicy}
     * will return {@link RoutingPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingPolicyAsString}.
     * </p>
     * 
     * @return The routing policy that you want to apply to all records that Route 53 creates when you register an
     *         instance and specify this service.</p> <note>
     *         <p>
     *         If you want to use this service to register instances that create alias records, specify
     *         <code>WEIGHTED</code> for the routing policy.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify the following values:
     *         </p>
     *         <p>
     *         <b>MULTIVALUE</b>
     *         </p>
     *         <p>
     *         If you define a health check for the service and the health check is healthy, Route 53 returns the
     *         applicable value for up to eight instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight
     *         healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with
     *         the IP addresses for all of the healthy instances.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the values for up to eight instances.
     *         </p>
     *         <p>
     *         For more information about the multivalue routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     *         >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     *         </p>
     *         <p>
     *         <b>WEIGHTED</b>
     *         </p>
     *         <p>
     *         Route 53 returns the applicable value from one randomly selected instance from among the instances that
     *         you registered using the same service. Currently, all records have the same weight, so you can't route
     *         more or less traffic to any instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
     *         randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
     *         responds to DNS queries as if all of the instances were healthy.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the applicable value for one randomly selected instance.
     *         </p>
     *         <p>
     *         For more information about the weighted routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     *         >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * @see RoutingPolicy
     */
    public RoutingPolicy routingPolicy() {
        return RoutingPolicy.fromValue(routingPolicy);
    }

    /**
     * <p>
     * The routing policy that you want to apply to all records that Route 53 creates when you register an instance and
     * specify this service.
     * </p>
     * <note>
     * <p>
     * If you want to use this service to register instances that create alias records, specify <code>WEIGHTED</code>
     * for the routing policy.
     * </p>
     * </note>
     * <p>
     * You can specify the following values:
     * </p>
     * <p>
     * <b>MULTIVALUE</b>
     * </p>
     * <p>
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
     * value for up to eight instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
     * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP addresses
     * for all of the healthy instances.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the values for up to eight instances.
     * </p>
     * <p>
     * For more information about the multivalue routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * <b>WEIGHTED</b>
     * </p>
     * <p>
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service. Currently, all records have the same weight, so you can't route more or less
     * traffic to any instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly selected
     * instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS queries as if
     * all of the instances were healthy.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the applicable value for one randomly selected instance.
     * </p>
     * <p>
     * For more information about the weighted routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingPolicy}
     * will return {@link RoutingPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingPolicyAsString}.
     * </p>
     * 
     * @return The routing policy that you want to apply to all records that Route 53 creates when you register an
     *         instance and specify this service.</p> <note>
     *         <p>
     *         If you want to use this service to register instances that create alias records, specify
     *         <code>WEIGHTED</code> for the routing policy.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify the following values:
     *         </p>
     *         <p>
     *         <b>MULTIVALUE</b>
     *         </p>
     *         <p>
     *         If you define a health check for the service and the health check is healthy, Route 53 returns the
     *         applicable value for up to eight instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight
     *         healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with
     *         the IP addresses for all of the healthy instances.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the values for up to eight instances.
     *         </p>
     *         <p>
     *         For more information about the multivalue routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     *         >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     *         </p>
     *         <p>
     *         <b>WEIGHTED</b>
     *         </p>
     *         <p>
     *         Route 53 returns the applicable value from one randomly selected instance from among the instances that
     *         you registered using the same service. Currently, all records have the same weight, so you can't route
     *         more or less traffic to any instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
     *         randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
     *         responds to DNS queries as if all of the instances were healthy.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the applicable value for one randomly selected instance.
     *         </p>
     *         <p>
     *         For more information about the weighted routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     *         >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * @see RoutingPolicy
     */
    public String routingPolicyAsString() {
        return routingPolicy;
    }

    /**
     * <p>
     * An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to create when
     * you register an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to create
     *         when you register an instance.
     */
    public List<DnsRecord> dnsRecords() {
        return dnsRecords;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespaceId());
        hashCode = 31 * hashCode + Objects.hashCode(routingPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dnsRecords());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DnsConfig)) {
            return false;
        }
        DnsConfig other = (DnsConfig) obj;
        return Objects.equals(namespaceId(), other.namespaceId())
                && Objects.equals(routingPolicyAsString(), other.routingPolicyAsString())
                && Objects.equals(dnsRecords(), other.dnsRecords());
    }

    @Override
    public String toString() {
        return ToString.builder("DnsConfig").add("NamespaceId", namespaceId()).add("RoutingPolicy", routingPolicyAsString())
                .add("DnsRecords", dnsRecords()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NamespaceId":
            return Optional.ofNullable(clazz.cast(namespaceId()));
        case "RoutingPolicy":
            return Optional.ofNullable(clazz.cast(routingPolicyAsString()));
        case "DnsRecords":
            return Optional.ofNullable(clazz.cast(dnsRecords()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DnsConfigMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DnsConfig> {
        /**
         * <p>
         * The ID of the namespace to use for DNS configuration.
         * </p>
         * 
         * @param namespaceId
         *        The ID of the namespace to use for DNS configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaceId(String namespaceId);

        /**
         * <p>
         * The routing policy that you want to apply to all records that Route 53 creates when you register an instance
         * and specify this service.
         * </p>
         * <note>
         * <p>
         * If you want to use this service to register instances that create alias records, specify
         * <code>WEIGHTED</code> for the routing policy.
         * </p>
         * </note>
         * <p>
         * You can specify the following values:
         * </p>
         * <p>
         * <b>MULTIVALUE</b>
         * </p>
         * <p>
         * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
         * value for up to eight instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
         * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP
         * addresses for all of the healthy instances.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the values for up to eight instances.
         * </p>
         * <p>
         * For more information about the multivalue routing policy, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * <p>
         * <b>WEIGHTED</b>
         * </p>
         * <p>
         * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
         * registered using the same service. Currently, all records have the same weight, so you can't route more or
         * less traffic to any instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly
         * selected instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS
         * queries as if all of the instances were healthy.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the applicable value for one randomly selected instance.
         * </p>
         * <p>
         * For more information about the weighted routing policy, see <a
         * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * 
         * @param routingPolicy
         *        The routing policy that you want to apply to all records that Route 53 creates when you register an
         *        instance and specify this service.</p> <note>
         *        <p>
         *        If you want to use this service to register instances that create alias records, specify
         *        <code>WEIGHTED</code> for the routing policy.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify the following values:
         *        </p>
         *        <p>
         *        <b>MULTIVALUE</b>
         *        </p>
         *        <p>
         *        If you define a health check for the service and the health check is healthy, Route 53 returns the
         *        applicable value for up to eight instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to
         *        eight healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS
         *        query with the IP addresses for all of the healthy instances.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the values for up to eight instances.
         *        </p>
         *        <p>
         *        For more information about the multivalue routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         *        >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         *        </p>
         *        <p>
         *        <b>WEIGHTED</b>
         *        </p>
         *        <p>
         *        Route 53 returns the applicable value from one randomly selected instance from among the instances
         *        that you registered using the same service. Currently, all records have the same weight, so you can't
         *        route more or less traffic to any instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
         *        randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
         *        responds to DNS queries as if all of the instances were healthy.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the applicable value for one randomly selected instance.
         *        </p>
         *        <p>
         *        For more information about the weighted routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         *        >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * @see RoutingPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingPolicy
         */
        Builder routingPolicy(String routingPolicy);

        /**
         * <p>
         * The routing policy that you want to apply to all records that Route 53 creates when you register an instance
         * and specify this service.
         * </p>
         * <note>
         * <p>
         * If you want to use this service to register instances that create alias records, specify
         * <code>WEIGHTED</code> for the routing policy.
         * </p>
         * </note>
         * <p>
         * You can specify the following values:
         * </p>
         * <p>
         * <b>MULTIVALUE</b>
         * </p>
         * <p>
         * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
         * value for up to eight instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
         * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP
         * addresses for all of the healthy instances.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the values for up to eight instances.
         * </p>
         * <p>
         * For more information about the multivalue routing policy, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * <p>
         * <b>WEIGHTED</b>
         * </p>
         * <p>
         * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
         * registered using the same service. Currently, all records have the same weight, so you can't route more or
         * less traffic to any instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly
         * selected instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS
         * queries as if all of the instances were healthy.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the applicable value for one randomly selected instance.
         * </p>
         * <p>
         * For more information about the weighted routing policy, see <a
         * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * 
         * @param routingPolicy
         *        The routing policy that you want to apply to all records that Route 53 creates when you register an
         *        instance and specify this service.</p> <note>
         *        <p>
         *        If you want to use this service to register instances that create alias records, specify
         *        <code>WEIGHTED</code> for the routing policy.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify the following values:
         *        </p>
         *        <p>
         *        <b>MULTIVALUE</b>
         *        </p>
         *        <p>
         *        If you define a health check for the service and the health check is healthy, Route 53 returns the
         *        applicable value for up to eight instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to
         *        eight healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS
         *        query with the IP addresses for all of the healthy instances.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the values for up to eight instances.
         *        </p>
         *        <p>
         *        For more information about the multivalue routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         *        >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         *        </p>
         *        <p>
         *        <b>WEIGHTED</b>
         *        </p>
         *        <p>
         *        Route 53 returns the applicable value from one randomly selected instance from among the instances
         *        that you registered using the same service. Currently, all records have the same weight, so you can't
         *        route more or less traffic to any instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
         *        randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
         *        responds to DNS queries as if all of the instances were healthy.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the applicable value for one randomly selected instance.
         *        </p>
         *        <p>
         *        For more information about the weighted routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         *        >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * @see RoutingPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingPolicy
         */
        Builder routingPolicy(RoutingPolicy routingPolicy);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to create
         * when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to
         *        create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(Collection<DnsRecord> dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to create
         * when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to
         *        create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(DnsRecord... dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each record that you want Route 53 to create
         * when you register an instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DnsRecord>.Builder} avoiding the need to
         * create one manually via {@link List<DnsRecord>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DnsRecord>.Builder#build()} is called immediately and its
         * result is passed to {@link #dnsRecords(List<DnsRecord>)}.
         * 
         * @param dnsRecords
         *        a consumer that will call methods on {@link List<DnsRecord>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dnsRecords(List<DnsRecord>)
         */
        Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords);
    }

    static final class BuilderImpl implements Builder {
        private String namespaceId;

        private String routingPolicy;

        private List<DnsRecord> dnsRecords = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DnsConfig model) {
            namespaceId(model.namespaceId);
            routingPolicy(model.routingPolicy);
            dnsRecords(model.dnsRecords);
        }

        public final String getNamespaceId() {
            return namespaceId;
        }

        @Override
        public final Builder namespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
            return this;
        }

        public final void setNamespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
        }

        public final String getRoutingPolicy() {
            return routingPolicy;
        }

        @Override
        public final Builder routingPolicy(String routingPolicy) {
            this.routingPolicy = routingPolicy;
            return this;
        }

        @Override
        public final Builder routingPolicy(RoutingPolicy routingPolicy) {
            this.routingPolicy(routingPolicy.toString());
            return this;
        }

        public final void setRoutingPolicy(String routingPolicy) {
            this.routingPolicy = routingPolicy;
        }

        public final Collection<DnsRecord.Builder> getDnsRecords() {
            return dnsRecords != null ? dnsRecords.stream().map(DnsRecord::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dnsRecords(Collection<DnsRecord> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copy(dnsRecords);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(DnsRecord... dnsRecords) {
            dnsRecords(Arrays.asList(dnsRecords));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords) {
            dnsRecords(Stream.of(dnsRecords).map(c -> DnsRecord.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDnsRecords(Collection<DnsRecord.BuilderImpl> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copyFromBuilder(dnsRecords);
        }

        @Override
        public DnsConfig build() {
            return new DnsConfig(this);
        }
    }
}
