/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.paginators;

import java.util.Collections;
import java.util.Iterator;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.core.pagination.SyncPageFetcher;
import software.amazon.awssdk.services.servicediscovery.ServiceDiscoveryClient;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.servicediscovery.ServiceDiscoveryClient#listInstancesPaginator(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client
 *             .listInstancesPaginator(request);
 *     for (software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listInstances(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListInstancesIterable implements SdkIterable<ListInstancesResponse> {
    private final ServiceDiscoveryClient client;

    private final ListInstancesRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListInstancesIterable(final ServiceDiscoveryClient client, final ListInstancesRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListInstancesResponseFetcher();
    }

    @Override
    public Iterator<ListInstancesResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListInstancesIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    public final ListInstancesIterable resume(final ListInstancesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListInstancesIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new ListInstancesIterable(client, firstRequest) {
            @Override
            public Iterator<ListInstancesResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListInstancesResponseFetcher implements SyncPageFetcher<ListInstancesResponse> {
        @Override
        public boolean hasNextPage(ListInstancesResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public ListInstancesResponse nextPage(ListInstancesResponse previousPage) {
            if (previousPage == null) {
                return client.listInstances(firstRequest);
            }
            return client.listInstances(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
