/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CustomHealthNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.DuplicateRequestException;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.InstanceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.InvalidInputException;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.OperationNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.ResourceInUseException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryRequest;
import software.amazon.awssdk.services.servicediscovery.model.ServiceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusIterable;
import software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable;
import software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesIterable;
import software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsIterable;
import software.amazon.awssdk.services.servicediscovery.paginators.ListServicesIterable;
import software.amazon.awssdk.services.servicediscovery.transform.CreateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeregisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DiscoverInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstancesHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListNamespacesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.RegisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateInstanceCustomHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateServiceRequestMarshaller;

/**
 * Internal implementation of {@link ServiceDiscoveryClient}.
 *
 * @see ServiceDiscoveryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceDiscoveryClient implements ServiceDiscoveryClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceDiscoveryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an HTTP namespace. Service instances that you register using an HTTP namespace can be discovered using a
     * <code>DiscoverInstances</code> request but can't be discovered using DNS.
     * </p>
     * <p>
     * For the current limit on the number of namespaces that you can create using the same AWS account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map Limits</a> in the
     * <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createHttpNamespaceRequest
     * @return Result of the CreateHttpNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the limit on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreateHttpNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHttpNamespaceResponse createHttpNamespace(CreateHttpNamespaceRequest createHttpNamespaceRequest)
            throws InvalidInputException, NamespaceAlreadyExistsException, ResourceLimitExceededException,
            DuplicateRequestException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHttpNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateHttpNamespaceRequest, CreateHttpNamespaceResponse>()
                .withOperationName("CreateHttpNamespace").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createHttpNamespaceRequest)
                .withMarshaller(new CreateHttpNamespaceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC. The namespace
     * defines your service naming scheme. For example, if you name your namespace <code>example.com</code> and name
     * your service <code>backend</code>, the resulting DNS name for the service will be
     * <code>backend.example.com</code>. For the current limit on the number of namespaces that you can create using the
     * same AWS account, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud
     * Map Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createPrivateDnsNamespaceRequest
     * @return Result of the CreatePrivateDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the limit on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreatePrivateDnsNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePrivateDnsNamespaceResponse createPrivateDnsNamespace(
            CreatePrivateDnsNamespaceRequest createPrivateDnsNamespaceRequest) throws InvalidInputException,
            NamespaceAlreadyExistsException, ResourceLimitExceededException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePrivateDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreatePrivateDnsNamespaceRequest, CreatePrivateDnsNamespaceResponse>()
                        .withOperationName("CreatePrivateDnsNamespace").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createPrivateDnsNamespaceRequest)
                        .withMarshaller(new CreatePrivateDnsNamespaceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a public namespace based on DNS, which will be visible on the internet. The namespace defines your
     * service naming scheme. For example, if you name your namespace <code>example.com</code> and name your service
     * <code>backend</code>, the resulting DNS name for the service will be <code>backend.example.com</code>. For the
     * current limit on the number of namespaces that you can create using the same AWS account, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map Limits</a> in the
     * <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createPublicDnsNamespaceRequest
     * @return Result of the CreatePublicDnsNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceAlreadyExistsException
     *         The namespace that you're trying to create already exists.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the limit on the number of resources.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreatePublicDnsNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePublicDnsNamespaceResponse createPublicDnsNamespace(
            CreatePublicDnsNamespaceRequest createPublicDnsNamespaceRequest) throws InvalidInputException,
            NamespaceAlreadyExistsException, ResourceLimitExceededException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePublicDnsNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreatePublicDnsNamespaceRequest, CreatePublicDnsNamespaceResponse>()
                        .withOperationName("CreatePublicDnsNamespace").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createPublicDnsNamespaceRequest)
                        .withMarshaller(new CreatePublicDnsNamespaceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a service, which defines the configuration for the following entities:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For public and private DNS namespaces, one of the following combinations of DNS records in Amazon Route 53:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A
     * </p>
     * </li>
     * <li>
     * <p>
     * AAAA
     * </p>
     * </li>
     * <li>
     * <p>
     * A and AAAA
     * </p>
     * </li>
     * <li>
     * <p>
     * SRV
     * </p>
     * </li>
     * <li>
     * <p>
     * CNAME
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Optionally, a health check
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you create the service, you can submit a <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_RegisterInstance.html">RegisterInstance</a> request,
     * and AWS Cloud Map uses the values in the configuration to create the specified entities.
     * </p>
     * <p>
     * For the current limit on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map
     * Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createServiceRequest
     * @return Result of the CreateService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the limit on the number of resources.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ServiceAlreadyExistsException
     *         The service can't be created because a service with the same name already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.CreateService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateServiceResponse createService(CreateServiceRequest createServiceRequest) throws InvalidInputException,
            ResourceLimitExceededException, NamespaceNotFoundException, ServiceAlreadyExistsException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                .withOperationName("CreateService").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createServiceRequest)
                .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a namespace from the current account. If the namespace still contains one or more services, the request
     * fails.
     * </p>
     *
     * @param deleteNamespaceRequest
     * @return Result of the DeleteNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeleteNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNamespaceResponse deleteNamespace(DeleteNamespaceRequest deleteNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, ResourceInUseException, DuplicateRequestException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteNamespaceRequest, DeleteNamespaceResponse>()
                .withOperationName("DeleteNamespace").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteNamespaceRequest)
                .withMarshaller(new DeleteNamespaceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a specified service. If the service still contains one or more registered instances, the request fails.
     * </p>
     *
     * @param deleteServiceRequest
     * @return Result of the DeleteService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeleteService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteServiceResponse deleteService(DeleteServiceRequest deleteServiceRequest) throws InvalidInputException,
            ServiceNotFoundException, ResourceInUseException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                .withOperationName("DeleteService").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteServiceRequest)
                .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the Amazon Route 53 DNS records and health check, if any, that AWS Cloud Map created for the specified
     * instance.
     * </p>
     *
     * @param deregisterInstanceRequest
     * @return Result of the DeregisterInstance operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DeregisterInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeregisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterInstanceResponse deregisterInstance(DeregisterInstanceRequest deregisterInstanceRequest)
            throws DuplicateRequestException, InvalidInputException, InstanceNotFoundException, ResourceInUseException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeregisterInstanceRequest, DeregisterInstanceResponse>()
                .withOperationName("DeregisterInstance").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deregisterInstanceRequest)
                .withMarshaller(new DeregisterInstanceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Discovers registered instances for a specified namespace and service. You can use <code>DiscoverInstances</code>
     * to discover instances for any type of namespace. For public and private DNS namespaces, you can also use DNS
     * queries to discover instances.
     * </p>
     *
     * @param discoverInstancesRequest
     * @return Result of the DiscoverInstances operation returned by the service.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.DiscoverInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DiscoverInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DiscoverInstancesResponse discoverInstances(DiscoverInstancesRequest discoverInstancesRequest)
            throws ServiceNotFoundException, NamespaceNotFoundException, InvalidInputException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        String hostPrefix = "data-";
        String resolvedHostExpression = "data-";
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DiscoverInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DiscoverInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DiscoverInstancesRequest, DiscoverInstancesResponse>()
                .withOperationName("DiscoverInstances").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                .withInput(discoverInstancesRequest).withMarshaller(new DiscoverInstancesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about a specified instance.
     * </p>
     *
     * @param getInstanceRequest
     * @return Result of the GetInstance operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetInstanceResponse getInstance(GetInstanceRequest getInstanceRequest) throws InstanceNotFoundException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetInstanceRequest, GetInstanceResponse>()
                .withOperationName("GetInstance").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getInstanceRequest)
                .withMarshaller(new GetInstanceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There is a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note>
     *
     * @param getInstancesHealthStatusRequest
     * @return Result of the GetInstancesHealthStatus operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetInstancesHealthStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInstancesHealthStatusResponse getInstancesHealthStatus(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) throws InstanceNotFoundException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInstancesHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInstancesHealthStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetInstancesHealthStatusRequest, GetInstancesHealthStatusResponse>()
                        .withOperationName("GetInstancesHealthStatus").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getInstancesHealthStatusRequest)
                        .withMarshaller(new GetInstancesHealthStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There is a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getInstancesHealthStatus(software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusIterable responses = client.getInstancesHealthStatusPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusIterable responses = client
     *             .getInstancesHealthStatusPaginator(request);
     *     for (software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusIterable responses = client.getInstancesHealthStatusPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getInstancesHealthStatus(software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getInstancesHealthStatusRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetInstancesHealthStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInstancesHealthStatusIterable getInstancesHealthStatusPaginator(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) throws InstanceNotFoundException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        return new GetInstancesHealthStatusIterable(this, applyPaginatorUserAgent(getInstancesHealthStatusRequest));
    }

    /**
     * <p>
     * Gets information about a namespace.
     * </p>
     *
     * @param getNamespaceRequest
     * @return Result of the GetNamespace operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws NamespaceNotFoundException
     *         No namespace exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetNamespaceResponse getNamespace(GetNamespaceRequest getNamespaceRequest) throws InvalidInputException,
            NamespaceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNamespaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetNamespaceRequest, GetNamespaceResponse>()
                .withOperationName("GetNamespace").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getNamespaceRequest)
                .withMarshaller(new GetNamespaceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about any operation that returns an operation ID in the response, such as a
     * <code>CreateService</code> request.
     * </p>
     * <note>
     * <p>
     * To get a list of operations that match specified criteria, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_ListOperations.html">ListOperations</a>.
     * </p>
     * </note>
     *
     * @param getOperationRequest
     * @return Result of the GetOperation operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws OperationNotFoundException
     *         No operation exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetOperationResponse getOperation(GetOperationRequest getOperationRequest) throws InvalidInputException,
            OperationNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                .withOperationName("GetOperation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getOperationRequest)
                .withMarshaller(new GetOperationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the settings for a specified service.
     * </p>
     *
     * @param getServiceRequest
     * @return Result of the GetService operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.GetService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetServiceResponse getService(GetServiceRequest getServiceRequest) throws InvalidInputException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>()
                .withOperationName("GetService").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getServiceRequest)
                .withMarshaller(new GetServiceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     *
     * @param listInstancesRequest
     * @return Result of the ListInstances operation returned by the service.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInstancesResponse listInstances(ListInstancesRequest listInstancesRequest) throws ServiceNotFoundException,
            InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                .withOperationName("ListInstances").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInstancesRequest)
                .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstances(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client
     *             .listInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesIterable responses = client.listInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstances(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInstancesIterable listInstancesPaginator(ListInstancesRequest listInstancesRequest)
            throws ServiceNotFoundException, InvalidInputException, AwsServiceException, SdkClientException,
            ServiceDiscoveryException {
        return new ListInstancesIterable(this, applyPaginatorUserAgent(listInstancesRequest));
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current AWS account.
     * </p>
     *
     * @param listNamespacesRequest
     * @return Result of the ListNamespaces operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListNamespaces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNamespacesResponse listNamespaces(ListNamespacesRequest listNamespacesRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNamespacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListNamespacesRequest, ListNamespacesResponse>()
                .withOperationName("ListNamespaces").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listNamespacesRequest)
                .withMarshaller(new ListNamespacesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamespaces(software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesIterable responses = client.listNamespacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesIterable responses = client
     *             .listNamespacesPaginator(request);
     *     for (software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesIterable responses = client.listNamespacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamespaces(software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNamespacesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListNamespaces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNamespacesIterable listNamespacesPaginator(ListNamespacesRequest listNamespacesRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        return new ListNamespacesIterable(this, applyPaginatorUserAgent(listNamespacesRequest));
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     *
     * @param listOperationsRequest
     * @return Result of the ListOperations operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                .withOperationName("ListOperations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listOperationsRequest)
                .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOperations(software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsIterable responses = client
     *             .listOperationsPaginator(request);
     *     for (software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsIterable responses = client.listOperationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOperations(software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOperationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListOperationsIterable listOperationsPaginator(ListOperationsRequest listOperationsRequest)
            throws InvalidInputException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        return new ListOperationsIterable(this, applyPaginatorUserAgent(listOperationsRequest));
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                .withOperationName("ListServices").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listServicesRequest)
                .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServices(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesIterable responses = client.listServicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.servicediscovery.paginators.ListServicesIterable responses = client
     *             .listServicesPaginator(request);
     *     for (software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesIterable responses = client.listServicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServices(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)} operation.</b>
     * </p>
     *
     * @param listServicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServicesIterable listServicesPaginator(ListServicesRequest listServicesRequest) throws InvalidInputException,
            AwsServiceException, SdkClientException, ServiceDiscoveryException {
        return new ListServicesIterable(this, applyPaginatorUserAgent(listServicesRequest));
    }

    /**
     * <p>
     * Creates or updates one or more records and, optionally, creates a health check based on the settings in a
     * specified service. When you submit a <code>RegisterInstance</code> request, the following occurs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For each DNS record that you define in the service that is specified by <code>ServiceId</code>, a record is
     * created or updated in the hosted zone that is associated with the corresponding namespace.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the service includes <code>HealthCheckConfig</code>, a health check is created based on the settings in the
     * health check configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * The health check, if any, is associated with each of the new or updated records.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * One <code>RegisterInstance</code> request must complete before you can submit another request and specify the
     * same service ID and instance ID.
     * </p>
     * </important>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html">CreateService</a>.
     * </p>
     * <p>
     * When AWS Cloud Map receives a DNS query for the specified DNS name, it returns the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>If the health check is healthy</b>: returns all the records
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If the health check is unhealthy</b>: returns the applicable value for the last healthy instance
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you didn't specify a health check configuration</b>: returns all the records
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the current limit on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="https://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map
     * Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param registerInstanceRequest
     * @return Result of the RegisterInstance operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ResourceInUseException
     *         The specified resource can't be deleted because it contains other resources. For example, you can't
     *         delete a service that contains any instances.
     * @throws ResourceLimitExceededException
     *         The resource can't be created because you've reached the limit on the number of resources.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.RegisterInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/RegisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterInstanceResponse registerInstance(RegisterInstanceRequest registerInstanceRequest)
            throws DuplicateRequestException, InvalidInputException, ResourceInUseException, ResourceLimitExceededException,
            ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RegisterInstanceRequest, RegisterInstanceResponse>()
                .withOperationName("RegisterInstance").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(registerInstanceRequest)
                .withMarshaller(new RegisterInstanceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Submits a request to change the health status of a custom health check to healthy or unhealthy.
     * </p>
     * <p>
     * You can use <code>UpdateInstanceCustomHealthStatus</code> to change the status only for custom health checks,
     * which you define using <code>HealthCheckCustomConfig</code> when you create a service. You can't use it to change
     * the status for Route 53 health checks, which you define using <code>HealthCheckConfig</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/cloud-map/latest/api/API_HealthCheckCustomConfig.html"
     * >HealthCheckCustomConfig</a>.
     * </p>
     *
     * @param updateInstanceCustomHealthStatusRequest
     * @return Result of the UpdateInstanceCustomHealthStatus operation returned by the service.
     * @throws InstanceNotFoundException
     *         No instance exists with the specified ID, or the instance was recently registered, and information about
     *         the instance hasn't propagated yet.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws CustomHealthNotFoundException
     *         The health check for the instance that is specified by <code>ServiceId</code> and <code>InstanceId</code>
     *         is not a custom health check.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdateInstanceCustomHealthStatus
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateInstanceCustomHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateInstanceCustomHealthStatusResponse updateInstanceCustomHealthStatus(
            UpdateInstanceCustomHealthStatusRequest updateInstanceCustomHealthStatusRequest) throws InstanceNotFoundException,
            ServiceNotFoundException, CustomHealthNotFoundException, InvalidInputException, AwsServiceException,
            SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInstanceCustomHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateInstanceCustomHealthStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateInstanceCustomHealthStatusRequest, UpdateInstanceCustomHealthStatusResponse>()
                        .withOperationName("UpdateInstanceCustomHealthStatus").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateInstanceCustomHealthStatusRequest)
                        .withMarshaller(new UpdateInstanceCustomHealthStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Submits a request to perform the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Update the TTL setting for existing <code>DnsRecords</code> configurations
     * </p>
     * </li>
     * <li>
     * <p>
     * Add, update, or delete <code>HealthCheckConfig</code> for a specified service
     * </p>
     * <note>
     * <p>
     * You can't add, update, or delete a <code>HealthCheckCustomConfig</code> configuration.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * For public and private DNS namespaces, note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you omit any existing <code>DnsRecords</code> or <code>HealthCheckConfig</code> configurations from an
     * <code>UpdateService</code> request, the configurations are deleted from the service.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you omit an existing <code>HealthCheckCustomConfig</code> configuration from an <code>UpdateService</code>
     * request, the configuration is not deleted from the service.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you update settings for a service, AWS Cloud Map also updates the corresponding settings in all the records
     * and health checks that were created by using the specified service.
     * </p>
     *
     * @param updateServiceRequest
     * @return Result of the UpdateService operation returned by the service.
     * @throws DuplicateRequestException
     *         The operation is already in progress.
     * @throws InvalidInputException
     *         One or more specified values aren't valid. For example, a required value might be missing, a numeric
     *         value might be outside the allowed range, or a string value might exceed length constraints.
     * @throws ServiceNotFoundException
     *         No service exists with the specified ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceDiscoveryClient.UpdateService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateServiceResponse updateService(UpdateServiceRequest updateServiceRequest) throws DuplicateRequestException,
            InvalidInputException, ServiceNotFoundException, AwsServiceException, SdkClientException, ServiceDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                .withOperationName("UpdateService").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceRequest)
                .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceDiscoveryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotFound")
                                .exceptionBuilderSupplier(OperationNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceAlreadyExists")
                                .exceptionBuilderSupplier(ServiceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceNotFound")
                                .exceptionBuilderSupplier(ServiceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceAlreadyExists")
                                .exceptionBuilderSupplier(NamespaceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceNotFound")
                                .exceptionBuilderSupplier(InstanceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceNotFound")
                                .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomHealthNotFound")
                                .exceptionBuilderSupplier(CustomHealthNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceeded")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ServiceDiscoveryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
