/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.paginators;

import java.util.concurrent.CompletableFuture;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.servicediscovery.ServiceDiscoveryAsyncClient;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.servicediscovery.ServiceDiscoveryAsyncClient#listServicesPaginator(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse}
 * response pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the subscribe helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
 * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listServices(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListServicesPublisher implements SdkPublisher<ListServicesResponse> {
    private final ServiceDiscoveryAsyncClient client;

    private final ListServicesRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public ListServicesPublisher(ServiceDiscoveryAsyncClient client, ListServicesRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private ListServicesPublisher(ServiceDiscoveryAsyncClient client, ListServicesRequest firstRequest, boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new ListServicesResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super ListServicesResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListServicesPublisher} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final ListServicesPublisher resume(ListServicesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListServicesPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new ListServicesPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super ListServicesResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class ListServicesResponseFetcher implements AsyncPageFetcher<ListServicesResponse> {
        @Override
        public boolean hasNextPage(final ListServicesResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<ListServicesResponse> nextPage(final ListServicesResponse previousPage) {
            if (previousPage == null) {
                return client.listServices(firstRequest);
            }
            return client.listServices(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
