/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateHttpNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePrivateDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreatePublicDnsNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.CreateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.CustomHealthNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeleteServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.DeregisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.DiscoverInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.DuplicateRequestException;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetNamespaceResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetOperationResponse;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.GetServiceResponse;
import software.amazon.awssdk.services.servicediscovery.model.InstanceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.InvalidInputException;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest;
import software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.NamespaceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.OperationNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceRequest;
import software.amazon.awssdk.services.servicediscovery.model.RegisterInstanceResponse;
import software.amazon.awssdk.services.servicediscovery.model.ResourceInUseException;
import software.amazon.awssdk.services.servicediscovery.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceAlreadyExistsException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryException;
import software.amazon.awssdk.services.servicediscovery.model.ServiceDiscoveryRequest;
import software.amazon.awssdk.services.servicediscovery.model.ServiceNotFoundException;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateInstanceCustomHealthStatusResponse;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceRequest;
import software.amazon.awssdk.services.servicediscovery.model.UpdateServiceResponse;
import software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusPublisher;
import software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesPublisher;
import software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesPublisher;
import software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsPublisher;
import software.amazon.awssdk.services.servicediscovery.paginators.ListServicesPublisher;
import software.amazon.awssdk.services.servicediscovery.transform.CreateHttpNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePrivateDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreatePublicDnsNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DeregisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.DiscoverInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetInstancesHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetNamespaceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListNamespacesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.RegisterInstanceRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateInstanceCustomHealthStatusRequestMarshaller;
import software.amazon.awssdk.services.servicediscovery.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ServiceDiscoveryAsyncClient}.
 *
 * @see ServiceDiscoveryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceDiscoveryAsyncClient implements ServiceDiscoveryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultServiceDiscoveryAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceDiscoveryAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an HTTP namespace. Service instances that you register using an HTTP namespace can be discovered using a
     * <code>DiscoverInstances</code> request but can't be discovered using DNS.
     * </p>
     * <p>
     * For the current limit on the number of namespaces that you can create using the same AWS account, see <a
     * href="http://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map Limits</a> in the
     * <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createHttpNamespaceRequest
     * @return A Java Future containing the result of the CreateHttpNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the limit on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreateHttpNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateHttpNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHttpNamespaceResponse> createHttpNamespace(
            CreateHttpNamespaceRequest createHttpNamespaceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHttpNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateHttpNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateHttpNamespaceRequest, CreateHttpNamespaceResponse>()
                    .withOperationName("CreateHttpNamespace")
                    .withMarshaller(new CreateHttpNamespaceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createHttpNamespaceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a private namespace based on DNS, which will be visible only inside a specified Amazon VPC. The namespace
     * defines your service naming scheme. For example, if you name your namespace <code>example.com</code> and name
     * your service <code>backend</code>, the resulting DNS name for the service will be
     * <code>backend.example.com</code>. For the current limit on the number of namespaces that you can create using the
     * same AWS account, see <a href="http://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud
     * Map Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createPrivateDnsNamespaceRequest
     * @return A Java Future containing the result of the CreatePrivateDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the limit on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreatePrivateDnsNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePrivateDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePrivateDnsNamespaceResponse> createPrivateDnsNamespace(
            CreatePrivateDnsNamespaceRequest createPrivateDnsNamespaceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePrivateDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePrivateDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePrivateDnsNamespaceRequest, CreatePrivateDnsNamespaceResponse>()
                            .withOperationName("CreatePrivateDnsNamespace")
                            .withMarshaller(new CreatePrivateDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPrivateDnsNamespaceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a public namespace based on DNS, which will be visible on the internet. The namespace defines your
     * service naming scheme. For example, if you name your namespace <code>example.com</code> and name your service
     * <code>backend</code>, the resulting DNS name for the service will be <code>backend.example.com</code>. For the
     * current limit on the number of namespaces that you can create using the same AWS account, see <a
     * href="http://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map Limits</a> in the
     * <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createPublicDnsNamespaceRequest
     * @return A Java Future containing the result of the CreatePublicDnsNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceAlreadyExistsException The namespace that you're trying to create already exists.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the limit on the
     *         number of resources.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreatePublicDnsNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreatePublicDnsNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePublicDnsNamespaceResponse> createPublicDnsNamespace(
            CreatePublicDnsNamespaceRequest createPublicDnsNamespaceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePublicDnsNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePublicDnsNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePublicDnsNamespaceRequest, CreatePublicDnsNamespaceResponse>()
                            .withOperationName("CreatePublicDnsNamespace")
                            .withMarshaller(new CreatePublicDnsNamespaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createPublicDnsNamespaceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service, which defines the configuration for the following entities:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For public and private DNS namespaces, one of the following combinations of DNS records in Amazon Route 53:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A
     * </p>
     * </li>
     * <li>
     * <p>
     * AAAA
     * </p>
     * </li>
     * <li>
     * <p>
     * A and AAAA
     * </p>
     * </li>
     * <li>
     * <p>
     * SRV
     * </p>
     * </li>
     * <li>
     * <p>
     * CNAME
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Optionally, a health check
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you create the service, you can submit a <a>RegisterInstance</a> request, and AWS Cloud Map uses the values
     * in the configuration to create the specified entities.
     * </p>
     * <p>
     * For the current limit on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="http://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map
     * Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param createServiceRequest
     * @return A Java Future containing the result of the CreateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the limit on the
     *         number of resources.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ServiceAlreadyExistsException The service can't be created because a service with the same name
     *         already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.CreateService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/CreateService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServiceResponse> createService(CreateServiceRequest createServiceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                    .withOperationName("CreateService").withMarshaller(new CreateServiceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createServiceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a namespace from the current account. If the namespace still contains one or more services, the request
     * fails.
     * </p>
     *
     * @param deleteNamespaceRequest
     * @return A Java Future containing the result of the DeleteNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeleteNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteNamespace"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNamespaceResponse> deleteNamespace(DeleteNamespaceRequest deleteNamespaceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteNamespaceRequest, DeleteNamespaceResponse>()
                    .withOperationName("DeleteNamespace").withMarshaller(new DeleteNamespaceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteNamespaceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified service. If the service still contains one or more registered instances, the request fails.
     * </p>
     *
     * @param deleteServiceRequest
     * @return A Java Future containing the result of the DeleteService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeleteService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeleteService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceResponse> deleteService(DeleteServiceRequest deleteServiceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                    .withOperationName("DeleteService").withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteServiceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon Route 53 DNS records and health check, if any, that AWS Cloud Map created for the specified
     * instance.
     * </p>
     *
     * @param deregisterInstanceRequest
     * @return A Java Future containing the result of the DeregisterInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DeregisterInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DeregisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterInstanceResponse> deregisterInstance(DeregisterInstanceRequest deregisterInstanceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeregisterInstanceRequest, DeregisterInstanceResponse>()
                    .withOperationName("DeregisterInstance")
                    .withMarshaller(new DeregisterInstanceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deregisterInstanceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Discovers registered instances for a specified namespace and service.
     * </p>
     *
     * @param discoverInstancesRequest
     * @return A Java Future containing the result of the DiscoverInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.DiscoverInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/DiscoverInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DiscoverInstancesResponse> discoverInstances(DiscoverInstancesRequest discoverInstancesRequest) {
        try {
            String hostPrefix = "data-";
            String resolvedHostExpression = "data-";
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DiscoverInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DiscoverInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DiscoverInstancesRequest, DiscoverInstancesResponse>()
                    .withOperationName("DiscoverInstances")
                    .withMarshaller(new DiscoverInstancesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(discoverInstancesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified instance.
     * </p>
     *
     * @param getInstanceRequest
     * @return A Java Future containing the result of the GetInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstanceResponse> getInstance(GetInstanceRequest getInstanceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetInstanceRequest, GetInstanceResponse>()
                    .withOperationName("GetInstance").withMarshaller(new GetInstanceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getInstanceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There is a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note>
     *
     * @param getInstancesHealthStatusRequest
     * @return A Java Future containing the result of the GetInstancesHealthStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetInstancesHealthStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInstancesHealthStatusResponse> getInstancesHealthStatus(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInstancesHealthStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInstancesHealthStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<GetInstancesHealthStatusRequest, GetInstancesHealthStatusResponse>()
                            .withOperationName("GetInstancesHealthStatus")
                            .withMarshaller(new GetInstancesHealthStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getInstancesHealthStatusRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the current health status (<code>Healthy</code>, <code>Unhealthy</code>, or <code>Unknown</code>) of one or
     * more instances that are associated with a specified service.
     * </p>
     * <note>
     * <p>
     * There is a brief delay between when you register an instance and when the health status for the instance is
     * available.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getInstancesHealthStatus(software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusPublisher publisher = client.getInstancesHealthStatusPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.GetInstancesHealthStatusPublisher publisher = client.getInstancesHealthStatusPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getInstancesHealthStatus(software.amazon.awssdk.services.servicediscovery.model.GetInstancesHealthStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getInstancesHealthStatusRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetInstancesHealthStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetInstancesHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    public GetInstancesHealthStatusPublisher getInstancesHealthStatusPaginator(
            GetInstancesHealthStatusRequest getInstancesHealthStatusRequest) {
        return new GetInstancesHealthStatusPublisher(this, applyPaginatorUserAgent(getInstancesHealthStatusRequest));
    }

    /**
     * <p>
     * Gets information about a namespace.
     * </p>
     *
     * @param getNamespaceRequest
     * @return A Java Future containing the result of the GetNamespace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>NamespaceNotFoundException No namespace exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetNamespace
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetNamespaceResponse> getNamespace(GetNamespaceRequest getNamespaceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNamespaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetNamespaceRequest, GetNamespaceResponse>()
                    .withOperationName("GetNamespace").withMarshaller(new GetNamespaceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getNamespaceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about any operation that returns an operation ID in the response, such as a
     * <code>CreateService</code> request.
     * </p>
     * <note>
     * <p>
     * To get a list of operations that match specified criteria, see <a>ListOperations</a>.
     * </p>
     * </note>
     *
     * @param getOperationRequest
     * @return A Java Future containing the result of the GetOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>OperationNotFoundException No operation exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOperationResponse> getOperation(GetOperationRequest getOperationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                    .withOperationName("GetOperation").withMarshaller(new GetOperationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getOperationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the settings for a specified service.
     * </p>
     *
     * @param getServiceRequest
     * @return A Java Future containing the result of the GetService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.GetService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/GetService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceResponse> getService(GetServiceRequest getServiceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>()
                    .withOperationName("GetService").withMarshaller(new GetServiceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getServiceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     *
     * @param listInstancesRequest
     * @return A Java Future containing the result of the ListInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstancesResponse> listInstances(ListInstancesRequest listInstancesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                    .withOperationName("ListInstances").withMarshaller(new ListInstancesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listInstancesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the instances that you registered by using a specified service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstances(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesPublisher publisher = client.listInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListInstancesPublisher publisher = client.listInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicediscovery.model.ListInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstances(software.amazon.awssdk.services.servicediscovery.model.ListInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListInstances
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListInstances" target="_top">AWS
     *      API Documentation</a>
     */
    public ListInstancesPublisher listInstancesPaginator(ListInstancesRequest listInstancesRequest) {
        return new ListInstancesPublisher(this, applyPaginatorUserAgent(listInstancesRequest));
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current AWS account.
     * </p>
     *
     * @param listNamespacesRequest
     * @return A Java Future containing the result of the ListNamespaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListNamespaces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNamespacesResponse> listNamespaces(ListNamespacesRequest listNamespacesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNamespacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNamespacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListNamespacesRequest, ListNamespacesResponse>()
                    .withOperationName("ListNamespaces").withMarshaller(new ListNamespacesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listNamespacesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the namespaces that were created by the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamespaces(software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesPublisher publisher = client.listNamespacesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListNamespacesPublisher publisher = client.listNamespacesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicediscovery.model.ListNamespacesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamespaces(software.amazon.awssdk.services.servicediscovery.model.ListNamespacesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNamespacesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListNamespaces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListNamespaces"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNamespacesPublisher listNamespacesPaginator(ListNamespacesRequest listNamespacesRequest) {
        return new ListNamespacesPublisher(this, applyPaginatorUserAgent(listNamespacesRequest));
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     *
     * @param listOperationsRequest
     * @return A Java Future containing the result of the ListOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOperationsResponse> listOperations(ListOperationsRequest listOperationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                    .withOperationName("ListOperations").withMarshaller(new ListOperationsRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listOperationsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists operations that match the criteria that you specify.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOperations(software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsPublisher publisher = client.listOperationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListOperationsPublisher publisher = client.listOperationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicediscovery.model.ListOperationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOperations(software.amazon.awssdk.services.servicediscovery.model.ListOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listOperationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListOperations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListOperationsPublisher listOperationsPaginator(ListOperationsRequest listOperationsRequest) {
        return new ListOperationsPublisher(this, applyPaginatorUserAgent(listOperationsRequest));
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(listServicesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information for all the services that are associated with one or more specified namespaces.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServices(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicediscovery.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicediscovery.model.ListServicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServices(software.amazon.awssdk.services.servicediscovery.model.ListServicesRequest)} operation.</b>
     * </p>
     *
     * @param listServicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    public ListServicesPublisher listServicesPaginator(ListServicesRequest listServicesRequest) {
        return new ListServicesPublisher(this, applyPaginatorUserAgent(listServicesRequest));
    }

    /**
     * <p>
     * Creates or updates one or more records and, optionally, creates a health check based on the settings in a
     * specified service. When you submit a <code>RegisterInstance</code> request, the following occurs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For each DNS record that you define in the service that is specified by <code>ServiceId</code>, a record is
     * created or updated in the hosted zone that is associated with the corresponding namespace.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the service includes <code>HealthCheckConfig</code>, a health check is created based on the settings in the
     * health check configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * The health check, if any, is associated with each of the new or updated records.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * One <code>RegisterInstance</code> request must complete before you can submit another request and specify the
     * same service ID and instance ID.
     * </p>
     * </important>
     * <p>
     * For more information, see <a>CreateService</a>.
     * </p>
     * <p>
     * When AWS Cloud Map receives a DNS query for the specified DNS name, it returns the applicable value:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>If the health check is healthy</b>: returns all the records
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If the health check is unhealthy</b>: returns the applicable value for the last healthy instance
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>If you didn't specify a health check configuration</b>: returns all the records
     * </p>
     * </li>
     * </ul>
     * <p>
     * For the current limit on the number of instances that you can register using the same namespace and using the
     * same service, see <a href="http://docs.aws.amazon.com/cloud-map/latest/dg/cloud-map-limits.html">AWS Cloud Map
     * Limits</a> in the <i>AWS Cloud Map Developer Guide</i>.
     * </p>
     *
     * @param registerInstanceRequest
     * @return A Java Future containing the result of the RegisterInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ResourceInUseException The specified resource can't be deleted because it contains other resources.
     *         For example, you can't delete a service that contains any instances.</li>
     *         <li>ResourceLimitExceededException The resource can't be created because you've reached the limit on the
     *         number of resources.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.RegisterInstance
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/RegisterInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterInstanceResponse> registerInstance(RegisterInstanceRequest registerInstanceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<RegisterInstanceRequest, RegisterInstanceResponse>()
                    .withOperationName("RegisterInstance").withMarshaller(new RegisterInstanceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(registerInstanceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a request to change the health status of a custom health check to healthy or unhealthy.
     * </p>
     * <p>
     * You can use <code>UpdateInstanceCustomHealthStatus</code> to change the status only for custom health checks,
     * which you define using <code>HealthCheckCustomConfig</code> when you create a service. You can't use it to change
     * the status for Route 53 health checks, which you define using <code>HealthCheckConfig</code>.
     * </p>
     * <p>
     * For more information, see <a>HealthCheckCustomConfig</a>.
     * </p>
     *
     * @param updateInstanceCustomHealthStatusRequest
     * @return A Java Future containing the result of the UpdateInstanceCustomHealthStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InstanceNotFoundException No instance exists with the specified ID, or the instance was recently
     *         registered, and information about the instance hasn't propagated yet.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>CustomHealthNotFoundException The health check for the instance that is specified by
     *         <code>ServiceId</code> and <code>InstanceId</code> is not a custom health check.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdateInstanceCustomHealthStatus
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateInstanceCustomHealthStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInstanceCustomHealthStatusResponse> updateInstanceCustomHealthStatus(
            UpdateInstanceCustomHealthStatusRequest updateInstanceCustomHealthStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInstanceCustomHealthStatusResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateInstanceCustomHealthStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateInstanceCustomHealthStatusRequest, UpdateInstanceCustomHealthStatusResponse>()
                            .withOperationName("UpdateInstanceCustomHealthStatus")
                            .withMarshaller(new UpdateInstanceCustomHealthStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateInstanceCustomHealthStatusRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a request to perform the following operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Add or delete <code>DnsRecords</code> configurations
     * </p>
     * </li>
     * <li>
     * <p>
     * Update the TTL setting for existing <code>DnsRecords</code> configurations
     * </p>
     * </li>
     * <li>
     * <p>
     * Add, update, or delete <code>HealthCheckConfig</code> for a specified service
     * </p>
     * </li>
     * </ul>
     * <p>
     * For public and private DNS namespaces, you must specify all <code>DnsRecords</code> configurations (and,
     * optionally, <code>HealthCheckConfig</code>) that you want to appear in the updated service. Any current
     * configurations that don't appear in an <code>UpdateService</code> request are deleted.
     * </p>
     * <p>
     * When you update the TTL setting for a service, AWS Cloud Map also updates the corresponding settings in all the
     * records and health checks that were created by using the specified service.
     * </p>
     *
     * @param updateServiceRequest
     * @return A Java Future containing the result of the UpdateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateRequestException The operation is already in progress.</li>
     *         <li>InvalidInputException One or more specified values aren't valid. For example, a required value might
     *         be missing, a numeric value might be outside the allowed range, or a string value might exceed length
     *         constraints.</li>
     *         <li>ServiceNotFoundException No service exists with the specified ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceDiscoveryException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ServiceDiscoveryAsyncClient.UpdateService
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/servicediscovery-2017-03-14/UpdateService" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceResponse> updateService(UpdateServiceRequest updateServiceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                    .withOperationName("UpdateService").withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateServiceRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceDiscoveryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotFound")
                                .exceptionBuilderSupplier(OperationNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUse")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInput")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceAlreadyExists")
                                .exceptionBuilderSupplier(ServiceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceNotFound")
                                .exceptionBuilderSupplier(ServiceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceAlreadyExists")
                                .exceptionBuilderSupplier(NamespaceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InstanceNotFound")
                                .exceptionBuilderSupplier(InstanceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NamespaceNotFound")
                                .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CustomHealthNotFound")
                                .exceptionBuilderSupplier(CustomHealthNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceeded")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
    }

    private <T extends ServiceDiscoveryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
