/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicediscovery.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that contains information about the Amazon Route 53 DNS records that you want AWS Cloud Map to create
 * when you register an instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DnsConfig implements SdkPojo, Serializable, ToCopyableBuilder<DnsConfig.Builder, DnsConfig> {
    private static final SdkField<String> NAMESPACE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DnsConfig::namespaceId)).setter(setter(Builder::namespaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NamespaceId").build()).build();

    private static final SdkField<String> ROUTING_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DnsConfig::routingPolicyAsString)).setter(setter(Builder::routingPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoutingPolicy").build()).build();

    private static final SdkField<List<DnsRecord>> DNS_RECORDS_FIELD = SdkField
            .<List<DnsRecord>> builder(MarshallingType.LIST)
            .getter(getter(DnsConfig::dnsRecords))
            .setter(setter(Builder::dnsRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsRecords").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DnsRecord> builder(MarshallingType.SDK_POJO)
                                            .constructor(DnsRecord::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_ID_FIELD,
            ROUTING_POLICY_FIELD, DNS_RECORDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespaceId;

    private final String routingPolicy;

    private final List<DnsRecord> dnsRecords;

    private DnsConfig(BuilderImpl builder) {
        this.namespaceId = builder.namespaceId;
        this.routingPolicy = builder.routingPolicy;
        this.dnsRecords = builder.dnsRecords;
    }

    /**
     * <p>
     * The ID of the namespace to use for DNS configuration.
     * </p>
     * 
     * @return The ID of the namespace to use for DNS configuration.
     */
    public String namespaceId() {
        return namespaceId;
    }

    /**
     * <p>
     * The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
     * register an instance and specify this service.
     * </p>
     * <note>
     * <p>
     * If you want to use this service to register instances that create alias records, specify <code>WEIGHTED</code>
     * for the routing policy.
     * </p>
     * </note>
     * <p>
     * You can specify the following values:
     * </p>
     * <p>
     * <b>MULTIVALUE</b>
     * </p>
     * <p>
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
     * value for up to eight instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
     * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP addresses
     * for all of the healthy instances.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the values for up to eight instances.
     * </p>
     * <p>
     * For more information about the multivalue routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * <b>WEIGHTED</b>
     * </p>
     * <p>
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service. Currently, all records have the same weight, so you can't route more or less
     * traffic to any instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly selected
     * instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS queries as if
     * all of the instances were healthy.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the applicable value for one randomly selected instance.
     * </p>
     * <p>
     * For more information about the weighted routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingPolicy}
     * will return {@link RoutingPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingPolicyAsString}.
     * </p>
     * 
     * @return The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
     *         register an instance and specify this service.</p> <note>
     *         <p>
     *         If you want to use this service to register instances that create alias records, specify
     *         <code>WEIGHTED</code> for the routing policy.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify the following values:
     *         </p>
     *         <p>
     *         <b>MULTIVALUE</b>
     *         </p>
     *         <p>
     *         If you define a health check for the service and the health check is healthy, Route 53 returns the
     *         applicable value for up to eight instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight
     *         healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with
     *         the IP addresses for all of the healthy instances.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the values for up to eight instances.
     *         </p>
     *         <p>
     *         For more information about the multivalue routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     *         >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     *         </p>
     *         <p>
     *         <b>WEIGHTED</b>
     *         </p>
     *         <p>
     *         Route 53 returns the applicable value from one randomly selected instance from among the instances that
     *         you registered using the same service. Currently, all records have the same weight, so you can't route
     *         more or less traffic to any instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
     *         randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
     *         responds to DNS queries as if all of the instances were healthy.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the applicable value for one randomly selected instance.
     *         </p>
     *         <p>
     *         For more information about the weighted routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     *         >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * @see RoutingPolicy
     */
    public RoutingPolicy routingPolicy() {
        return RoutingPolicy.fromValue(routingPolicy);
    }

    /**
     * <p>
     * The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
     * register an instance and specify this service.
     * </p>
     * <note>
     * <p>
     * If you want to use this service to register instances that create alias records, specify <code>WEIGHTED</code>
     * for the routing policy.
     * </p>
     * </note>
     * <p>
     * You can specify the following values:
     * </p>
     * <p>
     * <b>MULTIVALUE</b>
     * </p>
     * <p>
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
     * value for up to eight instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
     * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP addresses
     * for all of the healthy instances.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the values for up to eight instances.
     * </p>
     * <p>
     * For more information about the multivalue routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * <b>WEIGHTED</b>
     * </p>
     * <p>
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service. Currently, all records have the same weight, so you can't route more or less
     * traffic to any instances.
     * </p>
     * <p>
     * For example, suppose the service includes configurations for one A record and a health check, and you use the
     * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly selected
     * instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS queries as if
     * all of the instances were healthy.
     * </p>
     * <p>
     * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and returns
     * the applicable value for one randomly selected instance.
     * </p>
     * <p>
     * For more information about the weighted routing policy, see <a
     * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routingPolicy}
     * will return {@link RoutingPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #routingPolicyAsString}.
     * </p>
     * 
     * @return The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
     *         register an instance and specify this service.</p> <note>
     *         <p>
     *         If you want to use this service to register instances that create alias records, specify
     *         <code>WEIGHTED</code> for the routing policy.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify the following values:
     *         </p>
     *         <p>
     *         <b>MULTIVALUE</b>
     *         </p>
     *         <p>
     *         If you define a health check for the service and the health check is healthy, Route 53 returns the
     *         applicable value for up to eight instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight
     *         healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with
     *         the IP addresses for all of the healthy instances.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the values for up to eight instances.
     *         </p>
     *         <p>
     *         For more information about the multivalue routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
     *         >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
     *         </p>
     *         <p>
     *         <b>WEIGHTED</b>
     *         </p>
     *         <p>
     *         Route 53 returns the applicable value from one randomly selected instance from among the instances that
     *         you registered using the same service. Currently, all records have the same weight, so you can't route
     *         more or less traffic to any instances.
     *         </p>
     *         <p>
     *         For example, suppose the service includes configurations for one A record and a health check, and you use
     *         the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
     *         randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
     *         responds to DNS queries as if all of the instances were healthy.
     *         </p>
     *         <p>
     *         If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
     *         returns the applicable value for one randomly selected instance.
     *         </p>
     *         <p>
     *         For more information about the weighted routing policy, see <a href=
     *         "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
     *         >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
     * @see RoutingPolicy
     */
    public String routingPolicyAsString() {
        return routingPolicy;
    }

    /**
     * <p>
     * An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want AWS Cloud Map
     * to create when you register an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want AWS
     *         Cloud Map to create when you register an instance.
     */
    public List<DnsRecord> dnsRecords() {
        return dnsRecords;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespaceId());
        hashCode = 31 * hashCode + Objects.hashCode(routingPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dnsRecords());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DnsConfig)) {
            return false;
        }
        DnsConfig other = (DnsConfig) obj;
        return Objects.equals(namespaceId(), other.namespaceId())
                && Objects.equals(routingPolicyAsString(), other.routingPolicyAsString())
                && Objects.equals(dnsRecords(), other.dnsRecords());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DnsConfig").add("NamespaceId", namespaceId()).add("RoutingPolicy", routingPolicyAsString())
                .add("DnsRecords", dnsRecords()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NamespaceId":
            return Optional.ofNullable(clazz.cast(namespaceId()));
        case "RoutingPolicy":
            return Optional.ofNullable(clazz.cast(routingPolicyAsString()));
        case "DnsRecords":
            return Optional.ofNullable(clazz.cast(dnsRecords()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DnsConfig, T> g) {
        return obj -> g.apply((DnsConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DnsConfig> {
        /**
         * <p>
         * The ID of the namespace to use for DNS configuration.
         * </p>
         * 
         * @param namespaceId
         *        The ID of the namespace to use for DNS configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaceId(String namespaceId);

        /**
         * <p>
         * The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
         * register an instance and specify this service.
         * </p>
         * <note>
         * <p>
         * If you want to use this service to register instances that create alias records, specify
         * <code>WEIGHTED</code> for the routing policy.
         * </p>
         * </note>
         * <p>
         * You can specify the following values:
         * </p>
         * <p>
         * <b>MULTIVALUE</b>
         * </p>
         * <p>
         * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
         * value for up to eight instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
         * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP
         * addresses for all of the healthy instances.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the values for up to eight instances.
         * </p>
         * <p>
         * For more information about the multivalue routing policy, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * <p>
         * <b>WEIGHTED</b>
         * </p>
         * <p>
         * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
         * registered using the same service. Currently, all records have the same weight, so you can't route more or
         * less traffic to any instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly
         * selected instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS
         * queries as if all of the instances were healthy.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the applicable value for one randomly selected instance.
         * </p>
         * <p>
         * For more information about the weighted routing policy, see <a
         * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * 
         * @param routingPolicy
         *        The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when
         *        you register an instance and specify this service.</p> <note>
         *        <p>
         *        If you want to use this service to register instances that create alias records, specify
         *        <code>WEIGHTED</code> for the routing policy.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify the following values:
         *        </p>
         *        <p>
         *        <b>MULTIVALUE</b>
         *        </p>
         *        <p>
         *        If you define a health check for the service and the health check is healthy, Route 53 returns the
         *        applicable value for up to eight instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to
         *        eight healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS
         *        query with the IP addresses for all of the healthy instances.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the values for up to eight instances.
         *        </p>
         *        <p>
         *        For more information about the multivalue routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         *        >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         *        </p>
         *        <p>
         *        <b>WEIGHTED</b>
         *        </p>
         *        <p>
         *        Route 53 returns the applicable value from one randomly selected instance from among the instances
         *        that you registered using the same service. Currently, all records have the same weight, so you can't
         *        route more or less traffic to any instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
         *        randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
         *        responds to DNS queries as if all of the instances were healthy.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the applicable value for one randomly selected instance.
         *        </p>
         *        <p>
         *        For more information about the weighted routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         *        >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * @see RoutingPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingPolicy
         */
        Builder routingPolicy(String routingPolicy);

        /**
         * <p>
         * The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when you
         * register an instance and specify this service.
         * </p>
         * <note>
         * <p>
         * If you want to use this service to register instances that create alias records, specify
         * <code>WEIGHTED</code> for the routing policy.
         * </p>
         * </note>
         * <p>
         * You can specify the following values:
         * </p>
         * <p>
         * <b>MULTIVALUE</b>
         * </p>
         * <p>
         * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable
         * value for up to eight instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to eight healthy
         * instances. If fewer than eight instances are healthy, Route 53 responds to every DNS query with the IP
         * addresses for all of the healthy instances.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the values for up to eight instances.
         * </p>
         * <p>
         * For more information about the multivalue routing policy, see <a href=
         * "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         * >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * <p>
         * <b>WEIGHTED</b>
         * </p>
         * <p>
         * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
         * registered using the same service. Currently, all records have the same weight, so you can't route more or
         * less traffic to any instances.
         * </p>
         * <p>
         * For example, suppose the service includes configurations for one A record and a health check, and you use the
         * service to register 10 instances. Route 53 responds to DNS queries with the IP address for one randomly
         * selected instance from among the healthy instances. If no instances are healthy, Route 53 responds to DNS
         * queries as if all of the instances were healthy.
         * </p>
         * <p>
         * If you don't define a health check for the service, Route 53 assumes that all instances are healthy and
         * returns the applicable value for one randomly selected instance.
         * </p>
         * <p>
         * For more information about the weighted routing policy, see <a
         * href="http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         * >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * </p>
         * 
         * @param routingPolicy
         *        The routing policy that you want to apply to all Route 53 DNS records that AWS Cloud Map creates when
         *        you register an instance and specify this service.</p> <note>
         *        <p>
         *        If you want to use this service to register instances that create alias records, specify
         *        <code>WEIGHTED</code> for the routing policy.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify the following values:
         *        </p>
         *        <p>
         *        <b>MULTIVALUE</b>
         *        </p>
         *        <p>
         *        If you define a health check for the service and the health check is healthy, Route 53 returns the
         *        applicable value for up to eight instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with IP addresses for up to
         *        eight healthy instances. If fewer than eight instances are healthy, Route 53 responds to every DNS
         *        query with the IP addresses for all of the healthy instances.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the values for up to eight instances.
         *        </p>
         *        <p>
         *        For more information about the multivalue routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-multivalue"
         *        >Multivalue Answer Routing</a> in the <i>Route 53 Developer Guide</i>.
         *        </p>
         *        <p>
         *        <b>WEIGHTED</b>
         *        </p>
         *        <p>
         *        Route 53 returns the applicable value from one randomly selected instance from among the instances
         *        that you registered using the same service. Currently, all records have the same weight, so you can't
         *        route more or less traffic to any instances.
         *        </p>
         *        <p>
         *        For example, suppose the service includes configurations for one A record and a health check, and you
         *        use the service to register 10 instances. Route 53 responds to DNS queries with the IP address for one
         *        randomly selected instance from among the healthy instances. If no instances are healthy, Route 53
         *        responds to DNS queries as if all of the instances were healthy.
         *        </p>
         *        <p>
         *        If you don't define a health check for the service, Route 53 assumes that all instances are healthy
         *        and returns the applicable value for one randomly selected instance.
         *        </p>
         *        <p>
         *        For more information about the weighted routing policy, see <a href=
         *        "http://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-policy.html#routing-policy-weighted"
         *        >Weighted Routing</a> in the <i>Route 53 Developer Guide</i>.
         * @see RoutingPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingPolicy
         */
        Builder routingPolicy(RoutingPolicy routingPolicy);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want AWS Cloud
         * Map to create when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want
         *        AWS Cloud Map to create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(Collection<DnsRecord> dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want AWS Cloud
         * Map to create when you register an instance.
         * </p>
         * 
         * @param dnsRecords
         *        An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want
         *        AWS Cloud Map to create when you register an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsRecords(DnsRecord... dnsRecords);

        /**
         * <p>
         * An array that contains one <code>DnsRecord</code> object for each Route 53 DNS record that you want AWS Cloud
         * Map to create when you register an instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DnsRecord>.Builder} avoiding the need to
         * create one manually via {@link List<DnsRecord>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DnsRecord>.Builder#build()} is called immediately and its
         * result is passed to {@link #dnsRecords(List<DnsRecord>)}.
         * 
         * @param dnsRecords
         *        a consumer that will call methods on {@link List<DnsRecord>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dnsRecords(List<DnsRecord>)
         */
        Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords);
    }

    static final class BuilderImpl implements Builder {
        private String namespaceId;

        private String routingPolicy;

        private List<DnsRecord> dnsRecords = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DnsConfig model) {
            namespaceId(model.namespaceId);
            routingPolicy(model.routingPolicy);
            dnsRecords(model.dnsRecords);
        }

        public final String getNamespaceId() {
            return namespaceId;
        }

        @Override
        public final Builder namespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
            return this;
        }

        public final void setNamespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
        }

        public final String getRoutingPolicyAsString() {
            return routingPolicy;
        }

        @Override
        public final Builder routingPolicy(String routingPolicy) {
            this.routingPolicy = routingPolicy;
            return this;
        }

        @Override
        public final Builder routingPolicy(RoutingPolicy routingPolicy) {
            this.routingPolicy(routingPolicy.toString());
            return this;
        }

        public final void setRoutingPolicy(String routingPolicy) {
            this.routingPolicy = routingPolicy;
        }

        public final Collection<DnsRecord.Builder> getDnsRecords() {
            return dnsRecords != null ? dnsRecords.stream().map(DnsRecord::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dnsRecords(Collection<DnsRecord> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copy(dnsRecords);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(DnsRecord... dnsRecords) {
            dnsRecords(Arrays.asList(dnsRecords));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsRecords(Consumer<DnsRecord.Builder>... dnsRecords) {
            dnsRecords(Stream.of(dnsRecords).map(c -> DnsRecord.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDnsRecords(Collection<DnsRecord.BuilderImpl> dnsRecords) {
            this.dnsRecords = DnsRecordListCopier.copyFromBuilder(dnsRecords);
        }

        @Override
        public DnsConfig build() {
            return new DnsConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
