/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicequotas;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicequotas.model.AccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.AwsServiceAccessNotEnabledException;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.DependencyAccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeRequest;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.IllegalArgumentException;
import software.amazon.awssdk.services.servicequotas.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.servicequotas.model.InvalidResourceStateException;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServicesRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServicesResponse;
import software.amazon.awssdk.services.servicequotas.model.NoAvailableOrganizationException;
import software.amazon.awssdk.services.servicequotas.model.NoSuchResourceException;
import software.amazon.awssdk.services.servicequotas.model.OrganizationNotInAllFeaturesModeException;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.QuotaExceededException;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseRequest;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseResponse;
import software.amazon.awssdk.services.servicequotas.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.servicequotas.model.ServiceException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotaTemplateNotInUseException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotasException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.TemplatesNotAvailableInRegionException;
import software.amazon.awssdk.services.servicequotas.model.TooManyRequestsException;
import software.amazon.awssdk.services.servicequotas.paginators.ListAWSDefaultServiceQuotasPublisher;
import software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryByQuotaPublisher;
import software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryPublisher;
import software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotaIncreaseRequestsInTemplatePublisher;
import software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotasPublisher;
import software.amazon.awssdk.services.servicequotas.paginators.ListServicesPublisher;
import software.amazon.awssdk.services.servicequotas.transform.AssociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DisassociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAssociationForServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAwsDefaultServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetRequestedServiceQuotaChangeRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListAwsDefaultServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.RequestServiceQuotaIncreaseRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ServiceQuotasAsyncClient}.
 *
 * @see ServiceQuotasAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceQuotasAsyncClient implements ServiceQuotasAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultServiceQuotasAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceQuotasAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates the Service Quotas template with your organization so that when new accounts are created in your
     * organization, the template submits increase requests for the specified service quotas. Use the Service Quotas
     * template to request an increase for any adjustable quota value. After you define the Service Quotas template, use
     * this operation to associate, or enable, the template.
     * </p>
     *
     * @param associateServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the AssociateServiceQuotaTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>OrganizationNotInAllFeaturesModeException The organization that your account belongs to, is not in
     *         All Features mode. To enable all features mode, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html"
     *         >EnableAllFeatures</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.AssociateServiceQuotaTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/AssociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateServiceQuotaTemplateResponse> associateServiceQuotaTemplate(
            AssociateServiceQuotaTemplateRequest associateServiceQuotaTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateServiceQuotaTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateServiceQuotaTemplateRequest, AssociateServiceQuotaTemplateResponse>()
                            .withOperationName("AssociateServiceQuotaTemplate")
                            .withMarshaller(new AssociateServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateServiceQuotaTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateServiceQuotaTemplateRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<AssociateServiceQuotaTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a service quota increase request from the Service Quotas template.
     * </p>
     *
     * @param deleteServiceQuotaIncreaseRequestFromTemplateRequest
     * @return A Java Future containing the result of the DeleteServiceQuotaIncreaseRequestFromTemplate operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.DeleteServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DeleteServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> deleteServiceQuotaIncreaseRequestFromTemplate(
            DeleteServiceQuotaIncreaseRequestFromTemplateRequest deleteServiceQuotaIncreaseRequestFromTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServiceQuotaIncreaseRequestFromTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteServiceQuotaIncreaseRequestFromTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceQuotaIncreaseRequestFromTemplateRequest, DeleteServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("DeleteServiceQuotaIncreaseRequestFromTemplate")
                            .withMarshaller(new DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceQuotaIncreaseRequestFromTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteServiceQuotaIncreaseRequestFromTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the Service Quotas template. Once the template is disabled, it does not request quota increases for new
     * accounts in your organization. Disabling the quota template does not apply the quota increase requests from the
     * template.
     * </p>
     * <p>
     * <b>Related operations</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * To enable the quota template, call <a>AssociateServiceQuotaTemplate</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To delete a specific service quota from the template, use <a>DeleteServiceQuotaIncreaseRequestFromTemplate</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param disassociateServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the DisassociateServiceQuotaTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceQuotaTemplateNotInUseException The quota request template is not associated with your
     *         organization. </p>
     *         <p>
     *         To use the template, call <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.DisassociateServiceQuotaTemplate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DisassociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateServiceQuotaTemplateResponse> disassociateServiceQuotaTemplate(
            DisassociateServiceQuotaTemplateRequest disassociateServiceQuotaTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateServiceQuotaTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateServiceQuotaTemplateRequest, DisassociateServiceQuotaTemplateResponse>()
                            .withOperationName("DisassociateServiceQuotaTemplate")
                            .withMarshaller(new DisassociateServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateServiceQuotaTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateServiceQuotaTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DisassociateServiceQuotaTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the default service quotas values. The Value returned for each quota is the AWS default value, even if
     * the quotas have been increased..
     * </p>
     *
     * @param getAwsDefaultServiceQuotaRequest
     * @return A Java Future containing the result of the GetAWSDefaultServiceQuota operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetAWSDefaultServiceQuota
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAWSDefaultServiceQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAwsDefaultServiceQuotaResponse> getAWSDefaultServiceQuota(
            GetAwsDefaultServiceQuotaRequest getAwsDefaultServiceQuotaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAwsDefaultServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAWSDefaultServiceQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAwsDefaultServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAwsDefaultServiceQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAwsDefaultServiceQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAwsDefaultServiceQuotaRequest, GetAwsDefaultServiceQuotaResponse>()
                            .withOperationName("GetAWSDefaultServiceQuota")
                            .withMarshaller(new GetAwsDefaultServiceQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAwsDefaultServiceQuotaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAwsDefaultServiceQuotaRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetAwsDefaultServiceQuotaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>ServiceQuotaTemplateAssociationStatus</code> value from the service. Use this action to
     * determine if the Service Quota template is associated, or enabled.
     * </p>
     *
     * @param getAssociationForServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the GetAssociationForServiceQuotaTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceQuotaTemplateNotInUseException The quota request template is not associated with your
     *         organization. </p>
     *         <p>
     *         To use the template, call <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetAssociationForServiceQuotaTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAssociationForServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> getAssociationForServiceQuotaTemplate(
            GetAssociationForServiceQuotaTemplateRequest getAssociationForServiceQuotaTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAssociationForServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssociationForServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAssociationForServiceQuotaTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAssociationForServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAssociationForServiceQuotaTemplateRequest, GetAssociationForServiceQuotaTemplateResponse>()
                            .withOperationName("GetAssociationForServiceQuotaTemplate")
                            .withMarshaller(new GetAssociationForServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAssociationForServiceQuotaTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAssociationForServiceQuotaTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details for a particular increase request.
     * </p>
     *
     * @param getRequestedServiceQuotaChangeRequest
     * @return A Java Future containing the result of the GetRequestedServiceQuotaChange operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetRequestedServiceQuotaChange
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetRequestedServiceQuotaChange"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRequestedServiceQuotaChangeResponse> getRequestedServiceQuotaChange(
            GetRequestedServiceQuotaChangeRequest getRequestedServiceQuotaChangeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRequestedServiceQuotaChangeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRequestedServiceQuotaChange");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRequestedServiceQuotaChangeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRequestedServiceQuotaChangeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRequestedServiceQuotaChangeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRequestedServiceQuotaChangeRequest, GetRequestedServiceQuotaChangeResponse>()
                            .withOperationName("GetRequestedServiceQuotaChange")
                            .withMarshaller(new GetRequestedServiceQuotaChangeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRequestedServiceQuotaChangeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRequestedServiceQuotaChangeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetRequestedServiceQuotaChangeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details for the specified service quota. This operation provides a different Value than the
     * <code>GetAWSDefaultServiceQuota</code> operation. This operation returns the applied value for each quota.
     * <code>GetAWSDefaultServiceQuota</code> returns the default AWS value for each quota.
     * </p>
     *
     * @param getServiceQuotaRequest
     * @return A Java Future containing the result of the GetServiceQuota operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetServiceQuota
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuota" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceQuotaResponse> getServiceQuota(GetServiceQuotaRequest getServiceQuotaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceQuotaRequest, GetServiceQuotaResponse>()
                            .withOperationName("GetServiceQuota")
                            .withMarshaller(new GetServiceQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getServiceQuotaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getServiceQuotaRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetServiceQuotaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of the service quota increase request in your template.
     * </p>
     *
     * @param getServiceQuotaIncreaseRequestFromTemplateRequest
     * @return A Java Future containing the result of the GetServiceQuotaIncreaseRequestFromTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> getServiceQuotaIncreaseRequestFromTemplate(
            GetServiceQuotaIncreaseRequestFromTemplateRequest getServiceQuotaIncreaseRequestFromTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuotaIncreaseRequestFromTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetServiceQuotaIncreaseRequestFromTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceQuotaIncreaseRequestFromTemplateRequest, GetServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("GetServiceQuotaIncreaseRequestFromTemplate")
                            .withMarshaller(new GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceQuotaIncreaseRequestFromTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getServiceQuotaIncreaseRequestFromTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all default service quotas for the specified AWS service or all AWS services. ListAWSDefaultServiceQuotas
     * is similar to <a>ListServiceQuotas</a> except for the Value object. The Value object returned by
     * <code>ListAWSDefaultServiceQuotas</code> is the default value assigned by AWS. This request returns a list of all
     * service quotas for the specified service. The listing of each you'll see the default values are the values that
     * AWS provides for the quotas.
     * </p>
     * <note>
     * <p>
     * Always check the <code>NextToken</code> response parameter when calling any of the <code>List*</code> operations.
     * These operations can return an unexpected list of results, even when there are more results available. When this
     * happens, the <code>NextToken</code> response parameter contains a value to pass the next call to the same API to
     * request the next part of the list.
     * </p>
     * </note>
     *
     * @param listAwsDefaultServiceQuotasRequest
     * @return A Java Future containing the result of the ListAWSDefaultServiceQuotas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListAWSDefaultServiceQuotas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListAWSDefaultServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAwsDefaultServiceQuotasResponse> listAWSDefaultServiceQuotas(
            ListAwsDefaultServiceQuotasRequest listAwsDefaultServiceQuotasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAwsDefaultServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAWSDefaultServiceQuotas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAwsDefaultServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAwsDefaultServiceQuotasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAwsDefaultServiceQuotasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAwsDefaultServiceQuotasRequest, ListAwsDefaultServiceQuotasResponse>()
                            .withOperationName("ListAWSDefaultServiceQuotas")
                            .withMarshaller(new ListAwsDefaultServiceQuotasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAwsDefaultServiceQuotasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAwsDefaultServiceQuotasRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAwsDefaultServiceQuotasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all default service quotas for the specified AWS service or all AWS services. ListAWSDefaultServiceQuotas
     * is similar to <a>ListServiceQuotas</a> except for the Value object. The Value object returned by
     * <code>ListAWSDefaultServiceQuotas</code> is the default value assigned by AWS. This request returns a list of all
     * service quotas for the specified service. The listing of each you'll see the default values are the values that
     * AWS provides for the quotas.
     * </p>
     * <note>
     * <p>
     * Always check the <code>NextToken</code> response parameter when calling any of the <code>List*</code> operations.
     * These operations can return an unexpected list of results, even when there are more results available. When this
     * happens, the <code>NextToken</code> response parameter contains a value to pass the next call to the same API to
     * request the next part of the list.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listAWSDefaultServiceQuotas(software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListAWSDefaultServiceQuotasPublisher publisher = client.listAWSDefaultServiceQuotasPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListAWSDefaultServiceQuotasPublisher publisher = client.listAWSDefaultServiceQuotasPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAWSDefaultServiceQuotas(software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAwsDefaultServiceQuotasRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListAWSDefaultServiceQuotas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListAWSDefaultServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAWSDefaultServiceQuotasPublisher listAWSDefaultServiceQuotasPaginator(
            ListAwsDefaultServiceQuotasRequest listAwsDefaultServiceQuotasRequest) {
        return new ListAWSDefaultServiceQuotasPublisher(this, applyPaginatorUserAgent(listAwsDefaultServiceQuotasRequest));
    }

    /**
     * <p>
     * Requests a list of the changes to quotas for a service.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryRequest
     * @return A Java Future containing the result of the ListRequestedServiceQuotaChangeHistory operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistory
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> listRequestedServiceQuotaChangeHistory(
            ListRequestedServiceQuotaChangeHistoryRequest listRequestedServiceQuotaChangeHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryRequest, ListRequestedServiceQuotaChangeHistoryResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistory")
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRequestedServiceQuotaChangeHistoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRequestedServiceQuotaChangeHistoryRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests a list of the changes to specific service quotas. This command provides additional granularity over the
     * <code>ListRequestedServiceQuotaChangeHistory</code> command. Once a quota change request has reached
     * <code>CASE_CLOSED, APPROVED,</code> or <code>DENIED</code>, the history has been kept for 90 days.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryByQuotaRequest
     * @return A Java Future containing the result of the ListRequestedServiceQuotaChangeHistoryByQuota operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistoryByQuota
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistoryByQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> listRequestedServiceQuotaChangeHistoryByQuota(
            ListRequestedServiceQuotaChangeHistoryByQuotaRequest listRequestedServiceQuotaChangeHistoryByQuotaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryByQuotaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistoryByQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryByQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryByQuotaRequest, ListRequestedServiceQuotaChangeHistoryByQuotaResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistoryByQuota")
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listRequestedServiceQuotaChangeHistoryByQuotaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRequestedServiceQuotaChangeHistoryByQuotaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests a list of the changes to specific service quotas. This command provides additional granularity over the
     * <code>ListRequestedServiceQuotaChangeHistory</code> command. Once a quota change request has reached
     * <code>CASE_CLOSED, APPROVED,</code> or <code>DENIED</code>, the history has been kept for 90 days.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRequestedServiceQuotaChangeHistoryByQuota(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryByQuotaPublisher publisher = client.listRequestedServiceQuotaChangeHistoryByQuotaPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryByQuotaPublisher publisher = client.listRequestedServiceQuotaChangeHistoryByQuotaPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRequestedServiceQuotaChangeHistoryByQuota(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryByQuotaRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistoryByQuota
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistoryByQuota"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRequestedServiceQuotaChangeHistoryByQuotaPublisher listRequestedServiceQuotaChangeHistoryByQuotaPaginator(
            ListRequestedServiceQuotaChangeHistoryByQuotaRequest listRequestedServiceQuotaChangeHistoryByQuotaRequest) {
        return new ListRequestedServiceQuotaChangeHistoryByQuotaPublisher(this,
                applyPaginatorUserAgent(listRequestedServiceQuotaChangeHistoryByQuotaRequest));
    }

    /**
     * <p>
     * Requests a list of the changes to quotas for a service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRequestedServiceQuotaChangeHistory(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryPublisher publisher = client.listRequestedServiceQuotaChangeHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListRequestedServiceQuotaChangeHistoryPublisher publisher = client.listRequestedServiceQuotaChangeHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRequestedServiceQuotaChangeHistory(software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistory
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistory"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRequestedServiceQuotaChangeHistoryPublisher listRequestedServiceQuotaChangeHistoryPaginator(
            ListRequestedServiceQuotaChangeHistoryRequest listRequestedServiceQuotaChangeHistoryRequest) {
        return new ListRequestedServiceQuotaChangeHistoryPublisher(this,
                applyPaginatorUserAgent(listRequestedServiceQuotaChangeHistoryRequest));
    }

    /**
     * <p>
     * Returns a list of the quota increase requests in the template.
     * </p>
     *
     * @param listServiceQuotaIncreaseRequestsInTemplateRequest
     * @return A Java Future containing the result of the ListServiceQuotaIncreaseRequestsInTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotaIncreaseRequestsInTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotaIncreaseRequestsInTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> listServiceQuotaIncreaseRequestsInTemplate(
            ListServiceQuotaIncreaseRequestsInTemplateRequest listServiceQuotaIncreaseRequestsInTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listServiceQuotaIncreaseRequestsInTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotaIncreaseRequestsInTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceQuotaIncreaseRequestsInTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListServiceQuotaIncreaseRequestsInTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceQuotaIncreaseRequestsInTemplateRequest, ListServiceQuotaIncreaseRequestsInTemplateResponse>()
                            .withOperationName("ListServiceQuotaIncreaseRequestsInTemplate")
                            .withMarshaller(new ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceQuotaIncreaseRequestsInTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listServiceQuotaIncreaseRequestsInTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the quota increase requests in the template.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServiceQuotaIncreaseRequestsInTemplate(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotaIncreaseRequestsInTemplatePublisher publisher = client.listServiceQuotaIncreaseRequestsInTemplatePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotaIncreaseRequestsInTemplatePublisher publisher = client.listServiceQuotaIncreaseRequestsInTemplatePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServiceQuotaIncreaseRequestsInTemplate(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateRequest)}
     * operation.</b>
     * </p>
     *
     * @param listServiceQuotaIncreaseRequestsInTemplateRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotaIncreaseRequestsInTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotaIncreaseRequestsInTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    public ListServiceQuotaIncreaseRequestsInTemplatePublisher listServiceQuotaIncreaseRequestsInTemplatePaginator(
            ListServiceQuotaIncreaseRequestsInTemplateRequest listServiceQuotaIncreaseRequestsInTemplateRequest) {
        return new ListServiceQuotaIncreaseRequestsInTemplatePublisher(this,
                applyPaginatorUserAgent(listServiceQuotaIncreaseRequestsInTemplateRequest));
    }

    /**
     * <p>
     * Lists all service quotas for the specified AWS service. This request returns a list of the service quotas for the
     * specified service. you'll see the default values are the values that AWS provides for the quotas.
     * </p>
     * <note>
     * <p>
     * Always check the <code>NextToken</code> response parameter when calling any of the <code>List*</code> operations.
     * These operations can return an unexpected list of results, even when there are more results available. When this
     * happens, the <code>NextToken</code> response parameter contains a value to pass the next call to the same API to
     * request the next part of the list.
     * </p>
     * </note>
     *
     * @param listServiceQuotasRequest
     * @return A Java Future containing the result of the ListServiceQuotas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceQuotasResponse> listServiceQuotas(ListServiceQuotasRequest listServiceQuotasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceQuotasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceQuotasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceQuotasRequest, ListServiceQuotasResponse>()
                            .withOperationName("ListServiceQuotas")
                            .withMarshaller(new ListServiceQuotasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listServiceQuotasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listServiceQuotasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListServiceQuotasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all service quotas for the specified AWS service. This request returns a list of the service quotas for the
     * specified service. you'll see the default values are the values that AWS provides for the quotas.
     * </p>
     * <note>
     * <p>
     * Always check the <code>NextToken</code> response parameter when calling any of the <code>List*</code> operations.
     * These operations can return an unexpected list of results, even when there are more results available. When this
     * happens, the <code>NextToken</code> response parameter contains a value to pass the next call to the same API to
     * request the next part of the list.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listServiceQuotas(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotasPublisher publisher = client.listServiceQuotasPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServiceQuotasPublisher publisher = client.listServiceQuotasPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServiceQuotas(software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasRequest)}
     * operation.</b>
     * </p>
     *
     * @param listServiceQuotasRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotas
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    public ListServiceQuotasPublisher listServiceQuotasPaginator(ListServiceQuotasRequest listServiceQuotasRequest) {
        return new ListServiceQuotasPublisher(this, applyPaginatorUserAgent(listServiceQuotasRequest));
    }

    /**
     * <p>
     * Lists the AWS services available in Service Quotas. Not all AWS services are available in Service Quotas. To list
     * the see the list of the service quotas for a specific service, use <a>ListServiceQuotas</a>.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listServicesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listServicesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the AWS services available in Service Quotas. Not all AWS services are available in Service Quotas. To list
     * the see the list of the service quotas for a specific service, use <a>ListServiceQuotas</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listServices(software.amazon.awssdk.services.servicequotas.model.ListServicesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.servicequotas.paginators.ListServicesPublisher publisher = client.listServicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.servicequotas.model.ListServicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.servicequotas.model.ListServicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listServices(software.amazon.awssdk.services.servicequotas.model.ListServicesRequest)} operation.</b>
     * </p>
     *
     * @param listServicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    public ListServicesPublisher listServicesPaginator(ListServicesRequest listServicesRequest) {
        return new ListServicesPublisher(this, applyPaginatorUserAgent(listServicesRequest));
    }

    /**
     * <p>
     * Defines and adds a quota to the service quota template. To add a quota to the template, you must provide the
     * <code>ServiceCode</code>, <code>QuotaCode</code>, <code>AwsRegion</code>, and <code>DesiredValue</code>. Once you
     * add a quota to the template, use <a>ListServiceQuotaIncreaseRequestsInTemplate</a> to see the list of quotas in
     * the template.
     * </p>
     *
     * @param putServiceQuotaIncreaseRequestIntoTemplateRequest
     * @return A Java Future containing the result of the PutServiceQuotaIncreaseRequestIntoTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>QuotaExceededException You have exceeded your service quota. To perform the requested action, remove
     *         some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization. To enable, call
     *         <a>AssociateServiceQuotaTemplate</a>.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in the Region
     *         where you are making the request. Please make the request in us-east-1.</li>
     *         <li>NoAvailableOrganizationException The account making this call is not a member of an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.PutServiceQuotaIncreaseRequestIntoTemplate
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/PutServiceQuotaIncreaseRequestIntoTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> putServiceQuotaIncreaseRequestIntoTemplate(
            PutServiceQuotaIncreaseRequestIntoTemplateRequest putServiceQuotaIncreaseRequestIntoTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putServiceQuotaIncreaseRequestIntoTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutServiceQuotaIncreaseRequestIntoTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutServiceQuotaIncreaseRequestIntoTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutServiceQuotaIncreaseRequestIntoTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutServiceQuotaIncreaseRequestIntoTemplateRequest, PutServiceQuotaIncreaseRequestIntoTemplateResponse>()
                            .withOperationName("PutServiceQuotaIncreaseRequestIntoTemplate")
                            .withMarshaller(new PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(putServiceQuotaIncreaseRequestIntoTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putServiceQuotaIncreaseRequestIntoTemplateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of a service quota increase request. The response to this command provides the details in
     * the <a>RequestedServiceQuotaChange</a> object.
     * </p>
     *
     * @param requestServiceQuotaIncreaseRequest
     * @return A Java Future containing the result of the RequestServiceQuotaIncrease operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>QuotaExceededException You have exceeded your service quota. To perform the requested action, remove
     *         some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidResourceStateException Invalid input was provided for the .</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.RequestServiceQuotaIncrease
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/RequestServiceQuotaIncrease"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RequestServiceQuotaIncreaseResponse> requestServiceQuotaIncrease(
            RequestServiceQuotaIncreaseRequest requestServiceQuotaIncreaseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestServiceQuotaIncreaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RequestServiceQuotaIncrease");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RequestServiceQuotaIncreaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RequestServiceQuotaIncreaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RequestServiceQuotaIncreaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RequestServiceQuotaIncreaseRequest, RequestServiceQuotaIncreaseResponse>()
                            .withOperationName("RequestServiceQuotaIncrease")
                            .withMarshaller(new RequestServiceQuotaIncreaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(requestServiceQuotaIncreaseRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = requestServiceQuotaIncreaseRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RequestServiceQuotaIncreaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceQuotasException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalArgumentException")
                                .exceptionBuilderSupplier(IllegalArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                                .exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException")
                                .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchResourceException")
                                .exceptionBuilderSupplier(NoSuchResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QuotaExceededException")
                                .exceptionBuilderSupplier(QuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException")
                                .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                                .exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                                .exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ServiceQuotasRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
