/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicequotas;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.servicequotas.internal.ServiceQuotasServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.servicequotas.model.AccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.AwsServiceAccessNotEnabledException;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.DependencyAccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeRequest;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.IllegalArgumentException;
import software.amazon.awssdk.services.servicequotas.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.servicequotas.model.InvalidResourceStateException;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServicesRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServicesResponse;
import software.amazon.awssdk.services.servicequotas.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.servicequotas.model.NoAvailableOrganizationException;
import software.amazon.awssdk.services.servicequotas.model.NoSuchResourceException;
import software.amazon.awssdk.services.servicequotas.model.OrganizationNotInAllFeaturesModeException;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.QuotaExceededException;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseRequest;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseResponse;
import software.amazon.awssdk.services.servicequotas.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.servicequotas.model.ServiceException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotaTemplateNotInUseException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotasException;
import software.amazon.awssdk.services.servicequotas.model.TagPolicyViolationException;
import software.amazon.awssdk.services.servicequotas.model.TagResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.TagResourceResponse;
import software.amazon.awssdk.services.servicequotas.model.TemplatesNotAvailableInRegionException;
import software.amazon.awssdk.services.servicequotas.model.TooManyRequestsException;
import software.amazon.awssdk.services.servicequotas.model.TooManyTagsException;
import software.amazon.awssdk.services.servicequotas.model.UntagResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.UntagResourceResponse;
import software.amazon.awssdk.services.servicequotas.transform.AssociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DisassociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAssociationForServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAwsDefaultServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetRequestedServiceQuotaChangeRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListAwsDefaultServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.RequestServiceQuotaIncreaseRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ServiceQuotasAsyncClient}.
 *
 * @see ServiceQuotasAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceQuotasAsyncClient implements ServiceQuotasAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultServiceQuotasAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceQuotasAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates your quota request template with your organization. When a new Amazon Web Services account is created
     * in your organization, the quota increase requests in the template are automatically applied to the account. You
     * can add a quota increase request for any adjustable quota to your template.
     * </p>
     *
     * @param associateServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the AssociateServiceQuotaTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>OrganizationNotInAllFeaturesModeException The organization that your Amazon Web Services account
     *         belongs to is not in All Features mode.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.AssociateServiceQuotaTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/AssociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateServiceQuotaTemplateResponse> associateServiceQuotaTemplate(
            AssociateServiceQuotaTemplateRequest associateServiceQuotaTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateServiceQuotaTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateServiceQuotaTemplateRequest, AssociateServiceQuotaTemplateResponse>()
                            .withOperationName("AssociateServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateServiceQuotaTemplateRequest));
            CompletableFuture<AssociateServiceQuotaTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the quota increase request for the specified quota from your quota request template.
     * </p>
     *
     * @param deleteServiceQuotaIncreaseRequestFromTemplateRequest
     * @return A Java Future containing the result of the DeleteServiceQuotaIncreaseRequestFromTemplate operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.DeleteServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DeleteServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> deleteServiceQuotaIncreaseRequestFromTemplate(
            DeleteServiceQuotaIncreaseRequestFromTemplateRequest deleteServiceQuotaIncreaseRequestFromTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deleteServiceQuotaIncreaseRequestFromTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServiceQuotaIncreaseRequestFromTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteServiceQuotaIncreaseRequestFromTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceQuotaIncreaseRequestFromTemplateRequest, DeleteServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("DeleteServiceQuotaIncreaseRequestFromTemplate")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceQuotaIncreaseRequestFromTemplateRequest));
            CompletableFuture<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables your quota request template. After a template is disabled, the quota increase requests in the template
     * are not applied to new Amazon Web Services accounts in your organization. Disabling a quota request template does
     * not apply its quota increase requests.
     * </p>
     *
     * @param disassociateServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the DisassociateServiceQuotaTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceQuotaTemplateNotInUseException The quota request template is not associated with your
     *         organization.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.DisassociateServiceQuotaTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DisassociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateServiceQuotaTemplateResponse> disassociateServiceQuotaTemplate(
            DisassociateServiceQuotaTemplateRequest disassociateServiceQuotaTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateServiceQuotaTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateServiceQuotaTemplateRequest, DisassociateServiceQuotaTemplateResponse>()
                            .withOperationName("DisassociateServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateServiceQuotaTemplateRequest));
            CompletableFuture<DisassociateServiceQuotaTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the default value for the specified quota. The default value does not reflect any quota increases.
     * </p>
     *
     * @param getAwsDefaultServiceQuotaRequest
     * @return A Java Future containing the result of the GetAWSDefaultServiceQuota operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetAWSDefaultServiceQuota
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAWSDefaultServiceQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAwsDefaultServiceQuotaResponse> getAWSDefaultServiceQuota(
            GetAwsDefaultServiceQuotaRequest getAwsDefaultServiceQuotaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAwsDefaultServiceQuotaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAwsDefaultServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAWSDefaultServiceQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAwsDefaultServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAwsDefaultServiceQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAwsDefaultServiceQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAwsDefaultServiceQuotaRequest, GetAwsDefaultServiceQuotaResponse>()
                            .withOperationName("GetAWSDefaultServiceQuota").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAwsDefaultServiceQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAwsDefaultServiceQuotaRequest));
            CompletableFuture<GetAwsDefaultServiceQuotaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the status of the association for the quota request template.
     * </p>
     *
     * @param getAssociationForServiceQuotaTemplateRequest
     * @return A Java Future containing the result of the GetAssociationForServiceQuotaTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceQuotaTemplateNotInUseException The quota request template is not associated with your
     *         organization.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetAssociationForServiceQuotaTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAssociationForServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> getAssociationForServiceQuotaTemplate(
            GetAssociationForServiceQuotaTemplateRequest getAssociationForServiceQuotaTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAssociationForServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAssociationForServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssociationForServiceQuotaTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAssociationForServiceQuotaTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAssociationForServiceQuotaTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAssociationForServiceQuotaTemplateRequest, GetAssociationForServiceQuotaTemplateResponse>()
                            .withOperationName("GetAssociationForServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAssociationForServiceQuotaTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAssociationForServiceQuotaTemplateRequest));
            CompletableFuture<GetAssociationForServiceQuotaTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified quota increase request.
     * </p>
     *
     * @param getRequestedServiceQuotaChangeRequest
     * @return A Java Future containing the result of the GetRequestedServiceQuotaChange operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetRequestedServiceQuotaChange
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetRequestedServiceQuotaChange"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRequestedServiceQuotaChangeResponse> getRequestedServiceQuotaChange(
            GetRequestedServiceQuotaChangeRequest getRequestedServiceQuotaChangeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRequestedServiceQuotaChangeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRequestedServiceQuotaChangeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRequestedServiceQuotaChange");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRequestedServiceQuotaChangeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRequestedServiceQuotaChangeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRequestedServiceQuotaChangeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRequestedServiceQuotaChangeRequest, GetRequestedServiceQuotaChangeResponse>()
                            .withOperationName("GetRequestedServiceQuotaChange").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRequestedServiceQuotaChangeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRequestedServiceQuotaChangeRequest));
            CompletableFuture<GetRequestedServiceQuotaChangeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the applied quota value for the specified quota. For some quotas, only the default values are
     * available. If the applied quota value is not available for a quota, the quota is not retrieved.
     * </p>
     *
     * @param getServiceQuotaRequest
     * @return A Java Future containing the result of the GetServiceQuota operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetServiceQuota
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceQuotaResponse> getServiceQuota(GetServiceQuotaRequest getServiceQuotaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceQuotaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceQuotaRequest, GetServiceQuotaResponse>()
                            .withOperationName("GetServiceQuota").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceQuotaRequest));
            CompletableFuture<GetServiceQuotaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified quota increase request in your quota request template.
     * </p>
     *
     * @param getServiceQuotaIncreaseRequestFromTemplateRequest
     * @return A Java Future containing the result of the GetServiceQuotaIncreaseRequestFromTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.GetServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> getServiceQuotaIncreaseRequestFromTemplate(
            GetServiceQuotaIncreaseRequestFromTemplateRequest getServiceQuotaIncreaseRequestFromTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getServiceQuotaIncreaseRequestFromTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuotaIncreaseRequestFromTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetServiceQuotaIncreaseRequestFromTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceQuotaIncreaseRequestFromTemplateRequest, GetServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("GetServiceQuotaIncreaseRequestFromTemplate")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceQuotaIncreaseRequestFromTemplateRequest));
            CompletableFuture<GetServiceQuotaIncreaseRequestFromTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the default values for the quotas for the specified Amazon Web Service. A default value does not reflect
     * any quota increases.
     * </p>
     *
     * @param listAwsDefaultServiceQuotasRequest
     * @return A Java Future containing the result of the ListAWSDefaultServiceQuotas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListAWSDefaultServiceQuotas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListAWSDefaultServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAwsDefaultServiceQuotasResponse> listAWSDefaultServiceQuotas(
            ListAwsDefaultServiceQuotasRequest listAwsDefaultServiceQuotasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAwsDefaultServiceQuotasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAwsDefaultServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAWSDefaultServiceQuotas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAwsDefaultServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAwsDefaultServiceQuotasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAwsDefaultServiceQuotasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAwsDefaultServiceQuotasRequest, ListAwsDefaultServiceQuotasResponse>()
                            .withOperationName("ListAWSDefaultServiceQuotas").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAwsDefaultServiceQuotasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAwsDefaultServiceQuotasRequest));
            CompletableFuture<ListAwsDefaultServiceQuotasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the quota increase requests for the specified Amazon Web Service.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryRequest
     * @return A Java Future containing the result of the ListRequestedServiceQuotaChangeHistory operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistory
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> listRequestedServiceQuotaChangeHistory(
            ListRequestedServiceQuotaChangeHistoryRequest listRequestedServiceQuotaChangeHistoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRequestedServiceQuotaChangeHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryRequest, ListRequestedServiceQuotaChangeHistoryResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRequestedServiceQuotaChangeHistoryRequest));
            CompletableFuture<ListRequestedServiceQuotaChangeHistoryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the quota increase requests for the specified quota.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryByQuotaRequest
     * @return A Java Future containing the result of the ListRequestedServiceQuotaChangeHistoryByQuota operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListRequestedServiceQuotaChangeHistoryByQuota
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistoryByQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> listRequestedServiceQuotaChangeHistoryByQuota(
            ListRequestedServiceQuotaChangeHistoryByQuotaRequest listRequestedServiceQuotaChangeHistoryByQuotaRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listRequestedServiceQuotaChangeHistoryByQuotaRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryByQuotaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistoryByQuota");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryByQuotaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryByQuotaRequest, ListRequestedServiceQuotaChangeHistoryByQuotaResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistoryByQuota")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRequestedServiceQuotaChangeHistoryByQuotaRequest));
            CompletableFuture<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the quota increase requests in the specified quota request template.
     * </p>
     *
     * @param listServiceQuotaIncreaseRequestsInTemplateRequest
     * @return A Java Future containing the result of the ListServiceQuotaIncreaseRequestsInTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotaIncreaseRequestsInTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotaIncreaseRequestsInTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> listServiceQuotaIncreaseRequestsInTemplate(
            ListServiceQuotaIncreaseRequestsInTemplateRequest listServiceQuotaIncreaseRequestsInTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listServiceQuotaIncreaseRequestsInTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listServiceQuotaIncreaseRequestsInTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotaIncreaseRequestsInTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceQuotaIncreaseRequestsInTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListServiceQuotaIncreaseRequestsInTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceQuotaIncreaseRequestsInTemplateRequest, ListServiceQuotaIncreaseRequestsInTemplateResponse>()
                            .withOperationName("ListServiceQuotaIncreaseRequestsInTemplate")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceQuotaIncreaseRequestsInTemplateRequest));
            CompletableFuture<ListServiceQuotaIncreaseRequestsInTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the applied quota values for the specified Amazon Web Service. For some quotas, only the default values are
     * available. If the applied quota value is not available for a quota, the quota is not retrieved.
     * </p>
     *
     * @param listServiceQuotasRequest
     * @return A Java Future containing the result of the ListServiceQuotas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServiceQuotas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceQuotasResponse> listServiceQuotas(ListServiceQuotasRequest listServiceQuotasRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceQuotasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceQuotasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceQuotasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceQuotasRequest, ListServiceQuotasResponse>()
                            .withOperationName("ListServiceQuotas").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceQuotasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceQuotasRequest));
            CompletableFuture<ListServiceQuotasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the names and codes for the Amazon Web Services integrated with Service Quotas.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidPaginationTokenException Invalid input was provided.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesRequest));
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified applied quota.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a quota increase request to your quota request template.
     * </p>
     *
     * @param putServiceQuotaIncreaseRequestIntoTemplateRequest
     * @return A Java Future containing the result of the PutServiceQuotaIncreaseRequestIntoTemplate operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>QuotaExceededException You have exceeded your service quota. To perform the requested action, remove
     *         some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>AwsServiceAccessNotEnabledException The action you attempted is not allowed unless Service Access
     *         with Service Quotas is enabled in your organization.</li>
     *         <li>TemplatesNotAvailableInRegionException The Service Quotas template is not available in this Amazon
     *         Web Services Region.</li>
     *         <li>NoAvailableOrganizationException The Amazon Web Services account making this call is not a member of
     *         an organization.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.PutServiceQuotaIncreaseRequestIntoTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/PutServiceQuotaIncreaseRequestIntoTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> putServiceQuotaIncreaseRequestIntoTemplate(
            PutServiceQuotaIncreaseRequestIntoTemplateRequest putServiceQuotaIncreaseRequestIntoTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                putServiceQuotaIncreaseRequestIntoTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putServiceQuotaIncreaseRequestIntoTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutServiceQuotaIncreaseRequestIntoTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutServiceQuotaIncreaseRequestIntoTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutServiceQuotaIncreaseRequestIntoTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutServiceQuotaIncreaseRequestIntoTemplateRequest, PutServiceQuotaIncreaseRequestIntoTemplateResponse>()
                            .withOperationName("PutServiceQuotaIncreaseRequestIntoTemplate")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putServiceQuotaIncreaseRequestIntoTemplateRequest));
            CompletableFuture<PutServiceQuotaIncreaseRequestIntoTemplateResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits a quota increase request for the specified quota.
     * </p>
     *
     * @param requestServiceQuotaIncreaseRequest
     * @return A Java Future containing the result of the RequestServiceQuotaIncrease operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DependencyAccessDeniedException You can't perform this action because a dependency does not have
     *         access.</li>
     *         <li>QuotaExceededException You have exceeded your service quota. To perform the requested action, remove
     *         some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>InvalidResourceStateException The resource is in an invalid state.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.RequestServiceQuotaIncrease
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/RequestServiceQuotaIncrease"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RequestServiceQuotaIncreaseResponse> requestServiceQuotaIncrease(
            RequestServiceQuotaIncreaseRequest requestServiceQuotaIncreaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(requestServiceQuotaIncreaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestServiceQuotaIncreaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RequestServiceQuotaIncrease");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RequestServiceQuotaIncreaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RequestServiceQuotaIncreaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RequestServiceQuotaIncreaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RequestServiceQuotaIncreaseRequest, RequestServiceQuotaIncreaseResponse>()
                            .withOperationName("RequestServiceQuotaIncrease").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RequestServiceQuotaIncreaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(requestServiceQuotaIncreaseRequest));
            CompletableFuture<RequestServiceQuotaIncreaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified applied quota. You can include one or more tags to add to the quota.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>TooManyTagsException You've exceeded the number of tags allowed for a resource. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/servicequotas/latest/userguide/sq-tagging.html#sq-tagging-restrictions">Tag
     *         restrictions</a> in the <i>Service Quotas User Guide</i>.</li>
     *         <li>TagPolicyViolationException The specified tag is a reserved word and cannot be used.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified applied quota. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException Due to throttling, the request was denied. Slow down the rate of request
     *         calls, or request an increase for this quota.</li>
     *         <li>NoSuchResourceException The specified resource does not exist.</li>
     *         <li>IllegalArgumentException Invalid input was provided.</li>
     *         <li>AccessDeniedException You do not have sufficient permission to perform this action.</li>
     *         <li>ServiceException Something went wrong.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ServiceQuotasException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ServiceQuotasAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ServiceQuotasServiceClientConfiguration serviceClientConfiguration() {
        return new ServiceQuotasServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ServiceQuotasException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalArgumentException")
                                .exceptionBuilderSupplier(IllegalArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                                .exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException")
                                .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchResourceException")
                                .exceptionBuilderSupplier(NoSuchResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QuotaExceededException")
                                .exceptionBuilderSupplier(QuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException")
                                .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                                .exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyViolationException")
                                .exceptionBuilderSupplier(TagPolicyViolationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                                .exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).httpStatusCode(400)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ServiceQuotasServiceClientConfigurationBuilder serviceConfigBuilder = new ServiceQuotasServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
