/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.servicequotas;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.servicequotas.internal.ServiceQuotasServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.servicequotas.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.servicequotas.model.AccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.AssociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.AwsServiceAccessNotEnabledException;
import software.amazon.awssdk.services.servicequotas.model.CreateSupportCaseRequest;
import software.amazon.awssdk.services.servicequotas.model.CreateSupportCaseResponse;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DeleteServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.DependencyAccessDeniedException;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.DisassociateServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAssociationForServiceQuotaTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetAwsDefaultServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeRequest;
import software.amazon.awssdk.services.servicequotas.model.GetRequestedServiceQuotaChangeResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaIncreaseRequestFromTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.GetServiceQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.IllegalArgumentException;
import software.amazon.awssdk.services.servicequotas.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.servicequotas.model.InvalidResourceStateException;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListAwsDefaultServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryByQuotaResponse;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryRequest;
import software.amazon.awssdk.services.servicequotas.model.ListRequestedServiceQuotaChangeHistoryResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotaIncreaseRequestsInTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServiceQuotasResponse;
import software.amazon.awssdk.services.servicequotas.model.ListServicesRequest;
import software.amazon.awssdk.services.servicequotas.model.ListServicesResponse;
import software.amazon.awssdk.services.servicequotas.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.servicequotas.model.NoAvailableOrganizationException;
import software.amazon.awssdk.services.servicequotas.model.NoSuchResourceException;
import software.amazon.awssdk.services.servicequotas.model.OrganizationNotInAllFeaturesModeException;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateRequest;
import software.amazon.awssdk.services.servicequotas.model.PutServiceQuotaIncreaseRequestIntoTemplateResponse;
import software.amazon.awssdk.services.servicequotas.model.QuotaExceededException;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseRequest;
import software.amazon.awssdk.services.servicequotas.model.RequestServiceQuotaIncreaseResponse;
import software.amazon.awssdk.services.servicequotas.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.servicequotas.model.ServiceException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotaTemplateNotInUseException;
import software.amazon.awssdk.services.servicequotas.model.ServiceQuotasException;
import software.amazon.awssdk.services.servicequotas.model.TagPolicyViolationException;
import software.amazon.awssdk.services.servicequotas.model.TagResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.TagResourceResponse;
import software.amazon.awssdk.services.servicequotas.model.TemplatesNotAvailableInRegionException;
import software.amazon.awssdk.services.servicequotas.model.TooManyRequestsException;
import software.amazon.awssdk.services.servicequotas.model.TooManyTagsException;
import software.amazon.awssdk.services.servicequotas.model.UntagResourceRequest;
import software.amazon.awssdk.services.servicequotas.model.UntagResourceResponse;
import software.amazon.awssdk.services.servicequotas.transform.AssociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.CreateSupportCaseRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.DisassociateServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAssociationForServiceQuotaTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetAwsDefaultServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetRequestedServiceQuotaChangeRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.GetServiceQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListAwsDefaultServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListRequestedServiceQuotaChangeHistoryRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServiceQuotasRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.RequestServiceQuotaIncreaseRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.servicequotas.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ServiceQuotasClient}.
 *
 * @see ServiceQuotasClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultServiceQuotasClient implements ServiceQuotasClient {
    private static final Logger log = Logger.loggerFor(DefaultServiceQuotasClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultServiceQuotasClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Service_Quotas" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates your quota request template with your organization. When a new Amazon Web Services account is created
     * in your organization, the quota increase requests in the template are automatically applied to the account. You
     * can add a quota increase request for any adjustable quota to your template.
     * </p>
     *
     * @param associateServiceQuotaTemplateRequest
     * @return Result of the AssociateServiceQuotaTemplate operation returned by the service.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws OrganizationNotInAllFeaturesModeException
     *         The organization that your Amazon Web Services account belongs to is not in All Features mode.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.AssociateServiceQuotaTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/AssociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateServiceQuotaTemplateResponse associateServiceQuotaTemplate(
            AssociateServiceQuotaTemplateRequest associateServiceQuotaTemplateRequest) throws DependencyAccessDeniedException,
            AccessDeniedException, ServiceException, TooManyRequestsException, AwsServiceAccessNotEnabledException,
            OrganizationNotInAllFeaturesModeException, TemplatesNotAvailableInRegionException, NoAvailableOrganizationException,
            AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateServiceQuotaTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateServiceQuotaTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateServiceQuotaTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateServiceQuotaTemplateRequest, AssociateServiceQuotaTemplateResponse>()
                            .withOperationName("AssociateServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateServiceQuotaTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateServiceQuotaTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Support case for an existing quota increase request. This call only creates a Support case if the
     * request has a <code>Pending</code> status.
     * </p>
     *
     * @param createSupportCaseRequest
     * @return Result of the CreateSupportCase operation returned by the service.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ResourceAlreadyExistsException
     *         The specified resource already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidResourceStateException
     *         The resource is in an invalid state.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.CreateSupportCase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/CreateSupportCase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSupportCaseResponse createSupportCase(CreateSupportCaseRequest createSupportCaseRequest)
            throws DependencyAccessDeniedException, ResourceAlreadyExistsException, AccessDeniedException,
            NoSuchResourceException, IllegalArgumentException, InvalidResourceStateException, ServiceException,
            TooManyRequestsException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSupportCaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSupportCaseResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSupportCaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSupportCaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSupportCase");

            return clientHandler.execute(new ClientExecutionParams<CreateSupportCaseRequest, CreateSupportCaseResponse>()
                    .withOperationName("CreateSupportCase").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createSupportCaseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSupportCaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the quota increase request for the specified quota from your quota request template.
     * </p>
     *
     * @param deleteServiceQuotaIncreaseRequestFromTemplateRequest
     * @return Result of the DeleteServiceQuotaIncreaseRequestFromTemplate operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.DeleteServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DeleteServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteServiceQuotaIncreaseRequestFromTemplateResponse deleteServiceQuotaIncreaseRequestFromTemplate(
            DeleteServiceQuotaIncreaseRequestFromTemplateRequest deleteServiceQuotaIncreaseRequestFromTemplateRequest)
            throws AccessDeniedException, ServiceException, DependencyAccessDeniedException, TooManyRequestsException,
            NoSuchResourceException, IllegalArgumentException, AwsServiceAccessNotEnabledException,
            TemplatesNotAvailableInRegionException, NoAvailableOrganizationException, AwsServiceException, SdkClientException,
            ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DeleteServiceQuotaIncreaseRequestFromTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deleteServiceQuotaIncreaseRequestFromTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServiceQuotaIncreaseRequestFromTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceQuotaIncreaseRequestFromTemplateRequest, DeleteServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("DeleteServiceQuotaIncreaseRequestFromTemplate")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deleteServiceQuotaIncreaseRequestFromTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables your quota request template. After a template is disabled, the quota increase requests in the template
     * are not applied to new Amazon Web Services accounts in your organization. Disabling a quota request template does
     * not apply its quota increase requests.
     * </p>
     *
     * @param disassociateServiceQuotaTemplateRequest
     * @return Result of the DisassociateServiceQuotaTemplate operation returned by the service.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ServiceQuotaTemplateNotInUseException
     *         The quota request template is not associated with your organization.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.DisassociateServiceQuotaTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/DisassociateServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateServiceQuotaTemplateResponse disassociateServiceQuotaTemplate(
            DisassociateServiceQuotaTemplateRequest disassociateServiceQuotaTemplateRequest)
            throws DependencyAccessDeniedException, ServiceQuotaTemplateNotInUseException, AccessDeniedException,
            ServiceException, TooManyRequestsException, AwsServiceAccessNotEnabledException,
            TemplatesNotAvailableInRegionException, NoAvailableOrganizationException, AwsServiceException, SdkClientException,
            ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateServiceQuotaTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateServiceQuotaTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateServiceQuotaTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateServiceQuotaTemplateRequest, DisassociateServiceQuotaTemplateResponse>()
                            .withOperationName("DisassociateServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateServiceQuotaTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateServiceQuotaTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the default value for the specified quota. The default value does not reflect any quota increases.
     * </p>
     *
     * @param getAwsDefaultServiceQuotaRequest
     * @return Result of the GetAWSDefaultServiceQuota operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.GetAWSDefaultServiceQuota
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAWSDefaultServiceQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAwsDefaultServiceQuotaResponse getAWSDefaultServiceQuota(
            GetAwsDefaultServiceQuotaRequest getAwsDefaultServiceQuotaRequest) throws AccessDeniedException,
            NoSuchResourceException, IllegalArgumentException, ServiceException, TooManyRequestsException, AwsServiceException,
            SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAwsDefaultServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAwsDefaultServiceQuotaResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAwsDefaultServiceQuotaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAwsDefaultServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAWSDefaultServiceQuota");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAwsDefaultServiceQuotaRequest, GetAwsDefaultServiceQuotaResponse>()
                            .withOperationName("GetAWSDefaultServiceQuota").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getAwsDefaultServiceQuotaRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAwsDefaultServiceQuotaRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the status of the association for the quota request template.
     * </p>
     *
     * @param getAssociationForServiceQuotaTemplateRequest
     * @return Result of the GetAssociationForServiceQuotaTemplate operation returned by the service.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ServiceQuotaTemplateNotInUseException
     *         The quota request template is not associated with your organization.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.GetAssociationForServiceQuotaTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetAssociationForServiceQuotaTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAssociationForServiceQuotaTemplateResponse getAssociationForServiceQuotaTemplate(
            GetAssociationForServiceQuotaTemplateRequest getAssociationForServiceQuotaTemplateRequest)
            throws DependencyAccessDeniedException, ServiceQuotaTemplateNotInUseException, AccessDeniedException,
            ServiceException, TooManyRequestsException, AwsServiceAccessNotEnabledException,
            TemplatesNotAvailableInRegionException, NoAvailableOrganizationException, AwsServiceException, SdkClientException,
            ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssociationForServiceQuotaTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetAssociationForServiceQuotaTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAssociationForServiceQuotaTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAssociationForServiceQuotaTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAssociationForServiceQuotaTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAssociationForServiceQuotaTemplateRequest, GetAssociationForServiceQuotaTemplateResponse>()
                            .withOperationName("GetAssociationForServiceQuotaTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getAssociationForServiceQuotaTemplateRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAssociationForServiceQuotaTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified quota increase request.
     * </p>
     *
     * @param getRequestedServiceQuotaChangeRequest
     * @return Result of the GetRequestedServiceQuotaChange operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.GetRequestedServiceQuotaChange
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetRequestedServiceQuotaChange"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRequestedServiceQuotaChangeResponse getRequestedServiceQuotaChange(
            GetRequestedServiceQuotaChangeRequest getRequestedServiceQuotaChangeRequest) throws AccessDeniedException,
            NoSuchResourceException, IllegalArgumentException, ServiceException, TooManyRequestsException, AwsServiceException,
            SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRequestedServiceQuotaChangeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRequestedServiceQuotaChangeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRequestedServiceQuotaChangeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRequestedServiceQuotaChangeRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRequestedServiceQuotaChange");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRequestedServiceQuotaChangeRequest, GetRequestedServiceQuotaChangeResponse>()
                            .withOperationName("GetRequestedServiceQuotaChange").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getRequestedServiceQuotaChangeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRequestedServiceQuotaChangeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the applied quota value for the specified account-level or resource-level quota. For some quotas, only
     * the default values are available. If the applied quota value is not available for a quota, the quota is not
     * retrieved.
     * </p>
     *
     * @param getServiceQuotaRequest
     * @return Result of the GetServiceQuota operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.GetServiceQuota
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceQuotaResponse getServiceQuota(GetServiceQuotaRequest getServiceQuotaRequest) throws AccessDeniedException,
            NoSuchResourceException, IllegalArgumentException, ServiceException, TooManyRequestsException, AwsServiceException,
            SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceQuotaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceQuotaResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceQuotaRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceQuotaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuota");

            return clientHandler.execute(new ClientExecutionParams<GetServiceQuotaRequest, GetServiceQuotaResponse>()
                    .withOperationName("GetServiceQuota").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getServiceQuotaRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServiceQuotaRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about the specified quota increase request in your quota request template.
     * </p>
     *
     * @param getServiceQuotaIncreaseRequestFromTemplateRequest
     * @return Result of the GetServiceQuotaIncreaseRequestFromTemplate operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.GetServiceQuotaIncreaseRequestFromTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/GetServiceQuotaIncreaseRequestFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceQuotaIncreaseRequestFromTemplateResponse getServiceQuotaIncreaseRequestFromTemplate(
            GetServiceQuotaIncreaseRequestFromTemplateRequest getServiceQuotaIncreaseRequestFromTemplateRequest)
            throws AccessDeniedException, DependencyAccessDeniedException, ServiceException, TooManyRequestsException,
            NoSuchResourceException, IllegalArgumentException, AwsServiceAccessNotEnabledException,
            TemplatesNotAvailableInRegionException, NoAvailableOrganizationException, AwsServiceException, SdkClientException,
            ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceQuotaIncreaseRequestFromTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetServiceQuotaIncreaseRequestFromTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getServiceQuotaIncreaseRequestFromTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getServiceQuotaIncreaseRequestFromTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceQuotaIncreaseRequestFromTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<GetServiceQuotaIncreaseRequestFromTemplateRequest, GetServiceQuotaIncreaseRequestFromTemplateResponse>()
                            .withOperationName("GetServiceQuotaIncreaseRequestFromTemplate")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getServiceQuotaIncreaseRequestFromTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetServiceQuotaIncreaseRequestFromTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the default values for the quotas for the specified Amazon Web Services service. A default value does not
     * reflect any quota increases.
     * </p>
     *
     * @param listAwsDefaultServiceQuotasRequest
     * @return Result of the ListAWSDefaultServiceQuotas operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidPaginationTokenException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListAWSDefaultServiceQuotas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListAWSDefaultServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAwsDefaultServiceQuotasResponse listAWSDefaultServiceQuotas(
            ListAwsDefaultServiceQuotasRequest listAwsDefaultServiceQuotasRequest) throws AccessDeniedException,
            NoSuchResourceException, IllegalArgumentException, InvalidPaginationTokenException, ServiceException,
            TooManyRequestsException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAwsDefaultServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAwsDefaultServiceQuotasResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAwsDefaultServiceQuotasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAwsDefaultServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAWSDefaultServiceQuotas");

            return clientHandler
                    .execute(new ClientExecutionParams<ListAwsDefaultServiceQuotasRequest, ListAwsDefaultServiceQuotasResponse>()
                            .withOperationName("ListAWSDefaultServiceQuotas").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listAwsDefaultServiceQuotasRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListAwsDefaultServiceQuotasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the quota increase requests for the specified Amazon Web Services service. Filter responses to return
     * quota requests at either the account level, resource level, or all levels. Responses include any open or closed
     * requests within 90 days.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryRequest
     * @return Result of the ListRequestedServiceQuotaChangeHistory operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidPaginationTokenException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListRequestedServiceQuotaChangeHistory
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRequestedServiceQuotaChangeHistoryResponse listRequestedServiceQuotaChangeHistory(
            ListRequestedServiceQuotaChangeHistoryRequest listRequestedServiceQuotaChangeHistoryRequest)
            throws AccessDeniedException, NoSuchResourceException, IllegalArgumentException, InvalidPaginationTokenException,
            ServiceException, TooManyRequestsException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRequestedServiceQuotaChangeHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistory");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryRequest, ListRequestedServiceQuotaChangeHistoryResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(listRequestedServiceQuotaChangeHistoryRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the quota increase requests for the specified quota. Filter responses to return quota requests at
     * either the account level, resource level, or all levels.
     * </p>
     *
     * @param listRequestedServiceQuotaChangeHistoryByQuotaRequest
     * @return Result of the ListRequestedServiceQuotaChangeHistoryByQuota operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidPaginationTokenException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListRequestedServiceQuotaChangeHistoryByQuota
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListRequestedServiceQuotaChangeHistoryByQuota"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRequestedServiceQuotaChangeHistoryByQuotaResponse listRequestedServiceQuotaChangeHistoryByQuota(
            ListRequestedServiceQuotaChangeHistoryByQuotaRequest listRequestedServiceQuotaChangeHistoryByQuotaRequest)
            throws AccessDeniedException, NoSuchResourceException, IllegalArgumentException, InvalidPaginationTokenException,
            ServiceException, TooManyRequestsException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRequestedServiceQuotaChangeHistoryByQuotaResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListRequestedServiceQuotaChangeHistoryByQuotaResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listRequestedServiceQuotaChangeHistoryByQuotaRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRequestedServiceQuotaChangeHistoryByQuotaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequestedServiceQuotaChangeHistoryByQuota");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRequestedServiceQuotaChangeHistoryByQuotaRequest, ListRequestedServiceQuotaChangeHistoryByQuotaResponse>()
                            .withOperationName("ListRequestedServiceQuotaChangeHistoryByQuota")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(listRequestedServiceQuotaChangeHistoryByQuotaRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRequestedServiceQuotaChangeHistoryByQuotaRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the quota increase requests in the specified quota request template.
     * </p>
     *
     * @param listServiceQuotaIncreaseRequestsInTemplateRequest
     * @return Result of the ListServiceQuotaIncreaseRequestsInTemplate operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListServiceQuotaIncreaseRequestsInTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotaIncreaseRequestsInTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceQuotaIncreaseRequestsInTemplateResponse listServiceQuotaIncreaseRequestsInTemplate(
            ListServiceQuotaIncreaseRequestsInTemplateRequest listServiceQuotaIncreaseRequestsInTemplateRequest)
            throws AccessDeniedException, DependencyAccessDeniedException, ServiceException, TooManyRequestsException,
            IllegalArgumentException, AwsServiceAccessNotEnabledException, TemplatesNotAvailableInRegionException,
            NoAvailableOrganizationException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceQuotaIncreaseRequestsInTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListServiceQuotaIncreaseRequestsInTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listServiceQuotaIncreaseRequestsInTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listServiceQuotaIncreaseRequestsInTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotaIncreaseRequestsInTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<ListServiceQuotaIncreaseRequestsInTemplateRequest, ListServiceQuotaIncreaseRequestsInTemplateResponse>()
                            .withOperationName("ListServiceQuotaIncreaseRequestsInTemplate")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(listServiceQuotaIncreaseRequestsInTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListServiceQuotaIncreaseRequestsInTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the applied quota values for the specified Amazon Web Services service. For some quotas, only the default
     * values are available. If the applied quota value is not available for a quota, the quota is not retrieved. Filter
     * responses to return applied quota values at either the account level, resource level, or all levels.
     * </p>
     *
     * @param listServiceQuotasRequest
     * @return Result of the ListServiceQuotas operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidPaginationTokenException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListServiceQuotas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServiceQuotas"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceQuotasResponse listServiceQuotas(ListServiceQuotasRequest listServiceQuotasRequest)
            throws AccessDeniedException, NoSuchResourceException, IllegalArgumentException, InvalidPaginationTokenException,
            ServiceException, TooManyRequestsException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceQuotasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServiceQuotasResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceQuotasRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceQuotasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceQuotas");

            return clientHandler.execute(new ClientExecutionParams<ListServiceQuotasRequest, ListServiceQuotasResponse>()
                    .withOperationName("ListServiceQuotas").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServiceQuotasRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServiceQuotasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the names and codes for the Amazon Web Services services integrated with Service Quotas.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidPaginationTokenException
     *         Invalid input was provided.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListServices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws AccessDeniedException,
            IllegalArgumentException, InvalidPaginationTokenException, ServiceException, TooManyRequestsException,
            AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified applied quota.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws TooManyRequestsException, NoSuchResourceException, IllegalArgumentException, AccessDeniedException,
            ServiceException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a quota increase request to your quota request template.
     * </p>
     *
     * @param putServiceQuotaIncreaseRequestIntoTemplateRequest
     * @return Result of the PutServiceQuotaIncreaseRequestIntoTemplate operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws QuotaExceededException
     *         You have exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use Service Quotas to request a service quota increase.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws AwsServiceAccessNotEnabledException
     *         The action you attempted is not allowed unless Service Access with Service Quotas is enabled in your
     *         organization.
     * @throws TemplatesNotAvailableInRegionException
     *         The Service Quotas template is not available in this Amazon Web Services Region.
     * @throws NoAvailableOrganizationException
     *         The Amazon Web Services account making this call is not a member of an organization.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.PutServiceQuotaIncreaseRequestIntoTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/PutServiceQuotaIncreaseRequestIntoTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutServiceQuotaIncreaseRequestIntoTemplateResponse putServiceQuotaIncreaseRequestIntoTemplate(
            PutServiceQuotaIncreaseRequestIntoTemplateRequest putServiceQuotaIncreaseRequestIntoTemplateRequest)
            throws AccessDeniedException, DependencyAccessDeniedException, ServiceException, TooManyRequestsException,
            IllegalArgumentException, QuotaExceededException, NoSuchResourceException, AwsServiceAccessNotEnabledException,
            TemplatesNotAvailableInRegionException, NoAvailableOrganizationException, AwsServiceException, SdkClientException,
            ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutServiceQuotaIncreaseRequestIntoTemplateResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, PutServiceQuotaIncreaseRequestIntoTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                putServiceQuotaIncreaseRequestIntoTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putServiceQuotaIncreaseRequestIntoTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutServiceQuotaIncreaseRequestIntoTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<PutServiceQuotaIncreaseRequestIntoTemplateRequest, PutServiceQuotaIncreaseRequestIntoTemplateResponse>()
                            .withOperationName("PutServiceQuotaIncreaseRequestIntoTemplate")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(putServiceQuotaIncreaseRequestIntoTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutServiceQuotaIncreaseRequestIntoTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Submits a quota increase request for the specified quota at the account or resource level.
     * </p>
     *
     * @param requestServiceQuotaIncreaseRequest
     * @return Result of the RequestServiceQuotaIncrease operation returned by the service.
     * @throws DependencyAccessDeniedException
     *         You can't perform this action because a dependency does not have access.
     * @throws QuotaExceededException
     *         You have exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use Service Quotas to request a service quota increase.
     * @throws ResourceAlreadyExistsException
     *         The specified resource already exists.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws InvalidResourceStateException
     *         The resource is in an invalid state.
     * @throws ServiceException
     *         Something went wrong.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.RequestServiceQuotaIncrease
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/RequestServiceQuotaIncrease"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RequestServiceQuotaIncreaseResponse requestServiceQuotaIncrease(
            RequestServiceQuotaIncreaseRequest requestServiceQuotaIncreaseRequest) throws DependencyAccessDeniedException,
            QuotaExceededException, ResourceAlreadyExistsException, AccessDeniedException, NoSuchResourceException,
            IllegalArgumentException, InvalidResourceStateException, ServiceException, TooManyRequestsException,
            AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RequestServiceQuotaIncreaseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RequestServiceQuotaIncreaseResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(requestServiceQuotaIncreaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestServiceQuotaIncreaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RequestServiceQuotaIncrease");

            return clientHandler
                    .execute(new ClientExecutionParams<RequestServiceQuotaIncreaseRequest, RequestServiceQuotaIncreaseResponse>()
                            .withOperationName("RequestServiceQuotaIncrease").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(requestServiceQuotaIncreaseRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RequestServiceQuotaIncreaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to the specified applied quota. You can include one or more tags to add to the quota.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws TooManyTagsException
     *         You've exceeded the number of tags allowed for a resource. For more information, see <a href=
     *         "https://docs.aws.amazon.com/servicequotas/latest/userguide/sq-tagging.html#sq-tagging-restrictions">Tag
     *         restrictions</a> in the <i>Service Quotas User Guide</i>.
     * @throws TagPolicyViolationException
     *         The specified tag is a reserved word and cannot be used.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws TooManyRequestsException,
            NoSuchResourceException, TooManyTagsException, TagPolicyViolationException, IllegalArgumentException,
            AccessDeniedException, ServiceException, AwsServiceException, SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified applied quota. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws TooManyRequestsException
     *         Due to throttling, the request was denied. Slow down the rate of request calls, or request an increase
     *         for this quota.
     * @throws NoSuchResourceException
     *         The specified resource does not exist.
     * @throws IllegalArgumentException
     *         Invalid input was provided.
     * @throws AccessDeniedException
     *         You do not have sufficient permission to perform this action.
     * @throws ServiceException
     *         Something went wrong.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ServiceQuotasException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ServiceQuotasClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/service-quotas-2019-06-24/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws TooManyRequestsException,
            NoSuchResourceException, IllegalArgumentException, AccessDeniedException, ServiceException, AwsServiceException,
            SdkClientException, ServiceQuotasException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "InvalidResourceStateException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidResourceStateException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidResourceStateException::builder).build());
            case "IllegalArgumentException":
                return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                        .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
            case "ServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceException::builder).build());
            case "OrganizationNotInAllFeaturesModeException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .httpStatusCode(400).exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder)
                                .build());
            case "AWSServiceAccessNotEnabledException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AWSServiceAccessNotEnabledException")
                        .httpStatusCode(400).exceptionBuilderSupplier(AwsServiceAccessNotEnabledException::builder).build());
            case "DependencyAccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyAccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DependencyAccessDeniedException::builder).build());
            case "NoSuchResourceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoSuchResourceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoSuchResourceException::builder).build());
            case "QuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("QuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(QuotaExceededException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "NoAvailableOrganizationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoAvailableOrganizationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoAvailableOrganizationException::builder).build());
            case "TemplatesNotAvailableInRegionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplatesNotAvailableInRegionException")
                        .httpStatusCode(400).exceptionBuilderSupplier(TemplatesNotAvailableInRegionException::builder).build());
            case "InvalidPaginationTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "TagPolicyViolationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TagPolicyViolationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TagPolicyViolationException::builder).build());
            case "ServiceQuotaTemplateNotInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaTemplateNotInUseException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceQuotaTemplateNotInUseException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Service Quotas");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ServiceQuotasServiceClientConfigurationBuilder serviceConfigBuilder = new ServiceQuotasServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ServiceQuotasException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final ServiceQuotasServiceClientConfiguration serviceClientConfiguration() {
        return new ServiceQuotasServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
