/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * When included in a receipt rule, this action publishes a notification to Amazon Simple Notification Service (Amazon
 * SNS). This action includes a complete copy of the email content in the Amazon SNS notifications. Amazon SNS
 * notifications for all other actions simply provide information about the email. They do not include the email content
 * itself.
 * </p>
 * <p>
 * If you own the Amazon SNS topic, you don't need to do anything to give Amazon SES permission to publish emails to it.
 * However, if you don't own the Amazon SNS topic, you need to attach a policy to the topic to give Amazon SES
 * permissions to access it. For information about giving permissions, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 * <important>
 * <p>
 * You can only publish emails that are 150 KB or less (including the header) to Amazon SNS. Larger emails will bounce.
 * If you anticipate emails larger than 150 KB, use the S3 action instead.
 * </p>
 * </important>
 * <p>
 * For information about using a receipt rule to publish an Amazon SNS notification, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-sns.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SNSAction implements ToCopyableBuilder<SNSAction.Builder, SNSAction> {
    private final String topicArn;

    private final String encoding;

    private SNSAction(BuilderImpl builder) {
        this.topicArn = builder.topicArn;
        this.encoding = builder.encoding;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. An example of an Amazon SNS topic ARN is
     * <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about Amazon SNS topics, see the <a
     * href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. An example of an Amazon SNS topic ARN
     *         is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about Amazon SNS topics,
     *         see the <a href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer
     *         Guide</a>.
     */
    public String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves all
     * special characters. The default value is UTF-8.
     * </p>
     * 
     * @return The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
     *         preserve all special characters when a message was encoded with a different encoding format. Base64
     *         preserves all special characters. The default value is UTF-8.
     * @see SNSActionEncoding
     */
    public String encoding() {
        return encoding;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((topicArn() == null) ? 0 : topicArn().hashCode());
        hashCode = 31 * hashCode + ((encoding() == null) ? 0 : encoding().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SNSAction)) {
            return false;
        }
        SNSAction other = (SNSAction) obj;
        if (other.topicArn() == null ^ this.topicArn() == null) {
            return false;
        }
        if (other.topicArn() != null && !other.topicArn().equals(this.topicArn())) {
            return false;
        }
        if (other.encoding() == null ^ this.encoding() == null) {
            return false;
        }
        if (other.encoding() != null && !other.encoding().equals(this.encoding())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (topicArn() != null) {
            sb.append("TopicArn: ").append(topicArn()).append(",");
        }
        if (encoding() != null) {
            sb.append("Encoding: ").append(encoding()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, SNSAction> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. An example of an Amazon SNS topic ARN is
         * <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about Amazon SNS topics, see
         * the <a href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS Developer Guide</a>.
         * </p>
         * 
         * @param topicArn
         *        The Amazon Resource Name (ARN) of the Amazon SNS topic to notify. An example of an Amazon SNS topic
         *        ARN is <code>arn:aws:sns:us-west-2:123456789012:MyTopic</code>. For more information about Amazon SNS
         *        topics, see the <a href="http://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html">Amazon SNS
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
         * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves
         * all special characters. The default value is UTF-8.
         * </p>
         * 
         * @param encoding
         *        The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may
         *        not preserve all special characters when a message was encoded with a different encoding format.
         *        Base64 preserves all special characters. The default value is UTF-8.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SNSActionEncoding
         */
        Builder encoding(String encoding);

        /**
         * <p>
         * The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may not
         * preserve all special characters when a message was encoded with a different encoding format. Base64 preserves
         * all special characters. The default value is UTF-8.
         * </p>
         * 
         * @param encoding
         *        The encoding to use for the email within the Amazon SNS notification. UTF-8 is easier to use, but may
         *        not preserve all special characters when a message was encoded with a different encoding format.
         *        Base64 preserves all special characters. The default value is UTF-8.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SNSActionEncoding
         */
        Builder encoding(SNSActionEncoding encoding);
    }

    private static final class BuilderImpl implements Builder {
        private String topicArn;

        private String encoding;

        private BuilderImpl() {
        }

        private BuilderImpl(SNSAction model) {
            setTopicArn(model.topicArn);
            setEncoding(model.encoding);
        }

        public final String getTopicArn() {
            return topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        public final String getEncoding() {
            return encoding;
        }

        @Override
        public final Builder encoding(String encoding) {
            this.encoding = encoding;
            return this;
        }

        @Override
        public final Builder encoding(SNSActionEncoding encoding) {
            this.encoding(encoding.toString());
            return this;
        }

        public final void setEncoding(String encoding) {
            this.encoding = encoding;
        }

        public final void setEncoding(SNSActionEncoding encoding) {
            this.encoding(encoding.toString());
        }

        @Override
        public SNSAction build() {
            return new SNSAction(this);
        }
    }
}
