/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.AmazonWebServiceRequest;
import software.amazon.awssdk.runtime.SdkInternalList;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single raw email using Amazon SES. For more information, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-raw.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SendRawEmailRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<SendRawEmailRequest.Builder, SendRawEmailRequest> {
    private final String source;

    private final List<String> destinations;

    private final RawMessage rawMessage;

    private final String fromArn;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private SendRawEmailRequest(BuilderImpl builder) {
        this.source = builder.source;
        this.destinations = builder.destinations;
        this.rawMessage = builder.rawMessage;
        this.fromArn = builder.fromArn;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The identity's email address. If you do not provide a value for this parameter, you must specify a "From" address
     * in the raw text of the message. (You can also specify both.)
     * </p>
     * <p>
     * By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must use MIME
     * encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following form:
     * <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     * href="http://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     * </p>
     * <note>
     * <p>
     * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     * complaints will be sent to this email address. This takes precedence over any <i>Return-Path</i> header that you
     * might include in the raw text of the message.
     * </p>
     * </note>
     * 
     * @return The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
     *         address in the raw text of the message. (You can also specify both.)</p>
     *         <p>
     *         By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must
     *         use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the
     *         following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     *         href="http://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
     *         complaints will be sent to this email address. This takes precedence over any <i>Return-Path</i> header
     *         that you might include in the raw text of the message.
     *         </p>
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     * </p>
     * 
     * @return A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
     */
    public List<String> destinations() {
        return destinations;
    }

    /**
     * <p>
     * The raw text of the message. The client is responsible for ensuring the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Message must contain a header and a body, separated by a blank line.
     * </p>
     * </li>
     * <li>
     * <p>
     * All required header fields must be present.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each part of a multipart MIME message must be formatted properly.
     * </p>
     * </li>
     * <li>
     * <p>
     * MIME content types must be among those supported by Amazon SES. For more information, go to the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer Guide</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be base64-encoded.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The raw text of the message. The client is responsible for ensuring the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Message must contain a header and a body, separated by a blank line.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All required header fields must be present.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each part of a multipart MIME message must be formatted properly.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MIME content types must be among those supported by Amazon SES. For more information, go to the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer
     *         Guide</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be base64-encoded.
     *         </p>
     *         </li>
     */
    public RawMessage rawMessage() {
        return rawMessage;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to specify a particular "From" address in the header of the raw
     * email.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of the
     * email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>FromArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to specify a particular "From" address in the
     *         header of the raw email.</p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message
     *         of the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon
     *         SES uses the value of the <code>FromArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String fromArn() {
        return fromArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of the
     * email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES uses the
     * value of the <code>SourceArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
     *         message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw message
     * of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding X-header, Amazon SES
     * uses the value of the <code>ReturnPathArn</code> parameter.
     * </p>
     * <note>
     * <p>
     * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this guide,
     * or see the <a href=
     * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     * >Amazon SES Developer Guide</a>.
     * </p>
     * </note>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
     *         message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
     *         X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
     *         </p>
     *         <note>
     *         <p>
     *         For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
     *         this guide, or see the <a href=
     *         "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
     *         >Amazon SES Developer Guide</a>.
     *         </p>
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using
     * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     * publish email sending events.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you
     *         can publish email sending events.
     */
    public List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((source() == null) ? 0 : source().hashCode());
        hashCode = 31 * hashCode + ((destinations() == null) ? 0 : destinations().hashCode());
        hashCode = 31 * hashCode + ((rawMessage() == null) ? 0 : rawMessage().hashCode());
        hashCode = 31 * hashCode + ((fromArn() == null) ? 0 : fromArn().hashCode());
        hashCode = 31 * hashCode + ((sourceArn() == null) ? 0 : sourceArn().hashCode());
        hashCode = 31 * hashCode + ((returnPathArn() == null) ? 0 : returnPathArn().hashCode());
        hashCode = 31 * hashCode + ((tags() == null) ? 0 : tags().hashCode());
        hashCode = 31 * hashCode + ((configurationSetName() == null) ? 0 : configurationSetName().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendRawEmailRequest)) {
            return false;
        }
        SendRawEmailRequest other = (SendRawEmailRequest) obj;
        if (other.source() == null ^ this.source() == null) {
            return false;
        }
        if (other.source() != null && !other.source().equals(this.source())) {
            return false;
        }
        if (other.destinations() == null ^ this.destinations() == null) {
            return false;
        }
        if (other.destinations() != null && !other.destinations().equals(this.destinations())) {
            return false;
        }
        if (other.rawMessage() == null ^ this.rawMessage() == null) {
            return false;
        }
        if (other.rawMessage() != null && !other.rawMessage().equals(this.rawMessage())) {
            return false;
        }
        if (other.fromArn() == null ^ this.fromArn() == null) {
            return false;
        }
        if (other.fromArn() != null && !other.fromArn().equals(this.fromArn())) {
            return false;
        }
        if (other.sourceArn() == null ^ this.sourceArn() == null) {
            return false;
        }
        if (other.sourceArn() != null && !other.sourceArn().equals(this.sourceArn())) {
            return false;
        }
        if (other.returnPathArn() == null ^ this.returnPathArn() == null) {
            return false;
        }
        if (other.returnPathArn() != null && !other.returnPathArn().equals(this.returnPathArn())) {
            return false;
        }
        if (other.tags() == null ^ this.tags() == null) {
            return false;
        }
        if (other.tags() != null && !other.tags().equals(this.tags())) {
            return false;
        }
        if (other.configurationSetName() == null ^ this.configurationSetName() == null) {
            return false;
        }
        if (other.configurationSetName() != null && !other.configurationSetName().equals(this.configurationSetName())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (source() != null) {
            sb.append("Source: ").append(source()).append(",");
        }
        if (destinations() != null) {
            sb.append("Destinations: ").append(destinations()).append(",");
        }
        if (rawMessage() != null) {
            sb.append("RawMessage: ").append(rawMessage()).append(",");
        }
        if (fromArn() != null) {
            sb.append("FromArn: ").append(fromArn()).append(",");
        }
        if (sourceArn() != null) {
            sb.append("SourceArn: ").append(sourceArn()).append(",");
        }
        if (returnPathArn() != null) {
            sb.append("ReturnPathArn: ").append(returnPathArn()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (configurationSetName() != null) {
            sb.append("ConfigurationSetName: ").append(configurationSetName()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, SendRawEmailRequest> {
        /**
         * <p>
         * The identity's email address. If you do not provide a value for this parameter, you must specify a "From"
         * address in the raw text of the message. (You can also specify both.)
         * </p>
         * <p>
         * By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you must use
         * MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following
         * form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         * href="http://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * </p>
         * <note>
         * <p>
         * If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces and
         * complaints will be sent to this email address. This takes precedence over any <i>Return-Path</i> header that
         * you might include in the raw text of the message.
         * </p>
         * </note>
         * 
         * @param source
         *        The identity's email address. If you do not provide a value for this parameter, you must specify a
         *        "From" address in the raw text of the message. (You can also specify both.)</p>
         *        <p>
         *        By default, the string must be 7-bit ASCII. If the text must contain any other characters, then you
         *        must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax
         *        uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         *        href="http://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify the <code>Source</code> parameter and have feedback forwarding enabled, then bounces
         *        and complaints will be sent to this email address. This takes precedence over any <i>Return-Path</i>
         *        header that you might include in the raw text of the message.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<String> destinations);

        /**
         * <p>
         * A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setDestinations(java.util.Collection)} or {@link #withDestinations(java.util.Collection)} if you want
         * to override the existing values.
         * </p>
         * 
         * @param destinations
         *        A list of destinations for the message, consisting of To:, CC:, and BCC: addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(String... destinations);

        /**
         * <p>
         * The raw text of the message. The client is responsible for ensuring the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Message must contain a header and a body, separated by a blank line.
         * </p>
         * </li>
         * <li>
         * <p>
         * All required header fields must be present.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each part of a multipart MIME message must be formatted properly.
         * </p>
         * </li>
         * <li>
         * <p>
         * MIME content types must be among those supported by Amazon SES. For more information, go to the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer Guide</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be base64-encoded.
         * </p>
         * </li>
         * </ul>
         * 
         * @param rawMessage
         *        The raw text of the message. The client is responsible for ensuring the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Message must contain a header and a body, separated by a blank line.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All required header fields must be present.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each part of a multipart MIME message must be formatted properly.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MIME content types must be among those supported by Amazon SES. For more information, go to the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/mime-types.html">Amazon SES Developer
         *        Guide</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be base64-encoded.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rawMessage(RawMessage rawMessage);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to specify a particular "From" address in the header of the
         * raw email.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw message of
         * the email. If you use both the <code>FromArn</code> parameter and the corresponding X-header, Amazon SES uses
         * the value of the <code>FromArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param fromArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to specify a particular "From"
         *        address in the header of the raw email.</p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-FROM-ARN</code> in the raw
         *        message of the email. If you use both the <code>FromArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>FromArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromArn(String fromArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw message of
         * the email. If you use both the <code>SourceArn</code> parameter and the corresponding X-header, Amazon SES
         * uses the value of the <code>SourceArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-SOURCE-ARN</code> in the raw
         *        message of the email. If you use both the <code>SourceArn</code> parameter and the corresponding
         *        X-header, Amazon SES uses the value of the <code>SourceArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the raw
         * message of the email. If you use both the <code>ReturnPathArn</code> parameter and the corresponding
         * X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         * </p>
         * <note>
         * <p>
         * For information about when to use this parameter, see the description of <code>SendRawEmail</code> in this
         * guide, or see the <a href=
         * "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         * >Amazon SES Developer Guide</a>.
         * </p>
         * </note>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        Instead of using this parameter, you can use the X-header <code>X-SES-RETURN-PATH-ARN</code> in the
         *        raw message of the email. If you use both the <code>ReturnPathArn</code> parameter and the
         *        corresponding X-header, Amazon SES uses the value of the <code>ReturnPathArn</code> parameter.
         *        </p>
         *        <note>
         *        <p>
         *        For information about when to use this parameter, see the description of <code>SendRawEmail</code> in
         *        this guide, or see the <a href=
         *        "http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization-delegate-sender-tasks-email.html"
         *        >Amazon SES Developer Guide</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setTags(java.util.Collection)} or {@link #withTags(java.util.Collection)} if you want to override the
         * existing values.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendRawEmail</code>. Tags correspond to characteristics of the email that you define, so that
         *        you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendRawEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);
    }

    private static final class BuilderImpl implements Builder {
        private String source;

        private List<String> destinations = new SdkInternalList<>();

        private RawMessage rawMessage;

        private String fromArn;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags = new SdkInternalList<>();

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendRawEmailRequest model) {
            setSource(model.source);
            setDestinations(model.destinations);
            setRawMessage(model.rawMessage);
            setFromArn(model.fromArn);
            setSourceArn(model.sourceArn);
            setReturnPathArn(model.returnPathArn);
            setTags(model.tags);
            setConfigurationSetName(model.configurationSetName);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Collection<String> getDestinations() {
            return destinations;
        }

        @Override
        public final Builder destinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(String... destinations) {
            if (this.destinations == null) {
                this.destinations = new SdkInternalList<>(destinations.length);
            }
            for (String e : destinations) {
                this.destinations.add(e);
            }
            return this;
        }

        public final void setDestinations(Collection<String> destinations) {
            this.destinations = AddressListCopier.copy(destinations);
        }

        @SafeVarargs
        public final void setDestinations(String... destinations) {
            if (this.destinations == null) {
                this.destinations = new SdkInternalList<>(destinations.length);
            }
            for (String e : destinations) {
                this.destinations.add(e);
            }
        }

        public final RawMessage getRawMessage() {
            return rawMessage;
        }

        @Override
        public final Builder rawMessage(RawMessage rawMessage) {
            this.rawMessage = rawMessage;
            return this;
        }

        public final void setRawMessage(RawMessage rawMessage) {
            this.rawMessage = rawMessage;
        }

        public final String getFromArn() {
            return fromArn;
        }

        @Override
        public final Builder fromArn(String fromArn) {
            this.fromArn = fromArn;
            return this;
        }

        public final void setFromArn(String fromArn) {
            this.fromArn = fromArn;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final Collection<MessageTag> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            if (this.tags == null) {
                this.tags = new SdkInternalList<>(tags.length);
            }
            for (MessageTag e : tags) {
                this.tags.add(e);
            }
            return this;
        }

        public final void setTags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
        }

        @SafeVarargs
        public final void setTags(MessageTag... tags) {
            if (this.tags == null) {
                this.tags = new SdkInternalList<>(tags.length);
            }
            for (MessageTag e : tags) {
                this.tags.add(e);
            }
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public SendRawEmailRequest build() {
            return new SendRawEmailRequest(this);
        }
    }
}
