/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request for the status of Amazon SES Easy DKIM signing for an identity. For domain identities, this
 * request also returns the DKIM tokens that are required for Easy DKIM signing, and whether Amazon SES successfully
 * verified that these tokens were published. For more information about Easy DKIM, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetIdentityDkimAttributesRequest extends SESRequest implements
        ToCopyableBuilder<GetIdentityDkimAttributesRequest.Builder, GetIdentityDkimAttributesRequest> {
    private final List<String> identities;

    private GetIdentityDkimAttributesRequest(BuilderImpl builder) {
        super(builder);
        this.identities = builder.identities;
    }

    /**
     * <p>
     * A list of one or more verified identities - email addresses, domains, or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of one or more verified identities - email addresses, domains, or both.
     */
    public List<String> identities() {
        return identities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identities());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetIdentityDkimAttributesRequest)) {
            return false;
        }
        GetIdentityDkimAttributesRequest other = (GetIdentityDkimAttributesRequest) obj;
        return Objects.equals(identities(), other.identities());
    }

    @Override
    public String toString() {
        return ToString.builder("GetIdentityDkimAttributesRequest").add("Identities", identities()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identities":
            return Optional.of(clazz.cast(identities()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SESRequest.Builder, CopyableBuilder<Builder, GetIdentityDkimAttributesRequest> {
        /**
         * <p>
         * A list of one or more verified identities - email addresses, domains, or both.
         * </p>
         * 
         * @param identities
         *        A list of one or more verified identities - email addresses, domains, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(Collection<String> identities);

        /**
         * <p>
         * A list of one or more verified identities - email addresses, domains, or both.
         * </p>
         * 
         * @param identities
         *        A list of one or more verified identities - email addresses, domains, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(String... identities);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SESRequest.BuilderImpl implements Builder {
        private List<String> identities;

        private BuilderImpl() {
        }

        private BuilderImpl(GetIdentityDkimAttributesRequest model) {
            super(model);
            identities(model.identities);
        }

        public final Collection<String> getIdentities() {
            return identities;
        }

        @Override
        public final Builder identities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identities(String... identities) {
            identities(Arrays.asList(identities));
            return this;
        }

        public final void setIdentities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public GetIdentityDkimAttributesRequest build() {
            return new GetIdentityDkimAttributesRequest(this);
        }
    }
}
