/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the DKIM attributes of a verified email address or a domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class IdentityDkimAttributes implements ToCopyableBuilder<IdentityDkimAttributes.Builder, IdentityDkimAttributes> {
    private final Boolean dkimEnabled;

    private final String dkimVerificationStatus;

    private final List<String> dkimTokens;

    private IdentityDkimAttributes(BuilderImpl builder) {
        this.dkimEnabled = builder.dkimEnabled;
        this.dkimVerificationStatus = builder.dkimVerificationStatus;
        this.dkimTokens = builder.dkimTokens;
    }

    /**
     * <p>
     * True if DKIM signing is enabled for email sent from the identity; false otherwise. The default value is true.
     * </p>
     * 
     * @return True if DKIM signing is enabled for email sent from the identity; false otherwise. The default value is
     *         true.
     */
    public Boolean dkimEnabled() {
        return dkimEnabled;
    }

    /**
     * <p>
     * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
     * name's DNS. (This only applies to domain identities, not email address identities.)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dkimVerificationStatus} will return {@link VerificationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dkimVerificationStatusAsString}.
     * </p>
     * 
     * @return Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
     *         domain name's DNS. (This only applies to domain identities, not email address identities.)
     * @see VerificationStatus
     */
    public VerificationStatus dkimVerificationStatus() {
        return VerificationStatus.fromValue(dkimVerificationStatus);
    }

    /**
     * <p>
     * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
     * name's DNS. (This only applies to domain identities, not email address identities.)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dkimVerificationStatus} will return {@link VerificationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dkimVerificationStatusAsString}.
     * </p>
     * 
     * @return Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
     *         domain name's DNS. (This only applies to domain identities, not email address identities.)
     * @see VerificationStatus
     */
    public String dkimVerificationStatusAsString() {
        return dkimVerificationStatus;
    }

    /**
     * <p>
     * A set of character strings that represent the domain's identity. Using these tokens, you will need to create DNS
     * CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will eventually detect
     * that you have updated your DNS records; this detection process may take up to 72 hours. Upon successful
     * detection, Amazon SES will be able to DKIM-sign email originating from that domain. (This only applies to domain
     * identities, not email address identities.)
     * </p>
     * <p>
     * For more information about creating DNS records using DKIM tokens, go to the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A set of character strings that represent the domain's identity. Using these tokens, you will need to
     *         create DNS CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will
     *         eventually detect that you have updated your DNS records; this detection process may take up to 72 hours.
     *         Upon successful detection, Amazon SES will be able to DKIM-sign email originating from that domain. (This
     *         only applies to domain identities, not email address identities.)</p>
     *         <p>
     *         For more information about creating DNS records using DKIM tokens, go to the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES
     *         Developer Guide</a>.
     */
    public List<String> dkimTokens() {
        return dkimTokens;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dkimEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(dkimVerificationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dkimTokens());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IdentityDkimAttributes)) {
            return false;
        }
        IdentityDkimAttributes other = (IdentityDkimAttributes) obj;
        return Objects.equals(dkimEnabled(), other.dkimEnabled())
                && Objects.equals(dkimVerificationStatusAsString(), other.dkimVerificationStatusAsString())
                && Objects.equals(dkimTokens(), other.dkimTokens());
    }

    @Override
    public String toString() {
        return ToString.builder("IdentityDkimAttributes").add("DkimEnabled", dkimEnabled())
                .add("DkimVerificationStatus", dkimVerificationStatusAsString()).add("DkimTokens", dkimTokens()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DkimEnabled":
            return Optional.of(clazz.cast(dkimEnabled()));
        case "DkimVerificationStatus":
            return Optional.of(clazz.cast(dkimVerificationStatusAsString()));
        case "DkimTokens":
            return Optional.of(clazz.cast(dkimTokens()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, IdentityDkimAttributes> {
        /**
         * <p>
         * True if DKIM signing is enabled for email sent from the identity; false otherwise. The default value is true.
         * </p>
         * 
         * @param dkimEnabled
         *        True if DKIM signing is enabled for email sent from the identity; false otherwise. The default value
         *        is true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimEnabled(Boolean dkimEnabled);

        /**
         * <p>
         * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
         * name's DNS. (This only applies to domain identities, not email address identities.)
         * </p>
         * 
         * @param dkimVerificationStatus
         *        Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
         *        domain name's DNS. (This only applies to domain identities, not email address identities.)
         * @see VerificationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationStatus
         */
        Builder dkimVerificationStatus(String dkimVerificationStatus);

        /**
         * <p>
         * Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the domain
         * name's DNS. (This only applies to domain identities, not email address identities.)
         * </p>
         * 
         * @param dkimVerificationStatus
         *        Describes whether Amazon SES has successfully verified the DKIM DNS records (tokens) published in the
         *        domain name's DNS. (This only applies to domain identities, not email address identities.)
         * @see VerificationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerificationStatus
         */
        Builder dkimVerificationStatus(VerificationStatus dkimVerificationStatus);

        /**
         * <p>
         * A set of character strings that represent the domain's identity. Using these tokens, you will need to create
         * DNS CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will eventually
         * detect that you have updated your DNS records; this detection process may take up to 72 hours. Upon
         * successful detection, Amazon SES will be able to DKIM-sign email originating from that domain. (This only
         * applies to domain identities, not email address identities.)
         * </p>
         * <p>
         * For more information about creating DNS records using DKIM tokens, go to the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param dkimTokens
         *        A set of character strings that represent the domain's identity. Using these tokens, you will need to
         *        create DNS CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will
         *        eventually detect that you have updated your DNS records; this detection process may take up to 72
         *        hours. Upon successful detection, Amazon SES will be able to DKIM-sign email originating from that
         *        domain. (This only applies to domain identities, not email address identities.)</p>
         *        <p>
         *        For more information about creating DNS records using DKIM tokens, go to the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimTokens(Collection<String> dkimTokens);

        /**
         * <p>
         * A set of character strings that represent the domain's identity. Using these tokens, you will need to create
         * DNS CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will eventually
         * detect that you have updated your DNS records; this detection process may take up to 72 hours. Upon
         * successful detection, Amazon SES will be able to DKIM-sign email originating from that domain. (This only
         * applies to domain identities, not email address identities.)
         * </p>
         * <p>
         * For more information about creating DNS records using DKIM tokens, go to the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param dkimTokens
         *        A set of character strings that represent the domain's identity. Using these tokens, you will need to
         *        create DNS CNAME records that point to DKIM public keys hosted by Amazon SES. Amazon Web Services will
         *        eventually detect that you have updated your DNS records; this detection process may take up to 72
         *        hours. Upon successful detection, Amazon SES will be able to DKIM-sign email originating from that
         *        domain. (This only applies to domain identities, not email address identities.)</p>
         *        <p>
         *        For more information about creating DNS records using DKIM tokens, go to the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/easy-dkim-dns-records.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dkimTokens(String... dkimTokens);
    }

    static final class BuilderImpl implements Builder {
        private Boolean dkimEnabled;

        private String dkimVerificationStatus;

        private List<String> dkimTokens;

        private BuilderImpl() {
        }

        private BuilderImpl(IdentityDkimAttributes model) {
            dkimEnabled(model.dkimEnabled);
            dkimVerificationStatus(model.dkimVerificationStatus);
            dkimTokens(model.dkimTokens);
        }

        public final Boolean getDkimEnabled() {
            return dkimEnabled;
        }

        @Override
        public final Builder dkimEnabled(Boolean dkimEnabled) {
            this.dkimEnabled = dkimEnabled;
            return this;
        }

        public final void setDkimEnabled(Boolean dkimEnabled) {
            this.dkimEnabled = dkimEnabled;
        }

        public final String getDkimVerificationStatus() {
            return dkimVerificationStatus;
        }

        @Override
        public final Builder dkimVerificationStatus(String dkimVerificationStatus) {
            this.dkimVerificationStatus = dkimVerificationStatus;
            return this;
        }

        @Override
        public final Builder dkimVerificationStatus(VerificationStatus dkimVerificationStatus) {
            this.dkimVerificationStatus(dkimVerificationStatus.toString());
            return this;
        }

        public final void setDkimVerificationStatus(String dkimVerificationStatus) {
            this.dkimVerificationStatus = dkimVerificationStatus;
        }

        public final Collection<String> getDkimTokens() {
            return dkimTokens;
        }

        @Override
        public final Builder dkimTokens(Collection<String> dkimTokens) {
            this.dkimTokens = VerificationTokenListCopier.copy(dkimTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dkimTokens(String... dkimTokens) {
            dkimTokens(Arrays.asList(dkimTokens));
            return this;
        }

        public final void setDkimTokens(Collection<String> dkimTokens) {
            this.dkimTokens = VerificationTokenListCopier.copy(dkimTokens);
        }

        @Override
        public IdentityDkimAttributes build() {
            return new IdentityDkimAttributes(this);
        }
    }
}
