/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a templated email using Amazon SES. For more information, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-personalized-email-api.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SendTemplatedEmailRequest extends SESRequest implements
        ToCopyableBuilder<SendTemplatedEmailRequest.Builder, SendTemplatedEmailRequest> {
    private final String source;

    private final Destination destination;

    private final List<String> replyToAddresses;

    private final String returnPath;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private final String template;

    private final String templateArn;

    private final String templateData;

    private SendTemplatedEmailRequest(BuilderImpl builder) {
        super(builder);
        this.source = builder.source;
        this.destination = builder.destination;
        this.replyToAddresses = builder.replyToAddresses;
        this.returnPath = builder.returnPath;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
        this.template = builder.template;
        this.templateArn = builder.templateArn;
        this.templateData = builder.templateData;
    }

    /**
     * <p>
     * The email address that is sending the email. This email address must be either individually verified with Amazon
     * SES, or from a domain that has been verified with Amazon SES. For information about verifying identities, see the
     * <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     * Developer Guide</a>.
     * </p>
     * <p>
     * If you are sending on behalf of another user and have been permitted to do so by a sending authorization policy,
     * then you must also specify the <code>SourceArn</code> parameter. For more information about sending
     * authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <p>
     * In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then you must
     * use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following
     * form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     * </p>
     * 
     * @return The email address that is sending the email. This email address must be either individually verified with
     *         Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
     *         identities, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     *         Developer Guide</a>.</p>
     *         <p>
     *         If you are sending on behalf of another user and have been permitted to do so by a sending authorization
     *         policy, then you must also specify the <code>SourceArn</code> parameter. For more information about
     *         sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         <p>
     *         In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then
     *         you must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax
     *         uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     *         href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * The destination for this email, composed of To:, CC:, and BCC: fields. A Destination can include up to 50
     * recipients across these three fields.
     * </p>
     * 
     * @return The destination for this email, composed of To:, CC:, and BCC: fields. A Destination can include up to 50
     *         recipients across these three fields.
     */
    public Destination destination() {
        return destination;
    }

    /**
     * <p>
     * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to address
     * will receive the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
     *         address will receive the reply.
     */
    public List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If the
     * message cannot be delivered to the recipient, then an error message will be returned from the recipient's ISP;
     * this message will then be forwarded to the email address specified by the <code>ReturnPath</code> parameter. The
     * <code>ReturnPath</code> parameter is never overwritten. This email address must be either individually verified
     * with Amazon SES, or from a domain that has been verified with Amazon SES.
     * </p>
     * 
     * @return The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled.
     *         If the message cannot be delivered to the recipient, then an error message will be returned from the
     *         recipient's ISP; this message will then be forwarded to the email address specified by the
     *         <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This email
     *         address must be either individually verified with Amazon SES, or from a domain that has been verified
     *         with Amazon SES.
     */
    public String returnPath() {
        return returnPath;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using
     * <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     * publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so that
     *         you can publish email sending events.
     */
    public List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendTemplatedEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendTemplatedEmail</code>.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    /**
     * <p>
     * The template to use when sending this email.
     * </p>
     * 
     * @return The template to use when sending this email.
     */
    public String template() {
        return template;
    }

    /**
     * <p>
     * The ARN of the template to use when sending this email.
     * </p>
     * 
     * @return The ARN of the template to use when sending this email.
     */
    public String templateArn() {
        return templateArn;
    }

    /**
     * <p>
     * A list of replacement values to apply to the template. This parameter is a JSON object, typically consisting of
     * key-value pairs in which the keys correspond to replacement tags in the email template.
     * </p>
     * 
     * @return A list of replacement values to apply to the template. This parameter is a JSON object, typically
     *         consisting of key-value pairs in which the keys correspond to replacement tags in the email template.
     */
    public String templateData() {
        return templateData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(replyToAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(returnPath());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        hashCode = 31 * hashCode + Objects.hashCode(template());
        hashCode = 31 * hashCode + Objects.hashCode(templateArn());
        hashCode = 31 * hashCode + Objects.hashCode(templateData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendTemplatedEmailRequest)) {
            return false;
        }
        SendTemplatedEmailRequest other = (SendTemplatedEmailRequest) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(destination(), other.destination())
                && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(returnPath(), other.returnPath()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(returnPathArn(), other.returnPathArn()) && Objects.equals(tags(), other.tags())
                && Objects.equals(configurationSetName(), other.configurationSetName())
                && Objects.equals(template(), other.template()) && Objects.equals(templateArn(), other.templateArn())
                && Objects.equals(templateData(), other.templateData());
    }

    @Override
    public String toString() {
        return ToString.builder("SendTemplatedEmailRequest").add("Source", source()).add("Destination", destination())
                .add("ReplyToAddresses", replyToAddresses()).add("ReturnPath", returnPath()).add("SourceArn", sourceArn())
                .add("ReturnPathArn", returnPathArn()).add("Tags", tags()).add("ConfigurationSetName", configurationSetName())
                .add("Template", template()).add("TemplateArn", templateArn()).add("TemplateData", templateData()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.of(clazz.cast(source()));
        case "Destination":
            return Optional.of(clazz.cast(destination()));
        case "ReplyToAddresses":
            return Optional.of(clazz.cast(replyToAddresses()));
        case "ReturnPath":
            return Optional.of(clazz.cast(returnPath()));
        case "SourceArn":
            return Optional.of(clazz.cast(sourceArn()));
        case "ReturnPathArn":
            return Optional.of(clazz.cast(returnPathArn()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "ConfigurationSetName":
            return Optional.of(clazz.cast(configurationSetName()));
        case "Template":
            return Optional.of(clazz.cast(template()));
        case "TemplateArn":
            return Optional.of(clazz.cast(templateArn()));
        case "TemplateData":
            return Optional.of(clazz.cast(templateData()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SESRequest.Builder, CopyableBuilder<Builder, SendTemplatedEmailRequest> {
        /**
         * <p>
         * The email address that is sending the email. This email address must be either individually verified with
         * Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
         * identities, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
         * Developer Guide</a>.
         * </p>
         * <p>
         * If you are sending on behalf of another user and have been permitted to do so by a sending authorization
         * policy, then you must also specify the <code>SourceArn</code> parameter. For more information about sending
         * authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * <p>
         * In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then you
         * must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the
         * following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * </p>
         * 
         * @param source
         *        The email address that is sending the email. This email address must be either individually verified
         *        with Amazon SES, or from a domain that has been verified with Amazon SES. For information about
         *        verifying identities, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon
         *        SES Developer Guide</a>.</p>
         *        <p>
         *        If you are sending on behalf of another user and have been permitted to do so by a sending
         *        authorization policy, then you must also specify the <code>SourceArn</code> parameter. For more
         *        information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         *        </p>
         *        <p>
         *        In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters,
         *        then you must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word
         *        syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information,
         *        see <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields. A Destination can include up to 50
         * recipients across these three fields.
         * </p>
         * 
         * @param destination
         *        The destination for this email, composed of To:, CC:, and BCC: fields. A Destination can include up to
         *        50 recipients across these three fields.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Destination destination);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields. A Destination can include up to 50
         * recipients across these three fields.
         * </p>
         * This is a convenience that creates an instance of the {@link Destination.Builder} avoiding the need to create
         * one manually via {@link Destination#builder()}.
         *
         * When the {@link Consumer} completes, {@link Destination.Builder#build()} is called immediately and its result
         * is passed to {@link #destination(Destination)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(Destination)
         */
        default Builder destination(Consumer<Destination.Builder> destination) {
            return destination(Destination.builder().apply(destination).build());
        }

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If
         * the message cannot be delivered to the recipient, then an error message will be returned from the recipient's
         * ISP; this message will then be forwarded to the email address specified by the <code>ReturnPath</code>
         * parameter. The <code>ReturnPath</code> parameter is never overwritten. This email address must be either
         * individually verified with Amazon SES, or from a domain that has been verified with Amazon SES.
         * </p>
         * 
         * @param returnPath
         *        The email address that bounces and complaints will be forwarded to when feedback forwarding is
         *        enabled. If the message cannot be delivered to the recipient, then an error message will be returned
         *        from the recipient's ISP; this message will then be forwarded to the email address specified by the
         *        <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This
         *        email address must be either individually verified with Amazon SES, or from a domain that has been
         *        verified with Amazon SES.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPath(String returnPath);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         * can publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         * can publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so
         *        that you can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendTemplatedEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         * can publish email sending events.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageTag>.Builder} avoiding the need to
         * create one manually via {@link List<MessageTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<MessageTag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<MessageTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<MessageTag>)
         */
        Builder tags(Consumer<MessageTag.Builder>... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendTemplatedEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendTemplatedEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);

        /**
         * <p>
         * The template to use when sending this email.
         * </p>
         * 
         * @param template
         *        The template to use when sending this email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder template(String template);

        /**
         * <p>
         * The ARN of the template to use when sending this email.
         * </p>
         * 
         * @param templateArn
         *        The ARN of the template to use when sending this email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateArn(String templateArn);

        /**
         * <p>
         * A list of replacement values to apply to the template. This parameter is a JSON object, typically consisting
         * of key-value pairs in which the keys correspond to replacement tags in the email template.
         * </p>
         * 
         * @param templateData
         *        A list of replacement values to apply to the template. This parameter is a JSON object, typically
         *        consisting of key-value pairs in which the keys correspond to replacement tags in the email template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateData(String templateData);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);

        @Override
        Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SESRequest.BuilderImpl implements Builder {
        private String source;

        private Destination destination;

        private List<String> replyToAddresses;

        private String returnPath;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags;

        private String configurationSetName;

        private String template;

        private String templateArn;

        private String templateData;

        private BuilderImpl() {
        }

        private BuilderImpl(SendTemplatedEmailRequest model) {
            super(model);
            source(model.source);
            destination(model.destination);
            replyToAddresses(model.replyToAddresses);
            returnPath(model.returnPath);
            sourceArn(model.sourceArn);
            returnPathArn(model.returnPathArn);
            tags(model.tags);
            configurationSetName(model.configurationSetName);
            template(model.template);
            templateArn(model.templateArn);
            templateData(model.templateData);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Destination.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        @Override
        public final Builder destination(Destination destination) {
            this.destination = destination;
            return this;
        }

        public final void setDestination(Destination.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        public final Collection<String> getReplyToAddresses() {
            return replyToAddresses;
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
        }

        public final String getReturnPath() {
            return returnPath;
        }

        @Override
        public final Builder returnPath(String returnPath) {
            this.returnPath = returnPath;
            return this;
        }

        public final void setReturnPath(String returnPath) {
            this.returnPath = returnPath;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final Collection<MessageTag.Builder> getTags() {
            return tags != null ? tags.stream().map(MessageTag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<MessageTag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> MessageTag.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<MessageTag.BuilderImpl> tags) {
            this.tags = MessageTagListCopier.copyFromBuilder(tags);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        public final String getTemplate() {
            return template;
        }

        @Override
        public final Builder template(String template) {
            this.template = template;
            return this;
        }

        public final void setTemplate(String template) {
            this.template = template;
        }

        public final String getTemplateArn() {
            return templateArn;
        }

        @Override
        public final Builder templateArn(String templateArn) {
            this.templateArn = templateArn;
            return this;
        }

        public final void setTemplateArn(String templateArn) {
            this.templateArn = templateArn;
        }

        public final String getTemplateData() {
            return templateData;
        }

        @Override
        public final Builder templateData(String templateData) {
            this.templateData = templateData;
            return this;
        }

        public final void setTemplateData(String templateData) {
            this.templateData = templateData;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public SendTemplatedEmailRequest build() {
            return new SendTemplatedEmailRequest(this);
        }
    }
}
