/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the dimension configuration to use when you publish email sending events to Amazon CloudWatch.
 * </p>
 * <p>
 * For information about publishing email sending events to Amazon CloudWatch, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudWatchDimensionConfiguration implements
        ToCopyableBuilder<CloudWatchDimensionConfiguration.Builder, CloudWatchDimensionConfiguration> {
    private final String dimensionName;

    private final String dimensionValueSource;

    private final String defaultDimensionValue;

    private CloudWatchDimensionConfiguration(BuilderImpl builder) {
        this.dimensionName = builder.dimensionName;
        this.dimensionValueSource = builder.dimensionValueSource;
        this.defaultDimensionValue = builder.defaultDimensionValue;
    }

    /**
     * <p>
     * The name of an Amazon CloudWatch dimension associated with an email sending metric. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 256 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of an Amazon CloudWatch dimension associated with an email sending metric. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 256 characters.
     *         </p>
     *         </li>
     */
    public String dimensionName() {
        return dimensionName;
    }

    /**
     * <p>
     * The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want Amazon SES
     * to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a parameter to the
     * <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If you want Amazon SES to
     * use your own email headers, choose <code>emailHeader</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dimensionValueSource} will return {@link DimensionValueSource#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dimensionValueSourceAsString}.
     * </p>
     * 
     * @return The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want
     *         Amazon SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a
     *         parameter to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If
     *         you want Amazon SES to use your own email headers, choose <code>emailHeader</code>.
     * @see DimensionValueSource
     */
    public DimensionValueSource dimensionValueSource() {
        return DimensionValueSource.fromValue(dimensionValueSource);
    }

    /**
     * <p>
     * The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want Amazon SES
     * to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a parameter to the
     * <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If you want Amazon SES to
     * use your own email headers, choose <code>emailHeader</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dimensionValueSource} will return {@link DimensionValueSource#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dimensionValueSourceAsString}.
     * </p>
     * 
     * @return The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want
     *         Amazon SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a
     *         parameter to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If
     *         you want Amazon SES to use your own email headers, choose <code>emailHeader</code>.
     * @see DimensionValueSource
     */
    public String dimensionValueSourceAsString() {
        return dimensionValueSource;
    }

    /**
     * <p>
     * The default value of the dimension that is published to Amazon CloudWatch if you do not provide the value of the
     * dimension when you send an email. The default value must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 256 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The default value of the dimension that is published to Amazon CloudWatch if you do not provide the value
     *         of the dimension when you send an email. The default value must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 256 characters.
     *         </p>
     *         </li>
     */
    public String defaultDimensionValue() {
        return defaultDimensionValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dimensionName());
        hashCode = 31 * hashCode + Objects.hashCode(dimensionValueSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(defaultDimensionValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudWatchDimensionConfiguration)) {
            return false;
        }
        CloudWatchDimensionConfiguration other = (CloudWatchDimensionConfiguration) obj;
        return Objects.equals(dimensionName(), other.dimensionName())
                && Objects.equals(dimensionValueSourceAsString(), other.dimensionValueSourceAsString())
                && Objects.equals(defaultDimensionValue(), other.defaultDimensionValue());
    }

    @Override
    public String toString() {
        return ToString.builder("CloudWatchDimensionConfiguration").add("DimensionName", dimensionName())
                .add("DimensionValueSource", dimensionValueSourceAsString())
                .add("DefaultDimensionValue", defaultDimensionValue()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DimensionName":
            return Optional.ofNullable(clazz.cast(dimensionName()));
        case "DimensionValueSource":
            return Optional.ofNullable(clazz.cast(dimensionValueSourceAsString()));
        case "DefaultDimensionValue":
            return Optional.ofNullable(clazz.cast(defaultDimensionValue()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CloudWatchDimensionConfiguration> {
        /**
         * <p>
         * The name of an Amazon CloudWatch dimension associated with an email sending metric. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 256 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dimensionName
         *        The name of an Amazon CloudWatch dimension associated with an email sending metric. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 256 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionName(String dimensionName);

        /**
         * <p>
         * The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want Amazon
         * SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a parameter
         * to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If you want
         * Amazon SES to use your own email headers, choose <code>emailHeader</code>.
         * </p>
         * 
         * @param dimensionValueSource
         *        The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want
         *        Amazon SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or
         *        a parameter to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose
         *        <code>messageTag</code>. If you want Amazon SES to use your own email headers, choose
         *        <code>emailHeader</code>.
         * @see DimensionValueSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DimensionValueSource
         */
        Builder dimensionValueSource(String dimensionValueSource);

        /**
         * <p>
         * The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want Amazon
         * SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or a parameter
         * to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose <code>messageTag</code>. If you want
         * Amazon SES to use your own email headers, choose <code>emailHeader</code>.
         * </p>
         * 
         * @param dimensionValueSource
         *        The place where Amazon SES finds the value of a dimension to publish to Amazon CloudWatch. If you want
         *        Amazon SES to use the message tags that you specify using an <code>X-SES-MESSAGE-TAGS</code> header or
         *        a parameter to the <code>SendEmail</code>/<code>SendRawEmail</code> API, choose
         *        <code>messageTag</code>. If you want Amazon SES to use your own email headers, choose
         *        <code>emailHeader</code>.
         * @see DimensionValueSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DimensionValueSource
         */
        Builder dimensionValueSource(DimensionValueSource dimensionValueSource);

        /**
         * <p>
         * The default value of the dimension that is published to Amazon CloudWatch if you do not provide the value of
         * the dimension when you send an email. The default value must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 256 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param defaultDimensionValue
         *        The default value of the dimension that is published to Amazon CloudWatch if you do not provide the
         *        value of the dimension when you send an email. The default value must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 256 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultDimensionValue(String defaultDimensionValue);
    }

    static final class BuilderImpl implements Builder {
        private String dimensionName;

        private String dimensionValueSource;

        private String defaultDimensionValue;

        private BuilderImpl() {
        }

        private BuilderImpl(CloudWatchDimensionConfiguration model) {
            dimensionName(model.dimensionName);
            dimensionValueSource(model.dimensionValueSource);
            defaultDimensionValue(model.defaultDimensionValue);
        }

        public final String getDimensionName() {
            return dimensionName;
        }

        @Override
        public final Builder dimensionName(String dimensionName) {
            this.dimensionName = dimensionName;
            return this;
        }

        public final void setDimensionName(String dimensionName) {
            this.dimensionName = dimensionName;
        }

        public final String getDimensionValueSource() {
            return dimensionValueSource;
        }

        @Override
        public final Builder dimensionValueSource(String dimensionValueSource) {
            this.dimensionValueSource = dimensionValueSource;
            return this;
        }

        @Override
        public final Builder dimensionValueSource(DimensionValueSource dimensionValueSource) {
            this.dimensionValueSource(dimensionValueSource.toString());
            return this;
        }

        public final void setDimensionValueSource(String dimensionValueSource) {
            this.dimensionValueSource = dimensionValueSource;
        }

        public final String getDefaultDimensionValue() {
            return defaultDimensionValue;
        }

        @Override
        public final Builder defaultDimensionValue(String defaultDimensionValue) {
            this.defaultDimensionValue = defaultDimensionValue;
            return this;
        }

        public final void setDefaultDimensionValue(String defaultDimensionValue) {
            this.defaultDimensionValue = defaultDimensionValue;
        }

        @Override
        public CloudWatchDimensionConfiguration build() {
            return new CloudWatchDimensionConfiguration(this);
        }
    }
}
