/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to return the Amazon SES verification status of a list of identities. For domain identities,
 * this request also returns the verification token. For information about verifying identities with Amazon SES, see the
 * <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetIdentityVerificationAttributesRequest extends SesRequest implements
        ToCopyableBuilder<GetIdentityVerificationAttributesRequest.Builder, GetIdentityVerificationAttributesRequest> {
    private final List<String> identities;

    private GetIdentityVerificationAttributesRequest(BuilderImpl builder) {
        super(builder);
        this.identities = builder.identities;
    }

    /**
     * <p>
     * A list of identities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of identities.
     */
    public List<String> identities() {
        return identities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identities());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetIdentityVerificationAttributesRequest)) {
            return false;
        }
        GetIdentityVerificationAttributesRequest other = (GetIdentityVerificationAttributesRequest) obj;
        return Objects.equals(identities(), other.identities());
    }

    @Override
    public String toString() {
        return ToString.builder("GetIdentityVerificationAttributesRequest").add("Identities", identities()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identities":
            return Optional.ofNullable(clazz.cast(identities()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SesRequest.Builder, CopyableBuilder<Builder, GetIdentityVerificationAttributesRequest> {
        /**
         * <p>
         * A list of identities.
         * </p>
         * 
         * @param identities
         *        A list of identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(Collection<String> identities);

        /**
         * <p>
         * A list of identities.
         * </p>
         * 
         * @param identities
         *        A list of identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(String... identities);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private List<String> identities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetIdentityVerificationAttributesRequest model) {
            super(model);
            identities(model.identities);
        }

        public final Collection<String> getIdentities() {
            return identities;
        }

        @Override
        public final Builder identities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identities(String... identities) {
            identities(Arrays.asList(identities));
            return this;
        }

        public final void setIdentities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetIdentityVerificationAttributesRequest build() {
            return new GetIdentityVerificationAttributesRequest(this);
        }
    }
}
