/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information associated with an Amazon CloudWatch event destination to which email sending events are
 * published.
 * </p>
 * <p>
 * Event destinations, such as Amazon CloudWatch, are associated with configuration sets, which enable you to publish
 * email sending events. For information about using configuration sets, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudWatchDestination implements ToCopyableBuilder<CloudWatchDestination.Builder, CloudWatchDestination> {
    private final List<CloudWatchDimensionConfiguration> dimensionConfigurations;

    private CloudWatchDestination(BuilderImpl builder) {
        this.dimensionConfigurations = builder.dimensionConfigurations;
    }

    /**
     * <p>
     * A list of dimensions upon which to categorize your emails when you publish email sending events to Amazon
     * CloudWatch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of dimensions upon which to categorize your emails when you publish email sending events to Amazon
     *         CloudWatch.
     */
    public List<CloudWatchDimensionConfiguration> dimensionConfigurations() {
        return dimensionConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dimensionConfigurations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudWatchDestination)) {
            return false;
        }
        CloudWatchDestination other = (CloudWatchDestination) obj;
        return Objects.equals(dimensionConfigurations(), other.dimensionConfigurations());
    }

    @Override
    public String toString() {
        return ToString.builder("CloudWatchDestination").add("DimensionConfigurations", dimensionConfigurations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DimensionConfigurations":
            return Optional.ofNullable(clazz.cast(dimensionConfigurations()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CloudWatchDestination> {
        /**
         * <p>
         * A list of dimensions upon which to categorize your emails when you publish email sending events to Amazon
         * CloudWatch.
         * </p>
         * 
         * @param dimensionConfigurations
         *        A list of dimensions upon which to categorize your emails when you publish email sending events to
         *        Amazon CloudWatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionConfigurations(Collection<CloudWatchDimensionConfiguration> dimensionConfigurations);

        /**
         * <p>
         * A list of dimensions upon which to categorize your emails when you publish email sending events to Amazon
         * CloudWatch.
         * </p>
         * 
         * @param dimensionConfigurations
         *        A list of dimensions upon which to categorize your emails when you publish email sending events to
         *        Amazon CloudWatch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionConfigurations(CloudWatchDimensionConfiguration... dimensionConfigurations);

        /**
         * <p>
         * A list of dimensions upon which to categorize your emails when you publish email sending events to Amazon
         * CloudWatch.
         * </p>
         * This is a convenience that creates an instance of the {@link List<CloudWatchDimensionConfiguration>.Builder}
         * avoiding the need to create one manually via {@link List<CloudWatchDimensionConfiguration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CloudWatchDimensionConfiguration>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #dimensionConfigurations(List<CloudWatchDimensionConfiguration>)}.
         * 
         * @param dimensionConfigurations
         *        a consumer that will call methods on {@link List<CloudWatchDimensionConfiguration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensionConfigurations(List<CloudWatchDimensionConfiguration>)
         */
        Builder dimensionConfigurations(Consumer<CloudWatchDimensionConfiguration.Builder>... dimensionConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private List<CloudWatchDimensionConfiguration> dimensionConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CloudWatchDestination model) {
            dimensionConfigurations(model.dimensionConfigurations);
        }

        public final Collection<CloudWatchDimensionConfiguration.Builder> getDimensionConfigurations() {
            return dimensionConfigurations != null ? dimensionConfigurations.stream()
                    .map(CloudWatchDimensionConfiguration::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensionConfigurations(Collection<CloudWatchDimensionConfiguration> dimensionConfigurations) {
            this.dimensionConfigurations = CloudWatchDimensionConfigurationsCopier.copy(dimensionConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensionConfigurations(CloudWatchDimensionConfiguration... dimensionConfigurations) {
            dimensionConfigurations(Arrays.asList(dimensionConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensionConfigurations(
                Consumer<CloudWatchDimensionConfiguration.Builder>... dimensionConfigurations) {
            dimensionConfigurations(Stream.of(dimensionConfigurations)
                    .map(c -> CloudWatchDimensionConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensionConfigurations(
                Collection<CloudWatchDimensionConfiguration.BuilderImpl> dimensionConfigurations) {
            this.dimensionConfigurations = CloudWatchDimensionConfigurationsCopier.copyFromBuilder(dimensionConfigurations);
        }

        @Override
        public CloudWatchDestination build() {
            return new CloudWatchDestination(this);
        }
    }
}
