/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to specify the Amazon SNS topic to which Amazon SES will publish bounce, complaint, or delivery
 * notifications for emails sent with that identity as the Source. For information about Amazon SES notifications, see
 * the <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/notifications-via-sns.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SetIdentityNotificationTopicRequest extends SesRequest implements
        ToCopyableBuilder<SetIdentityNotificationTopicRequest.Builder, SetIdentityNotificationTopicRequest> {
    private final String identity;

    private final String notificationType;

    private final String snsTopic;

    private SetIdentityNotificationTopicRequest(BuilderImpl builder) {
        super(builder);
        this.identity = builder.identity;
        this.notificationType = builder.notificationType;
        this.snsTopic = builder.snsTopic;
    }

    /**
     * <p>
     * The identity for which the Amazon SNS topic will be set. You can specify an identity by using its name or by
     * using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     * </p>
     * 
     * @return The identity for which the Amazon SNS topic will be set. You can specify an identity by using its name or
     *         by using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>,
     *         <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
     */
    public String identity() {
        return identity;
    }

    /**
     * <p>
     * The type of notifications that will be published to the specified Amazon SNS topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of notifications that will be published to the specified Amazon SNS topic.
     * @see NotificationType
     */
    public NotificationType notificationType() {
        return NotificationType.fromValue(notificationType);
    }

    /**
     * <p>
     * The type of notifications that will be published to the specified Amazon SNS topic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #notificationType}
     * will return {@link NotificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #notificationTypeAsString}.
     * </p>
     * 
     * @return The type of notifications that will be published to the specified Amazon SNS topic.
     * @see NotificationType
     */
    public String notificationTypeAsString() {
        return notificationType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a null
     * value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a
     *         null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
     */
    public String snsTopic() {
        return snsTopic;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(notificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopic());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SetIdentityNotificationTopicRequest)) {
            return false;
        }
        SetIdentityNotificationTopicRequest other = (SetIdentityNotificationTopicRequest) obj;
        return Objects.equals(identity(), other.identity())
                && Objects.equals(notificationTypeAsString(), other.notificationTypeAsString())
                && Objects.equals(snsTopic(), other.snsTopic());
    }

    @Override
    public String toString() {
        return ToString.builder("SetIdentityNotificationTopicRequest").add("Identity", identity())
                .add("NotificationType", notificationTypeAsString()).add("SnsTopic", snsTopic()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "NotificationType":
            return Optional.ofNullable(clazz.cast(notificationTypeAsString()));
        case "SnsTopic":
            return Optional.ofNullable(clazz.cast(snsTopic()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends SesRequest.Builder, CopyableBuilder<Builder, SetIdentityNotificationTopicRequest> {
        /**
         * <p>
         * The identity for which the Amazon SNS topic will be set. You can specify an identity by using its name or by
         * using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>, <code>example.com</code>,
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * </p>
         * 
         * @param identity
         *        The identity for which the Amazon SNS topic will be set. You can specify an identity by using its name
         *        or by using its Amazon Resource Name (ARN). Examples: <code>user@example.com</code>,
         *        <code>example.com</code>, <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(String identity);

        /**
         * <p>
         * The type of notifications that will be published to the specified Amazon SNS topic.
         * </p>
         * 
         * @param notificationType
         *        The type of notifications that will be published to the specified Amazon SNS topic.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(String notificationType);

        /**
         * <p>
         * The type of notifications that will be published to the specified Amazon SNS topic.
         * </p>
         * 
         * @param notificationType
         *        The type of notifications that will be published to the specified Amazon SNS topic.
         * @see NotificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotificationType
         */
        Builder notificationType(NotificationType notificationType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request or a
         * null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
         * </p>
         * 
         * @param snsTopic
         *        The Amazon Resource Name (ARN) of the Amazon SNS topic. If the parameter is omitted from the request
         *        or a null value is passed, <code>SnsTopic</code> is cleared and publishing is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopic(String snsTopic);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private String identity;

        private String notificationType;

        private String snsTopic;

        private BuilderImpl() {
        }

        private BuilderImpl(SetIdentityNotificationTopicRequest model) {
            super(model);
            identity(model.identity);
            notificationType(model.notificationType);
            snsTopic(model.snsTopic);
        }

        public final String getIdentity() {
            return identity;
        }

        @Override
        public final Builder identity(String identity) {
            this.identity = identity;
            return this;
        }

        public final void setIdentity(String identity) {
            this.identity = identity;
        }

        public final String getNotificationType() {
            return notificationType;
        }

        @Override
        public final Builder notificationType(String notificationType) {
            this.notificationType = notificationType;
            return this;
        }

        @Override
        public final Builder notificationType(NotificationType notificationType) {
            this.notificationType(notificationType.toString());
            return this;
        }

        public final void setNotificationType(String notificationType) {
            this.notificationType = notificationType;
        }

        public final String getSnsTopic() {
            return snsTopic;
        }

        @Override
        public final Builder snsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
            return this;
        }

        public final void setSnsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SetIdentityNotificationTopicRequest build() {
            return new SetIdentityNotificationTopicRequest(this);
        }
    }
}
