/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the event destination to which the specified email sending events are published.
 * </p>
 * <note>
 * <p>
 * When you create or update an event destination, you must provide one, and only one, destination. The destination can
 * be either Amazon CloudWatch or Amazon Kinesis Firehose.
 * </p>
 * </note>
 * <p>
 * Event destinations are associated with configuration sets, which enable you to publish email sending events to Amazon
 * CloudWatch or Amazon Kinesis Firehose. For information about using configuration sets, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/monitor-sending-activity.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class EventDestination implements ToCopyableBuilder<EventDestination.Builder, EventDestination> {
    private final String name;

    private final Boolean enabled;

    private final List<String> matchingEventTypes;

    private final KinesisFirehoseDestination kinesisFirehoseDestination;

    private final CloudWatchDestination cloudWatchDestination;

    private EventDestination(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.matchingEventTypes = builder.matchingEventTypes;
        this.kinesisFirehoseDestination = builder.kinesisFirehoseDestination;
        this.cloudWatchDestination = builder.cloudWatchDestination;
    }

    /**
     * <p>
     * The name of the event destination. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 64 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the event destination. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 64 characters.
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     * configuration set. Set to <code>true</code> to enable publishing to this destination; set to <code>false</code>
     * to prevent publishing to this destination. The default value is <code>false</code>.
     * </p>
     * 
     * @return Sets whether Amazon SES publishes events to this destination when you send an email with the associated
     *         configuration set. Set to <code>true</code> to enable publishing to this destination; set to
     *         <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The type of email sending events to publish to the event destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The type of email sending events to publish to the event destination.
     * @see EventType
     */
    public List<String> matchingEventTypes() {
        return matchingEventTypes;
    }

    /**
     * <p>
     * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis Firehose
     * event destination.
     * </p>
     * 
     * @return An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
     *         Firehose event destination.
     */
    public KinesisFirehoseDestination kinesisFirehoseDestination() {
        return kinesisFirehoseDestination;
    }

    /**
     * <p>
     * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
     * CloudWatch event destination.
     * </p>
     * 
     * @return An object that contains the names, default values, and sources of the dimensions associated with an
     *         Amazon CloudWatch event destination.
     */
    public CloudWatchDestination cloudWatchDestination() {
        return cloudWatchDestination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((enabled() == null) ? 0 : enabled().hashCode());
        hashCode = 31 * hashCode + ((matchingEventTypes() == null) ? 0 : matchingEventTypes().hashCode());
        hashCode = 31 * hashCode + ((kinesisFirehoseDestination() == null) ? 0 : kinesisFirehoseDestination().hashCode());
        hashCode = 31 * hashCode + ((cloudWatchDestination() == null) ? 0 : cloudWatchDestination().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventDestination)) {
            return false;
        }
        EventDestination other = (EventDestination) obj;
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.enabled() == null ^ this.enabled() == null) {
            return false;
        }
        if (other.enabled() != null && !other.enabled().equals(this.enabled())) {
            return false;
        }
        if (other.matchingEventTypes() == null ^ this.matchingEventTypes() == null) {
            return false;
        }
        if (other.matchingEventTypes() != null && !other.matchingEventTypes().equals(this.matchingEventTypes())) {
            return false;
        }
        if (other.kinesisFirehoseDestination() == null ^ this.kinesisFirehoseDestination() == null) {
            return false;
        }
        if (other.kinesisFirehoseDestination() != null
                && !other.kinesisFirehoseDestination().equals(this.kinesisFirehoseDestination())) {
            return false;
        }
        if (other.cloudWatchDestination() == null ^ this.cloudWatchDestination() == null) {
            return false;
        }
        if (other.cloudWatchDestination() != null && !other.cloudWatchDestination().equals(this.cloudWatchDestination())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (enabled() != null) {
            sb.append("Enabled: ").append(enabled()).append(",");
        }
        if (matchingEventTypes() != null) {
            sb.append("MatchingEventTypes: ").append(matchingEventTypes()).append(",");
        }
        if (kinesisFirehoseDestination() != null) {
            sb.append("KinesisFirehoseDestination: ").append(kinesisFirehoseDestination()).append(",");
        }
        if (cloudWatchDestination() != null) {
            sb.append("CloudWatchDestination: ").append(cloudWatchDestination()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, EventDestination> {
        /**
         * <p>
         * The name of the event destination. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 64 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the event destination. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 64 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Sets whether Amazon SES publishes events to this destination when you send an email with the associated
         * configuration set. Set to <code>true</code> to enable publishing to this destination; set to
         * <code>false</code> to prevent publishing to this destination. The default value is <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        Sets whether Amazon SES publishes events to this destination when you send an email with the
         *        associated configuration set. Set to <code>true</code> to enable publishing to this destination; set
         *        to <code>false</code> to prevent publishing to this destination. The default value is
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @see EventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventType
         */
        Builder matchingEventTypes(Collection<String> matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @see EventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventType
         */
        Builder matchingEventTypes(String... matchingEventTypes);

        /**
         * <p>
         * The type of email sending events to publish to the event destination.
         * </p>
         * 
         * @param matchingEventTypes
         *        The type of email sending events to publish to the event destination.
         * @see EventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventType
         */
        Builder matchingEventTypes(EventType... matchingEventTypes);

        /**
         * <p>
         * An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         * Firehose event destination.
         * </p>
         * 
         * @param kinesisFirehoseDestination
         *        An object that contains the delivery stream ARN and the IAM role ARN associated with an Amazon Kinesis
         *        Firehose event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination);

        /**
         * <p>
         * An object that contains the names, default values, and sources of the dimensions associated with an Amazon
         * CloudWatch event destination.
         * </p>
         * 
         * @param cloudWatchDestination
         *        An object that contains the names, default values, and sources of the dimensions associated with an
         *        Amazon CloudWatch event destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination);
    }

    private static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private List<String> matchingEventTypes;

        private KinesisFirehoseDestination kinesisFirehoseDestination;

        private CloudWatchDestination cloudWatchDestination;

        private BuilderImpl() {
        }

        private BuilderImpl(EventDestination model) {
            setName(model.name);
            setEnabled(model.enabled);
            setMatchingEventTypes(model.matchingEventTypes);
            setKinesisFirehoseDestination(model.kinesisFirehoseDestination);
            setCloudWatchDestination(model.cloudWatchDestination);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Collection<String> getMatchingEventTypes() {
            return matchingEventTypes;
        }

        @Override
        public final Builder matchingEventTypes(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypes(String... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchingEventTypes(EventType... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes).stream().map(Object::toString).collect(Collectors.toList()));
            return this;
        }

        public final void setMatchingEventTypes(Collection<String> matchingEventTypes) {
            this.matchingEventTypes = EventTypesCopier.copy(matchingEventTypes);
        }

        @SafeVarargs
        public final void setMatchingEventTypes(String... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes));
        }

        @SafeVarargs
        public final void setMatchingEventTypes(EventType... matchingEventTypes) {
            matchingEventTypes(Arrays.asList(matchingEventTypes).stream().map(Object::toString).collect(Collectors.toList()));
        }

        public final KinesisFirehoseDestination getKinesisFirehoseDestination() {
            return kinesisFirehoseDestination;
        }

        @Override
        public final Builder kinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination;
            return this;
        }

        public final void setKinesisFirehoseDestination(KinesisFirehoseDestination kinesisFirehoseDestination) {
            this.kinesisFirehoseDestination = kinesisFirehoseDestination;
        }

        public final CloudWatchDestination getCloudWatchDestination() {
            return cloudWatchDestination;
        }

        @Override
        public final Builder cloudWatchDestination(CloudWatchDestination cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination;
            return this;
        }

        public final void setCloudWatchDestination(CloudWatchDestination cloudWatchDestination) {
            this.cloudWatchDestination = cloudWatchDestination;
        }

        @Override
        public EventDestination build() {
            return new EventDestination(this);
        }
    }
}
