/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Receipt rules enable you to specify which actions Amazon SES should take when it receives mail on behalf of one or
 * more email addresses or domains that you own.
 * </p>
 * <p>
 * Each receipt rule defines a set of email addresses or domains to which it applies. If the email addresses or domains
 * match at least one recipient address of the message, Amazon SES executes all of the receipt rule's actions on the
 * message.
 * </p>
 * <p>
 * For information about setting up receipt rules, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-receipt-rules.html">Amazon SES Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ReceiptRule implements ToCopyableBuilder<ReceiptRule.Builder, ReceiptRule> {
    private final String name;

    private final Boolean enabled;

    private final String tlsPolicy;

    private final List<String> recipients;

    private final List<ReceiptAction> actions;

    private final Boolean scanEnabled;

    private ReceiptRule(BuilderImpl builder) {
        this.name = builder.name;
        this.enabled = builder.enabled;
        this.tlsPolicy = builder.tlsPolicy;
        this.recipients = builder.recipients;
        this.actions = builder.actions;
        this.scanEnabled = builder.scanEnabled;
    }

    /**
     * <p>
     * The name of the receipt rule. The name must:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-).
     * </p>
     * </li>
     * <li>
     * <p>
     * Start and end with a letter or number.
     * </p>
     * </li>
     * <li>
     * <p>
     * Contain less than 64 characters.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the receipt rule. The name must:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start and end with a letter or number.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Contain less than 64 characters.
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
     * </p>
     * 
     * @return If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
     * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce emails
     * that are not received over TLS. The default is <code>Optional</code>.
     * </p>
     * 
     * @return Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted
     *         with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will
     *         bounce emails that are not received over TLS. The default is <code>Optional</code>.
     * @see TlsPolicy
     */
    public String tlsPolicy() {
        return tlsPolicy;
    }

    /**
     * <p>
     * The recipient domains and email addresses to which the receipt rule applies. If this field is not specified, this
     * rule will match all recipients under all verified domains.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The recipient domains and email addresses to which the receipt rule applies. If this field is not
     *         specified, this rule will match all recipients under all verified domains.
     */
    public List<String> recipients() {
        return recipients;
    }

    /**
     * <p>
     * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
     * domains specified in the receipt rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An ordered list of actions to perform on messages that match at least one of the recipient email
     *         addresses or domains specified in the receipt rule.
     */
    public List<ReceiptAction> actions() {
        return actions;
    }

    /**
     * <p>
     * If <code>true</code>, then messages to which this receipt rule applies are scanned for spam and viruses. The
     * default value is <code>false</code>.
     * </p>
     * 
     * @return If <code>true</code>, then messages to which this receipt rule applies are scanned for spam and viruses.
     *         The default value is <code>false</code>.
     */
    public Boolean scanEnabled() {
        return scanEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((name() == null) ? 0 : name().hashCode());
        hashCode = 31 * hashCode + ((enabled() == null) ? 0 : enabled().hashCode());
        hashCode = 31 * hashCode + ((tlsPolicy() == null) ? 0 : tlsPolicy().hashCode());
        hashCode = 31 * hashCode + ((recipients() == null) ? 0 : recipients().hashCode());
        hashCode = 31 * hashCode + ((actions() == null) ? 0 : actions().hashCode());
        hashCode = 31 * hashCode + ((scanEnabled() == null) ? 0 : scanEnabled().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReceiptRule)) {
            return false;
        }
        ReceiptRule other = (ReceiptRule) obj;
        if (other.name() == null ^ this.name() == null) {
            return false;
        }
        if (other.name() != null && !other.name().equals(this.name())) {
            return false;
        }
        if (other.enabled() == null ^ this.enabled() == null) {
            return false;
        }
        if (other.enabled() != null && !other.enabled().equals(this.enabled())) {
            return false;
        }
        if (other.tlsPolicy() == null ^ this.tlsPolicy() == null) {
            return false;
        }
        if (other.tlsPolicy() != null && !other.tlsPolicy().equals(this.tlsPolicy())) {
            return false;
        }
        if (other.recipients() == null ^ this.recipients() == null) {
            return false;
        }
        if (other.recipients() != null && !other.recipients().equals(this.recipients())) {
            return false;
        }
        if (other.actions() == null ^ this.actions() == null) {
            return false;
        }
        if (other.actions() != null && !other.actions().equals(this.actions())) {
            return false;
        }
        if (other.scanEnabled() == null ^ this.scanEnabled() == null) {
            return false;
        }
        if (other.scanEnabled() != null && !other.scanEnabled().equals(this.scanEnabled())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (enabled() != null) {
            sb.append("Enabled: ").append(enabled()).append(",");
        }
        if (tlsPolicy() != null) {
            sb.append("TlsPolicy: ").append(tlsPolicy()).append(",");
        }
        if (recipients() != null) {
            sb.append("Recipients: ").append(recipients()).append(",");
        }
        if (actions() != null) {
            sb.append("Actions: ").append(actions()).append(",");
        }
        if (scanEnabled() != null) {
            sb.append("ScanEnabled: ").append(scanEnabled()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    public interface Builder extends CopyableBuilder<Builder, ReceiptRule> {
        /**
         * <p>
         * The name of the receipt rule. The name must:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-).
         * </p>
         * </li>
         * <li>
         * <p>
         * Start and end with a letter or number.
         * </p>
         * </li>
         * <li>
         * <p>
         * Contain less than 64 characters.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the receipt rule. The name must:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start and end with a letter or number.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Contain less than 64 characters.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        If <code>true</code>, the receipt rule is active. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
         * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce
         * emails that are not received over TLS. The default is <code>Optional</code>.
         * </p>
         * 
         * @param tlsPolicy
         *        Specifies whether Amazon SES should require that incoming email is delivered over a connection
         *        encrypted with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>,
         *        Amazon SES will bounce emails that are not received over TLS. The default is <code>Optional</code>.
         * @see TlsPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsPolicy
         */
        Builder tlsPolicy(String tlsPolicy);

        /**
         * <p>
         * Specifies whether Amazon SES should require that incoming email is delivered over a connection encrypted with
         * Transport Layer Security (TLS). If this parameter is set to <code>Require</code>, Amazon SES will bounce
         * emails that are not received over TLS. The default is <code>Optional</code>.
         * </p>
         * 
         * @param tlsPolicy
         *        Specifies whether Amazon SES should require that incoming email is delivered over a connection
         *        encrypted with Transport Layer Security (TLS). If this parameter is set to <code>Require</code>,
         *        Amazon SES will bounce emails that are not received over TLS. The default is <code>Optional</code>.
         * @see TlsPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TlsPolicy
         */
        Builder tlsPolicy(TlsPolicy tlsPolicy);

        /**
         * <p>
         * The recipient domains and email addresses to which the receipt rule applies. If this field is not specified,
         * this rule will match all recipients under all verified domains.
         * </p>
         * 
         * @param recipients
         *        The recipient domains and email addresses to which the receipt rule applies. If this field is not
         *        specified, this rule will match all recipients under all verified domains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipients(Collection<String> recipients);

        /**
         * <p>
         * The recipient domains and email addresses to which the receipt rule applies. If this field is not specified,
         * this rule will match all recipients under all verified domains.
         * </p>
         * 
         * @param recipients
         *        The recipient domains and email addresses to which the receipt rule applies. If this field is not
         *        specified, this rule will match all recipients under all verified domains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipients(String... recipients);

        /**
         * <p>
         * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
         * domains specified in the receipt rule.
         * </p>
         * 
         * @param actions
         *        An ordered list of actions to perform on messages that match at least one of the recipient email
         *        addresses or domains specified in the receipt rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ReceiptAction> actions);

        /**
         * <p>
         * An ordered list of actions to perform on messages that match at least one of the recipient email addresses or
         * domains specified in the receipt rule.
         * </p>
         * 
         * @param actions
         *        An ordered list of actions to perform on messages that match at least one of the recipient email
         *        addresses or domains specified in the receipt rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ReceiptAction... actions);

        /**
         * <p>
         * If <code>true</code>, then messages to which this receipt rule applies are scanned for spam and viruses. The
         * default value is <code>false</code>.
         * </p>
         * 
         * @param scanEnabled
         *        If <code>true</code>, then messages to which this receipt rule applies are scanned for spam and
         *        viruses. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanEnabled(Boolean scanEnabled);
    }

    private static final class BuilderImpl implements Builder {
        private String name;

        private Boolean enabled;

        private String tlsPolicy;

        private List<String> recipients;

        private List<ReceiptAction> actions;

        private Boolean scanEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(ReceiptRule model) {
            setName(model.name);
            setEnabled(model.enabled);
            setTlsPolicy(model.tlsPolicy);
            setRecipients(model.recipients);
            setActions(model.actions);
            setScanEnabled(model.scanEnabled);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final String getTlsPolicy() {
            return tlsPolicy;
        }

        @Override
        public final Builder tlsPolicy(String tlsPolicy) {
            this.tlsPolicy = tlsPolicy;
            return this;
        }

        @Override
        public final Builder tlsPolicy(TlsPolicy tlsPolicy) {
            this.tlsPolicy(tlsPolicy.toString());
            return this;
        }

        public final void setTlsPolicy(String tlsPolicy) {
            this.tlsPolicy = tlsPolicy;
        }

        public final void setTlsPolicy(TlsPolicy tlsPolicy) {
            this.tlsPolicy(tlsPolicy.toString());
        }

        public final Collection<String> getRecipients() {
            return recipients;
        }

        @Override
        public final Builder recipients(Collection<String> recipients) {
            this.recipients = RecipientsListCopier.copy(recipients);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recipients(String... recipients) {
            recipients(Arrays.asList(recipients));
            return this;
        }

        public final void setRecipients(Collection<String> recipients) {
            this.recipients = RecipientsListCopier.copy(recipients);
        }

        @SafeVarargs
        public final void setRecipients(String... recipients) {
            recipients(Arrays.asList(recipients));
        }

        public final Collection<ReceiptAction> getActions() {
            return actions;
        }

        @Override
        public final Builder actions(Collection<ReceiptAction> actions) {
            this.actions = ReceiptActionsListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ReceiptAction... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        public final void setActions(Collection<ReceiptAction> actions) {
            this.actions = ReceiptActionsListCopier.copy(actions);
        }

        @SafeVarargs
        public final void setActions(ReceiptAction... actions) {
            actions(Arrays.asList(actions));
        }

        public final Boolean getScanEnabled() {
            return scanEnabled;
        }

        @Override
        public final Builder scanEnabled(Boolean scanEnabled) {
            this.scanEnabled = scanEnabled;
            return this;
        }

        public final void setScanEnabled(Boolean scanEnabled) {
            this.scanEnabled = scanEnabled;
        }

        @Override
        public ReceiptRule build() {
            return new ReceiptRule(this);
        }
    }
}
