/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AmazonWebServiceRequest;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to send a single formatted email using Amazon SES. For more information, see the <a
 * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/send-email-formatted.html">Amazon SES Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class SendEmailRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<SendEmailRequest.Builder, SendEmailRequest> {
    private final String source;

    private final Destination destination;

    private final Message message;

    private final List<String> replyToAddresses;

    private final String returnPath;

    private final String sourceArn;

    private final String returnPathArn;

    private final List<MessageTag> tags;

    private final String configurationSetName;

    private SendEmailRequest(BuilderImpl builder) {
        this.source = builder.source;
        this.destination = builder.destination;
        this.message = builder.message;
        this.replyToAddresses = builder.replyToAddresses;
        this.returnPath = builder.returnPath;
        this.sourceArn = builder.sourceArn;
        this.returnPathArn = builder.returnPathArn;
        this.tags = builder.tags;
        this.configurationSetName = builder.configurationSetName;
    }

    /**
     * <p>
     * The email address that is sending the email. This email address must be either individually verified with Amazon
     * SES, or from a domain that has been verified with Amazon SES. For information about verifying identities, see the
     * <a href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     * Developer Guide</a>.
     * </p>
     * <p>
     * If you are sending on behalf of another user and have been permitted to do so by a sending authorization policy,
     * then you must also specify the <code>SourceArn</code> parameter. For more information about sending
     * authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * <p>
     * In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then you must
     * use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the following
     * form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     * </p>
     * 
     * @return The email address that is sending the email. This email address must be either individually verified with
     *         Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
     *         identities, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
     *         Developer Guide</a>.</p>
     *         <p>
     *         If you are sending on behalf of another user and have been permitted to do so by a sending authorization
     *         policy, then you must also specify the <code>SourceArn</code> parameter. For more information about
     *         sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     *         </p>
     *         <p>
     *         In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then
     *         you must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax
     *         uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
     *         href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
     */
    public String source() {
        return source;
    }

    /**
     * <p>
     * The destination for this email, composed of To:, CC:, and BCC: fields.
     * </p>
     * 
     * @return The destination for this email, composed of To:, CC:, and BCC: fields.
     */
    public Destination destination() {
        return destination;
    }

    /**
     * <p>
     * The message to be sent.
     * </p>
     * 
     * @return The message to be sent.
     */
    public Message message() {
        return message;
    }

    /**
     * <p>
     * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to address
     * will receive the reply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
     *         address will receive the reply.
     */
    public List<String> replyToAddresses() {
        return replyToAddresses;
    }

    /**
     * <p>
     * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If the
     * message cannot be delivered to the recipient, then an error message will be returned from the recipient's ISP;
     * this message will then be forwarded to the email address specified by the <code>ReturnPath</code> parameter. The
     * <code>ReturnPath</code> parameter is never overwritten. This email address must be either individually verified
     * with Amazon SES, or from a domain that has been verified with Amazon SES.
     * </p>
     * 
     * @return The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled.
     *         If the message cannot be delivered to the recipient, then an error message will be returned from the
     *         recipient's ISP; this message will then be forwarded to the email address specified by the
     *         <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This email
     *         address must be either individually verified with Amazon SES, or from a domain that has been verified
     *         with Amazon SES.
     */
    public String returnPath() {
        return returnPath;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to send for the email address specified in the <code>Source</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
     * <code>user@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to send for the email address specified in the
     *         <code>Source</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to send from <code>user@example.com</code>, then you would specify the
     *         <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
     *         the <code>Source</code> to be <code>user@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with the
     * sending authorization policy that permits you to use the email address specified in the <code>ReturnPath</code>
     * parameter.
     * </p>
     * <p>
     * For example, if the owner of <code>example.com</code> (which has ARN
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes you
     * to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
     * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
     * <code>feedback@example.com</code>.
     * </p>
     * <p>
     * For more information about sending authorization, see the <a
     * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
     * Guide</a>.
     * </p>
     * 
     * @return This parameter is used only for sending authorization. It is the ARN of the identity that is associated
     *         with the sending authorization policy that permits you to use the email address specified in the
     *         <code>ReturnPath</code> parameter.</p>
     *         <p>
     *         For example, if the owner of <code>example.com</code> (which has ARN
     *         <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
     *         authorizes you to use <code>feedback@example.com</code>, then you would specify the
     *         <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
     *         and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
     *         </p>
     *         <p>
     *         For more information about sending authorization, see the <a
     *         href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
     *         Developer Guide</a>.
     */
    public String returnPathArn() {
        return returnPathArn;
    }

    /**
     * <p>
     * A list of tags, in the form of name/value pairs, to apply to an email that you send using <code>SendEmail</code>.
     * Tags correspond to characteristics of the email that you define, so that you can publish email sending events.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of tags, in the form of name/value pairs, to apply to an email that you send using
     *         <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
     *         publish email sending events.
     */
    public List<MessageTag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the configuration set to use when you send an email using <code>SendEmail</code>.
     * </p>
     * 
     * @return The name of the configuration set to use when you send an email using <code>SendEmail</code>.
     */
    public String configurationSetName() {
        return configurationSetName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(replyToAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(returnPath());
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(returnPathArn());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSetName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SendEmailRequest)) {
            return false;
        }
        SendEmailRequest other = (SendEmailRequest) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(destination(), other.destination())
                && Objects.equals(message(), other.message()) && Objects.equals(replyToAddresses(), other.replyToAddresses())
                && Objects.equals(returnPath(), other.returnPath()) && Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(returnPathArn(), other.returnPathArn()) && Objects.equals(tags(), other.tags())
                && Objects.equals(configurationSetName(), other.configurationSetName());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (source() != null) {
            sb.append("Source: ").append(source()).append(",");
        }
        if (destination() != null) {
            sb.append("Destination: ").append(destination()).append(",");
        }
        if (message() != null) {
            sb.append("Message: ").append(message()).append(",");
        }
        if (replyToAddresses() != null) {
            sb.append("ReplyToAddresses: ").append(replyToAddresses()).append(",");
        }
        if (returnPath() != null) {
            sb.append("ReturnPath: ").append(returnPath()).append(",");
        }
        if (sourceArn() != null) {
            sb.append("SourceArn: ").append(sourceArn()).append(",");
        }
        if (returnPathArn() != null) {
            sb.append("ReturnPathArn: ").append(returnPathArn()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (configurationSetName() != null) {
            sb.append("ConfigurationSetName: ").append(configurationSetName()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.of(clazz.cast(source()));
        case "Destination":
            return Optional.of(clazz.cast(destination()));
        case "Message":
            return Optional.of(clazz.cast(message()));
        case "ReplyToAddresses":
            return Optional.of(clazz.cast(replyToAddresses()));
        case "ReturnPath":
            return Optional.of(clazz.cast(returnPath()));
        case "SourceArn":
            return Optional.of(clazz.cast(sourceArn()));
        case "ReturnPathArn":
            return Optional.of(clazz.cast(returnPathArn()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "ConfigurationSetName":
            return Optional.of(clazz.cast(configurationSetName()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, SendEmailRequest> {
        /**
         * <p>
         * The email address that is sending the email. This email address must be either individually verified with
         * Amazon SES, or from a domain that has been verified with Amazon SES. For information about verifying
         * identities, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
         * Developer Guide</a>.
         * </p>
         * <p>
         * If you are sending on behalf of another user and have been permitted to do so by a sending authorization
         * policy, then you must also specify the <code>SourceArn</code> parameter. For more information about sending
         * authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * <p>
         * In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters, then you
         * must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word syntax uses the
         * following form: <code>=?charset?encoding?encoded-text?=</code>. For more information, see <a
         * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * </p>
         * 
         * @param source
         *        The email address that is sending the email. This email address must be either individually verified
         *        with Amazon SES, or from a domain that has been verified with Amazon SES. For information about
         *        verifying identities, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon
         *        SES Developer Guide</a>.</p>
         *        <p>
         *        If you are sending on behalf of another user and have been permitted to do so by a sending
         *        authorization policy, then you must also specify the <code>SourceArn</code> parameter. For more
         *        information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         *        </p>
         *        <p>
         *        In all cases, the email address must be 7-bit ASCII. If the text must contain any other characters,
         *        then you must use MIME encoded-word syntax (RFC 2047) instead of a literal string. MIME encoded-word
         *        syntax uses the following form: <code>=?charset?encoding?encoded-text?=</code>. For more information,
         *        see <a href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields.
         * </p>
         * 
         * @param destination
         *        The destination for this email, composed of To:, CC:, and BCC: fields.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Destination destination);

        /**
         * <p>
         * The destination for this email, composed of To:, CC:, and BCC: fields.
         * </p>
         * This is a convenience that creates an instance of the {@link Destination.Builder} avoiding the need to create
         * one manually via {@link Destination#builder()}.
         *
         * When the {@link Consumer} completes, {@link Destination.Builder#build()} is called immediately and its result
         * is passed to {@link #destination(Destination)}.
         * 
         * @param destination
         *        a consumer that will call methods on {@link Destination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destination(Destination)
         */
        default Builder destination(Consumer<Destination.Builder> destination) {
            return destination(Destination.builder().apply(destination).build());
        }

        /**
         * <p>
         * The message to be sent.
         * </p>
         * 
         * @param message
         *        The message to be sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(Message message);

        /**
         * <p>
         * The message to be sent.
         * </p>
         * This is a convenience that creates an instance of the {@link Message.Builder} avoiding the need to create one
         * manually via {@link Message#builder()}.
         *
         * When the {@link Consumer} completes, {@link Message.Builder#build()} is called immediately and its result is
         * passed to {@link #message(Message)}.
         * 
         * @param message
         *        a consumer that will call methods on {@link Message.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #message(Message)
         */
        default Builder message(Consumer<Message.Builder> message) {
            return message(Message.builder().apply(message).build());
        }

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(Collection<String> replyToAddresses);

        /**
         * <p>
         * The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         * address will receive the reply.
         * </p>
         * 
         * @param replyToAddresses
         *        The reply-to email address(es) for the message. If the recipient replies to the message, each reply-to
         *        address will receive the reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToAddresses(String... replyToAddresses);

        /**
         * <p>
         * The email address that bounces and complaints will be forwarded to when feedback forwarding is enabled. If
         * the message cannot be delivered to the recipient, then an error message will be returned from the recipient's
         * ISP; this message will then be forwarded to the email address specified by the <code>ReturnPath</code>
         * parameter. The <code>ReturnPath</code> parameter is never overwritten. This email address must be either
         * individually verified with Amazon SES, or from a domain that has been verified with Amazon SES.
         * </p>
         * 
         * @param returnPath
         *        The email address that bounces and complaints will be forwarded to when feedback forwarding is
         *        enabled. If the message cannot be delivered to the recipient, then an error message will be returned
         *        from the recipient's ISP; this message will then be forwarded to the email address specified by the
         *        <code>ReturnPath</code> parameter. The <code>ReturnPath</code> parameter is never overwritten. This
         *        email address must be either individually verified with Amazon SES, or from a domain that has been
         *        verified with Amazon SES.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPath(String returnPath);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to send for the email address specified in the
         * <code>Source</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to send from <code>user@example.com</code>, then you would specify the <code>SourceArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>Source</code> to be
         * <code>user@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param sourceArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to send for the email address
         *        specified in the <code>Source</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to send from <code>user@example.com</code>, then you would specify the
         *        <code>SourceArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and
         *        the <code>Source</code> to be <code>user@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * This parameter is used only for sending authorization. It is the ARN of the identity that is associated with
         * the sending authorization policy that permits you to use the email address specified in the
         * <code>ReturnPath</code> parameter.
         * </p>
         * <p>
         * For example, if the owner of <code>example.com</code> (which has ARN
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that authorizes
         * you to use <code>feedback@example.com</code>, then you would specify the <code>ReturnPathArn</code> to be
         * <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>, and the <code>ReturnPath</code> to be
         * <code>feedback@example.com</code>.
         * </p>
         * <p>
         * For more information about sending authorization, see the <a
         * href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES Developer
         * Guide</a>.
         * </p>
         * 
         * @param returnPathArn
         *        This parameter is used only for sending authorization. It is the ARN of the identity that is
         *        associated with the sending authorization policy that permits you to use the email address specified
         *        in the <code>ReturnPath</code> parameter.</p>
         *        <p>
         *        For example, if the owner of <code>example.com</code> (which has ARN
         *        <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>) attaches a policy to it that
         *        authorizes you to use <code>feedback@example.com</code>, then you would specify the
         *        <code>ReturnPathArn</code> to be <code>arn:aws:ses:us-east-1:123456789012:identity/example.com</code>,
         *        and the <code>ReturnPath</code> to be <code>feedback@example.com</code>.
         *        </p>
         *        <p>
         *        For more information about sending authorization, see the <a
         *        href="http://docs.aws.amazon.com/ses/latest/DeveloperGuide/sending-authorization.html">Amazon SES
         *        Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnPathArn(String returnPathArn);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         *        can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<MessageTag> tags);

        /**
         * <p>
         * A list of tags, in the form of name/value pairs, to apply to an email that you send using
         * <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you can
         * publish email sending events.
         * </p>
         * 
         * @param tags
         *        A list of tags, in the form of name/value pairs, to apply to an email that you send using
         *        <code>SendEmail</code>. Tags correspond to characteristics of the email that you define, so that you
         *        can publish email sending events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(MessageTag... tags);

        /**
         * <p>
         * The name of the configuration set to use when you send an email using <code>SendEmail</code>.
         * </p>
         * 
         * @param configurationSetName
         *        The name of the configuration set to use when you send an email using <code>SendEmail</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSetName(String configurationSetName);
    }

    static final class BuilderImpl implements Builder {
        private String source;

        private Destination destination;

        private Message message;

        private List<String> replyToAddresses;

        private String returnPath;

        private String sourceArn;

        private String returnPathArn;

        private List<MessageTag> tags;

        private String configurationSetName;

        private BuilderImpl() {
        }

        private BuilderImpl(SendEmailRequest model) {
            source(model.source);
            destination(model.destination);
            message(model.message);
            replyToAddresses(model.replyToAddresses);
            returnPath(model.returnPath);
            sourceArn(model.sourceArn);
            returnPathArn(model.returnPathArn);
            tags(model.tags);
            configurationSetName(model.configurationSetName);
        }

        public final String getSource() {
            return source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        public final Destination.Builder getDestination() {
            return destination != null ? destination.toBuilder() : null;
        }

        @Override
        public final Builder destination(Destination destination) {
            this.destination = destination;
            return this;
        }

        public final void setDestination(Destination.BuilderImpl destination) {
            this.destination = destination != null ? destination.build() : null;
        }

        public final Message.Builder getMessage() {
            return message != null ? message.toBuilder() : null;
        }

        @Override
        public final Builder message(Message message) {
            this.message = message;
            return this;
        }

        public final void setMessage(Message.BuilderImpl message) {
            this.message = message != null ? message.build() : null;
        }

        public final Collection<String> getReplyToAddresses() {
            return replyToAddresses;
        }

        @Override
        public final Builder replyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replyToAddresses(String... replyToAddresses) {
            replyToAddresses(Arrays.asList(replyToAddresses));
            return this;
        }

        public final void setReplyToAddresses(Collection<String> replyToAddresses) {
            this.replyToAddresses = AddressListCopier.copy(replyToAddresses);
        }

        public final String getReturnPath() {
            return returnPath;
        }

        @Override
        public final Builder returnPath(String returnPath) {
            this.returnPath = returnPath;
            return this;
        }

        public final void setReturnPath(String returnPath) {
            this.returnPath = returnPath;
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        public final String getReturnPathArn() {
            return returnPathArn;
        }

        @Override
        public final Builder returnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
            return this;
        }

        public final void setReturnPathArn(String returnPathArn) {
            this.returnPathArn = returnPathArn;
        }

        public final Collection<MessageTag.Builder> getTags() {
            return tags != null ? tags.stream().map(MessageTag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<MessageTag> tags) {
            this.tags = MessageTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(MessageTag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<MessageTag.BuilderImpl> tags) {
            this.tags = MessageTagListCopier.copyFromBuilder(tags);
        }

        public final String getConfigurationSetName() {
            return configurationSetName;
        }

        @Override
        public final Builder configurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
            return this;
        }

        public final void setConfigurationSetName(String configurationSetName) {
            this.configurationSetName = configurationSetName;
        }

        @Override
        public SendEmailRequest build() {
            return new SendEmailRequest(this);
        }
    }
}
